<?php
/*
  This code is part of GOsa (https://gosa.gonicus.de)
  Copyright (C) 2003  Cajus Pollmeier

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/* Configuration file location */
define ("CONFIG_DIR", "/etc/gosa");

/* Include required files */
require_once ("ldap.inc");
require_once ("config.inc");
require_once ("userinfo.inc");
require_once ("plugin.inc");
require_once ("pluglist.inc");
require_once ("tabs.inc");
require_once ("debuglib.inc");


/* Define constants for debugging */
define ("DEBUG_TRACE",   1);
define ("DEBUG_LDAP",    2);
define ("DEBUG_MYSQL",   4);
define ("DEBUG_SHELL",   8);
define ("DEBUG_POST",   16);
define ("DEBUG_SESSION",32);
define ("DEBUG_CONFIG", 64);


/* Rewrite german 'umlauts' to get better results */
$REWRITE= array( "" => "ae",
                 "" => "oe",
                 "" => "ue",
                 "" => "ss" );


/* Function to include all class_ files starting at a
   given directory base */
function get_dir_list($folder= ".")
{
	$currdir=getcwd();
	if ($folder){
		@chdir("$folder");
	}

	$dh = opendir(".");
	while(false !== ($file = readdir($dh))){

    		if (preg_match("/^class_/", $file)) {
			require_once($file);
		}

	        /* insert all directories in an array */
	        if(is_dir($file) && $file!="." && $file!=".." && $file!="CVS"){
			get_dir_list($file);
		}
	}

	closedir($dh);
	chdir($currdir);
}


/* seed with microseconds */
function make_seed() {
	list($usec, $sec) = explode(' ', microtime());
	return (float) $sec + ((float) $usec * 100000);
}

/* Debug level action */
function DEBUG($level, $line, $function, $file, $data, $info="")
{
	if ($_SESSION['DEBUGLEVEL'] & $level){
		$output= "DEBUG[$level] ";
		if ($function != ""){
			$output.= "($file:$function():$line) - $info: ";
		} else {
			$output.= "($file:$line) - $info: ";
		}
		echo $output;
		if (is_array($data)){
			print_a($data);
		} else {
			echo "'$data'";
		}
		echo "<br>";
	}
}


/* Simple function to get browser language and convert it to
   xx_XY needed by locales. Ignores sublanguages and weights. */
function get_browser_language()
{
	/* Get list of languages */
	if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])){
		$lang= preg_replace("/\s+/", "", $_SERVER['HTTP_ACCEPT_LANGUAGE']);
		$languages= split (',', $lang);
	} else {
		$languages= array("C");
	}

	/* Walk through languages and get first supported */
	foreach ($languages as $val){

		/* Strip off weight */
		$lang= preg_replace("/;q=.*$/i", "", $val);

		/* Simplify sub language handling */
		$lang= preg_replace("/-.*$/", "", $lang);

		return (strtolower($lang)."_".strtoupper($lang));
	}
}


/* Rewrite ui object to another dn */
function change_ui_dn($dn, $newdn)
{
	$ui= $_SESSION['ui'];
	if ($ui->dn == $dn){
		$ui->dn= $newdn;
		$_SESSION['ui']= $ui;
	}
}


/* Return theme path for specified file */
function get_template_path($filename= '', $plugin= FALSE, $path= "")
{
	global $config, $BASE_DIR;

	if (!isset($config->data['MAIN']['THEME'])){
		$theme= 'default';
	} else {
		$theme= $config->data['MAIN']['THEME'];
	}

	/* Return path for empty filename */
	if ($filename == ''){
		return ("themes/$theme/");
	}
	
	/* Return plugin dir or root directory? */
	if ($plugin){
		if ($path == ""){
			$nf= preg_replace("!^".$BASE_DIR."/!", "", $_SESSION['plugin_dir']);
		} else {
			$nf= preg_replace("!^".$BASE_DIR."/!", "", $path);
		}
		if (file_exists("$BASE_DIR/ihtml/themes/$theme/$nf")){
			return ("$BASE_DIR/ihtml/themes/$theme/$nf");
		}
		if (file_exists("$BASE_DIR/ihtml/themes/default/$nf")){
			return ("$BASE_DIR/ihtml/themes/default/$nf");
		}
		if ($path == ""){
			return ($_SESSION['plugin_dir']."/$filename");
		} else {
			return ($path."/$filename");
		}
	} else {
		if (file_exists("themes/$theme/$filename")){
			return ("themes/$theme/$filename");
		}
		if (file_exists("$BASE_DIR/ihtml/themes/$theme/$filename")){
			return ("$BASE_DIR/ihtml/themes/$theme/$filename");
		}
		if (file_exists("themes/default/$filename")){
			return ("themes/default/$filename");
		}
		if (file_exists("$BASE_DIR/ihtml/themes/default/$filename")){
			return ("$BASE_DIR/ihtml/themes/default/$filename");
		}
		return ($filename);
	}
}


function array_remove_entry($entry, $stack)
{
	$temp= array();
	foreach ($stack as $key => $value){
		if ($value != $entry){
			$temp[$key]= $value;
		}
	}
	return($temp);
}

## array_strip ($toberemoved, $data)
##
function array_strip($toberemoved, $data)
{
    $tmp= array();
    
    foreach ($data as $needle){
      if (!in_array($needle, $toberemoved)){
        $tmp[]= $needle;
      }
    }

    return ($tmp);
}


## log (message)
##
function gosa_log ($message)
{
  global $ui;
 
  $username= "";

  if (isset($ui)){
    if ($ui->username != ""){
      $username= "[$ui->username]";
    }
  }
  
  syslog(LOG_WARNING,"GOsa$username: $message");
}


## ldap_init (server, base [, binddn, pass])
##
function ldap_init ($server, $base, $binddn='', $pass='')
{
  $ldap = new LDAP ($binddn, $pass, $server);

  if (!(strpos($ldap->error, "Could not bind to") === false)) {
    return (0);
  }

  if ($ldap->error != "Success"){
	print_red(sprintf(_("Error when connecting the LDAP. Server said '%s'."), $ldap->get_error()));
	exit;
  }

  $ldap->cd ($base);

  return $ldap;
}


## ldap_login_user (username, password)
##
function ldap_login_user ($username, $password)
{
  /* Include global configuration */
  global $config;

  /* look through the entire ldap */
  $tldap = new LDAP($config->current['ADMIN'],$config->current['PASSWORD'],$config->current['SERVER'], "true");
  if ($tldap->error != "Success"){
	print_red(sprintf(_("User login failed. LDAP server said '%s'."), $tldap->get_error()));
	exit;
  }
  $tldap->cd($config->current['BASE']);
  $tldap->search("(&(uid=$username)(objectClass=gosaAccount))");
 
  /* get results, only a count of 1 is valid */
  switch ($tldap->count()){
  
	/* user not found */
	case 0:	return (0);
		break;

	/* valid uniq user */
	case 1: 
		break;

	/* found more than one matching id */
	default:
		print_red(_("Username / UID is not unique. Please check your LDAP database."));
		exit;
  }

  /* got user dn, fill acl's */
  $tldap->fetch();
  $ui= new userinfo($config, $tldap->getDN());
  $ui->username= $username;

  /* password check, bind as user with supplied password  */
  $tldap = new LDAP($ui->dn, $password, $config->current['SERVER'], "true");
  if ($tldap->error != "Success"){
	return (0);
  }
 
  /* Username is set, load subtreeACL's now */
  $ui->loadACL();

  return ($ui);
}


## must(string)
#  like _(), but adds a red * to the end
function must($string)
{
  echo _($string);
  echo "<font class=\"must\">*</font>";
}

## add_lock (object, user)
##
function add_lock ($object, $user)
{
  global $config;

  if ($object == "" || $user == ""){
	print_red(_("Error while adding a lock. Parameters are not set correctly, please check the source!"));
	exit;
  }

  $ldap= $config->get_ldap_link();
  $ldap->cd ($config->current['CONFIG']);
  $ldap->search("(&(objectClass=gosaLockEntry)(gosaUser=$user)(gosaObject=$object))");
  if ($ldap->error != "Success"){
        print_red (_("Can't set locking information in LDAP database. Please check the 'config' entry in gosa.conf!"));
        exit;
  }

  # Lock exists?
  if ($ldap->count() == 0){
    # create lock
    $attrs= array();
    $name= md5($object);
    $ldap->cd("cn=$name,".$config->current['CONFIG']);
    $attrs["objectClass"] = "gosaLockEntry";
    $attrs["gosaUser"] = $user;
    $attrs["gosaObject"] = $object;
    $attrs["cn"] = "$name";
    $ldap->add($attrs);
    if ($ldap->error != "Success"){
	print_red(sprintf(_("Adding a lock failed. LDAP server says '%s'."), $ldap->get_error()));
	exit;
    }

  }

  # success
  return (1);
}


## del_lock (object)
##
function del_lock ($object)
{
  global $config;

  if ($object == ""){
    return (0);
  }

  $ldap= $config->get_ldap_link();
  $ldap->cd ($config->current['CONFIG']);
  $ldap->search ("(&(objectClass=gosaLockEntry)(gosaObject=$object))");
  $attrs= $ldap->fetch();
  if ($ldap->getDN() != "" && $ldap->error == "Success"){
    $ldap->rmdir ($ldap->getDN());
  }
  
  if ($ldap->error != "Success"){
    return (0);
  }

  # Success
  return (1);
}


## user get_lock (object);
##
function get_lock ($object)
{
  global $config;
  
  if ($object == ""){
	print_red(_("Getting the lock from LDAP failed. Parameters are not sed correctly, please check the source!"));
	exit;
  }

  $user= "";
  $ldap= $config->get_ldap_link();
  $ldap->cd ($config->current['CONFIG']);
  
  $ldap->search("(&(objectClass=gosaLockEntry)(gosaObject=$object))");
  if ($ldap->error != "Success"){
        print_red (_("Can't get locking information in LDAP database. Please check the 'config' entry in gosa.conf!"));
        exit;
  }

  # validity check
  if ($ldap->count() > 1){
	print_red(_("Found multiple locks for object to be locked. This should not be possible, please check your LDAP."));
	exit;
  } elseif ($ldap->count() == 1){
    $attrs = $ldap->fetch();
    $user= $attrs['gosaUser'][0];
  }

  return $user;
}

## array get_list (subtreeACL, filter, [base])
##
function get_list($subtreeACL, $filter, $base="")
{
  global $config;

  $ldap= $config->get_ldap_link();
  if ($base == ""){
    $ldap->cd ($config->current['BASE']);
  } else {
    $ldap->cd ($base);
  }
  $ldap->search ($filter);
  $result= array();

  while($attrs = $ldap->fetch()) {
    $dn= preg_replace("/[ ]*,[ ]*/", ",", $ldap->getDN());
    foreach ($subtreeACL as $key => $value){
      if (preg_match("/$key/", $dn)){
        $attrs["dn"]= $dn;
        $result[]= $attrs;
        break;
      }
    }
  }
  return ($result);
}


## print_sort_ranges()
##
function print_sort_ranges($sortranges)
{
	echo get_sort_ranges($sortranges);
}

## get_sort_ranges()
##
function get_sort_ranges($sortranges)
{
  global $plug;

  $res= "";

  if (isset($plug)){
    $plugin= "plug=$plug&amp;";
  } else {
    $plugin= "";
  }
  foreach ($sortranges as $key => $value){
    $res.= "[<a href=\"main.php?".$plugin."sorting=$key\">$key</a>]&nbsp;";
  }

  return $res;
}


## get_permissions ($userinfo)
##
function get_permissions ($dn, $subtreeACL)
{
  global $config;

  $base= $config->current['BASE'];
  $tmp= "d,".$dn;
  $sacl= array();

  # sort subacl's for lenght
  foreach ($subtreeACL as $key => $value){
    $sacl[$key]= strlen($key);
  }
  arsort ($sacl);
  reset ($sacl);

  #echo "Sorted acl's:<br>";
  #print_a ($sacl);
  #echo "<br>";

  while (preg_match('/,/', $tmp)){
    $tmp= ltrim(strstr($tmp, ","), ",");

    # check for acl that applies
    foreach ($sacl as $key => $value){
      if (preg_match("/$tmp$/", $key)){
      	  #echo "Returning: ";
	  #print_a($subtreeACL[$key]);
          return ($subtreeACL[$key]);
      }
    }
  }

  return array("");
}


## get_module_permission($acl, $module)
##
function get_module_permission($acl_array, $module, $dn)
{
  global $ui;

  foreach($acl_array as $acl){

	  /* Check for selfflag */
	  if (preg_match("/^!/", $acl)){
		if ($dn != "" && $dn != $ui->dn){
			continue;
		} else {
			$acl= preg_replace("/^!/", "", $acl);
		}
	  }
	  $acl= preg_replace("/^:/", "", $acl);

	  $tmp= split(",", $acl);
	  foreach ($tmp as $mod){
		if (preg_match("/$module#/", $mod)){
			return (strstr($mod, "#")."#");
		}
		if (preg_match("/[^#]$module$/", $mod)){
			return ("#all#");
		}
		if (preg_match("/^all$/", $mod)){
			return ("#all#");
		}
	  }
  }
  return ("#none#");
}


function get_userinfo()
{
  global $ui;

  return $ui;
}


/* Yam, yam function */
function get_smarty()
{
  global $smarty;

  return $smarty;
}


## convert_department_dn($dn)
##
function convert_department_dn($dn)
{
  $dep= "";
  foreach (split (",", $dn) as $val){
    if (preg_match ("/ou=/", $val)){
      $dep= preg_replace("/ou=([^,]+)/", "\\1", $val)."/$dep";
    }
    if (preg_match ("/l=/", $val)){
      $dep= preg_replace("/l=([^,]+)/", "\\1", $val)."/$dep";
    }
  }
  return rtrim($dep, "/");
}

## get_people_ou()
##
function get_people_ou()
{
  global $config;
  return "ou=".$config->current['PEOPLE'];
}

## get_groups_ou()
##
function get_groups_ou()
{
  global $config;

  return "ou=".$config->current['GROUPS'];
}

##
## get_base_from_people
function get_base_from_people($dn)
{
  return preg_replace('/^[^,]+,[^,]+,/', '', $dn);
}


## get_departments
##
function get_departments()
{
  global $config;

  $result= array();

  $ldap= $config->get_ldap_link();
  $ldap->cd ($config->current['BASE']);
  $ldap->search ("(objectClass=gosaDepartment)");
  $result['/']= $config->current['BASE'];
  while ($attrs= $ldap->fetch()){
    $dn= $ldap->getDN();
    $result[utf8_decode(convert_department_dn($dn))]= $dn;
  }

  return ($result);
}


## chkacl($acl, $name)
##
function chkacl($acl, $name)
{

  # matches?
  if (preg_match("/#$name#/", $acl) || $acl == "#all#"){
    return ("");
  }

  # no restrictions
  return (" disabled ");
}


## is_phone_nr($nr)
##
function is_phone_nr($nr)
{
  if ($nr == ""){
    return (TRUE);
  }
  return preg_match ("/^[0-9 ()+*-]+$/", $nr);
}


## is_url($url)
##
function is_url($url)
{
  if ($url == ""){
    return (TRUE);
  }
  return preg_match ("/^(http|https):\/\/((?:[a-zA-Z0-9_-]+\.?)+):?(\d*)/", $url);
}


## is_dn($dn)
##
function is_dn($dn)
{
  if ($dn == ""){
    return (TRUE);
  }
  return preg_match ("/^[a-z0-9 _-]+$/i", $dn);
}

## is_uid($uid)
##
function is_uid($uid)
{
  global $config;

  if ($uid == ""){
    return (TRUE);
  }
  if (isset($config->current['STRICT']) && preg_match('/^no$/i', $config->current['STRICT'])){
	return preg_match ("/^[a-z0-9 -]+$/i", $uid);
  } else {
	return preg_match ("/^[a-z0-9-]+$/", $uid);
  }
}

## is_uid($id)
##
function is_id($id)
{
  if ($id == ""){
    return (FALSE);
  }
  return preg_match ("/^[0-9]+$/", $id);
}

## is_path($path)
##
function is_path($path)
{
  if ($path == ""){
    return (TRUE);
  }
  return preg_match ("/\/.+$/", $path);
}

## is_email($address)
##
function is_email($address, $template= FALSE)
{
  if ($address == ""){
    return (TRUE);
  }
  if ($template){
  	return preg_match ("/^[._a-z0-9%-]+@[_a-z0-9-]+(\.[a-z0-9-]+)(\.[a-z0-9-]+)*$/i", $address);
  } else {
  	return preg_match ("/^[._a-z0-9-]+@[_a-z0-9-]+(\.[a-z0-9i-]+)(\.[a-z0-9-]+)*$/i", $address);
  }
}

## is_dns_notation($input)
##
function is_dns_notation($input)
{
  if ($input == ""){
    return (TRUE);
  }
  return preg_match ("/^[a-z0-9.-]+\.$/i", $input);
}


## print_red($string)
##
function print_red (){
	/* Check number of arguments */
	if (func_num_args() < 1){
		return;
	}

	/* Get arguments, save string */
	$array = func_get_args();
	$string= $array[0];

	/* Step through arguments */
	for ($i= 1; $i<count($array); $i++){
		$string= preg_replace ("/%s/", $array[$i], $string, 1);
	}

	if (isset($_SESSION['DEBUGLEVEL'])){
	echo ("<div align=center style=\"border-width:3px; border-style:solid; ".
		"border-color:red; background-color:black;margin:4px;\"><br>".
		"<font color=#FFFFFF><b>$string</b></font><br><br></div>\n");
	} else {
		echo "Error: $string\n";
	}
}

## print_orange($string)
##
function print_orange (){
	/* Check number of arguments */
	if (func_num_args() < 1){
		return;
	}

	/* Get arguments, save string */
	$array = func_get_args();
	$string= $array[0];

	/* Step through arguments */
	for ($i= 1; $i<count($array); $i++){
		$string= preg_replace ("/%s/", $array[$i], $string, 1);
	}

	echo ("<div align=center style=\"border-width:3px; border-style:solid; ".
		"border-color:orange; background-color:black;margin:4px;\"><br>".
		"<font color=#FFFFFF><b>$string</b></font><br><br></div>\n");
}

## gen_locked_message (user, dn)
##
function gen_locked_message($user, $dn)
{
  global $plug, $config;

  $_SESSION['dn']= $dn;
  $ldap= $config->get_ldap_link();
  $ldap->cat ($user);
  $attrs= $ldap->fetch();
  $uid= $attrs["uid"][0];

  /* Prepare and show template */
  $smarty= get_smarty();
  $smarty->assign ("dn", $dn);
  $smarty->assign ("message", sprintf(_("You're going to edit the LDAP entry '%s' which appears to be used by '%s'. Please contact the person in order to clarify proceedings."), utf8_decode($dn), "<a href=main.php?plug=0&viewid=$uid>$uid</a>"));
  $smarty->display (get_template_path('islocked.tpl'));
}

function to_string ($value)
{
  if (is_array($value)){
    $ret= "";
    foreach ($value as $line){
      $ret.= $line."<br>\n";
    }
    return ($ret);
  } else {
    return ($value);
  }
}

function minimum_version($vercheck)
{
  $minver = (int)str_replace('.', '', $vercheck);
  $curver = (int)str_replace('.', '', phpversion());

  if($curver >= $minver){
    return (true);
  }

  return (false);
}

function get_dn ($uid)
{
  global $config;

  $ldap= $config->get_ldap_link();
  $ldap->cd ($config->current['BASE']);
  $ldap->search ("(&(uid=$uid)(objectClass=person)".
                 "(objectClass=organizationalPerson)(objectClass=inetOrgPerson))");
  if ($ldap->count() != 1){
	print_red(_("Username / UID is not unique in function get_dn! Aborting."));
	exit;
  }
  $ldap->fetch ();
  return ($ldap->getDN());
}


function get_printer_list($cups_server)
{
  $res= array();
  if (function_exists('cups_get_dest_list')){
    $dest_list= cups_get_dest_list ($cups_server);

    foreach ($dest_list as $prt){
      $attr= cups_get_printer_attributes ($cups_server, $prt->name);

      foreach ($attr as $prt_info){
        if ($prt_info->name == "printer-info"){
          $info= $prt_info->value;
          break;
        }
      }
      $res[$prt->name]= "$info [$prt->name]";
    }
  } else {
    $ret= shell_exec("lpstat -p | awk '/^[^@]+$/ {print $2}'");
    $ar= preg_split("/\n/", $ret);
    foreach($ar as $val){
      if (!preg_match('/^\s*$/', $val)){
	$res[$val]= "$val";
      }
    }
  }

  return $res;
}


/*
* sess_del
* Remove variables from session.
*/
function sess_del ($var)
{
  /* New style */
  unset ($_SESSION[$var]);

  /* ... work around, since the first one
     doesn't seem to work all the time */
  session_unregister ($var);
}


function show_errors($message)
{
	$complete= "";
	foreach ($message as $error){
		if ($complete == ""){
			$complete= $error;
		} else {
			$complete= "$error<br>$complete";
		}
	}
	print_red($complete);
}


function show_ldap_error($message)
{
	if ($message != "Success"){
		print_red (_("LDAP error:")." $message");
		return TRUE;
	}
	return FALSE;
}


function rewrite($s)
{
  global $REWRITE;

  foreach ($REWRITE as $key => $val){
    $s= preg_replace("/$key/i", "$val", $s);
  }

  return ($s);
}

function dn2base($dn)
{
	global $config;

        $search = '/^[^,]+,(('.get_people_ou().')|('.get_groups_ou().')),/i';
	$base= preg_replace ($search, '', $dn);

	return ($base);
}


function change_password ($dn, $password, $mode=0, $hash= "")
{
  global $config;
  $newpass= "";

  # Read current password entry for $dn
  $ldap= $config->get_ldap_link();
  $ldap->cat ($dn);
  $attrs= $ldap->fetch ();

  # Preset encryption type if not set

  # Check for encryption type and do what's needed
  if ((isset($attrs['userPassword'][0]) && preg_match ("/^{([^}]+)}(.+)/", $attrs['userPassword'][0], $matches))
	|| $hash != ""){

    # Check for supported algorithm
    mt_srand((double) microtime()*1000000);

    if ($hash == ""){
      $hash= strtolower($matches[1]);
    }
    switch ($hash){
      case "crypt":
                $newpass= "{crypt}".crypt($password, substr(session_id(),0,2));
                break;

      case "md5":
                $newpass = "{md5}".base64_encode(pack("H*",md5($password)));
                break;

      case "ssha":
                $salt=mhash_keygen_s2k(MHASH_SHA1,$password,substr(pack("h*",md5(mt_rand())),0,8),4);
                $newpass= "{ssha}".base64_encode(mhash(MHASH_SHA1, $password.$salt).$salt);
                break;

      case "kerberos":
		$cfg= $config->data['SERVERS']['KERBEROS'];
		if (!function_exists('kadm5_init_with_password')){
			print_orange (_("Warning: Can't set kerberos password. Your PHP version has no kerberos support, password has not been changed."));
		} else {
	                $handle = kadm5_init_with_password($cfg['SERVER'],
	                                $cfg['REALM'], $cfg['ADMIN'], $cfg['PASSWORD']);
	                if ($handle === FALSE){
	                  print_red (_("Kerberos database communication failed!"));
	                }
	                $ret= kadm5_chpass_principal($handle, $attrs['uid'][0]."@".$cfg['REALM'],
	                                $password);
	                if ($ret === FALSE){
	                  print_red (_("Changing password in kerberos database failed!"));
	                }
	                kadm5_destroy($handle);

	                $newpass= "{kerberos}".$attrs['uid'][0]."@".$cfg['REALM'];
		}
                break;

      default:  print_red (_("Unsupported password hash detected. Can't generate compatible password."));
                break;
    }

  } else {
    # Crypt it by default
    $newpass= "{crypt}".crypt($password, substr(session_id(),0,2));
  }

  # Update shadow timestamp?
  if (isset($attrs["shadowLastChange"][0])){
    $shadow= (int)(date("U") / 86400);
  } else {
    $shadow= 0;
  }

  # Write back modified entry
  $ldap->cd($dn);
  $attrs= array();
  if ($password == ""){
    $attrs['userPassword']= array();
  } else {
    $attrs['userPassword']= $newpass;
  }

  # Not for groups
  if ($mode == 0){
    if ($shadow != 0){
      $attrs['shadowLastChange']= $shadow;
    }

    # Create SMB Password
    $tmp= $config->data['MAIN']['SMBHASH']." ".escapeshellarg($password);
    @DEBUG (DEBUG_LDAP, __LINE__, __FUNCTION__, __FILE__,
	$tmp, "Execute");

    $hash= shell_exec ($tmp);
    if ($hash == ""){
	print_red (_("Setting for SMBHASH in gosa.conf is incorrect! Can't change Samba password."));
    } else {
      list($lm,$nt)= split (":", trim($hash));
      if ($config->current['SAMBAVERSION'] == 3){
        $attrs['sambaLMPassword']= $lm;
        $attrs['sambaNTPassword']= $nt;
        $attrs['sambaPwdLastSet']= date('U');
      } else {
        $attrs['lmPassword']= $lm;
        $attrs['ntPassword']= $nt;
        $attrs['pwdLastSet']= date('U');
      }
    }
  }

  $ldap->modify($attrs);
  if ($ldap->error != 'Success'){
	print_red(sprintf(_("Setting the password failed. LDAP server says '%s'."), $ldap->get_error()));
	exit;
  }
}

function check_command($cmdline)
{
	$cmd= preg_replace("/ .*$/", "", $cmdline);

	/* Check if command exists in filesystem */
	if (!file_exists($cmd)){
		return (FALSE);
	}

	/* Check if command is executable */
	if (!is_executable($cmd)){
		return (FALSE);
	}

	return (TRUE);
}


function print_header($image, $headline)
{
	echo "<div class=\"plugtop\">\n";
	echo "  <img src=\"$image\" align=\"middle\" alt=\"*\">&nbsp;$headline\n";
	echo "</div>\n";
}


function register_global($name, $object)
{
	$_SESSION[$name]= $object;
}


function is_global($name)
{
	return isset($_SESSION[$name]);
}


function get_global($name)
{
	return $_SESSION[$name];
}


function range_selector($data, $start)
{
	$output= "";

	if ($start != 0){
		$output.= "<a href=\"main.php?plug=".$_GET['plug']."&start=".
			($start-25)."\"><img src=\"images/back.png\" align=\"center\"".
			" border=0>&nbsp;";
	}
	$num= (int)(count($data) / 25);
	$gstart= (int)((int)($start / 25) / 25) * 25;
	$gend= $gstart + 25;
	if ($gend > $num){
		$gend= $num;
	}
	if ($gstart != 0){
		$output.= "...&nbsp;";
	}
	if ($num != 0){
		for ($i= $gstart; $i<= $gend; $i++){
			if ($i == (int)($start / 25)){
				$output.= ($i+1)."&nbsp;";
			} else {
				$output.= "<a href=\"main.php?plug=".$_GET['plug'].
					"&start=".($i*25)."\" style=\"color:#A0A0A0;\">".
					($i+1)."</a>&nbsp;";
				if ($i >= $num){
					$output.= "<a href=\"main.php?plug=".$_GET['plug'].
						"&start=".($start+25)."\"><img src=\"images/".
						"forward.png\" align=\"center\" border=0>";
				} elseif ($i == $gend){
					$output.= "...&nbsp;";
					$output.= "<a href=\"main.php?plug=".$_GET['plug'].
						"&start=".($start+25)."\"><img src=\"images/".
						"forward.png\" align=\"center\" border=0>";
				}
			}
		}
	}

	return ($output);
}

function apply_filter()
{
	$apply= "";
	
	if ($_SESSION['js'] == FALSE){
		$apply= '<p class="contentboxb" style="border-top:1px solid #B0B0B0"><table width="100%"><tr><td width="100%" align="right"><input type="submit" name="apply" value="'._("Apply").'"></td></tr></table></p>';
	}

	return ($apply);
}

function back_to_main()
{
	$string= '<br><p class="plugbottom"><input type=submit name="password_back" value="'._("Back").'"></p>';
	return ($string);
}


function time_diff($start, $stop)
{
	list($a_dec, $a_sec) = explode(" ", $start);
	list($b_dec, $b_sec) = explode(" ", $stop);
	return ($b_sec - $a_sec + $b_dec - $a_dec);
}

function normalize_netmask($netmask)
{
	/* Check for notation of netmask */
	if (!preg_match('/^([0-9]+\.){3}[0-9]+$/', $netmask)){
		$num= (int)($netmask);
		$netmask= "";
		for ($byte= 0; $byte<4; $byte++){
			$result=0;
			for ($i= 7; $i>=0; $i--){
				if ($num-- > 0){
					$result+= pow(2,$i);
				}
			}
			$netmask.= $result.".";
		}
		return (preg_replace('/\.$/', '', $netmask));
	}

	return ($netmask);
}

function netmask_to_bits($netmask)
{
        list($nm0, $nm1, $nm2, $nm3)= split('\.', $netmask);
        $res= 0;
        for ($n= 0; $n<4; $n++){
                $start= 255;
                $name= "nm$n";
                for ($i= 0; $i<8; $i++){
                        if ($start == (int)($$name)){
                                $res+= 8 - $i;
                                break;
                        }
                        $start-= pow(2,$i);
                }
        }
	
	return ($res);
}

function recurse($rule, $variables)
{
	$result= array();

	if (!count($variables)){
		return array($rule);
	}

	reset($variables);
	$key= key($variables);
	$val= current($variables);
	unset ($variables[$key]);

	foreach($val as $possibility){
		$nrule= preg_replace("/\{$key\}/", $possibility, $rule);
		$result= array_merge($result, recurse($nrule, $variables));
	}

	return ($result);
}


function expand_id($rule, $attributes)
{
	/* Check for id rule */
	if(preg_match('/^id(:|#)\d+$/',$rule)){
		return (array("\{$rule}"));
	}

	/* Check for clean attribute */
	if (preg_match('/^%[a-zA-Z0-9]+$/', $rule)){
		$rule= preg_replace('/^%/', '', $rule);
		$val= rewrite(preg_replace('/ /', '', strtolower($attributes[$rule])));
		return (array($val));
	}

	/* Check for attribute with parameters */
	if (preg_match('/^%[a-zA-Z0-9]+\[[0-9-]+\]$/', $rule)){
		$param= preg_replace('/^[^[]+\[([^]]+)]$/', '\\1', $rule);
		$part= preg_replace('/^%/', '', preg_replace('/\[.*$/', '', $rule));
		$val= rewrite(preg_replace('/ /', '', strtolower($attributes[$part])));
		$start= preg_replace ('/-.*$/', '', $param);
		$stop = preg_replace ('/^[^-]+-/', '', $param);

		/* Assemble results */
		$result= array();
		for ($i= $start; $i<= $stop; $i++){
			$result[]= substr($val, 0, $i);
		}
		return ($result);
	}
	
	echo "Error in idgen string: don't know how to handle rule $rule.\n";
	return (array($rule));
}


function gen_uids($rule, $attributes)
{
	global $config;

	/* Search for keys and fill the variables array with all 
	   possible values for that key. */
	$part= "";
	$trigger= false;
	$stripped= "";
	$variables= array();
	for ($pos= 0; $pos < strlen($rule); $pos++){
		if ($rule[$pos] == "{" ){
			$trigger= true;
			$part= "";
			continue;
		}
		if ($rule[$pos] == "}" ){
			$variables[$pos]= expand_id($part, $attributes);
			$stripped.= "\{$pos}";
			$trigger= false;
			continue;
		}
		if ($trigger){
			$part.= $rule[$pos];
		} else {
			$stripped.= $rule[$pos];
		}
	}

	/* Recurse through all possible combinations */
	$proposed= recurse($stripped, $variables);

	/* Get list of used ID's */
	$used= array();
	$ldap= $config->get_ldap_link();
	$ldap->cd($config->current['BASE']);
	$ldap->search('(uid=*)');
	while($attrs= $ldap->fetch()){
		$used[]= $attrs['uid'][0];
	}
	
	/* Remove used uids and watch out for id tags */
	$ret= array();
	foreach($proposed as $uid){
		
		/* Check for id tag and modify uid if needed */
		if(preg_match('/\{id:\d+}/',$uid)){
			$size= preg_replace('/^.*{id:(\d+)}.*$/', '\\1', $uid);
			
			for ($i= 0; $i < pow(10,$size); $i++){
				$number= sprintf("%0".$size."d", $i);
				$res= preg_replace('/{id:(\d+)}/', $number, $uid);
				if (!in_array($res, $used)){
					$uid= $res;
					break;
				}
			}
		}
		if(preg_match('/\{id#\d+}/',$uid)){
			$size= preg_replace('/^.*{id#(\d+)}.*$/', '\\1', $uid);
			
			while (true){
				mt_srand((double) microtime()*1000000);
				$number= sprintf("%0".$size."d", mt_rand(0, pow(10, $size)-1));
				$res= preg_replace('/{id#(\d+)}/', $number, $uid);
				if (!in_array($res, $used)){
					$uid= $res;
					break;
				}
			}
		}

		/* Don't assign used ones */
		if (!in_array($uid, $used)){
			$ret[]= $uid;
		}
	}

	return($ret);
}


function check_schema_version($description, $version)
{
	$desc= preg_replace("/^.* DESC\s+\(*\s*'([^']+)'\s*\)*.*$/", '\\1', $description);

	return preg_match("/\(v$version\)/", $desc);
}

function array_search_r($needle, $key, $haystack){
	foreach($haystack as $index => $value){
		$match= 0;
		if (is_array($value)){
			$match= array_search_r($needle, $key, $value);
		}
		if ($index==$key && !is_array($value) && preg_match("/$needle/i", $value)){
			$match=1;
		}
		if ($match) return 1;
	}
	return 0;
} 


function class_is_needed($name)
{
	global $config;

	return array_search_r($name, "CLASS", $config);
}


function schema_check($server, $admin, $password)
{
	global $config;

        $messages= array();
        $required_classes= array(
                "gosaObject"            => array("version" => "2.1"),
                "gosaAccount"           => array("version" => "2.1"),
                "gosaLockEntry"         => array("version" => "2.1"),
                "gosaCacheEntry"        => array("version" => "2.1"),
                "gosaDepartment"        => array("version" => "2.1"),

                "goFaxAccount"          => array("version" => "1.0.3", "class" => "gofaxAccount"),
                "goFaxSBlock"           => array("version" => "1.0.3", "class" => "gofaxAccount"),
                "goFaxRBlock"           => array("version" => "1.0.3", "class" => "gofaxAccount"),

                "gosaUserTemplate"      => array("version" => "2.1", "class" => "posixAccount"),
                "gosaMailAccount"       => array("version" => "2.1", "class" => "mailAccount"),
                "gosaProxyAccount"      => array("version" => "2.1", "class" => "proxyAccount"),
                "gosaApplication"       => array("version" => "2.1", "class" => "appgroup"),
                "gosaApplicationGroup"  => array("version" => "2.1", "class" => "appgroup"),

                "GOhard"                => array("version" => "2.0", "class" => "terminals"),
                "gotoTerminal"          => array("version" => "2.0", "class" => "terminals"),
                "goServer"              => array("version" => "2.1"),
                "goTerminalServer"      => array("version" => "2.1", "class" => "terminals"),
                "goNfsServer"           => array("version" => "2.1", "class" => "terminals"),
                "goNtpServer"           => array("version" => "2.1", "class" => "terminals"),
                "goSyslogServer"        => array("version" => "2.1", "class" => "terminals"),
                "goLdapServer"          => array("version" => "2.1"),
                "goCupsServer"          => array("version" => "2.1", "class" => array("posixAccount", "terminals")),
                "goImapServer"          => array("version" => "2.1", "class" => array("mailAccount", "mailgroup")),
                "goKrbServer"           => array("version" => "2.1"),
                "goFaxServer"           => array("version" => "2.1", "class" => "gofaxAccount")
                );

        /* Build LDAP connection */
        $ds= ldap_connect ($server);
        if (!$ds) {
                return (array(_("Can't bind to LDAP. No schema check possible!")));
        }
	ldap_set_option($ds, LDAP_OPT_PROTOCOL_VERSION, 3);
        $r= ldap_bind ($ds, $admin, $password);

        /* Get base to look for schema */
        $sr= ldap_read ($ds, "", "objectClass=*", array("subschemaSubentry"));
        $attr= ldap_get_entries($ds,$sr);
        if (!isset($attr[0]['subschemasubentry'][0])){
                return (array(_("Can't get schema information from server. No schema check possible!")));
        }

        /* Get list of objectclasses */
        $objectclasses= array();
        $sr= ldap_read ($ds, "cn=subschema", "objectClass=*", array("objectclasses"));
        $attrs= ldap_get_entries($ds,$sr);
        if (!isset($attrs[0])){
                return (array(_("Can't get schema information from server. No schema check possible!")));
        }
        foreach ($attrs[0]['objectclasses'] as $val){
                $name= preg_replace("/^.* NAME\s+\(*\s*'([^']+)'\s*\)*.*$/", '\\1', $val);
                if ($name != $val){
                        $objectclasses[$name]= $val;
                }
        }

        /* Walk through objectclasses and check if they are needed or not */
        foreach ($required_classes as $key => $value){
		if (isset($value['class'])){
			if (!is_array($value['class'])){
				$classes= array($value['class']);
			} else {
				$classes= $value['class'];
			}
				
			/* Check if we are using the class that requires */
			foreach($classes as $class){
				if (!isset($objectclasses[$key])){
	                                $messages[$key]= sprintf(_("Optional objectclass '%s' required by plugin '%s' is not present in LDAP setup"), $key, $class);
	                        } else {
	                                if (!check_schema_version($objectclasses[$key], $value['version'])){
	                                        $messages[$key]= sprintf(_("Optional objectclass '%s' required by plugin '%s' does not have version %s"), $key, $class, $value['version']);
	                                }
	                        }

			}
		} else {
			/* Required class */
			if (!isset($objectclasses[$key])){
				$messages[$key]= sprintf(_("Required objectclass '%s' is not present in LDAP setup"), $key);
			} else {
				if (!check_schema_version($objectclasses[$key], $value['version'])){
					$messages[$key]= sprintf(_("Required objectclass '%s' does not have version %s"), $key, $value['version']);
				}
			}
		}
        }

	/* Check for correct samba parameters */
	if (!isset($objectclasses['sambaSamAccount'])){
		$messages['samba3']= _("SAMBA 3 support disabled, but no schema seems to be installed");
	}
	if (!isset($objectclasses['sambaAccount'])){
		$messages['samba2']= _("SAMBA 2 support disabled, but no schema seems to be installed");
	}

	/* Check pureftp/dns/ */
	if (!isset($objectclasses['PureFTPdUser'])){
		$messages['pureftp']= _("Support for pureftp disabled, but no schema seems not to be installed");
	}
	if (!isset($objectclasses['dNSZone'])){
		$messages['dns']= _("Support for dns management disabled, but no schema seems not to be installed");
	}
	if (!isset($objectclasses['dhcpServer'])){
		$messages['dhcp']= _("Support for dhcp management disabled, but no schema seems not to be installed");
	}

        return ($messages);
}

/* Sadly values like memory_limit are perpended by K, M, G, etc.
   Need to convert... */
function to_byte($value) {

    $value= strtolower(trim($value));
    if(!is_numeric(substr($value, -1))) {
        switch(substr($value, -1)) {
            case 'g':
               $mult= 1073741824;
               break;
            case 'm':
               $mult= 1048576;
               break;
            case 'k':
               $mult= 1024;
               break;
        }
        return ($mult * (int)substr($value, 0, -1));
    }
    return $calue;
}

?>
