/***************************************************************************
 *            expenses-gtk.c
 *
 *  Mon Nov 21 22:56:18 2005
 *  Copyright  2005  Neil Williams <linux@codehelp.co.uk>
 *  Copyright (C) 2002, 2003, 2004 Philip Blundell <philb@gnu.org>
 *  Copyright (C) 2005, 2006 Florian Boor <florian@kernelconcepts.de>
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
 
#include "config.h"
#include <stdlib.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <glib/gprintf.h>
#include <qof.h>
#include <math.h>
#include <locale.h>
 /* GTK and GPE includes */
#include <gtk/gtkmain.h>
#include <gpe/pixmaps.h>
#include <gpe/init.h>
#include <gpe/pim-categories.h>
#include <gpe/pim-categories-ui.h>
#include <gpe/spacing.h>
#include "expenses-gtk.h"
#include "qof-main.h"
#include "gpe-expenses.h"
#include "qof-expenses.h"

#define _(String) gettext (String)

/** Used as a default. qof-expense sorts out the rest. */
#define SHOW_DECIMAL_PLACES 2

static QofLogModule log_module = GPE_MOD_GUI;

enum { 
	COL_EXP_DATE, 
	COL_EXP_TYPE, 
	COL_EXP_SYMBOL, 
	COL_EXP_AMOUNT,
	COL_ENTITY,
	COL_MAX
};
static void exp_refresh_list (GpeExpenseData *context);

/** \brief Only update modified parameters

It probably isn't best to have the g_free here
but it is convenient. Just remember to only
pass newly allocated strings to this function
and that text_entry has \b been freed before
it returns.

@param text_entry parameter value as a string - this will
	be \b freed if it exists! 
@param ent The entity to set this value, if modified
@param param The parameter of the entity in use.
*/
static inline void
compare_cache (gchar * text_entry, QofEntity * ent, const QofParam * param)
{
	gchar * check;
	check = qof_util_param_to_string (ent, param);
	g_return_if_fail (check);
	if (safe_strcmp(check, text_entry))
	{
		qof_util_param_edit ((QofInstance*)ent, param);
		qof_util_param_set_string (ent, param, text_entry);
		qof_util_param_commit ((QofInstance*)ent, param);
	}
	if (text_entry)
		g_free (text_entry);
	text_entry = NULL;
}

/** \brief check each parameter and update backend

Checks each entity for a dirty flag, then
compares the live (dirty) data against the cached backend
data and updates modified parameters.
*/
static void
edit_ok_clicked(GtkWidget G_GNUC_UNUSED *widget, gpointer data)
{
	GpeExpenseData *context;
	QofCurrency * currency;
	GtkTextBuffer *buf;
	GtkTextIter start, end;
	QofEntity *ent;
	gchar * mnemonic;

	context = (GpeExpenseData*)data;
	g_return_if_fail(context);
	ent = context->entity;

	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (context->text_view));
	gtk_text_buffer_get_bounds (buf, &start, &end);
	/* exp_note */
	compare_cache (g_strdup(gtk_text_buffer_get_text (GTK_TEXT_BUFFER (buf), 
		&start, &end, FALSE)), ent, 
		qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_NOTE));
	/* exp_vendor */
	compare_cache ( g_strdup (gtk_entry_get_text
		(GTK_ENTRY (context->vendor_entry))), ent, 
		qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_VENDOR));
	/* exp_city */
	compare_cache (g_strdup (gtk_entry_get_text
		(GTK_ENTRY (context->city_entry))), ent, 
	qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_CITY));
	/* exp_attendees */
	compare_cache (g_strdup (gtk_entry_get_text
		(GTK_ENTRY (context->attendees_entry))), ent,
	qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_ATTENDEES));
	/* exp_type */
	compare_cache (g_strdup (ExpenseTypeasString
		(gtk_combo_box_get_active(context->edit_type_list))), ent, 
		qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_TYPE));
	/* exp_payment */
	compare_cache (g_strdup (ExpensePaymentasString
		(gtk_combo_box_get_active(context->payment_list))), ent, 
		qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_PAYMENT));
	/* exp_category */
	compare_cache (g_strdup (gtk_label_get_text (GTK_LABEL(context->cat_label))),
		ent, qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_CATEGORY));
	/* exp_currency */
	mnemonic = gtk_combo_box_get_active_text (context->currency_list);
	currency = qof_currency_lookup_name ((QofInstance*)ent, mnemonic);
	/* if currency is not found, preserve the cache version. */
	if (currency)
		compare_cache (g_strdup_printf ("%d", currency->pq_code), ent, 
			qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_CURRENCY));
	gtk_widget_destroy(context->window);
	context->entity = NULL;
	exp_refresh_list(context);
}

static void
edit_cancel_clicked(GtkWidget G_GNUC_UNUSED *widget, gpointer data)
{
	GpeExpenseData *context;

	context = (GpeExpenseData*)data;
	g_return_if_fail(context);
	exp_refresh_list(context);
	gtk_widget_destroy(context->window);
	context->entity = NULL;
}

static void
edit_delete_clicked(GtkWidget G_GNUC_UNUSED *widget, gpointer data)
{
	GpeExpenseData *context;

	context = (GpeExpenseData*)data;
	g_return_if_fail(context);
	qof_event_gen (context->entity, QOF_EVENT_DESTROY, NULL);
	qof_entity_release(context->entity);
	exp_refresh_list(context);
	gtk_widget_destroy(context->window);
	context->entity = NULL;
}

static void
create_currency_list(gpointer G_GNUC_UNUSED key, gpointer value, gpointer user_data)
{
	QofCurrency *curr;
	GpeExpenseData *context;

	curr = (QofCurrency*)value;
	context = (GpeExpenseData*)user_data;
	gtk_combo_box_insert_text (context->currency_list, curr->pq_code, 
		curr->mnemonic);
}

/* ported from gpe-contacts */
static gchar *
build_categories_string (GSList * catlist)
{
	guint id;
	GSList * iter;
	const gchar *cat;
	gchar *s = NULL;

	if (!catlist) { return g_strdup(""); }
	for (iter = catlist; iter; iter = iter->next)
	{
		id = GPOINTER_TO_INT(iter->data);
		cat = gpe_pim_category_name (id);

		if (cat)
		{
			if (s)
			{
				/* append as a comma-separated list */
				gchar *ns = g_strdup_printf ("%s, %s", s, cat);
				g_free (s);
				s = ns;
			}
			else
				s = g_strdup (cat);
		}
	}
	return s;
}

static void
set_selected_category (GtkWidget G_GNUC_UNUSED *ui, GSList *selected, gpointer user_data)
{
	GpeExpenseData * context;
	gchar * str;

	context = (GpeExpenseData*)user_data;
	g_return_if_fail (context);

	str = build_categories_string (selected);
	if (str)
	{
		gtk_label_set_markup (GTK_LABEL (context->cat_label), str);
		g_free (str);
	}
}

/* based on gpe-contacts */
static void
on_categories_clicked (GtkButton G_GNUC_UNUSED *button, gpointer user_data)
{
	GtkWidget *w;

	w = gpe_pim_categories_dialog (gpe_pim_categories_list (), 
			G_CALLBACK (set_selected_category), user_data);
}

static gint
cat_compare (gconstpointer gpe_cat, gconstpointer qof_name)
{
	struct gpe_pim_category * c;

	c = (struct gpe_pim_category *) gpe_cat;
	return safe_strcasecmp (gpe_pim_category_name (c->id), qof_name);
}

/* merge the QOF category list into the GPE PIM category list */
static void
cat_populate (const gchar * cat_name)
{
	GSList * cat_list, *match;
	gint max_list;

	match = NULL;
	cat_list = gpe_pim_categories_list();
	if (cat_list)
	{
		/* if the category name exists, leave it alone */
		match = g_slist_find_custom (cat_list, cat_name, cat_compare);
	}
	if (!match)
	{
		DEBUG ("'%s' not found in GPE category list, adding.", cat_name);
		gpe_pim_category_new (cat_name, &max_list);
	}
}

static void 
edit_expense (GtkWidget G_GNUC_UNUSED *w, GpeExpenseData *context)
{
	GtkWidget *table, *top_vbox;
	GtkWidget *viewport, *scrolled_window;
	GtkWidget *buttonbox, *buttonok, *buttoncancel, *buttondelete;
	GtkWidget *type_label, *payment_label, *currency_label;
	GtkWidget *city_label, *vendor_label, *note_label, *attendees_label;
	GtkTextBuffer *buf;
	guint gpe_spacing, pos, i;
	const QofParam *param;
	gboolean mileage;
	gchar *text;

	g_return_if_fail(context);
	if(!context->entity) { return; }
	ENTER (" ");
	gpe_spacing = 0;
	pos = 0;
	mileage = FALSE;
	buttonok = gtk_button_new_from_stock (GTK_STOCK_SAVE);
	buttoncancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	buttondelete = gtk_button_new_from_stock (GTK_STOCK_DELETE);

	table = gtk_table_new(10, 6, FALSE);
	top_vbox = gtk_vbox_new (FALSE, 0);
	context->text_view = gtk_text_view_new ();
	buttonbox = gtk_hbox_new (FALSE, 0);
	type_label = gtk_label_new (_("Type:"));
	payment_label = gtk_label_new (_("Payment:"));
	currency_label = gtk_label_new (_("Currency:"));
	vendor_label = gtk_label_new(_("Vendor:"));
	city_label = gtk_label_new(_("City:"));
	note_label = gtk_label_new(_("Note:"));
	attendees_label = gtk_label_new(_("Attendees"));
	context->categories = gtk_button_new_with_label (_("Category"));
	context->cat_label = gtk_label_new ("");
	context->edit_type_list = GTK_COMBO_BOX(gtk_combo_box_new_text());
	context->currency_list = GTK_COMBO_BOX(gtk_combo_box_new_text());
	context->payment_list = GTK_COMBO_BOX(gtk_combo_box_new_text());
	context->vendor_entry = GTK_ENTRY (gtk_entry_new());
	context->city_entry = GTK_ENTRY (gtk_entry_new());
	context->attendees_entry = GTK_ENTRY (gtk_entry_new());
	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	viewport = gtk_viewport_new(NULL, NULL);

	context->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

	gtk_table_set_col_spacings (GTK_TABLE(table), gpe_spacing);
	gtk_table_set_row_spacings (GTK_TABLE(table), gpe_spacing);
	gtk_box_set_spacing (GTK_BOX(top_vbox), gpe_spacing);

	gtk_window_set_default_size (GTK_WINDOW (context->window), 240, 320);
	gtk_window_set_title (GTK_WINDOW (context->window), _("Expenses"));

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewport), GTK_SHADOW_NONE);
	gtk_container_add (GTK_CONTAINER (viewport), table);
	gtk_container_add (GTK_CONTAINER (scrolled_window), viewport);

	gtk_container_set_border_width(GTK_CONTAINER(buttonbox), 0);

	gtk_table_set_col_spacings (GTK_TABLE(table), gpe_spacing);
	gtk_table_set_row_spacings (GTK_TABLE(table), gpe_spacing);
	gtk_box_set_spacing (GTK_BOX(top_vbox), gpe_spacing);
    
	gtk_signal_connect (GTK_OBJECT (buttonok), "clicked",
			  GTK_SIGNAL_FUNC (edit_ok_clicked), context);
	gtk_signal_connect (GTK_OBJECT (buttoncancel), "clicked",
			  GTK_SIGNAL_FUNC (edit_cancel_clicked), context);
	gtk_signal_connect (GTK_OBJECT (buttondelete), "clicked",
			  GTK_SIGNAL_FUNC (edit_delete_clicked), context);

	gtk_box_pack_start (GTK_BOX (buttonbox), buttondelete, TRUE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (buttonbox), buttoncancel, TRUE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (buttonbox), buttonok, TRUE, FALSE, 0);

	gtk_misc_set_alignment (GTK_MISC (context->cat_label), 0.0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (type_label), 0.0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (payment_label), 0.0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (currency_label), 0.0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (vendor_label), 0.0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (city_label), 0.0, 0.5);
	gtk_misc_set_alignment (GTK_MISC (attendees_label), 0.0, 0.5);
  
	i = 0;
	param = qof_class_get_parameter(context->entity->e_type, EXP_TYPE);
	i = ExpenseTypefromString(param->param_getfcn(context->entity, param));
	gtk_combo_box_set_active(context->edit_type_list, i);
	/* Currency and PaymentType disabled for mileage */
	if(i == Mileage) { mileage = TRUE; }

	/* Category */
	pos++;
	gtk_table_attach (GTK_TABLE(table), context->categories, 0, 1, pos, pos+1, 
					  GTK_FILL, GTK_FILL, 0, 0);
	/* the label comes after the button as the label *is* the data. */
	gtk_table_attach (GTK_TABLE(table), context->cat_label, 2, 3, pos, pos+1, 
					  GTK_FILL, GTK_FILL, 0, 0);
	pos++;
	/* Type of expense */
	i = 0;
	gtk_table_attach(GTK_TABLE(table), type_label, 0, 1, pos, pos+1, 
				   GTK_FILL, GTK_FILL, 0, 0);
	while(0 != safe_strcmp(ExpenseTypeasString(i), ""))
	{
		gchar *check;

		gtk_combo_box_append_text (context->edit_type_list, 
			dgettext(LIBRARY_GETTEXT_PACKAGE, ExpenseTypeasString(i)));
		check = qof_util_param_to_string(context->entity, param);
		if(0 == safe_strcmp(check, ExpenseTypeasString(i)))
		{
			gtk_combo_box_set_active(context->edit_type_list, i);
		}
		g_free(check);
		i++;
	}
	gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(context->edit_type_list), 1, 
				3, pos, pos+1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);

	pos++;
	/* Payment method */
	i = 0;
	gtk_table_attach(GTK_TABLE(table), payment_label, 0, 1, pos, pos+1, 
				GTK_FILL, GTK_FILL, 0, 0);
	
	while((0 != safe_strcmp(ExpensePaymentasString(i), "")) &&
		(!mileage))
	{
		gtk_combo_box_append_text(context->payment_list, 
			dgettext(LIBRARY_GETTEXT_PACKAGE, ExpensePaymentasString(i)));
		i++;
	}
	gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(context->payment_list), 1, 
				3, pos, pos+1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
	pos++;
	/* Currency (to be replaced when using mileage) */
	if(!mileage)
	{
		gtk_table_attach(GTK_TABLE(table), currency_label, 0, 1, pos, pos+1, 
					GTK_FILL, GTK_FILL, 0, 0);
		qof_currency_foreach(create_currency_list, context);
		gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(context->currency_list), 1, 
					3, pos, pos+1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
		pos++;
	}
	/* Vendor */
	gtk_table_attach(GTK_TABLE(table), vendor_label, 0, 1, pos, pos+1, 
				GTK_FILL, GTK_FILL, 0, 0);
	gtk_table_attach(GTK_TABLE(table), GTK_WIDGET (context->vendor_entry), 
		1, 5, pos, pos+1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
	pos++;
	/* City */
	gtk_table_attach(GTK_TABLE(table), city_label, 0, 1, pos, pos+1, 
				GTK_FILL, GTK_FILL, 0, 0);
	gtk_table_attach(GTK_TABLE(table), GTK_WIDGET (context->city_entry), 
		1, 5, pos, pos+1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
	pos++;
	/* Attendees */
	gtk_table_attach(GTK_TABLE(table), attendees_label, 0, 1, pos, pos+1, 
				GTK_FILL, GTK_FILL, 0, 0);
	gtk_table_attach(GTK_TABLE(table), GTK_WIDGET (context->attendees_entry), 
		1, 5, pos, pos+1, GTK_FILL | GTK_EXPAND, GTK_FILL, 0, 0);
	pos++;
	/* Note */
	gtk_table_attach(GTK_TABLE(table), note_label, 0, 1, pos, pos+1, 
				GTK_FILL, GTK_FILL, 0, 0);
	gtk_table_attach(GTK_TABLE(table), context->text_view, 1, 5, pos, pos+1,
				GTK_FILL, GTK_FILL | GTK_EXPAND, 0, 0);
	pos++;

	gtk_text_view_set_editable (GTK_TEXT_VIEW (context->text_view), TRUE);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (context->text_view), GTK_WRAP_WORD_CHAR);

	gtk_box_pack_start (GTK_BOX (top_vbox), scrolled_window, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (top_vbox), buttonbox, FALSE, FALSE, 0);

	gtk_container_add (GTK_CONTAINER (context->window), top_vbox);

	/* use entity values to preset the edit window  */
	i = 0;
	param = qof_class_get_parameter(context->entity->e_type, EXP_CATEGORY);
	text = param->param_getfcn(context->entity, param);
	if(text) {
		/* check the QOF categories against the GPE categories
		 and add the OQF ones if they do not exist in GPE */
		cat_populate (text);
		PINFO ("setting cat_label=%s", text);
		gtk_label_set_markup (GTK_LABEL(context->cat_label), text);
	}

	param = qof_class_get_parameter(context->entity->e_type, EXP_PAYMENT);
	i = ExpensePaymentfromString(param->param_getfcn(context->entity, param));
	gtk_combo_box_set_active(context->payment_list, i);

	param = qof_class_get_parameter(context->entity->e_type, EXP_VENDOR);
	text = param->param_getfcn(context->entity, param);
	if(text) { gtk_entry_set_text(context->vendor_entry, text); }

	param = qof_class_get_parameter(context->entity->e_type, EXP_CITY);
	text = param->param_getfcn(context->entity, param);
	if(text) { gtk_entry_set_text(context->city_entry, text); }

	param = qof_class_get_parameter(context->entity->e_type, EXP_CURRENCY);
	{
		QofCurrency * currency;
		gint32 check, (*int32_getter)   (QofEntity*, const QofParam*);
		int32_getter = (gint32 (*)(QofEntity*, const QofParam*)) param->param_getfcn;
		check = int32_getter(context->entity, param);
		currency = qof_currency_lookup ((QofInstance*)context->entity, check);
		if (currency)
			PINFO (" currency=%d mnemonic=%s", check, currency->mnemonic);
		gtk_combo_box_set_active(context->currency_list, check);
	}

	param = qof_class_get_parameter(context->entity->e_type, EXP_ATTENDEES);
	text = param->param_getfcn(context->entity, param);
	if(text) { gtk_entry_set_text(context->attendees_entry, text); }

	param = qof_class_get_parameter(context->entity->e_type, EXP_NOTE);
	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW (context->text_view));
	text = param->param_getfcn(context->entity, param);
	if(text) 
	{ 
		gtk_text_buffer_set_text(buf, text, strlen(text));
		gtk_text_view_set_buffer(GTK_TEXT_VIEW (context->text_view), buf);
	}

	g_signal_connect (G_OBJECT (context->categories), "clicked",
					  G_CALLBACK (on_categories_clicked), context);
	gpe_set_window_icon (context->window, "icon");
	gtk_widget_show_all(context->window);
	LEAVE (" ");
}

static void 
open_about_dialog (void)
{
	GtkAboutDialog* ab;
	/* If you modify gpe-expenses, add your details here. */
	const gchar * authors[4] = {"Neil Williams <linux@codehelp.co.uk>\n"
		"Philip Blundell <philb@gnu.org>\n"
		"Florian Boor <florian@kernelconcepts.de>\n"
		, NULL };

	ab = GTK_ABOUT_DIALOG( gtk_about_dialog_new() );
	gtk_about_dialog_set_copyright(ab, 
		"Copyright 2005-2007 Neil Williams <linux@codehelp.co.uk>");
	gtk_about_dialog_set_version(ab, VERSION);
	gtk_about_dialog_set_comments(ab,
		/* Translators: line breaks allowed here. */
		_("Expenses records for GPE. Supports payment types, "
		"categories, expense types (mileage, meals, parking, etc.), "
		"notes and currency selection."));
	gtk_about_dialog_set_license (ab,
		" This package is free software; you can redistribute it and/or modify\n"
		" it under the terms of the GNU General Public License as published by\n"
		" the Free Software Foundation; either version 3 of the License, or\n"
		" (at your option) any later version.\n"
		"\n"
		" This program is distributed in the hope that it will be useful,\n"
		" but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		" GNU General Public License for more details.\n"
		"\n"
		" You should have received a copy of the GNU General Public License\n"
		" along with this program.  If not, see <http://www.gnu.org/licenses/>.\n");
	gtk_about_dialog_set_website (ab, "http://gpe-expenses.sourceforge.net/");
	gtk_about_dialog_set_authors (ab, authors);
	gtk_about_dialog_set_translator_credits (ab, _("translator-credits"));
	gtk_about_dialog_set_logo (ab, gpe_try_find_icon("icon", NULL));
	gpe_set_window_icon(GTK_WIDGET(ab), "icon");
	gtk_dialog_run(GTK_DIALOG(ab));
	gtk_widget_destroy (GTK_WIDGET(ab));
}

/*	Receives the selected expense, then list each
	column parameter. Use data selection for column1,
	string selection from list for column2, digit entry for column4.
	Column3 is set via selection of column2.
*/
static void
exp_show_entities(QofEntity *ent, gpointer data)
{
	GpeExpenseData *context;
	QofTime     *qt;
	QofDate     *qd;
	QofNumeric amount, (*numeric_getter) (QofEntity*, const QofParam*);
	GtkTreeIter      ent_data;
	const QofParam   *param;
	gchar            *type, *symbol, *date_string, *dis_string;
	gdouble           d_amount;

	context = (GpeExpenseData*)data;
	g_return_if_fail(context);
	symbol = "";
	/* param_getfcn each parameter */
	param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_DATE);
	qt = (QofTime*)param->param_getfcn (ent, param);
	qd = qof_date_from_qtime (qt);
	date_string = qof_date_print (qd, QOF_DATE_FORMAT_CE);
	qof_date_free (qd);
	param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_TYPE);
	dis_string = param->param_getfcn(ent, param);
	type = dgettext(LIBRARY_GETTEXT_PACKAGE, dis_string);
	/* If Mileage, use ExpenseDistance,
	else use ExpenseCustomCurrency->symbol */
	if(0 == safe_strcmp(dis_string, "Mileage"))
	{
		gint unit;
		/* EXP_DISTANCE */
		param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_DISTANCE);
		unit = ExpenseDistancefromString(param->param_getfcn(ent, param));
		switch (unit)
		{
			/* Translators: short form of 'miles' */
			case 0 : { symbol = _("mi"); break; }
			/* Translators: short form of 'kilometres' */
			case 1 : { symbol = _("km"); break; }
		}
	}
	else
	{
		QofCurrency * pqc;
		gint32 curr_code, (*int32_getter) (QofEntity *, const QofParam *);
		param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_CURRENCY);
		int32_getter = (gint32 (*)(QofEntity *, const QofParam *))
			param->param_getfcn;
		curr_code = int32_getter (ent, param);
		pqc = qof_currency_lookup ((QofInstance*)ent, curr_code);
		if (pqc)
			symbol = g_strdup(pqc->symbol);
	}
	param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_AMOUNT);
	numeric_getter = (QofNumeric (*)(QofEntity*, const QofParam*)) param->param_getfcn;
	amount = numeric_getter(ent, param);
	d_amount = qof_numeric_to_double(amount);
	/* Columns:  Date Type symbol Amount */
	gtk_list_store_append(context->list_store, &ent_data);
	gtk_list_store_set(context->list_store, &ent_data,
		COL_EXP_DATE,   date_string,
		COL_EXP_TYPE,   type,
		COL_EXP_SYMBOL, symbol,
		COL_EXP_AMOUNT, d_amount,
		COL_ENTITY,     ent,
		-1 );
	g_free(date_string);
}

/* button == 1 left mouse button, 3 = right click. */
static gboolean
button_press_event (GtkWidget *widget, GdkEventButton *b, gpointer data)
{
	GtkTreeViewColumn *col;
	GtkTreePath *path;
	GtkTreeIter iter;
	QofEntity  *ent;
	gdouble amount;
	GpeExpenseData *context;

	context = (GpeExpenseData*)data;
	g_return_val_if_fail(context, FALSE);
	amount = 0.00;
	ent = NULL;
	switch (b->button) {
		case 1 : {
			if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (widget),
						b->x, b->y, &path, &col, NULL, NULL))
			{
				GtkTreeView * treeview;

				treeview = GTK_TREE_VIEW 
					(gtk_tree_view_new_with_model 
					(GTK_TREE_MODEL(context->list_store)));
				gtk_tree_view_set_cursor (treeview, path, NULL, TRUE);
				gtk_widget_grab_focus (GTK_WIDGET (treeview));
				gtk_tree_model_get_iter (GTK_TREE_MODEL (context->list_store), 
						&iter, path);
				gtk_tree_model_get(GTK_TREE_MODEL (context->list_store),
						&iter, COL_ENTITY, &ent, -1);
				gtk_tree_model_get(GTK_TREE_MODEL (context->list_store),
						&iter, COL_EXP_AMOUNT, &amount, -1);
				context->entity = ent;
				gtk_tree_path_free (path);
			}
			break; 
		}
		default : { break; }
	}
	return FALSE;
}

/** \bug Need to provide a better date change mechanism

Should really use QOF_DATE_FORMAT_LOCALE for the display
without trying to parse that when supporting changing the date
itself - instead clicking date should probably raise a
calendar widget and/or some preference setting for the
chosen format string.

*/
static void 
change_date (GtkCellRendererText G_GNUC_UNUSED *cell, 
			gchar G_GNUC_UNUSED *path_string, gchar *new_text, gpointer data)
{
	void (*time_setter) (QofEntity*, QofTime*);
	GpeExpenseData *context;
	const QofParam *param;
	QofTime *qt;
	QofDate *qd;

	context = (GpeExpenseData*)data;
	g_return_if_fail(context);
	ENTER (" new_text=%s", new_text);
	param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_DATE);
	time_setter = (void(*)(QofEntity*, QofTime*))param->param_setfcn;
	/* convert the string to a QofTime */
	qd = qof_date_parse (new_text, QOF_DATE_FORMAT_CE);
	qt = qof_date_to_qtime (qd);
	qof_util_param_edit ((QofInstance*)context->entity, param);
	if ((time_setter && qof_time_is_valid (qt)))
		time_setter (context->entity, qt);
	qof_util_param_commit ((QofInstance*)context->entity, param);
	exp_refresh_list(context);
	LEAVE (" ");
}

static void 
change_amount (GtkCellRendererText G_GNUC_UNUSED *cell, 
			   gchar G_GNUC_UNUSED *path_string, gchar *new_text, gpointer data)
{
	GpeExpenseData *context;
	QofNumeric amount;
	void (*numeric_setter)   (QofEntity*, QofNumeric);
	void (*i32_setter)       (QofEntity*, gint32);
	const QofParam *param;
	gchar *numeric_char;
	gdouble d_amount;
	gint precision;
	/* detect local currency */
	QofCurrency * c;
	struct lconv lc;
	gchar * sy;
	guint def_pq_code;

	context = (GpeExpenseData*)data;
	g_return_if_fail(context);
	ENTER (" ");
	def_pq_code = -1;
	param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_CURRENCY);
	i32_setter = (void(*)(QofEntity*, gint32))param->param_setfcn;
	/* use locale currency */
	lc = *localeconv();
	DEBUG (" sym=%s", lc.currency_symbol);
	sy = g_strdup (lc.int_curr_symbol);
	sy = g_strstrip (sy);
	c = qof_currency_lookup_name ((QofInstance*)context->entity, sy);
	if (c)
		def_pq_code = c->pq_code;
	g_free (sy);
	if(i32_setter != NULL) { i32_setter(context->entity, def_pq_code); }
	param = qof_class_get_parameter(GPE_QOF_EXPENSES, EXP_AMOUNT);
	numeric_setter = (void(*)(QofEntity*, QofNumeric))param->param_setfcn;
	d_amount = 0.00;
	precision = pow(10, SHOW_DECIMAL_PLACES);
	amount = qof_numeric_zero();
	d_amount = strtod(new_text, NULL);
	amount = qof_numeric_from_double (d_amount, precision, 
		QOF_HOW_DENOM_EXACT | QOF_HOW_RND_ROUND);
	numeric_char = qof_numeric_to_string(amount);
	DEBUG (" numeric_char=%s", numeric_char);
	g_free(numeric_char);

	if ((qof_numeric_check (amount) == QOF_ERROR_OK) && (numeric_setter))
	{
		qof_util_param_edit ((QofInstance*)context->entity, param);
		numeric_setter(context->entity, amount);
		qof_util_param_commit ((QofInstance*)context->entity, param);
	}
	else 
	{
		qof_util_param_edit ((QofInstance*)context->entity, param);
		numeric_setter(context->entity, qof_numeric_zero ());
		qof_util_param_commit ((QofInstance*)context->entity, param);
	}
	exp_refresh_list(context);
	LEAVE (" ");
}

static void
rounding_func (GtkTreeViewColumn G_GNUC_UNUSED *tree_column,
        GtkCellRenderer *cell, GtkTreeModel *tree_model,
        GtkTreeIter *iter, gpointer data)
{
	GtkCellRendererText *cell_text;
	gdouble d;
	gchar *s;

	cell_text = (GtkCellRendererText *)cell;
	d = 0.00;
	s = NULL;
	g_free(cell_text->text);
	gtk_tree_model_get(tree_model, iter, GPOINTER_TO_INT(data), &d, -1);
	s = g_strdup_printf("%%.%i", SHOW_DECIMAL_PLACES);
	s = g_strconcat(s, "f", NULL);
	cell_text->text = g_strdup_printf(s, d);
	g_free(s);
}

static gint
type_compare (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
 		gpointer G_GNUC_UNUSED user_data)
{
	gint result;
	gchar *str_a, *str_b;

	result = 0;
	gtk_tree_model_get(model, a, COL_EXP_TYPE, &str_a, -1);
	gtk_tree_model_get(model, b, COL_EXP_TYPE, &str_b, -1);
	result = safe_strcmp(str_a, str_b);
	g_free(str_a);
	g_free(str_b);
	return result;
}

/** \bug need to support comparing the real date 

The value in the text field could be wrong or in
a weird locale format that is impossible to parse.

*/
static gint
date_compare (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
 		gpointer G_GNUC_UNUSED user_data)
{
	QofTime *qt_a, *qt_b;
	QofEntity * ent_a, * ent_b;
	const QofParam * param;
	gint result;

	ent_a = ent_b = NULL;
	result = 0;
	gtk_tree_model_get(model, a, COL_ENTITY, &ent_a, -1);
	gtk_tree_model_get(model, b, COL_ENTITY, &ent_b, -1);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_DATE);
	qt_a = param->param_getfcn (ent_a, param);
	qt_b = param->param_getfcn (ent_b, param);
	result = qof_time_cmp (qt_a, qt_b);
	qof_time_free (qt_a);
	qof_time_free (qt_b);
	return result;
}

static gint
amount_compare (GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
 		gpointer G_GNUC_UNUSED user_data)
{
	gint result;
	gdouble *dbl_a, *dbl_b;

	result = 0;
	gtk_tree_model_get(model, a, COL_EXP_AMOUNT, &dbl_a, -1);
	gtk_tree_model_get(model, b, COL_EXP_AMOUNT, &dbl_b, -1);
	if(dbl_a != dbl_b)
		result = (dbl_a > dbl_b) ? 1 : -1;
	return result;
}

static GtkWidget*
set_list_view(GpeExpenseData *context)
{
	GtkTreeView *view;
	GtkTreeSortable *sort;
	GtkWidget *scrolled;
	GtkCellRenderer *date_rend, *type_rend, *symbol_rend, *amount_rend;
	GtkTreeViewColumn *col;

	scrolled = gtk_scrolled_window_new (NULL, NULL);
	context->list_store = gtk_list_store_new(COL_MAX, G_TYPE_STRING, 
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_POINTER);
	sort = GTK_TREE_SORTABLE (context->list_store);
	view = GTK_TREE_VIEW (gtk_tree_view_new_with_model
		(GTK_TREE_MODEL (sort)));

	date_rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes ((_("Date")), date_rend,
			"text", COL_EXP_DATE, NULL);
	g_object_set(date_rend, "editable", TRUE, NULL);
	g_object_set(col, "reorderable", TRUE, NULL);
	g_object_set(col, "sort-indicator", TRUE, NULL);
	gtk_tree_view_column_set_sort_column_id(col, COL_EXP_DATE);
	gtk_tree_sortable_set_sort_func (sort, COL_EXP_DATE, date_compare, NULL, NULL);
	gtk_tree_view_column_set_expand(col, TRUE);
	gtk_tree_view_column_set_clickable(col, TRUE);
	gtk_tree_view_insert_column (GTK_TREE_VIEW (view), col, -1);
	g_signal_connect(date_rend, "edited", 
		(GCallback) change_date, context);

	type_rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Type"), type_rend,
			"text", COL_EXP_TYPE, NULL);
	gtk_tree_view_column_set_sort_column_id(col, COL_EXP_TYPE);
	gtk_tree_sortable_set_sort_func (sort, COL_EXP_TYPE, type_compare, NULL, NULL);
	gtk_tree_view_column_set_expand(col, TRUE);
	gtk_tree_view_column_set_clickable(col, TRUE);
	g_object_set(col, "reorderable", TRUE, NULL);
	g_object_set(col, "sort-indicator", TRUE, NULL);
	gtk_tree_view_insert_column (GTK_TREE_VIEW (view), col, -1);

	symbol_rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes ("", symbol_rend,
			"text", COL_EXP_SYMBOL, NULL);
	gtk_tree_view_column_set_expand(col, TRUE);
	gtk_tree_view_column_set_clickable(col, TRUE);
	gtk_tree_view_insert_column (GTK_TREE_VIEW (view), col, -1);

	amount_rend = gtk_cell_renderer_text_new ();
	col = gtk_tree_view_column_new_with_attributes (_("Amount"), amount_rend,
			"text", COL_EXP_AMOUNT, NULL);
	g_object_set(amount_rend, "editable", TRUE, NULL);
	g_object_set(col, "reorderable", TRUE, NULL);
	g_object_set(col, "sort-indicator", TRUE, NULL);
	gtk_tree_view_column_set_sort_column_id(col, COL_EXP_AMOUNT);
	gtk_tree_sortable_set_sort_func (sort, COL_EXP_AMOUNT, amount_compare, NULL, NULL);
	gtk_tree_view_column_set_expand(col, TRUE);
	gtk_tree_view_column_set_clickable(col, TRUE);
	gtk_tree_view_insert_column (GTK_TREE_VIEW (view), col, -1);
	gtk_tree_view_column_set_cell_data_func(col, amount_rend, 
		rounding_func, GINT_TO_POINTER(COL_EXP_AMOUNT), NULL);
	g_signal_connect(amount_rend, "edited", 
		(GCallback) change_amount, context);

	g_signal_connect (G_OBJECT (view), "button_press_event", 
			G_CALLBACK (button_press_event), context);

	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET(view));
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
			  GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_tree_sortable_set_sort_column_id (sort, COL_EXP_DATE, 
		GTK_SORT_DESCENDING);
	return GTK_WIDGET(scrolled);
}

static void
exp_refresh_list (GpeExpenseData *context)
{
	gtk_list_store_clear(context->list_store);
	/* Populate the list from qof_object_foreach */
	qof_object_foreach(GPE_QOF_EXPENSES, context->book, exp_show_entities, context);
}

static void 
open_new_expense (GtkWidget G_GNUC_UNUSED *w, GpeExpenseData *context)
{
	GtkTreeIter  ent_data;
	QofInstance  *inst;
	QofBook      *book;
	const QofParam *param;
	void (*i32_setter)       (QofEntity*, gint32);
	/* detect local currency */
	QofCurrency * c;
	struct lconv lc;
	gchar * sy;
	guint def_pq_code;

	g_return_if_fail(context);
	def_pq_code = -1;
	book = qof_session_get_book(context->qof.input_session);
	inst = (QofInstance*)qof_object_new_instance(GPE_QOF_EXPENSES, book);
	context->entity = &inst->entity;
	/* use locale currency */
	lc = *localeconv();
	DEBUG (" sym=%s", lc.currency_symbol);
	sy = g_strdup (lc.int_curr_symbol);
	sy = g_strstrip (sy);
	c = qof_currency_lookup_name (inst, sy);
	if (c)
		def_pq_code = c->pq_code;
	g_free (sy);
	param = qof_class_get_parameter (GPE_QOF_EXPENSES, EXP_CURRENCY);
	i32_setter = (void(*)(QofEntity*, gint32))param->param_setfcn;
	qof_util_param_edit ((QofInstance*)context->entity, param);
	i32_setter(context->entity, def_pq_code);
	qof_util_param_commit ((QofInstance*)context->entity, param);
	gtk_list_store_append(context->list_store, &ent_data);
	exp_refresh_list(context);
}

static GtkWidget*
set_toolbar (GpeExpenseData *context)
{
	GtkWidget *toolbar;
	GtkToolItem *new_exp, *quit_exp, *about_exp, *item, *edit_exp;
	toolbar = gtk_toolbar_new ();
	gtk_toolbar_set_orientation (GTK_TOOLBAR (toolbar), GTK_ORIENTATION_HORIZONTAL);

	new_exp = gtk_tool_button_new_from_stock(GTK_STOCK_NEW);
	g_signal_connect(G_OBJECT(new_exp), "clicked", 
		G_CALLBACK (open_new_expense), context);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), new_exp, -1);

	edit_exp = gtk_tool_button_new_from_stock(GTK_STOCK_PROPERTIES);
	g_signal_connect(G_OBJECT(edit_exp), "clicked",
		G_CALLBACK (edit_expense), context);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), edit_exp, -1);

	item = gtk_separator_tool_item_new();
	gtk_tool_item_set_expand(GTK_TOOL_ITEM(item), FALSE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	about_exp = gtk_tool_button_new_from_stock(GTK_STOCK_ABOUT);
	g_signal_connect(G_OBJECT(about_exp), "clicked", 
		G_CALLBACK (open_about_dialog), NULL);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), about_exp, -1);

	item = gtk_separator_tool_item_new();
	gtk_tool_item_set_expand(GTK_TOOL_ITEM(item), FALSE);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), item, -1);

	quit_exp = gtk_tool_button_new_from_stock(GTK_STOCK_QUIT);
	gtk_toolbar_insert(GTK_TOOLBAR(toolbar), quit_exp, -1);
	g_signal_connect (G_OBJECT (quit_exp), "clicked",
			G_CALLBACK (gtk_main_quit), NULL);
	return GTK_WIDGET(toolbar);
}

void
open_expenses_window (GpeExpenseData *context)
{
	gboolean large_screen;
	gboolean mode_landscape;
	GtkWidget *window;
	GtkWidget *vbox;

	g_return_if_fail(context);
	ENTER (" ");
	vbox = gtk_vbox_new (FALSE, 0);

	large_screen = (gdk_screen_width() > 400);
	mode_landscape = (gdk_screen_width() > gdk_screen_height());
	gpe_pim_categories_init ();
	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_default_size (GTK_WINDOW (window), 240, 320);
	gtk_window_set_title (GTK_WINDOW (window), _("Expenses"));
	gpe_set_window_icon (window, "icon");

	gtk_box_pack_start (GTK_BOX (vbox), set_toolbar(context), FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (window), "delete-event",
			G_CALLBACK (gtk_main_quit), NULL);
	
	gtk_box_pack_start (GTK_BOX (vbox), set_list_view(context), TRUE, TRUE, 0);

	gtk_container_add (GTK_CONTAINER (window), vbox);
	gtk_widget_show_all (window);
	/* Populate the list from qof_object_foreach */
	qof_object_foreach(GPE_QOF_EXPENSES, context->book, 
		exp_show_entities, context);
	LEAVE (" ");
}
