/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------
 
  gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
          libraries.

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of gpiv.

   Gpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------*/

/*
 * (callback) functions for Piv evaluation window/tabulator
 * $Log: piveval.c,v $
 * Revision 1.15  2006/01/31 14:28:12  gerber
 * version 0.3.0
 *
 * Revision 1.13  2005/02/26 09:43:30  gerber
 * parameter flags (parameter_logic) defined as gboolean
 *
 * Revision 1.12  2005/02/26 09:17:14  gerber
 * structured of interrogate function by using gpiv_piv_isiadapt
 *
 * Revision 1.11  2005/01/19 15:53:42  gerber
 * Initiation of Data Acquisition (DAC); trigerring of lasers and camera
 * by using RTAI and Realtime Linux, recording images from IEEE1394
 * (Firewire) IIDC compliant camera's
 *
 * Revision 1.10  2004/10/15 19:24:05  gerber
 * GPIV_ and Gpiv prefix to defines and structure names of libgpiv
 *
 * Revision 1.9  2004/06/14 21:19:23  gerber
 * Image depth up to 16 bits.
 * Improvement "single int" and "drag int" in Eval tab.
 * Viewer's pop-up menu.
 * Adaption for gpiv_matrix_* and gpiv_vector_*.
 * Resizing console.
 * See Changelog for further info.
 *
 * Revision 1.8  2003/09/01 11:17:15  gerber
 * improved monitoring of interrogation process
 *
 * Revision 1.7  2003/08/22 15:24:52  gerber
 * interactive spatial scaling
 *
 * Revision 1.6  2003/07/13 14:38:18  gerber
 * changed error handling of libgpiv
 *
 * Revision 1.5  2003/07/12 21:21:16  gerber
 * changed error handling libgpiv
 *
 * Revision 1.3  2003/07/10 11:56:07  gerber
 * added man page
 *
 * Revision 1.2  2003/06/27 13:47:26  gerber
 * display ruler, line/point evaluation
 *
 * Revision 1.1.1.1  2003/06/17 17:10:52  gerber
 * Imported gpiv
 *
 */


#include "gpiv_gui.h"
#include "utils.h"
#include "piveval.h"
#include "display.h"


/*
 * Private piv evaluation functions
 */
static gboolean toggle_button_changed;


static void
display_img(float **img,
            gint int_size,
            gint *int_size_old,
            gint i_min, 
            gint i_max, 
            gint j_min, 
            gint j_max,
            gint rgb_width,
            guchar *rgbbuf,
            GtkWidget *canvas,
            GnomeCanvasItem **gci,
            double affine[6],
            gboolean scale
            )
/* ----------------------------------------------------------------------------
 * Displaying routine in gnome canvas
 */
{
    guchar *pos;
    gint i, j;
    float min = 1000. ,max = -1000.;

    GdkPixbuf *pixbuf = NULL;
    gint depth = 8;    
    guint16 fact = 1;

    fact = fact << (image_par.depth - depth);

    if (*int_size_old != i_max - i_min && rgbbuf != NULL) {
        g_free(rgbbuf);
        rgbbuf = NULL;        
    }

    if (rgbbuf == NULL) {
/*
 * row stride; each row is a 4-byte buffer array
 */
        rgb_width = (i_max - i_min) * 3;
        while ((rgb_width) % 4 != 0) {
            rgb_width++;
        } 

        rgbbuf = g_malloc(rgb_width * 3 * 
                          (j_max - j_min));
    }


    if (*gci != NULL) {
        gtk_object_destroy(GTK_OBJECT(*gci));
        *gci = NULL;
    } 
    

    pixbuf = gdk_pixbuf_new_from_data(rgbbuf,
                                      GDK_COLORSPACE_RGB,
                                      FALSE,
                                      depth,
                                      i_max - i_min, 
                                      j_max - j_min,
                                      rgb_width,
                                      NULL, 
                                     NULL);



    *gci = gnome_canvas_item_new( gnome_canvas_root( GNOME_CANVAS
                                                    (canvas)),
                                 gnome_canvas_pixbuf_get_type (),
                                 "pixbuf", pixbuf,
                                 NULL);


    affine[4] = (GPIV_MAX_INTERR_SIZE - gpiv_var.piv_disproc_zoom 
                 * int_size) / 2.0;
    affine[5] = (GPIV_MAX_INTERR_SIZE - gpiv_var.piv_disproc_zoom 
         * int_size) / 2.0;
    gnome_canvas_item_affine_absolute(*gci, affine);

/* 
 * As the mean has been subtracted from the image data, image intensities
 * will have to be tilted above zero
 */
    if (scale) {
        for (j = j_min; j < j_max; j++) {
            for (i = i_min; i < i_max; i++) {
                if (img[j][i] < min)
                    min = img[j][i];
                if (img[j][i] > max)
                    max = img[j][i];
            }
        }

        pos = rgbbuf;
        for (j = j_min; j < j_max; j++) {
            for (i = i_min; i < i_max; i++) {
                img[j][i] -= min;
                img[j][i] = (255.0 / (max - min)) * img[j][i];
                *pos++ = (unsigned char) img[j][i];
                *pos++ = (unsigned char) img[j][i];
                *pos++ = (unsigned char) img[j][i];
            }
            *pos = *pos++;
            *pos = *pos++;
            *pos = *pos++;
        }


    } else {
        for (j = j_min; j < j_max; j++) {
            for (i = i_min; i < i_max; i++) {
                if (img[j][i] < min)
                    min = img[j][i];
            }
        }
        
/*         if ( min < display_act->img.img_mean) */
/*             display_act->img.img_mean = min; */


        pos = rgbbuf;
        for (j = j_min; j < j_max; j++) {
            for (i = i_min; i < i_max; i++) {
/*                 *pos++ = (unsigned char) img[j][i] - (unsigned char) min */
/*                     + (unsigned char) display_act->img.img_mean; */
/*                 *pos++ = (unsigned char) img[j][i] - (unsigned char) min */
/*                     + (unsigned char) display_act->img.img_mean; */
/*                 *pos++ = (unsigned char) img[j][i] - (unsigned char) min */
/*                     + (unsigned char) display_act->img.img_mean; */
/*
 * BUGFIX: display image intars with: 8 < depth < 16
 * 6 March 2004
 */
	    *pos++ =  (guchar) ((img[j][i] - /* (guchar) */ min + 
                                /* (guchar) */ display_act->img.img_mean) / fact);
	    *pos++ =  (guchar) ((img[j][i] - /* (guchar) */ min + 
                                /* (guchar) */ display_act->img.img_mean) / fact);
	    *pos++ =  (guchar) ((img[j][i] - /* (guchar) */ min + 
                                /* (guchar) */ display_act->img.img_mean) / fact);
            }
        }
    }


    gdk_pixbuf_unref (pixbuf);
    *int_size_old = int_size;
}



static void
alloc_bufmem_per_intarea(int index_y, 
                         int index_x, 
                         GpivData * gpd, 
                         GpivImagePar image_par, 
                         int int_size_0
                         )
/* ----------------------------------------------------------------------------
 * Memory allocation of covariance in a packed interrogation area array
 */ 
{ 
}



static void
free_bufmem_per_intarea(int index_y, 
                        int index_x, 
                        GpivData * gpd, 
                        GpivImagePar image_par, 
                        int int_size_0
                        )
/*-----------------------------------------------------------------------------
 */
{ 
}



static void
report_progress(GpivConsole * gpiv,
                int index_y,
                int index_x,
                PivEvalPar piv_eval_par,
/*                 PivEvalPar piv_eval_par_dest, */
                int sweep,
                gfloat cum_residu,
                PivData *piv_data,
                Display *disp,
                gint *progress_value_prev
                )
/*-----------------------------------------------------------------------------
 * Calculates progress of interrogation processing and other variables 
 * and prints to message bar of the console if progress value has been changed
 */
{
    gchar progress_string[GPIV_MAX_CHARS];
    gfloat progress_value = 100 * (index_y * piv_data->nx + index_x +1) / 
        (piv_data->nx * piv_data->ny);


    if (progress_value != *progress_value_prev) {
        *progress_value_prev = progress_value;
        if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
             || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
             || piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
                if (piv_eval_par.ad_int == TRUE) {

                        g_snprintf(progress_string, GPIV_MAX_CHARS, 
                                   "Interrogating image #%d:"
                                   " sweep #%d"
                                   " size=%d"
                                   " shift=%d" 
                                   " residu=%.3f"
                                   ,
                                   disp->id,
                                   sweep, 
                                   piv_eval_par.int_size_2, 
                                   piv_eval_par.int_shift,
                                   cum_residu
                                   );

                } else {

                        g_snprintf(progress_string, GPIV_MAX_CHARS, 
                                   "Interrogating image #%d:"
                                   " sweep #%d"
                                   " shift=%d"
                                   " residu=%.3f"
                                   ,
                                   disp->id,
                                   sweep,
                                   piv_eval_par.int_shift,
                                   cum_residu
                                   );
                        
                }
            

        } else {
            g_snprintf(progress_string, GPIV_MAX_CHARS, 
                       "Interrogating image #%d: "
                       " shift=%d "
                       ,
                       disp->id,
                       piv_eval_par/* _dest */.int_shift
                       );
        }
        
        while (g_main_iteration(FALSE));
        gtk_progress_set_value(GTK_PROGRESS (gnome_appbar_get_progress
                                             (GNOME_APPBAR (gpiv->appbar))), 
                               progress_value);
        gnome_appbar_push(GNOME_APPBAR(gpiv->appbar), 
                          progress_string);
        
    }
   
}



static void
adjust_radio_button_intsize_1(GpivEvalPar piv_eval_par, 
                              PivEval *eval
                              )
/* ----------------------------------------------------------------------------
 * Adjust radio button interrogation size 1 in PIV interrogation tab
 */
{
    if (piv_eval_par.int_size_1 == 16) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize1_2), TRUE);   
    } else if (piv_eval_par.int_size_1 == 32) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize1_3), TRUE);
    } else if (piv_eval_par.int_size_1 == 64) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize1_4), TRUE);
    } else if (piv_eval_par.int_size_1 == 128) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize1_5), TRUE);
    }
}



static void
adjust_radio_button_intsize_2(GpivEvalPar piv_eval_par, 
                              PivEval *eval
                              )
/* ----------------------------------------------------------------------------
 * Adjust radio button interrogation size 1 in PIV interrogation tab
 */
{
    if (piv_eval_par.int_size_2 == 16) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize2_2), TRUE);
    } else if (piv_eval_par.int_size_2 == 32) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize2_3), TRUE);
    } else if (piv_eval_par.int_size_2 == 64) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize2_4), TRUE);
    } else if (piv_eval_par.int_size_2 == 128) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intsize2_5), TRUE);
    }
}   




static void
adjust_radio_button_intshift(GpivEvalPar piv_eval_par, 
                             PivEval *eval)
/* ----------------------------------------------------------------------------
 * Adjust radio button interrogation area shift in PIV interrogation tab
 */
{
    if (piv_eval_par.int_shift == 8) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intshift_1), TRUE);
    } else if (piv_eval_par.int_shift == 16) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intshift_2), TRUE);
    } else if (piv_eval_par.int_shift == 32) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intshift_3), TRUE);
    } else if (piv_eval_par.int_shift == 64) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intshift_4), TRUE);
    } else if (piv_eval_par.int_shift == 128) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (eval->radiobutton_intshift_5), TRUE);
    }
}



static char*
alloc_pivdata_gridgen(GpivPivData *piv_data, 
		      GpivImagePar image_par, 
		      GpivEvalPar piv_eval_par
		      )
/*-----------------------------------------------------------------------------
 * Determining the number of grid points, allocating memory for output data
 * and calculate locations of Interrogation Area's. If ad_int is enabled, a
 * course grid is started with and adapted for subsequent sweeps.
 ----------------------------------------------------------------------------*/
{
        gchar *err_msg = NULL;
        GpivEvalPar piv_eval_par_ACT;

        gpiv_piv_cp_parameters(piv_eval_par, &piv_eval_par_ACT, TRUE, FALSE);
        if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
            || piv_eval_par.int_scheme == GPIV_IMG_DEFORM
            || piv_eval_par.ad_int) {
            piv_eval_par_ACT.int_size_1 = piv_eval_par_ACT.int_size_2;
/*             g_message("alloc_pivdata_gridgen:: 0 adapting size1=%d ?= size2=%d", */
/*                       piv_eval_par_ACT.int_size_1, piv_eval_par_ACT.int_size_2); */
        }

        if (piv_eval_par.int_shift < piv_eval_par.int_size_2 / GPIV_SHIFT_FACTOR) { 
                piv_eval_par_ACT.int_shift = 
                    piv_eval_par_ACT.int_size_2 / GPIV_SHIFT_FACTOR;
        }

        if ((err_msg = 
             gpiv_piv_count_pivdata_fromimage(piv_data, image_par, piv_eval_par_ACT))
            != NULL) error_gpiv ("%s: %s", RCSID, err_msg);
        gpiv_null_pivdata(piv_data);
        gpiv_alloc_pivdata(piv_data);
        display_act->gpd.exist_piv = TRUE;
        display_act->gpd.averaged_piv = FALSE;
        if ((err_msg = 
             gpiv_piv_gridgen(piv_data, image_par, piv_eval_par_ACT))
            != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

        return err_msg;
}


/*
 * Public piv evaluation functions
 */


void 
exec_piv(GpivConsole * gpiv
         )
/*-----------------------------------------------------------------------------
 */
{
    char *err_msg = NULL;
    char message[2 * GPIV_MAX_CHARS];
    gboolean int_scheme_changed = FALSE;

/*
 * Free memory of pivdata and clean the display from its vectors
 */
    if (display_act->gpd.exist_piv) {
        if (verbose) 
            gpiv_warning("exec_piv: calling destroy_all_vectors, nx=%d ny=%d",
                      display_act->gpd.piv_data.nx, 
                      display_act->gpd.piv_data.ny);
	destroy_all_vectors(&display_act->gpd);
        gpiv_free_pivdata (&display_act->gpd.piv_data);
	display_act->gpd.exist_piv = FALSE;
        display_act->gpd.averaged_piv = FALSE;
        if (display_act->gpd.scaled_piv) {
            gpiv_free_pivdata(&display_act->gpd.piv_data_scaled);
            display_act->gpd.scaled_piv = FALSE;
        }
    }

/*
 * Free eventually existing memory of vor_data and clean the display 
 * as they does not belong to the piv data anymore
 */
    free_post_bufmems(display_act);

    if (display_act->img.exist_img && !cancel_process) {
        exec_process = TRUE;


/*
 * Set mouse selection to None
 */
        if (m_select == SINGLE_POINT_MS
            || m_select == SINGLE_AREA_MS
            || m_select == DRAG_MS) {
            gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                        (gpiv->piveval->radiobutton_mouse_1), 
                                        TRUE);
        }

/*
 * Setting interrogation scheme to GPIV_ZERO_OFF_CENTRAL if image deformation 
 * is impossible with too small (initial or final) grid.
 * BUGFIX: display_act->intreg.data.nx /ny incorrect criterium for judging!
 * do a count_pivdata or so
 */
/*         if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM */
/*             && (display_act->intreg.data.nx  < 2  */
/*                 || display_act->intreg.data.ny < 2)) { */
/*             g_snprintf(message, 2 * GPIV_MAX_CHARS,  */
/*                        _("Image deformation is not possibe with grid of nx = %d ny =%d\n\ */
/* Setting Interrogation scheme to Central difference"), */
/*                        display_act->intreg.data.nx,  */
/*                        display_act->intreg.data.ny); */
/*             warning_gpiv(message); */
/*             piv_eval_par.int_scheme = GPIV_ZERO_OFF_CENTRAL; */
/*             int_scheme_changed = TRUE; */
/*         } */

        alloc_pivdata_gridgen(&display_act->gpd.piv_data, 
                                  image_par, piv_eval_par);
        interrogate(&display_act->gpd.piv_data, 
                    display_act->img.img1, 
                    display_act->img.img2, 
                    image_par, 
                    piv_eval_par, 
                    gpiv);
/*
 * Some actions for graphical features
 */
        display_act->gpd.scaled_piv = FALSE;
        display_act->gpd.saved_piv = FALSE;
        display_act->display_piv = TRUE;

        display_act->gpd.exist_cov = TRUE;
/*
 * update vectors to correct for colors/gray-scale
 */
        update_all_vectors(&display_act->gpd);



/*
 * Copy parameters in Buffer structure for saving and, eventual, later use
 */
        gpiv_img_cp_parameters(image_par, &display_act->img.image_par, TRUE, 
                               FALSE);
        gpiv_piv_cp_parameters(piv_eval_par, &display_act->gpd.piv_eval_par, 
                               TRUE, FALSE);
        exec_process = FALSE;

/*
 * BUGFIX: 
 * Resetting interrogation scheme
 */
/*         if (int_scheme_changed) { */
/*             int_scheme_changed = FALSE; */
/*             piv_eval_par.int_scheme = GPIV_IMG_DEFORM; */
/*             g_message ("interrogate:: int_scheme = %d (voorgoed)",  */
/*                        piv_eval_par.int_scheme); */
/*         } */
        gtk_progress_set_value(GTK_PROGRESS (gnome_appbar_get_progress
                                             (GNOME_APPBAR (gpiv->appbar))), 
                               0.0);
    } else {
	err_msg =
	    _("At first, open an image. \n"
              "Than we'll further see what will happen.");
	g_warning(err_msg);
	warning_gpiv(err_msg);
    }

}


void
interrogate (GpivPivData * piv_data,
             guint16 **img1, 
             guint16 **img2,
             GpivImagePar image_par, 
             GpivEvalPar piv_eval_par,
             GpivConsole * gpiv
             )
/* ----------------------------------------------------------------------------
 * PIV interrogation of an image pair at an entire grid or single point
 */
{
    char *err_msg = NULL;
    int index_x = 0, index_y = 0;

    guint16 **img1_ACT, **img2_ACT;
    GpivPivData lgpd, *piv_data_ACT;
    GpivEvalPar piv_eval_par_ACT;
    
    gfloat **intreg1 = display_act->gpd.intreg1;
    gfloat **intreg2 = display_act->gpd.intreg2;
    int int_size_0;

    Covariance *cov = &display_act->gpd.cov, w_k;
    int sweep = 1;
    gboolean sweep_last = FALSE, sweep_stop = FALSE; 
    gboolean grid_last = FALSE, isi_last = FALSE, cum_residu_reached = FALSE;
    gfloat sum_dxdy = 0.0, sum_dxdy_old = 0.0, cum_residu = 914.6;
    gint progress = 0;



/*
 * Testing parameters on consistency and initializing derived 
 * parameters/variables
 */
    if ((err_msg = 
         gpiv_piv_test_parameter(&image_par, &piv_eval_par))
        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
/*
 * Local (actualized) parameters
 * Setting initial parameters and variables for adaptive grid and 
 * Interrogation Area dimensions
 */
    gpiv_piv_cp_parameters(piv_eval_par, &piv_eval_par_ACT, TRUE, FALSE);

    if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
        || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
	|| piv_eval_par.int_scheme == GPIV_IMG_DEFORM
        || piv_eval_par.ad_int) {
        piv_eval_par_ACT.int_size_1 = piv_eval_par_ACT.int_size_2;
        sweep_last = FALSE;
    } else {
        sweep_last = TRUE;
    }
    
    if (piv_eval_par.int_shift < piv_eval_par.int_size_2 / 
        GPIV_SHIFT_FACTOR) {
        piv_eval_par_ACT.int_shift = 
            piv_eval_par_ACT.int_size_2 / GPIV_SHIFT_FACTOR;
    }
    
    if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
            img1_ACT = gpiv_alloc_img(image_par);
            img2_ACT = gpiv_alloc_img(image_par);
            gpiv_cp_img(img1, img1_ACT, image_par);
            gpiv_cp_img(img2, img2_ACT, image_par);
            piv_data_ACT = &lgpd;
            gpiv_null_pivdata(piv_data_ACT);
            piv_data_ACT->nx = piv_data->nx;
            piv_data_ACT->ny = piv_data->ny;
            gpiv_alloc_pivdata(piv_data_ACT);
            gpiv_cp_pivdata(piv_data, piv_data_ACT);
            gpiv_0_pivdata(piv_data_ACT);
    } else {
        img1_ACT = img1;
        img2_ACT = img2;
        piv_data_ACT = piv_data;
    }

/*
 * Reads eventually existing fftw wisdom
 */
    gpiv_fread_fftw_wisdom(1);
    gpiv_fread_fftw_wisdom(-1);

    while (sweep <= GPIV_MAX_EVAL_SWEEP 
           && !sweep_stop 
           && !cancel_process) {

/*
 * Memory allocation of interrogation area's and packed interrogation area 
 * arrays. Define weight kernel values
 */
        int_size_0 = GPIV_ZEROPAD_FACT * piv_eval_par_ACT.int_size_2;
	intreg1 = gpiv_matrix (int_size_0, int_size_0);
	intreg2 = gpiv_matrix(int_size_0, int_size_0);
        
        gpiv_piv_bounds_cov(cov, int_size_0, image_par);
        gpiv_piv_bounds_cov(&w_k, int_size_0, image_par);
        cov->z = gpiv_matrix_index(cov->z_rl, cov->z_rh, cov->z_cl, cov->z_ch);
        w_k.z = gpiv_matrix_index(w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);
        
        if (piv_eval_par_ACT.int_scheme == GPIV_LK_WEIGHT) {
            gpiv_piv_weight_kernel_lin(&w_k, int_size_0);
        } else {
            gpiv_piv_weight_kernel_1(&w_k);
        }
        
/*
 * Interrogates a single point
 */
        if (m_select != SINGLE_AREA_MS 
            && m_select != DRAG_MS) {
            destroy_all_vectors(&display_act->gpd);
            gnome_canvas_update_now(GNOME_CANVAS(display_act->canvas));
        }
        
        if (piv_eval_par_ACT.int_geo == GPIV_POINT
            || m_select == SINGLE_AREA_MS
            || m_select == SINGLE_POINT_MS 
            || m_select == DRAG_MS) {            
            
            if ((err_msg = 
                 gpiv_piv_interr_reg(m_select_index_y,
                                     m_select_index_x,
                                     img1_ACT,
                                     img2_ACT,
                                     intreg1,
                                     intreg2,
                                     cov,
                                     &w_k, 
                                     piv_data_ACT,
                                     sweep, 
                                     sweep_last, 
                                     image_par,
                                     piv_eval_par_ACT))
                != NULL) error_gpiv ("%s: %s", RCSID, err_msg);

/*
 * Printing the progress of processing 
 */

            report_progress(gpiv,
                            index_y,
                            index_x,
                            piv_eval_par_ACT,
                            sweep,
                            cum_residu,
                            piv_data,
                            display_act,
                            &progress);
            
/*
 * display piv values, draw interrogation areas and covariance function
 */
            if (piv_eval_par.print_cov == 1) {
                display_piv_vector(m_select_index_y, 
                                   m_select_index_x, 
                                   display_act->gpd.piv_data, 
                                   gpiv->piveval);
                display_img_intreg1(intreg1,
                                    piv_eval_par_ACT.
                                    int_size_2,
                                    gpiv->piveval);
                display_img_intreg2(intreg2, 
                                    piv_eval_par_ACT.
                                    int_size_2, 
                                    gpiv->piveval);
                display_img_cov(cov,
                                piv_eval_par_ACT.int_size_2, 
                                gpiv->piveval);
            }
        } else {

/*
 * Interrogates at a grid of points within the Area Of Interest of the image
 */
            for (index_y = 0; index_y < piv_data_ACT->ny; index_y++) {
                for (index_x = 0; index_x < piv_data_ACT->nx; index_x++) {
                    if (cancel_process) break;

/*
 * Evaluates an interrogation area.
 */
                    if ((err_msg = 
                         gpiv_piv_interr_reg(index_y, 
                                             index_x, 
                                             img1_ACT,
                                             img2_ACT,
                                             intreg1,
                                             intreg2,
                                             cov,
                                             &w_k, 
                                             piv_data_ACT, 
                                             sweep, 
                                             sweep_last,
                                             image_par,
                                             piv_eval_par_ACT))
                        != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
/* gpiv_warning("interrogate:: back: sweep=%d x[%d][%d]=%f y[%d][%d]=%f dx[%d][%d]=%f dy[%d][%d]=%f snr=%f p_no=%d", */
/*              sweep, */
/*              index_y, index_x, piv_data_ACT->point_x[index_y][index_x], */
/*              index_y, index_x, piv_data_ACT->point_y[index_y][index_x], */
/*              index_y, index_x, piv_data_ACT->dx[index_y][index_x], */
/*              index_y, index_x, piv_data_ACT->dy[index_y][index_x], */
/*              piv_data_ACT->snr[index_y][index_x], */
/*              piv_data_ACT->peak_no[index_y][index_x] */
/*              ); */

/*
 * Printing the progress of processing 
 */
                    report_progress(gpiv,
                                    index_y,
                                    index_x, 
                                    piv_eval_par_ACT,
                                    sweep,
                                    cum_residu,
                                    piv_data,
                                    display_act,
                                    &progress);

/*
 * Draw interrogation areas, covariance function, display piv vector to
 * monitor the process.
 */
                    if (piv_eval_par.print_cov == 1) {
                        display_piv_vector(index_y, 
                                           index_x, 
                                           display_act->gpd.piv_data, 
                                           gpiv->piveval);
                        display_img_intreg1(intreg1,
                                            piv_eval_par_ACT.int_size_2, 
                                            gpiv->piveval);
                        display_img_intreg2(intreg2,
                                            piv_eval_par_ACT.int_size_2, 
                                            gpiv->piveval);
                        display_img_cov(cov,
                                        piv_eval_par_ACT.int_size_2, 
                                        gpiv->piveval);
                    }
                }
            }
        }

/*
 * Freeing memory: smaller sizes are eventually needed for a next sweep
 */
        gpiv_free_matrix(intreg1);
        gpiv_free_matrix(intreg2);
        gpiv_free_matrix_index(cov->z, cov->z_rl, cov->z_rh, cov->z_cl, cov->z_ch);
        gpiv_free_matrix_index(w_k.z, w_k.z_rl, w_k.z_rh, w_k.z_cl, w_k.z_ch);
                
        if (sweep_last) {
            sweep_stop = TRUE;
        }

        if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
            || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL
            ) {
/*
 * Test on outliers
 */
            GpivBinData klass;
            GpivLinRegData linreg;
            GpivPivData gpd;
            
            gpiv_null_pivdata(&gpd);
            gpd.nx = piv_data_ACT->nx;
            gpd.ny = piv_data_ACT->ny;
            gpiv_alloc_pivdata(&gpd);
            if ((err_msg = 
                 gpiv_valid_errvec(image_par, 
                                   piv_eval_par,
                                   piv_valid_par, 
                                   *piv_data_ACT, 
                                   &gpd, 
                                   &klass, 
                                   &linreg, 
                                   display_act->img.img1, 
                                   display_act->img.img2,
                                   TRUE))
                != NULL) gpiv_error ("%s: %s", RCSID, err_msg);
            gpiv_cp_pivdata(&gpd, piv_data_ACT);
            gpiv_free_pivdata(&gpd);


            if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
/*
 * Deformation of original image with updated estimators.
 */
                gpiv_cp_img(img1, img1_ACT, image_par);
                gpiv_cp_img(img2, img2_ACT, image_par);
                gpiv_add_dxdy_pivdata(piv_data_ACT, piv_data);
                gpiv_0_pivdata(piv_data_ACT);
                gpiv_imgproc_deform(piv_data,
                                    &image_par,
                                    img1, 
                                    img2,
                                    img1_ACT,
                                    img2_ACT);

                if (sweep_last && verbose) {
                    if ((err_msg = 
                         gpiv_eval_write_deformed_image(img1_ACT, img2_ACT, 
                                                        image_par))
                        != NULL) {
                        gpiv_error ("%s", err_msg);
                    }
                }
            }

/*
 * Check the relative cumulative residu for convergence
 * if final grid has been reached
 */
            if (isi_last && grid_last) {
                sum_dxdy_old = sum_dxdy;
                sum_dxdy = 0.0;
                gpiv_sum_dxdy_pivdata(*piv_data, &sum_dxdy);
                cum_residu = fabsf((sum_dxdy - sum_dxdy_old) / 
                             ((gfloat)piv_data->nx * (gfloat)piv_data->ny/*  * 2.0 */));
                if (cum_residu < GPIV_CUM_RESIDU_MIN) {
/*                     gpiv_warning("interrogate:: cum_residu < GPIV_CUM_RESIDU_MIN"); */
                    cum_residu_reached = TRUE;
                }
            }
	} else {
            cum_residu_reached = TRUE;
        }

/*
 * Adaptive grid and interrogation size
 */
        if (/* piv_eval_par.ad_int && */
            piv_eval_par_ACT.int_shift > piv_eval_par.int_shift
            && !sweep_stop) {
            gpiv_piv_gridadapt(&image_par,
                               piv_eval_par, 
                               &piv_eval_par_ACT, 
                               piv_data, 
                               sweep, 
                               &grid_last);
            if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
                gpiv_free_pivdata(piv_data_ACT);
                gpiv_null_pivdata(piv_data_ACT);
                piv_data_ACT->nx = piv_data->nx;
                piv_data_ACT->ny = piv_data->ny;
                gpiv_alloc_pivdata(piv_data_ACT);
                gpiv_cp_pivdata(piv_data, piv_data_ACT);
                gpiv_0_pivdata(piv_data_ACT);
            }
        } else {
            grid_last = TRUE;
        }

        gpiv_piv_isizadapt(piv_eval_par, 
                           &piv_eval_par_ACT, 
                           &isi_last);
        
        if (cum_residu_reached && isi_last && grid_last) {
            sweep_last = TRUE;
            if (piv_eval_par.int_scheme == GPIV_ZERO_OFF_FORWARD
                || piv_eval_par.int_scheme == GPIV_ZERO_OFF_CENTRAL) {
                piv_eval_par_ACT.ifit = piv_eval_par.ifit;
            }
        }
        sweep++;
    }

/*
 * Drawing vectors
 */
    if (piv_eval_par_ACT.int_geo == GPIV_POINT
        || m_select == SINGLE_AREA_MS
        || m_select == SINGLE_POINT_MS 
        || m_select == DRAG_MS) {
    } else {
        if (display_act->display_piv) {
            create_all_vectors(&display_act->gpd);
        }
    }


/*
 * Writes existing fftw wisdom
 */
    gpiv_fwrite_fftw_wisdom(1);
    gpiv_fwrite_fftw_wisdom(-1);

    if (piv_eval_par.int_scheme == GPIV_IMG_DEFORM) {
        gpiv_free_img(img1_ACT, image_par);
        gpiv_free_img(img2_ACT, image_par);
        gpiv_free_pivdata(piv_data_ACT);
    }
}


/*
 * Display functions in PIV tabulator canvas
 */


void 
display_piv_vector(gint i, 
                   gint j, 
                   GpivPivData piv_data, 
                   PivEval * eval
                   )
/* ----------------------------------------------------------------------------
 * Displays values in PIV tabulator of the GUI
 */
{
/*     float **dx = piv_data.dx, **dy = piv_data.dy; */
/*     int **peak_no = piv_data.peak_no; */
    GnomeCanvasPoints *points;
    float dl = sqrt(piv_data.dx[i][j] * piv_data.dx[i][j] + 
                    piv_data.dy[i][j] * piv_data.dy[i][j]);
    gchar *color;
    points = gnome_canvas_points_new(2);

    eval->mon.pi_da.dx[0][0] = piv_data.dx[i][j];
    eval->mon.pi_da.dy[0][0] = piv_data.dy[i][j];
    eval->mon.pi_da.peak_no[0][0] = piv_data.peak_no[i][j];

        if (eval->mon.pi_da.peak_no[0][0] == -1) {
            color="red";
        } else if (eval->mon.pi_da.peak_no[0][0] == 0) {
            color="lightblue";
        } else if (eval->mon.pi_da.peak_no[0][0] == 1) {
            color="green";
        } else if (eval->mon.pi_da.peak_no[0][0] == 2) {
            color="yellow";
        } else  {
/* if (eval->mon.pi_da.peak_no[i][j] == 3) */
            color="gray";
        }

    points->coords[0] = GPIV_MAX_INTERR_SIZE / 2;
    points->coords[1] = GPIV_MAX_INTERR_SIZE / 2;
    points->coords[2] = GPIV_MAX_INTERR_SIZE / 2 + eval->mon.pi_da.dx[0][0]
        * gpiv_var.piv_disproc_vlength;
    points->coords[3] =  GPIV_MAX_INTERR_SIZE / 2 + eval->mon.pi_da.dy[0][0]
        * gpiv_var.piv_disproc_vlength;


    if (eval->mon.gci_vec != NULL) {
	 gnome_canvas_item_set(eval->mon.gci_vec,
			       "points", points,
			       "fill_color", color,
			       "width_units", (double) THICKNESS,
			       "first_arrowhead", FALSE,
			       "last_arrowhead", TRUE,
                                      "arrow_shape_a", (double) ARROW_LENGTH * 
                                      ARROW_FACT *dl * gpiv_par.vector_scale + 
                                      ARROW_ADD,
                                      "arrow_shape_b", (double) ARROW_EDGE *
                                      ARROW_FACT * dl * gpiv_par.vector_scale + 
                                      ARROW_ADD,
                                      "arrow_shape_c", (double) ARROW_WIDTH * 
                                      ARROW_FACT * dl * gpiv_par.vector_scale + 
                                      ARROW_ADD,
 
/* 			       "arrow_shape_a", (double) ARROW_LENGTH, */
/* 			       "arrow_shape_b", (double) ARROW_EDGE, */
/* 			       "arrow_shape_c", (double) ARROW_WIDTH, */
			       NULL);
    } else {
    eval->mon.gci_vec =
	gnome_canvas_item_new(gnome_canvas_root
			      (GNOME_CANVAS(eval->canvas_monitor_vec)),
			      gnome_canvas_line_get_type(), 
			      "points", points, 
			      "fill_color", color, 
			      "width_units",(double) THICKNESS, 
			      "first_arrowhead", TRUE,
                                      "arrow_shape_a", (double) ARROW_LENGTH * 
                                      ARROW_FACT *dl * gpiv_par.vector_scale + 
                                      ARROW_ADD,
                                      "arrow_shape_b", (double) ARROW_EDGE *
                                      ARROW_FACT * dl * gpiv_par.vector_scale + 
                                      ARROW_ADD,
                                      "arrow_shape_c", (double) ARROW_WIDTH * 
                                      ARROW_FACT * dl * gpiv_par.vector_scale + 
                                      ARROW_ADD,
 
/* 			      "arrow_shape_a", (double) THICKNESS,  */
			      NULL);

    }

    gnome_canvas_points_free(points);
}



void 
display_img_intreg1(float **intreg1, 
                    int int_size,
                    PivEval * eval
                    )
/* ----------------------------------------------------------------------------
 * Displays image of intreg1 for drawing area
 * row stride; each row is a 4-byte buffer array
 */
{
     display_img(intreg1, 
                 int_size,
                 &eval->mon.int_size_old,
                 0, 
                 int_size, 
                 0, 
                 int_size, 
                 eval->mon.rgb_int_width,
                 eval->mon.rgbbuf_int1,
                 eval->canvas_monitor_int1,
                 &eval->mon.gci_int1,
                 eval->mon.affine,
                 FALSE
                 );
}



void display_img_intreg2(float **intreg2, 
                         int int_size,
                         PivEval * eval
                         )
/* ----------------------------------------------------------------------------
 * Displays image of intreg2 for drawing area
 */
{
     display_img(intreg2, 
                 int_size,
                 &eval->mon.int_size_old,
                 0, 
                 int_size, 
                 0, 
                 int_size, 
                 eval->mon.rgb_int_width,
                 eval->mon.rgbbuf_int2,
                 eval->canvas_monitor_int2,
                 &eval->mon.gci_int2,
                 eval->mon.affine,
                 FALSE
                 );

}



void display_img_cov(Covariance * cov, 
                     int int_size, 
                     PivEval * eval
                     )
/* ----------------------------------------------------------------------------
 * Displays image of intreg1 for drawing area
 */
{
     display_img(cov->z, 
                 int_size,
                 &eval->mon.cov_size_old,
                 cov->z_cl, 
                 cov->z_ch, 
                 cov->z_rl, 
                 cov->z_rh, 
                 eval->mon.rgb_cov_width,
                 eval->mon.rgbbuf_cov,
                 eval->canvas_monitor_cov,
                 &eval->mon.gci_cov,
                 eval->mon.affine,
                 TRUE
                 );
}



/*
 * Piv evaluation window/tabulator callbacks
 */


void 
on_radiobutton_piv_mouse(GtkWidget * widget, 
                         gpointer data
                         )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    m_select = atoi(gtk_object_get_data(GTK_OBJECT(widget),
					"mouse_select"));

    if (m_select == NO_MS) {
	piv_eval_par.int_geo = GPIV_AOI;

    } else if  (m_select == AOI_MS) {
	piv_eval_par.int_geo = GPIV_AOI;
    } else if (m_select == SINGLE_POINT_MS) {
	piv_eval_par.int_geo = GPIV_POINT;
    } else if (m_select == V_LINE_MS) {
	piv_eval_par.int_geo =  GPIV_LINE_C;
    } else if (m_select == H_LINE_MS) {
	piv_eval_par.int_geo =  GPIV_LINE_R;
    } else if (m_select == SINGLE_AREA_MS ||
               m_select == DRAG_MS) {
	piv_eval_par.int_geo = GPIV_POINT;
    }

/*
 * (Re)setting interrogation scheme if necessary
 */
    if ((m_select == SINGLE_AREA_MS 
         || m_select == DRAG_MS
         || m_select == SINGLE_POINT_MS
         || m_select == V_LINE_MS
         || m_select == H_LINE_MS)
        && piv_eval_par.int_scheme == GPIV_IMG_DEFORM
        ) {
        char message[2 * GPIV_MAX_CHARS];
        
        g_snprintf(message, 2 * GPIV_MAX_CHARS, 
                   _("Image deformation is not possibe with this Mouse select\n\
Setting Interrogation scheme to Central difference\n\
This will be reset automatically"));
        warning_gpiv(message);
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (gpiv->piveval->radiobutton_centraldiff), 
                                    TRUE);
        toggle_button_changed = TRUE;
    } else {
        if (toggle_button_changed) {
        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                    (gpiv->piveval->radiobutton_imgdeform), 
                                    TRUE);
            toggle_button_changed = FALSE;
        }
    }
}



void 
on_radiobutton_piv_mouse1_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("No mouse activity within displayer");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_radiobutton_piv_mouse2_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Selects an area within the image to be analyzed");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_radiobutton_piv_mouse3_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Piv evaluation at a single interrogation area. "
                   "Conserves other existing data");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_radiobutton_piv_mouse4_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Piv evaluation at a single point in the image. "
                   "Rejects all existing data!!");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_radiobutton_piv_mouse5_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Displaces a single interrogation area and analyzes");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_radiobutton_piv_mouse6_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Evaluation at a vertical line. ");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_radiobutton_piv_mouse7_enter(GtkWidget * widget, 
                                gpointer data
                                )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Evaluation at a horizontal line. ");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_spinbutton_piv_int(GtkSpinButton * widget, 
                      GtkWidget * entry
                      )
/*-----------------------------------------------------------------------------
 */
{
    char *err_msg = NULL;
    PivEval * eval = gtk_object_get_data(GTK_OBJECT(widget), "eval");
    gint nx = 0, ny = 0;
    GpivPivData local_intreg;
    gboolean renew = TRUE, local_display_intregs = FALSE;

    enum VariableType {
	COL_START = 0,
	COL_END = 1,
	PRE_SHIFT_COL = 2,
	ROWSTART = 3,
	ROWEND = 4,
	PRESHIFTROW = 5,
	INT_SIZE_1 = 6,
	INT_SIZE_2 = 7,
	INT_SHIFT = 8
    } var_type;

/*     eval = gtk_object_get_data(GTK_OBJECT(widget), "eval"); */
/*
 * Locale vars
 */
/*     Display display_act; */
/*     display_act = gtk_object_get_data(widget, display_act); */

/*
 * Select which variable has to be modified
 */
    setby_spinbutton = 1;
    var_type = atoi(gtk_object_get_data(GTK_OBJECT(widget), "var_type"));

    if (var_type == COL_START) {
	piv_eval_par.col_start = gtk_spin_button_get_value_as_int(widget);

    } else if (var_type == COL_END) {
	piv_eval_par.col_end = gtk_spin_button_get_value_as_int(widget);

    } else if (var_type == PRE_SHIFT_COL) {
	piv_eval_par.pre_shift_col = gtk_spin_button_get_value_as_int(widget);

    } else if (var_type == ROWSTART) {
	piv_eval_par.row_start = gtk_spin_button_get_value_as_int(widget);

    } else if (var_type == ROWEND) {
	piv_eval_par.row_end = gtk_spin_button_get_value_as_int(widget);

    } else if (var_type == PRESHIFTROW) {
	piv_eval_par.pre_shift_row = gtk_spin_button_get_value_as_int(widget);

    } else if (var_type == INT_SIZE_1) {
	piv_eval_par.int_size_1 = gtk_spin_button_get_value_as_int(widget);
        GTK_ADJUSTMENT(eval->spinbutton_adj_intsize2)->lower = 
            (gfloat) piv_eval_par.int_size_1;
        GTK_ADJUSTMENT(eval->spinbutton_adj_intsize2)->upper = 
            (gfloat) 128.0;
        gtk_adjustment_changed (GTK_ADJUSTMENT(eval->spinbutton_adj_intsize2));
        gtk_adjustment_set_value(GTK_ADJUSTMENT(eval->spinbutton_adj_intsize2),
                                 (gfloat) piv_eval_par.int_size_1);

/*
 * Adjust radio buttons for interrogation size 1
 */
        adjust_radio_button_intsize_1(piv_eval_par, eval);

    } else if (var_type == INT_SIZE_2) {
        char message[GPIV_MAX_CHARS];

	piv_eval_par.int_size_2 = gtk_spin_button_get_value_as_int(widget);
        if ((piv_eval_par.int_scheme == GPIV_NO_CORR
             || piv_eval_par.int_scheme == GPIV_LK_WEIGHT)
            && piv_eval_par.int_size_2 >  piv_eval_par.int_size_1) {
            gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                        (eval->radiobutton_imgdeform),
                                        TRUE);
            g_snprintf(message, GPIV_MAX_CHARS, 
                       _("Int Size 2 > Int Size 1 \nSetting Interrogation scheme to Image deformation"));
            warning_gpiv(message);

/*         } else { */
/*             gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON */
/*                                         (eval->radiobutton_zerooff),  */
/*                                         FALSE); */
        }

/*
 * Adjust radio buttons for interrogation size 2
 */
        adjust_radio_button_intsize_2(piv_eval_par, eval);
        
    } else if (var_type == INT_SHIFT) {
	piv_eval_par.int_shift = gtk_spin_button_get_value_as_int(widget);

/*
 * Adjust radio buttons for interrogation area shift
 */
        adjust_radio_button_intshift(piv_eval_par, eval);
    }
    
    if (display_act != NULL
        && display_act->intreg.exist_int
        ) {
        nx = display_act->intreg.data.nx;
        ny = display_act->intreg.data.ny;
	if ((err_msg = 
             gpiv_piv_count_pivdata_fromimage(&local_intreg, image_par, 
                                              piv_eval_par))
            != NULL) {
            warning_gpiv ("on_spinbutton_piv_int: %s", err_msg);
            g_warning ("on_spinbutton_piv_int: %s", err_msg);
            return;
        }
/* BUGFIX: unstable if nx or ny = 0 or has been. */
/*         g_warning("on_spinbutton_piv_int:: BACK from gpiv_piv_count_pivdata_fromimage, nx=%d ny=%d", */
/*                   local_intreg.nx, local_intreg.ny); */

/*
 * For some operations the intregs have to be renewed unconditionally,
 * for others only in case if the number of intregs has been changed
 */
        if (var_type == COL_START || var_type == COL_END
            || var_type == ROWSTART || var_type == ROWEND) {
            if (nx == local_intreg.nx 
                && ny == local_intreg.ny) {
                renew = FALSE;
            }
        }
/*         gpiv_warning("on_spinbutton_piv_int:: 0"); */
        
        if (renew) {
            local_display_intregs = display_act->display_intregs;
            destroy_all_intregs(display_act);
            create_all_intregs(display_act);
            display_act->display_intregs = local_display_intregs;
            if (display_act->display_intregs) show_all_intregs(display_act);
        }

    }

    setby_spinbutton = 0;
}



void 
on_radiobutton_piv_int(GtkWidget * widget, 
                       gpointer data
                       )
/*-----------------------------------------------------------------------------
 */
{
    gint int_size_1_tmp;
    gboolean local_display_intregs = FALSE;
    gint nx_tmp = 0, ny_tmp = 0;
/*     gint setby_spinbutton = atoi(gtk_object_get_data(GTK_OBJECT(widget),  */
/*                                                 "setby_spinbutton")); */
    PivEval *eval = gtk_object_get_data(GTK_OBJECT(widget), "eval");
    Display *disp = display_act;
    int nx, ny;
    enum VariableType {
	SIZE_1 = 0,
	SIZE_2 = 1,
	SHIFT = 2
    } var_type;

    if (disp != NULL) {
        nx = disp->intreg.data.nx;
        ny = disp->intreg.data.ny;
    } else {
        nx = 0;
        ny = 0;
    }

    if (GTK_TOGGLE_BUTTON(widget)->active) {
	var_type = atoi(gtk_object_get_data(GTK_OBJECT(widget),
					    "var_type"));

/*
 * Select which variable has to be modified
 */
	if (var_type == SIZE_1) {
            int_size_1_tmp = piv_eval_par.int_size_1;
	    piv_eval_par.int_size_1 = atoi(gtk_object_get_data
                                           (GTK_OBJECT(widget),
						  "intsize1"));
/*
 * switch off temporarly display_intregs
 */
            if (disp != NULL) {
                local_display_intregs = disp->display_intregs;
                disp->display_intregs = FALSE;
            }
	    nx_tmp = nx;
	    ny_tmp = ny;
	    adjust_radiobutton_piv_int(eval, piv_eval_par.int_size_1);

/*
 * return display_intregs to original value
 */
            if (disp != NULL) {
                disp->display_intregs = local_display_intregs;
            }
	    nx = nx_tmp;
	    ny = ny_tmp;

/*
 * BUGFIX: REMOVE Reset weight and zero_off
 */
            if (piv_eval_par.int_size_2 > int_size_1_tmp) {
/*             gtk_widget_set_sensitive(eval->radiobutton_zerooff, TRUE); */
/*             gtk_widget_set_sensitive(checkbutton_piv_weightkernel, TRUE); */

/*
                if (int_scheme_tmp != GPIV_ZERO_OFF_FORWARD
                    && int_scheme_tmp != GPIV_ZERO_OFF_CENTRAL) {
                    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                (eval->radiobutton_zerooff), 
                                                FALSE);
                    gtk_widget_set_sensitive(eval->radiobutton_zerooff, TRUE);
                    gtk_widget_set_sensitive(eval->radiobutton_weightkernel, 
                                             TRUE);
                    if (int_scheme_tmp == GPIV_LK_WEIGHT) {
                        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                    (eval->radiobutton_weightkernel), 
                                                    FALSE);
                        gtk_widget_set_sensitive(eval->radiobutton_zerooff, 
                                                 TRUE);
                        gtk_widget_set_sensitive(eval->radiobutton_weightkernel, TRUE);
                    } else {
                        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                    (eval->radiobutton_weightkernel), 
                                                    TRUE);
                        gtk_widget_set_sensitive(eval->radiobutton_zerooff, 
                                                 FALSE);
                        gtk_widget_set_sensitive(eval->radiobutton_weightkernel, 
                                                 TRUE);
                    }
                } else {
                    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                (eval->radiobutton_zerooff), 
                                                TRUE);
                    gtk_widget_set_sensitive(eval->radiobutton_zerooff, TRUE);
                    gtk_widget_set_sensitive(eval->radiobutton_weightkernel, 
                                             FALSE);
                }
*/
            }

/*
 * Also calls on_spinbutton_piv_int, which, on its turn, may call this function
 * again resulting into an infinite loop
*/
            gtk_spin_button_set_value(GTK_SPIN_BUTTON
                                      (eval->spinbutton_intsize1), 
                                      piv_eval_par.int_size_1);


	} else if (var_type == SIZE_2) {
	    piv_eval_par.int_size_2 = atoi(gtk_object_get_data(GTK_OBJECT(widget),
						  "intsize2"));

/*
 * Adjust spinner
 */
            gtk_adjustment_set_value(GTK_ADJUSTMENT
                                     (eval->spinbutton_adj_intsize2),
                                     (gfloat) piv_eval_par.int_size_2);
/*
 * Save origanal settings of weight and zero_off for later resetting
 */
            if (piv_eval_par.int_size_2 >  piv_eval_par.int_size_1) {
                int_scheme_tmp = piv_eval_par.int_scheme;
/*
                gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                            (eval->radiobutton_weightkernel), FALSE);
                gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                            (eval->radiobutton_zerooff), TRUE);
                gtk_widget_set_sensitive(eval->radiobutton_zerooff, FALSE);
*/

/*
 * Reset weight and zero_off
 */
            } else {
/*                if (zero_off_tmp == 0) {
                    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                (eval->radiobutton_zerooff), FALSE);
                    if (weight_tmp == 0) {
                        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                    (eval->radiobutton_weightkernel), FALSE);
                    } else {
                        gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                    (eval->radiobutton_weightkernel), TRUE);
                    }
                } else {
                    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
                                                (eval->radiobutton_zerooff), TRUE);
                }
	    gtk_widget_set_sensitive(eval->radiobutton_zerooff, TRUE);
*/
            }


	} else if (var_type == SHIFT) {
	    piv_eval_par.int_shift = atoi(gtk_object_get_data(GTK_OBJECT(widget),
						 "intshift"));
/*
 * Adjust spinner
 */
            gtk_spin_button_set_value(GTK_SPIN_BUTTON
                                      (eval->spinbutton_intshift), 
                                      piv_eval_par.int_shift);
	}

	if (disp != NULL 
            && display_act->intreg.exist_int
            ) {
            local_display_intregs = disp->display_intregs;
            destroy_all_intregs(display_act);
            create_all_intregs(display_act);
            disp->display_intregs = local_display_intregs;
            if (disp->display_intregs) show_all_intregs(disp);
        }

    }

}



void 
on_radiobutton_fit_enter(GtkWidget *widget, 
                         gpointer data
                         )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Interpolation scheme for sub-pixel estimation");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);

}



void 
on_radiobutton_peak_enter(GtkWidget *widget, 
                                        gpointer data
                                        )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Chooses n-th highest top number of correlation as estimator");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);

}



void 
on_radiobutton_interrogatescheme_enter(GtkWidget *widget, 
                                        gpointer data
                                        )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Bias correction scheme");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);

}



void 
on_radiobutton_interrogatescheme_imgdeform_enter(GtkWidget *widget, 
                                        gpointer data
                                        )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole *gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Deforms interrogation area's from previous PIV iteration");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);

}



void 
on_toggle_piv(GtkWidget * widget, 
              gpointer data
              )
/*-----------------------------------------------------------------------------
 */
{
    PivEval * eval = gtk_object_get_data(GTK_OBJECT(widget), "eval");
    enum VariableType {
	IFIT,
	PEAK_NR,
	SCHEME,
	CORRELATION
    } var_type;

/*
 * Select which variable has to be modified
 */
    var_type = atoi(gtk_object_get_data(GTK_OBJECT(widget), "var_type"));

    if (var_type == IFIT) {
	piv_eval_par.ifit = atoi(gtk_object_get_data(GTK_OBJECT(widget), 
                                                     "ifit"));

    } else if (var_type == PEAK_NR) {
	piv_eval_par.peak = atoi(gtk_object_get_data(GTK_OBJECT(widget), 
                                                     "peak"));


    } else if (var_type == SCHEME) {
        piv_eval_par.int_scheme = atoi(gtk_object_get_data(GTK_OBJECT(widget), 
                                                     "scheme"));

    } else if (var_type == CORRELATION) {
	image_par.x_corr = atoi(gtk_object_get_data(GTK_OBJECT(widget), 
                                                    "x_corr"));
    }

}



void 
adjust_radiobutton_piv_int(PivEval * eval,
                           int int_size_1
                           )
/* ----------------------------------------------------------------------------
 * adjusting and disabling int_size_2 and its toggle buttons
 */
{
/*     if (int_size_1 == 8) { */
/* 	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON */
/* 				    (eval->radiobutton_intsize2_1), TRUE); */
    /* } else  */if (int_size_1 == 16) {
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
				    (eval->radiobutton_intsize2_2), TRUE);
    } else if (int_size_1 == 32) {
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
				    (eval->radiobutton_intsize2_3), TRUE);
    } else if (int_size_1 == 64) {
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
				    (eval->radiobutton_intsize2_4), TRUE);
    } else if (int_size_1 >= 128) {
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON
				    (eval->radiobutton_intsize2_5), TRUE);
    }


    if (int_size_1 <= 64) {
	gtk_widget_set_sensitive(GTK_WIDGET(eval->radiobutton_intsize2_4),
				 TRUE);
	if (int_size_1 <= 32) {
	    gtk_widget_set_sensitive(GTK_WIDGET
				     (eval->radiobutton_intsize2_3), 
                                     TRUE);
	    if (int_size_1 <= 16) {
		gtk_widget_set_sensitive(GTK_WIDGET
					 (eval->radiobutton_intsize2_2),
					 TRUE);
/* 		if (int_size_1 <= 8) { */
/* 		    gtk_widget_set_sensitive(GTK_WIDGET */
/* 					     (eval->radiobutton_intsize2_1), */
/* 					     TRUE); */
/* 		} */
	    }
	}
    }



    if (int_size_1 >= 16) {
/* 	gtk_widget_set_sensitive(GTK_WIDGET(eval->radiobutton_intsize2_1), */
/* 				 FALSE); */
	if (int_size_1 >= 32) {
	    gtk_widget_set_sensitive(GTK_WIDGET
				     (eval->radiobutton_intsize2_2), 
                                     FALSE);
	    if (int_size_1 >= 64) {
		gtk_widget_set_sensitive(GTK_WIDGET
					 (eval->radiobutton_intsize2_3),
					 FALSE);
		if (int_size_1 >= 128) {
		    gtk_widget_set_sensitive(GTK_WIDGET
					     (eval->radiobutton_intsize2_4),
					     FALSE);
		}
	    }
	}
    }

}



void 
on_checkbutton_piv_monitor_enter(GtkWidget *widget, 
                                 gpointer data
                                 )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg =
	_("Displays subimages, correlation function and PIV vector");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void on_checkbutton_piv_monitor(GtkWidget * widget, gpointer data)
{
    PivEval * eval = gtk_object_get_data(GTK_OBJECT(widget), "eval");


    if (GTK_TOGGLE_BUTTON(widget)->active) {
	piv_eval_par.print_cov = 1;

        eval->mon.gci_vec_background = 
            gnome_canvas_item_new(gnome_canvas_root
                                  (GNOME_CANVAS(eval->canvas_monitor_vec)),
			       gnome_canvas_rect_get_type(),
			       "x1", (double) 0,
			       "y1", (double) 0,
			       "x2", (double) GPIV_MAX_INTERR_SIZE,
			       "y2", (double) GPIV_MAX_INTERR_SIZE,
			        "fill_color", "darkblue",
			       "width_units", 1.0, 
			       NULL);


        eval->mon.gci_int1_background = 
            gnome_canvas_item_new(gnome_canvas_root
                                  (GNOME_CANVAS(eval->canvas_monitor_int1)),
			       gnome_canvas_rect_get_type(),
			       "x1", (double) 0,
			       "y1", (double) 0,
			       "x2", (double) GPIV_MAX_INTERR_SIZE,
			       "y2", (double) GPIV_MAX_INTERR_SIZE,
			        "fill_color", "darkgreen",
			       "width_units", 1.0, 
			       NULL);


        eval->mon.gci_int2_background = 
            gnome_canvas_item_new(gnome_canvas_root
                                  (GNOME_CANVAS(eval->canvas_monitor_int2)),
			       gnome_canvas_rect_get_type(),
			       "x1", (double) 0,
			       "y1", (double) 0,
			       "x2", (double) GPIV_MAX_INTERR_SIZE,
			       "y2", (double) GPIV_MAX_INTERR_SIZE,
			        "fill_color", "darkgreen",
			       "width_units", 1.0, 
			       NULL);


        eval->mon.gci_background_cov = 
            gnome_canvas_item_new(gnome_canvas_root
                                  (GNOME_CANVAS(eval->canvas_monitor_cov)),
			       gnome_canvas_rect_get_type(),
			       "x1", (double) 0,
			       "y1", (double) 0,
			       "x2", (double) GPIV_MAX_INTERR_SIZE,
			       "y2", (double) GPIV_MAX_INTERR_SIZE,
			        "fill_color", "darkred",
			       "width_units", 1.0, 
			       NULL);
    } else {
	piv_eval_par.print_cov = 0;
/*
 * Destroy intreg1 image
 */
        if (eval->mon.gci_int1_background != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.
                                          gci_int1_background));
            eval->mon.gci_int1_background = NULL;
        }


        if (eval->mon.rgbbuf_int1 != NULL) {
            g_free(eval->mon.rgbbuf_int1);
            eval->mon.rgbbuf_int1 = NULL;
        }

        if (eval->mon.gci_int1 != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.gci_int1));
            eval->mon.gci_int1 = NULL;
        }

/*
 * Destroy intreg2 image
 */
        if (eval->mon.gci_int2_background != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.
                                          gci_int2_background));
            eval->mon.gci_int2_background = NULL;
        }


        if (eval->mon.rgbbuf_int2 != NULL) {
            g_free(eval->mon.rgbbuf_int2);
            eval->mon.rgbbuf_int2 = NULL;
        }

        if (eval->mon.gci_int2 != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.gci_int2));
            eval->mon.gci_int2 = NULL;
        }

/*
 * Destroy correlation image
 */
        if (eval->mon.gci_background_cov != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.gci_background_cov));
            eval->mon.gci_background_cov = NULL;
        }


        if (eval->mon.rgbbuf_cov != NULL) {
            g_free(eval->mon.rgbbuf_cov);
            eval->mon.rgbbuf_cov = NULL;
        }

        if (eval->mon.gci_cov != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.gci_cov));
            eval->mon.gci_cov = NULL;
        }

/*
 * Destroy vector
 */
        if (eval->mon.gci_vec_background != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.gci_vec_background));
            eval->mon.gci_vec_background = NULL;
        }


        if (eval->mon.gci_vec != NULL) {
            gtk_object_destroy(GTK_OBJECT(eval->mon.gci_vec));
            eval->mon.gci_vec = NULL;
        }
    }

}



void
on_spinbutton_piv_monitor_zoom(GtkSpinButton *widget, 
                               gpointer data
                               )
/*-----------------------------------------------------------------------------
 */
{
    PivEval * eval = gtk_object_get_data(GTK_OBJECT(widget), "eval");
/* gfloat piv_disproc_zoom */

    gpiv_var.piv_disproc_zoom = gtk_spin_button_get_value_as_float(widget);
    gnome_config_push_prefix("/gpiv/RuntimeVariables/");
    gnome_config_set_float("zoom_factor", gpiv_var.piv_disproc_zoom);
    gnome_config_pop_prefix();
    gnome_config_sync();

    eval->mon.affine[0] = gpiv_var.piv_disproc_zoom;
    eval->mon.affine[3] = gpiv_var.piv_disproc_zoom;
    eval->mon.affine[4] = 
        (GPIV_MAX_INTERR_SIZE - gpiv_var.piv_disproc_zoom 
        * piv_eval_par.int_size_1) /2.0;
    eval->mon.affine[5] = 
        (GPIV_MAX_INTERR_SIZE - gpiv_var.piv_disproc_zoom 
         * piv_eval_par.int_size_1) /2.0;

    if(eval->mon.gci_int1 != NULL) {
        gnome_canvas_item_affine_absolute(eval->mon.gci_int1,
                                          eval->mon.affine);
    }

    if(eval->mon.gci_int2 != NULL) {
        gnome_canvas_item_affine_absolute(eval->mon.gci_int2,
                                          eval->mon.affine);
    }

 
    if(eval->mon.gci_cov != NULL) {
        gnome_canvas_item_affine_absolute(eval->mon.gci_cov,
                                          eval->mon.affine);
    }

 
}



void
on_spinbutton_piv_monitor_vectorscale(GtkSpinButton *widget, 
                                      gpointer data
                                      )
/*-----------------------------------------------------------------------------
 */
{
    PivEval * eval = gtk_object_get_data(GTK_OBJECT(widget), "eval");
/* gint piv_disproc_vlength */

    gpiv_var.piv_disproc_vlength = gtk_spin_button_get_value_as_int(widget);
    gnome_config_set_int("vector_length", gpiv_var.piv_disproc_vlength);
    gnome_config_sync();

/*
 * Try to re-scale vector
*/
    if(eval->mon.gci_vec != NULL) {
        GnomeCanvasPoints *points = gnome_canvas_points_new(2);
        
        points->coords[0] = GPIV_MAX_INTERR_SIZE / 2;
        points->coords[1] = GPIV_MAX_INTERR_SIZE / 2;
        points->coords[2] = GPIV_MAX_INTERR_SIZE / 2 + eval->mon.pi_da.dx[0][0] * 
            gpiv_var.piv_disproc_vlength;
        points->coords[3] =  GPIV_MAX_INTERR_SIZE / 2 + eval->mon.pi_da.dy[0][0] * 
            gpiv_var.piv_disproc_vlength;
        
        gnome_canvas_item_set(eval->mon.gci_vec,
                              "points", points,
                              NULL);
        
        gnome_canvas_points_free(points);
    }
}



void 
on_button_piv_enter(GtkWidget *widget, 
                    gpointer data
                    )
/*-----------------------------------------------------------------------------
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gchar *msg = _("Analyses a PIV image (pair)");
    gnome_appbar_set_status(GNOME_APPBAR(gpiv->appbar), msg);
}



void 
on_button_piv(GtkWidget * widget, 
              gpointer data
              )
/* ----------------------------------------------------------------------------
 * The actual calculation of particle image displacements
 */
{
    GpivConsole * gpiv = gtk_object_get_data(GTK_OBJECT(widget), "gpiv");
    gint row, ibuf;

    cancel_process = FALSE;

    if (nbufs > 0) {
        for (row = gpiv->first_selected_row; row <= gpiv->last_selected_row; 
             row++) {
            display_act = gtk_clist_get_row_data(GTK_CLIST(gpiv->clist_buf), 
                                                 row);
            ibuf = display_act->id;
            if (display[ibuf] != NULL 
                && display_act->mwin != NULL 
                && GTK_WIDGET_VISIBLE(GTK_WIDGET(display_act->mwin)) ) { 
                gdk_window_show(GTK_WIDGET(display_act->mwin)->window);
                gdk_window_raise(GTK_WIDGET(display_act->mwin)->window);
            }
            
            exec_piv(gpiv);
        }
    }
}



