#!/usr/bin/env python
# GPixPod - organize photos on your iPod, freely!
# Copyright (C) 2006 Flavio Gargiulo (FLAGAR.com)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

from glob import glob
from threading import Thread
import gettext
import gtk, sys, os, mh, optparse, cPickle, time
import signal
from utils import *
gettext.bindtextdomain('gpixpod', 'po')
gettext.textdomain('gpixpod')
gettext.install('gpixpod', 'po', True)

class GPixPodCLI:
    """ GPixPod command line interface class """
    
    def __init__(self, verbose, ipod, model, album, recursive, fullres, behaviour, autorotate, force, args):
        """ Initialize class getting command line options and arguments """
        try:
            homedir = os.path.expanduser('~')
            prefsdir = os.path.join(homedir, '.gpixpod')
            if not os.path.isdir(prefsdir):
                os.mkdir(prefsdir)
            prefsfile = open(os.path.join(prefsdir, 'config'))
            self.prefs = cPickle.load(prefsfile)
            self.thumbscache = None
            if 'path_thumbscache' in self.prefs:
                self.thumbscache = self.prefs['path_thumbscache']
            if 'ipod_model' in self.prefs:
                self.ipodmodel = self.prefs['ipod_model']
            prefsfile.close()
        except:
            self.thumbscache = None
            self.ipodmodel = None
        if model != None:
            self.ipodmodel = model
        if not force:
            try:
                syncfile = open(os.path.join(os.path.expanduser('~'), '.gpixpod', 'sync'))
                self.sync = list(cPickle.load(syncfile))
                syncfile.close()
            except:
                self.sync = []
        self.tosave = False
        self.verbose = verbose
        self.ipod_mountpoint = ipod
        self.album_name = album
        self.album_id = 0
        self.recursive = recursive
        self.fullres = fullres
        self.behaviour = behaviour
        self.autorotate = autorotate
        self.skipsync = force
        self.filenames = args
        self.continue_processing = True

    def Open(self):
        """ Open the Photo Database and begin the process """
        dbfilename = os.path.join(self.ipod_mountpoint, 'Photos', 'Photo Database')
        if self.verbose:
            sys.stdout.write(_("Loading Photo Database '%s'... ") % dbfilename)
            sys.stdout.flush()
        self.db = mh.MH(dbfilename, self.thumbscache, ipodmodel=self.ipodmodel)
        if self.verbose:
            sys.stdout.write("loaded.\n")
            if self.ipodmodel in ('5G', 'Photo', 'Color', 'Nano'):
                print _("Selected iPod model is: %s") % self.ipodmodel
            else:
                print _("Autodetected iPod model is: %s") % self.db.ipod_model
        albums = self.db.Albums()
        if self.verbose:
            print _("Photo Database contains %s albums") % len(albums[1:])
            for album in albums[1:]:
                print _("- '%s', containing %s pictures") % (album.name, len(album.pics))
        for x in range(len(albums)):
            if albums[x].name == self.album_name:
                self.album_id = x
        if self.album_id == 0:
            if self.verbose:
                sys.stdout.write(_("Photo album '%s' does not exist, creating it... ") % self.album_name)
                sys.stdout.flush()
            self.album_id = len(albums)
            self.db.AddAlbum(self.album_name)
            self.tosave = True
            if self.verbose:
                print _("created (n. %s).") % self.album_id
        if self.verbose:
            if self.fullres:
                print _("Full resolution image will be regularly copied")
            else:
                print _("Full resolution image will NOT be copied")
            if self.autorotate:
                autorotate_methods = {'CW':_('clockwise'), 'CCW':_('counter-clockwise')}
                print _("Detected portrait images will be automatically rotated %s") % autorotate_methods[self.autorotate]
            print _("Image uploading behaviour is: %s") % self.behaviour
            sys.stdout.write(_("Adding pictures to the photo album '%s'") % self.album_name)

    def Process(self):
        """ Process images addition """
        if self.verbose:
            sys.stdout.write("\n")
        sys.stdout.write(_("Press Ctrl-c to stop adding images"))
        sys.stdout.flush()
        fls = expand_paths(self.filenames, self.recursive)
        x = 0
        fls_len = len(fls)
        while x < fls_len:
            if not self.continue_processing:
                break
            if gtk.gdk.pixbuf_get_file_info(fls[x]):
                if not self.skipsync and fls[x] in self.sync:
                    if self.verbose:
                        sys.stdout.write(_("\nPhoto '%s' is already stored in Photo Database.") % fls[x])
                else:
                    if self.verbose:
                        sys.stdout.write(_("\nAdding photo '%s'... ") % fls[x])
                        sys.stdout.flush()
                    addp = Thread(target=self.db.AddPhoto, args=[fls[x], self.album_id, self.fullres, self.behaviour, self.autorotate])
                    addp.start()
                    addp.join()
                    #self.db.AddPhoto(fls[x], self.album_id, self.fullres, self.behaviour, self.autorotate)
                    self.tosave = True
                    if not self.skipsync:
                        self.sync.append(fls[x])
                    if self.verbose:
                        sys.stdout.write(_("added (%s of %s - %s%%).") % (x+1, len(fls), 100*(x+1)/len(fls)))
            x += 1

    def Save(self):
        """ Save the Photo Database and terminate the process """
        sys.stdout.write("\n")
        if self.tosave:
            if self.verbose:
                sys.stdout.write(_("Saving Photo Database...     "))
                sys.stdout.flush()
            self.db.progress = 0.0
            thsave = Thread(target=self.db.Save)
            thsave.start()  # Actually saving the new-generated Photo Database file
            if self.verbose:
                while thsave.isAlive():
                    progress = str(int(self.db.progress*100)).rjust(4)
                    sys.stdout.write('\b\b\b\b\b%s%%' % progress)
                    #time.sleep(0.01)
                print _("\b\b\b\bsaved.")
        else:
            if self.verbose:
                print _("\nNo changes done. Nothing to save!")
        if not self.skipsync:
            if self.verbose:
                sys.stdout.write(_("Saving synchronization status... "))
            syncfile = open(os.path.join(os.path.expanduser('~'), '.gpixpod', 'sync'), 'w+')
            cPickle.dump(set(self.sync), syncfile)
            syncfile.close()
            if self.verbose:
                print _("saved.")
        if self.verbose:
            print _("All done.")

    def Stop(self):
        """ Stop the process, saving at the reached point """
        sys.stdout.write("\n%s" % _("Interruption requested. Please wait while saving at the current status..."))
        sys.stdout.flush()
        self.continue_processing = False

    def Kill(self, signal_no, stack):
        """ Kill the program saving the Photo Database """
        self.Stop()


def option_parse():
    parser = optparse.OptionParser()
    parser.add_option('-i', '--ipod', dest='ipod', help=_('The mountpoint of your connected iPod'))
    parser.add_option('-m', '--model', dest='model', default=None, help=_('Manually provide iPod model (Nano|Photo|Color|5G) avoiding autodetection'))
    parser.add_option('-f', '--force', dest='force', action='store_true', help=_('Skip synchronization and duplicate checking'))
    parser.add_option('-q', '--quiet', dest='verbose', default=True, action='store_false', help=_('Do not display detailed output'))
    parser.add_option('-a', '--album', dest='album', default='New Photo Album', help=_('Name of the photo album to use. It will be created if absent.'))
    parser.add_option('-r', '--recursive', dest='recursive', action='store_true', help=_('Add images from directories recursively'))
    parser.add_option('-t', '--no-fullresolution', dest='fullres', default=True, action='store_false', help=_('Do not copy full resolution image'))
    parser.add_option('-z', '--zoom', dest='zoom', action='store_true', help=_('Zoom images instead of fit'))
    parser.add_option('-s', '--stretch', dest='stretch', action='store_true', help=_('Stretch images instead of fit'))
    parser.add_option('-j', '--rotate-cw', dest='rotatecw', action='store_true', help=_('Rotate automatically portrait images clockwise'))
    parser.add_option('-k', '--rotate-ccw', dest='rotateccw', action='store_true', help=_('Rotate automatically portrait images counter-clockwise'))
    opts, args = parser.parse_args()
    if opts.ipod and os.path.isdir(opts.ipod):
        behaviour = 'Fit'
        if opts.stretch:
            behaviour = 'Stretch'
        elif opts.zoom:
            behaviour = 'Zoom'
        autorotate = None
        if opts.rotatecw:
            autorotate = 'CW'
        elif opts.rotateccw:
            autorotate = 'CCW'
        gpixpodcli = GPixPodCLI(opts.verbose, opts.ipod, opts.model, opts.album, opts.recursive, opts.fullres, behaviour, autorotate, opts.force, args)
        signal.signal(signal.SIGTERM, gpixpodcli.Kill)
        gpixpodcli.Open()
        try:
            gpixpodcli.Process()
        except KeyboardInterrupt:
            gpixpodcli.Stop()
        gpixpodcli.Save()
    else:
        print _("Invalid iPod mountpoint")
        print parser.get_usage()


if __name__ == '__main__':
    option_parse()
