/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satmap-multi-aos.c:  Helper functions accompaning satmap.c

  Copyright (C)  2002-2005   Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-window-icon.h>
#include <math.h>
#include "defaults.h"
#include "qth.h"
#include "satlog.h"
#include "util.h"
#include "fileio.h"
#include "satdata.h"
#include "satplot.h"
#include "satprint.h"
#include "sataos.h"   /**** FIXME: should be included here! */
#include "satmap-single-aos.h"
#include "satmap-multi-aos.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


extern GtkWidget *app;
extern qth_struc qth;    /* defined in qth.c */
extern GConfClient *client;  /* main.c */


/* Private function prototypes */
static void satmap_multi_aos_details_cb (GtkWidget *, gpointer);
static void satmap_multi_aos_print_cb   (GtkWidget *, gpointer);
static void satmap_multi_aos_save_cb    (GtkWidget *, gpointer);
static void satmap_multi_aos_save_save  (GtkButton *, gpointer);
static void satmap_multi_aos_close_cb   (GtkWidget *, gpointer);


void satmap_multi_aos_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the
	   AOS/LOS button.
	*/
	GnomeCanvasItem *item;
	gint index;
	guint n,i,row2, minelev;
	sat_t *sat;
	gchar **vbuff,*buff,*titles[3] = { N_(" AOS"), N_(" LOS"), N_("Duration (mn)") };
	GtkWidget *swin,*list,*dialog;

	gchar **slist;

	/* The right footer canvas item contains info about the selection */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "sati"));
	index = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (item), "index"));

	if (index != -1) {
		/* get satellite */
		sat = satdata_get_sat (index);

		switch ( sat->status ) {

		case SAT_STATUS_DECAYED:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite has decayed (no AOS)."));
			gnome_app_message (GNOME_APP (app), _("This satellite has DECAYED!"));
			break;

		case SAT_STATUS_GEOSTAT:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite is geostationary."));
			gnome_app_message (GNOME_APP (app), _("This satellite is GEOSTATIONARY!"));
			break;

		case SAT_STATUS_NOAOS:
			satlog_log (SAT_LOG_MESSAGE, _("Selected satellite has no AOS."));
			gnome_app_message (GNOME_APP (app), _("This satellite NEVER reaches AOS!"));
			break;

		default:
			/* how many passes to predict? */
			n = gconf_client_get_int (client, AOS_SIMPLE_NUM_PATH, NULL) ?
				gconf_client_get_int (client, AOS_SIMPLE_NUM_PATH, NULL) :
					AOS_SIMPLE_DEF_NUM;

			/* minimum elevation */
			minelev = gconf_client_get_int (client, AOS_MIN_ELEV_PATH, NULL) ? 
				gconf_client_get_int (client, AOS_MIN_ELEV_PATH, NULL) : 0;

			slist = aos_simple_list_new ( *sat, n, minelev, TFORM_AOS);

			/* create a clist widget */
			list = gtk_clist_new_with_titles ( 3, titles );
			gtk_clist_column_titles_passive ( GTK_CLIST(list) );

			/* Add AOS/LOS time to clist */
			for ( i=0; i<n; i++ ) {
				vbuff = g_strsplit ( slist[i], ";", 4 );
				row2 = gtk_clist_append (GTK_CLIST (list), vbuff );
				gtk_clist_set_row_data (GTK_CLIST (list), row2, g_strdup (vbuff[3]));
				g_strfreev (vbuff);
			}

			/* try to fix the height and width ... */
			gtk_object_set ( GTK_OBJECT(list), "height", 200,
					 "width",
					 gtk_clist_optimal_column_width (GTK_CLIST (list),0) +
					 gtk_clist_optimal_column_width (GTK_CLIST (list),1) +
					 gtk_clist_optimal_column_width (GTK_CLIST (list),2) + 50,
					 NULL);

			/* adjust the width of the columns */
			gtk_clist_set_column_width ( GTK_CLIST(list), 0,
						     gtk_clist_optimal_column_width (GTK_CLIST (list),0) + 20 );
			gtk_clist_set_column_resizeable (GTK_CLIST (list), 0, TRUE );
			gtk_clist_set_column_width (GTK_CLIST (list), 1,
						    gtk_clist_optimal_column_width (GTK_CLIST (list), 1) + 20 );
			gtk_clist_set_column_resizeable (GTK_CLIST (list), 1, TRUE);
			gtk_clist_set_column_width (GTK_CLIST (list), 2,
						    gtk_clist_optimal_column_width (GTK_CLIST (list), 2) + 10 );
			gtk_clist_set_column_resizeable (GTK_CLIST (list), 2, TRUE);

			swin = gtk_scrolled_window_new ( NULL, NULL );
			gtk_scrolled_window_set_policy ( GTK_SCROLLED_WINDOW (swin),
							 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
			gtk_container_add (GTK_CONTAINER (swin), list );


			/* create the dialog window */
			buff = g_strdup_printf (_("Upcoming passes for %s (EL: %d)"), sat->name, minelev);
			dialog = gnome_dialog_new (buff, NULL);
			g_free (buff);

			/* window icon */
			buff = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
					    G_DIR_SEPARATOR_S, "stock_timer.png", NULL);
			gnome_window_icon_set_from_file (GTK_WINDOW (dialog), buff);
			g_free (buff);

			/* This is hack, I know, but otherwise some of the dialogs popping up
			   from thius one will not be able to come to the foreground!
			*/
			gtk_object_set (GTK_OBJECT (dialog), "type", GTK_WINDOW_TOPLEVEL, NULL);

			/* attach data */
			gtk_object_set_data (GTK_OBJECT (dialog), "clist", list);
			gtk_object_set_data (GTK_OBJECT (dialog), "sat", sat);

			/* Dialog buttons */
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Save"),
								GNOME_STOCK_PIXMAP_SAVE_AS);
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Print"),
								GNOME_STOCK_PIXMAP_PRINT);
			gnome_dialog_append_button_with_pixmap (GNOME_DIALOG (dialog),
								_("Details"),
								GNOME_STOCK_PIXMAP_PROPERTIES);
			gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);

			/* connect signals */
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 0,
						     GTK_SIGNAL_FUNC (satmap_multi_aos_save_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 1,
						     GTK_SIGNAL_FUNC (satmap_multi_aos_print_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 2,
						     GTK_SIGNAL_FUNC (satmap_multi_aos_details_cb),
						     dialog);
			gnome_dialog_button_connect (GNOME_DIALOG (dialog), 3,
						     GTK_SIGNAL_FUNC (satmap_multi_aos_close_cb),
						     dialog);
			
			/* "Close" is the default button */
			gnome_dialog_set_default (GNOME_DIALOG (dialog), 3);
			
			/* This will (?) automatically free the memory occupied by slist, when
			   the dialog is closed. */
			gtk_object_set_data_full (GTK_OBJECT (dialog), "list", slist, free_vector_notify_cb);

			/* Also add satellite index - needed for list title when printing from
			   several open dialogs. */
			gtk_object_set_data (GTK_OBJECT (dialog), "index", GINT_TO_POINTER (index));
			gtk_object_set_data (GTK_OBJECT (dialog), "size", GUINT_TO_POINTER (n));
			gtk_object_set_data (GTK_OBJECT (dialog), "satname", sat->name);

			gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin,
					    TRUE, TRUE , 0);
			gtk_widget_show_all (dialog);

			break; /* well... */
		}


	}
	else
		gnome_app_error (GNOME_APP (app), _("Please select a satellite!"));
	
}



static void
satmap_multi_aos_save_cb (GtkWidget *widget, gpointer data)
{
	/* This function is called when the user clicks on the
	   "save predicted passes" button. It prompts for a
	   filename and, if user presses OK, calls a function
	   which in turn will call the general data file saving
	   routine, whith the correct parameters. If the user
	   presses cancel, nothing will happen.
	*/
	GtkWidget *filesel;

	filesel = gtk_file_selection_new (_("Select a filename"));

	/* attach satellite data to file selector, so that it can be passed to the save_save function */
	gtk_object_set_data (GTK_OBJECT (filesel), "index", gtk_object_get_data (data, "index"));
	gtk_object_set_data (GTK_OBJECT (filesel), "size", gtk_object_get_data (data, "size"));
	gtk_object_set_data (GTK_OBJECT (filesel), "satname", gtk_object_get_data (data, "satname"));
	gtk_object_set_data (GTK_OBJECT (filesel), "orbit", gtk_object_get_data (data, "orbit"));
	gtk_object_set_data (GTK_OBJECT (filesel), "list", gtk_object_get_data (data, "list"));

	/* set default path if exists */
	if ( gconf_client_get_string (client, AOS_SIMPLE_SAVE_PATH, NULL) )
		gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
						 gconf_client_get_string (client, AOS_SIMPLE_SAVE_PATH, NULL) );
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_multi_aos_save_save), (gpointer) filesel);
                             
	/* Ensure that the dialog box is destroyed when the user clicks a button. */
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->ok_button),
				   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   (gpointer) filesel);
	gtk_signal_connect_object (GTK_OBJECT (GTK_FILE_SELECTION (filesel)->cancel_button),
				   "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   (gpointer) filesel);

	gtk_widget_show (filesel);
}


static void
satmap_multi_aos_save_save (GtkButton *button, gpointer data)
{
	/* This function is called by the above function, and
	   it calls the general data file saving function in
	   fileio.c . The size should contain the number of
	   entries in the vector.
	*/
	gchar *fname,**slist,*header;
	guint n=0;

	/* get the data to be saved */
	slist = gtk_object_get_data (GTK_OBJECT (data), "list");
	n = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "size"));
	fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (data));
	if ( !g_file_test (fname, G_FILE_TEST_ISDIR) ) {
		/* create header */
		header = g_strdup_printf (_("Upcoming %d passes for %s at\n"\
					    "%s %.2f%c %.2f%c\n"\
					    "--------------------------------------------------------\n"\
					    " AOS                          LOS\n"\
					    "--------------------------------------------------------\n"),
					    n, (gchar *)gtk_object_get_data (GTK_OBJECT (data), "satname"),
					  qth.name,
					  fabs (qth.lat), (qth.lat < 0) ? 'S' : 'N',
					  fabs (qth.lon), (qth.lon < 0) ? 'E' : 'W');

		fileio_save_vector (fname, header, NULL, slist, 2, n, 0, ";", "   ");
		g_free (header);

		/* save path */
		gconf_client_set_string (client, AOS_SIMPLE_SAVE_PATH, fname, NULL);
	}
}


static void
satmap_multi_aos_close_cb (GtkWidget *widget, gpointer data)
{
	/* This function is called when the user clicks on
	   the "Close" button in the AOS/LOS predictions
	   dialog (verbose). It frees allocated vector and
	   closes the dialog.
	*/

	gnome_dialog_close (GNOME_DIALOG (data));
}


static void
satmap_multi_aos_details_cb (GtkWidget *widget, gpointer dialog)
{
	GList *selection;
	GtkCList *clist;
	gint row;
	gdouble aos;

	clist = GTK_CLIST (gtk_object_get_data (GTK_OBJECT (dialog), "clist"));
	selection = clist->selection;
	if (selection) {
		row = GPOINTER_TO_INT (selection->data);
		/* AOS time is attached to the row */
		aos = g_strtod ((gchar *)gtk_clist_get_row_data (clist, row), NULL);
		satmap_single_aos_from_multi (gtk_object_get_data (GTK_OBJECT (dialog), "sat"), aos);
	}	
}


static void
satmap_multi_aos_print_cb (GtkWidget *widget, gpointer data)
{
	/* This function is called when the user clicks on
	   the "Print" button in the AOS/LOS predictions
	   dialog. The function calls the sat_print_sat_aos
	   function in satprint.c
	   "data" is a pointer to the dialog widget, which
	   has pointers to the textlist and satellite index.
	*/
	guint index,n=0;
	sat_t *sat;
	gchar **slist;

	/* count lines in the list */
	slist = gtk_object_get_data (GTK_OBJECT (data), "list");;
	n = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "size") );
	/* get the satellite */
	index = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (data), "index") );
	sat = satdata_get_sat (index);

	sat_print_sat_aos (sat, slist, n);
}
