/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satpref_qth.c:  Gnome Predict configuration module (locations).

  Copyright (C)  2001-2005  Alexandru Csete.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include "satpref.h"
#include "satpref-qth.h"
#include "qth.h"
#include "sat-ui-utils.h"
#include "satlog.h"
#include "satloc-tree.h"
#include "satconfig-locations.h"


#if defined (HAVE_HAMLIB)
#  include <hamlib/rotator.h>
static void satpref_qth_latlon_cb         (GtkAdjustment *adj, gpointer data);
static void satpref_qth_latlon_toggled_cb (GtkToggleButton *button, gpointer locbut);
static void satpref_qth_qra_changed_cb    (GtkWidget *qra, gpointer data);
#endif

static GtkWidget *latsp,*lonsp;
static GtkWidget *qra_loc;                             /* QRA locator */
static GtkWidget *radio1,*radio2;
static GtkWidget *qth_loc,*altsp;        /* location, lat, lon and alt entries */
static GtkWidget *wxsta;                               /* nearest weather station */
static GtkWidget *qth_name,*qth_desc;                  /* name and descr. entries */


extern GConfClient *client;     /* shared client connection to GConf in main.c */
extern GtkWidget *app;

/* widgets */
static GtkWidget *newb,*editb,*delb,*defb,*clist;

/* other */
static gchar *titles[] = {
	N_("#"),
	N_(" Name "),
	N_("Description"),
	N_("Location"),
	N_(" QRA "),
	N_(" Lat "),
	N_(" Lon "),
	N_(" Alt "),
	N_(" WX "),
	N_("Default")
};

/* private function prototypes */
static gboolean satpref_qth_exists              (guint i);
static void     satpref_qth_row_clicked_cb      (GtkCList *, gint, gint, GdkEventButton *, gpointer);
static void     satpref_qth_add_cb              (GtkWidget *button, gpointer clist);
static void     satpref_qth_delete_cb           (GtkWidget *button, gpointer clist);
static void     satpref_qth_edit_cb             (GtkWidget *button, gpointer clist);
static void     satpref_qth_default_cb          (GtkWidget *button, gpointer clist);
static void     satpref_qth_locbut_cb           (GtkWidget *button, gpointer data);
static void     satpref_qth_loctree_select_cb   (GtkCTree *ctree, GList *node, gint column, gpointer dialog);
static void     satpref_qth_loctree_unselect_cb (GtkCTree *ctree, GList *node, gint column, gpointer dialog);
static void     satpref_qth_wxbut_cb            (GtkWidget *button, gpointer data);


/* This structure is used as buffer when the user
   selects a row in the locations tree.
*/
static struct {
	gchar *location;
	gdouble lat;
	gdouble lon;
	gint alt;
	gchar *wx;
} selection;



GtkWidget *
satpref_qth_create ()
{
	GtkWidget *butbox,*hbox;
	GtkWidget *swin;
	guint i,defqth;
	gchar *p1,*p2,*p3,*p4,*p5,*p6,*p7,*p8,*buff,**bufv;


	/* CList widget */
	clist = gtk_clist_new_with_titles (10, titles);
	gtk_clist_column_titles_passive (GTK_CLIST (clist));

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (clist), "select-row",
			    GTK_SIGNAL_FUNC (satpref_qth_row_clicked_cb),
			    NULL);
	gtk_signal_connect (GTK_OBJECT (clist), "unselect-row",
			    GTK_SIGNAL_FUNC (satpref_qth_row_clicked_cb),
			    NULL);

	/* adjust columns automatically */
	for (i=0; i<10; i++)
		gtk_clist_set_column_auto_resize (GTK_CLIST (clist), i, TRUE);

	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), clist);

	/* default location */
	p1 = g_strdup_printf ("%s/default", QTH_ROOT_PATH);
	defqth = gconf_client_get_int (client, p1, NULL);
	g_free (p1);


	/* read all defined locations into GtkCList */
	for (i=0; satpref_qth_exists (i); i++) {
		p1 = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, i);
		p2 = g_strdup_printf ("%s/qth%d_description", QTH_ROOT_PATH, i);
		p3 = g_strdup_printf ("%s/qth%d_location", QTH_ROOT_PATH, i);
		p4 = g_strdup_printf ("%s/qth%d_qra", QTH_ROOT_PATH, i);
		p5 = g_strdup_printf ("%s/qth%d_lat", QTH_ROOT_PATH, i);
		p6 = g_strdup_printf ("%s/qth%d_lon", QTH_ROOT_PATH, i);
		p7 = g_strdup_printf ("%s/qth%d_alt", QTH_ROOT_PATH, i);
		p8 = g_strdup_printf ("%s/qth%d_wxsta", QTH_ROOT_PATH, i);
		buff = g_strdup_printf ("%i;%s;%s;%s;%s;%.4f;%.4f;%d;%s ", i,
					gconf_client_get_string (client, p1, NULL),
					gconf_client_get_string (client, p2, NULL),
					gconf_client_get_string (client, p3, NULL),
					gconf_client_get_string (client, p4, NULL),
					gconf_client_get_float  (client, p5, NULL),
					gconf_client_get_float  (client, p6, NULL),
					gconf_client_get_int    (client, p7, NULL),
					gconf_client_get_string (client, p8, NULL));
		bufv = g_strsplit (buff, ";", 0);
		gtk_clist_append (GTK_CLIST (clist), bufv);
		g_strfreev (bufv);
		g_free (buff);
		g_free (p1);
		g_free (p2);
		g_free (p3);
		g_free (p4);
		g_free (p5);
		g_free (p6);
		g_free (p7);
		g_free (p8);

		/* Is this the default location */
		if (defqth == i)
			gtk_clist_set_text (GTK_CLIST (clist), i, 9, "YES");
		else
			gtk_clist_set_text (GTK_CLIST (clist), i, 9, " ");
	}


	/* buttons */
	newb  = sat_ui_utils_pixmap_button (_("Add"), GNOME_STOCK_PIXMAP_ADD,
					    _("Add a new location to the list"));
	delb  = sat_ui_utils_pixmap_button (_("Delete"), GNOME_STOCK_PIXMAP_REMOVE,
					    _("Delete the selected location from the list"));
	editb = sat_ui_utils_pixmap_button (_("Edit"), GNOME_STOCK_PIXMAP_PROPERTIES,
					    _("Edit the selected location"));
	defb  = sat_ui_utils_pixmap_button (_("Default"), GNOME_STOCK_PIXMAP_ABOUT,
					    _("Make the selected location default"));

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (newb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_qth_add_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (delb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_qth_delete_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (editb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_qth_edit_cb),
			    clist);
	gtk_signal_connect (GTK_OBJECT (defb), "clicked",
			    GTK_SIGNAL_FUNC (satpref_qth_default_cb),
			    clist);

	/* should only be clickable when a row is selected */
	gtk_widget_set_sensitive (delb, FALSE);
	gtk_widget_set_sensitive (editb, FALSE);
	gtk_widget_set_sensitive (defb, FALSE);

	/* vertical button box */
	butbox = gtk_vbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (butbox), GTK_BUTTONBOX_SPREAD);

	gtk_box_pack_start_defaults (GTK_BOX (butbox), newb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), delb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), editb);
	gtk_box_pack_start_defaults (GTK_BOX (butbox), defb);

	/* main horizontal box */
	hbox = gtk_hbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);

	gtk_box_pack_start (GTK_BOX (hbox), swin, TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (hbox), butbox, FALSE, FALSE, 0);

	return hbox;
}

void
satpref_qth_close ()
{
/*  	gtk_widget_destroy (newb); */
/*  	gtk_widget_destroy (delb); */
/*  	gtk_widget_destroy (editb); */
/*  	gtk_widget_destroy (defb); */
/*  	gtk_widget_destroy (testb); */
/*  	gtk_widget_destroy (clist); */
}


void
satpref_qth_apply ()
{
	/* This function is called by the preferences container.
	   It scans the rows of the list, and stores the values
	   using GConf.
	*/
	gint i, rows;
	gboolean ok;
	gchar *key,*buf[] = {" "};
	gboolean has_default = FALSE;

	/* get number of rows */
	rows = (GTK_CLIST (clist)->rows);
	for (i=0; i<rows; i++) {
		ok = TRUE;

		/* QTH name */
		key = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 1, buf))
			ok = ok && gconf_client_set_string (client, key, buf[0], NULL);
		else
			ok = ok && gconf_client_set_string (client, key, _("noname"), NULL);
		g_free (key);

		/* QTH description */
		key = g_strdup_printf ("%s/qth%d_description", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 2, buf))
			ok = ok && gconf_client_set_string (client, key, buf[0], NULL);
		else
			ok = ok && gconf_client_set_string (client, key, _("nodesc"), NULL);
		g_free (key);

		/* QTH location */
		key = g_strdup_printf ("%s/qth%d_location", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 3, buf))
			ok = ok && gconf_client_set_string (client, key, buf[0], NULL);
		else
			ok = ok && gconf_client_set_string (client, key, _("noloc"), NULL);
		g_free (key);

		/* QRA locator */
		key = g_strdup_printf ("%s/qth%d_qra", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 4, buf)) {
			ok = ok && gconf_client_set_string (client, key, buf[0], NULL);
		}
		else
			ok = ok && gconf_client_unset (client, key, NULL);
		g_free (key);
		
		/* Latitude */
		key = g_strdup_printf ("%s/qth%d_lat", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 5, buf))
			ok = ok && gconf_client_set_float (client, key, g_strtod (buf[0], NULL), NULL);
		else
			ok = ok && gconf_client_set_float (client, key, 0.0, NULL);
		g_free (key);

		/* Longitude */
		key = g_strdup_printf ("%s/qth%d_lon", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 6, buf))
			ok = ok && gconf_client_set_float (client, key, g_strtod (buf[0], NULL), NULL);
		else
			ok = ok && gconf_client_set_float (client, key, 0.0, NULL);
		g_free (key);

		/* Altitude */
		key = g_strdup_printf ("%s/qth%d_alt", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 7, buf))
			ok = ok && gconf_client_set_int (client, key, (guint) g_strtod (buf[0], NULL), NULL);
		else
			ok = ok && gconf_client_set_int (client, key, 0, NULL);
		g_free (key);

		/* Weather station */
		key = g_strdup_printf ("%s/qth%d_wxsta", QTH_ROOT_PATH, i);
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 8, buf)) {
			ok = ok && gconf_client_set_string (client, key, buf[0], NULL);
		}
		else
			ok = ok && gconf_client_unset (client, key, NULL);
		g_free (key);

		/* Is this the default location */
		if (gtk_clist_get_text (GTK_CLIST (clist), i, 9, buf)) {
			if (!g_strcasecmp (buf[0], "YES")) {
				key = g_strconcat (QTH_ROOT_PATH, "/default", NULL);
				ok = ok && gconf_client_set_int (client, key, i, NULL);
				g_free (key);
				has_default = TRUE;
			}
		}	
		if (!ok) {
			key = g_strdup_printf ("%s: Error saving location %d", __FUNCTION__, i);
			satlog_log (SAT_LOG_CRITICAL, key);
			g_free (key);
		}
	}

	/* set default QTH to 0 if not already set */
	if (!has_default) {
		key = g_strdup_printf ("%s/default", QTH_ROOT_PATH);
		gconf_client_set_int (client, key, 0, NULL);
		g_free (key);
	}

	/* Delete the rest from the Gconf config */
	for (i=rows; satpref_qth_exists (i); i++) {
		/* name */
		key = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* description */
		key = g_strdup_printf ("%s/qth%d_description", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* location */
		key = g_strdup_printf ("%s/qth%d_location", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* weather station */
		key = g_strdup_printf ("%s/qth%d_wxsta", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* QRA locator */
		key = g_strdup_printf ("%s/qth%d_qra", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);
		
		/* latitude */
		key = g_strdup_printf ("%s/qth%d_lat", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* longitude */
		key = g_strdup_printf ("%s/qth%d_lon", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);

		/* altitude */
		key = g_strdup_printf ("%s/qth%d_alt", QTH_ROOT_PATH, i);
		gconf_client_unset (client, key, NULL);
		g_free (key);
	}

	gconf_client_suggest_sync (client, NULL);
}


void
satpref_qth_cancel ()
{
	gtk_widget_destroy (newb);
	gtk_widget_destroy (delb);
	gtk_widget_destroy (editb);
	gtk_widget_destroy (defb);
	gtk_widget_destroy (clist);
}

void
satpref_qth_help ()
{
}



static gboolean
satpref_qth_exists (guint i)
{
	/* This function checks whether the Gnome Predict
	   configuration contains a location with  number
	   'i'.
	*/
	gchar *key;

	key = g_strdup_printf ("%s/qth%d_name", QTH_ROOT_PATH, i);
	if (gconf_client_get_string (client, key, NULL) != NULL) {
		g_free (key);
		return TRUE;
	}
	
	return FALSE;
}


static void
satpref_qth_row_clicked_cb (GtkCList *clist,
			    gint row,
			    gint column,
			    GdkEventButton *button,
			    gpointer data)
{
	/* This function is called when a row is selected
	   or unselected. It check whether the CList has a
	   valid selection and enables the buttons if so.
	   NOTE: The delete button is enabled only if there
	   are more than one QTH in the list.
	*/
	GList *selection = clist->selection;

	if (selection) {
		/* If only one qth is in list don't enable Delete button */
		if (clist->rows > 1) {
			gtk_widget_set_sensitive (delb, TRUE);
		}
		else {
			gtk_widget_set_sensitive (delb, FALSE);
		}
		gtk_widget_set_sensitive (editb, TRUE);
		gtk_widget_set_sensitive (defb, TRUE);
	}
	else {
		gtk_widget_set_sensitive (delb, FALSE);
		gtk_widget_set_sensitive (editb, FALSE);
		gtk_widget_set_sensitive (defb, FALSE);
	}
}


static void   
satpref_qth_add_cb         (GtkWidget *button, gpointer clist)
{
	/* This function is called, when the user clicks on
	   the "Add" location button. It is also called by
	   the "Edit" button, which is why it checks for the
	   "edit" flag.
	*/
	GtkWidget *dialog;
	gint yesno;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *locbut,*wxbut;
	GtkTooltips *tips;
	GtkObject *adj1,*adj2,*adj3;
	gint rows,new=0;

	/* create a new table */
	table = gtk_table_new (10, 4, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (table), 5);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);

	/* Name / Call */
	label = gtk_label_new (_("Name/Call:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 2, 0, 1);
	qth_name = gtk_entry_new ();
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qth_name,
			      _("Enter the name of your ground station or your callsign, "\
				"if you happen to be a radio amateur. This field is optional."),
			      _("This name will be used on various occasions like prints or saved images."));
	gtk_table_attach_defaults (GTK_TABLE (table), qth_name, 2, 4, 0, 1);

	/* Description */
	label = gtk_label_new (_("Description:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 2, 1, 2);
	qth_desc = gtk_entry_new ();
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qth_desc,
			      _("You can enter a description of your ground station here, if you wish."),
			      _("You can enter a description of your ground station here, if you wish."));
	gtk_table_attach_defaults (GTK_TABLE (table), qth_desc, 2, 4, 1, 2);

	/* separator */
	gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 4, 2, 3);
	
	/* Latitude */
	label = gtk_label_new (_("Latitude:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 3, 4);
	adj1 = gtk_adjustment_new (0.0, -90, 90, 0.1, 1.0, 1);
	latsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj1), 0.1, 4);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (latsp), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (latsp),
					   GTK_UPDATE_IF_VALID);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, latsp,
			      _("The latitude of your ground station in decimal degrees North (South = negative)."),
			      _("The latitude of your ground station in decimal degrees North (South = negative)."));
	gtk_table_attach (GTK_TABLE (table), latsp, 2, 3, 3, 4, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	/* Longitude */
	label = gtk_label_new (_("Longitude:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 4, 5);
	adj2 = gtk_adjustment_new (0.0, -180.0, 180.0, 0.1, 1.0, 1);
	lonsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj2), 0.1, 4);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (lonsp), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (lonsp),
					   GTK_UPDATE_IF_VALID);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, lonsp,
			      _("The longitude of your ground station in decimal degrees West (East = negative)."),
			      _("The longitude of your ground station in decimal degrees West (East = negative)."));
	gtk_table_attach (GTK_TABLE (table), lonsp, 2, 3, 4, 5, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 6, 6, 7);

	/* Location */
	label = gtk_label_new (_("Location:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 7, 8);
	qth_loc = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (qth_loc), 100);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qth_loc,
			      _("The location of your ground station (City, Country)."),
			      _("The location of your ground station (City, Country)."));
	gtk_table_attach_defaults (GTK_TABLE (table), qth_loc, 2, 3, 7, 8);

	/* button */
	locbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_INDEX),
				      _("Select Location"));
	gtk_signal_connect (GTK_OBJECT (locbut), "clicked",
			    GTK_SIGNAL_FUNC (satpref_qth_locbut_cb), NULL);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, locbut,
			      _("Click to select a location from a list of more than 2500 "\
				"predefined locations around the world."),
			      _("Click to select a location from a list of more than 2500 "\
				"predefined locations around the world."));
	gtk_table_attach_defaults (GTK_TABLE (table), locbut, 3, 4, 7, 8);

	/* altitude */
	label = gtk_label_new (_("Altitude:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 8, 9);
	adj3 = gtk_adjustment_new (0, -1000, 9000, 5, 100, 1);
	altsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj3), 1, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (altsp), TRUE);
	gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (altsp),
					   GTK_UPDATE_IF_VALID);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, altsp,
			      _("The altitude above sea level in meters."),
			      _("The altitude above sea level in meters."));
	gtk_table_attach (GTK_TABLE (table), altsp, 2, 3, 8, 9, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	/* weather station */
	label = gtk_label_new (_("Weather:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 9, 10);
	wxsta = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (wxsta), 4);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, wxsta,
			      _("Nearest weather station. This is a four letter combination "\
				"and is used to obtain weather information from the internet."),
			      _("Nearest weather station. This is a four letter combination "\
				"and is used to obtain weather information from the internet."));
	gtk_table_attach_defaults (GTK_TABLE (table), wxsta, 2, 3, 9, 10);

	/* button */
	wxbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_INDEX),
				     _("Select Station"));
	gtk_signal_connect (GTK_OBJECT (wxbut), "clicked",
			    GTK_SIGNAL_FUNC (satpref_qth_wxbut_cb), NULL);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, wxbut,
			      _("Click to select a weather station from a list of more than 2500 "\
				"predefined stations around the world. You will probably find one "\
				"near your location."),
			      _("Click to select a weather station from a list of more than 2500 "\
				"predefined stations around the world. You will probably find one "\
				"near your location."));
	gtk_table_attach_defaults (GTK_TABLE (table), wxbut, 3, 4, 9, 10);

#if defined (HAVE_HAMLIB)
	/* QRA locator */
	label = gtk_label_new (_("Locator:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 5, 6);
	qra_loc = gtk_entry_new ();
	gtk_signal_connect (GTK_OBJECT (qra_loc), "changed",
			    GTK_SIGNAL_FUNC (satpref_qth_qra_changed_cb),
			    NULL);
	gtk_entry_set_max_length (GTK_ENTRY (qra_loc), 6);
	gtk_widget_set_sensitive (qra_loc, FALSE);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, qra_loc,
			      _("You can enter your Mainhead grid locator here."),
			      _("You can enter your Mainhead grid locator here."));
	gtk_table_attach (GTK_TABLE (table), qra_loc, 2, 3, 5, 6, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

	/* connect changed signals to calc functions */
	gtk_signal_connect (adj1, "value-changed",
			    GTK_SIGNAL_FUNC (satpref_qth_latlon_cb),
			    NULL);
	gtk_signal_connect (adj2, "value-changed",
			    GTK_SIGNAL_FUNC (satpref_qth_latlon_cb),
			    NULL);

	/* Add radio buttons */
	radio1 = gtk_radio_button_new (NULL);
	gtk_signal_connect (GTK_OBJECT (radio1), "toggled",
			    GTK_SIGNAL_FUNC (satpref_qth_latlon_toggled_cb),
			    locbut);
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, radio1,
			      _("Select this option if you wish to specify your location "\
				"using geographical coordinates"),
			      _("Select this option if you wish to specify your location "\
				"using geographical coordinates"));
	radio2 = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON (radio1));
	tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (tips, radio2,
			      _("Select this option if you wish to specify your location "\
				"using the Mainhead locator system"),
			      _("Select this option if you wish to specify your location "\
				"using the Mainhead locator system"));
	gtk_table_attach (GTK_TABLE (table), radio1, 0, 1, 3, 4, GTK_SHRINK, GTK_SHRINK, 0, 0);
	gtk_table_attach (GTK_TABLE (table), radio2, 0, 1, 5, 6, GTK_SHRINK, GTK_SHRINK, 0, 0);
#endif
	gtk_widget_show_all (table);

	/* create dialog */
	dialog = gnome_dialog_new (_("Add Location"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);

	yesno = gnome_dialog_run (GNOME_DIALOG (dialog));

	if (!yesno) {
		/* user pressed 0'th button which is "OK" */
		gchar *buff,**vbuff;
	
		rows = (GTK_CLIST (clist)->rows);
		buff = g_strdup_printf ("%d<SEP>%s<SEP>%s<SEP>%s<SEP>%s<SEP>%.4f<SEP>%.4f<SEP>%d<SEP>%s<SEP> ",
					rows,
					gtk_entry_get_text (GTK_ENTRY (qth_name)),
					gtk_entry_get_text (GTK_ENTRY (qth_desc)),
					gtk_entry_get_text (GTK_ENTRY (qth_loc)),
					gtk_entry_get_text (GTK_ENTRY (qra_loc)),
					gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (latsp)),
					gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (lonsp)),
					gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (altsp)),
					gtk_entry_get_text (GTK_ENTRY (wxsta)));
		vbuff = g_strsplit (buff, "<SEP>", 9);
		new = gtk_clist_append (GTK_CLIST (clist), vbuff);
		g_free (buff);
		g_strfreev (vbuff);

		if (new != rows) {
			buff = g_strdup_printf ("%s: Error inserting row (%d != %d)", __FUNCTION__, rows, new);
			satlog_log (SAT_LOG_CRITICAL, buff);
			g_free (buff);
		}
		satpref_changed ();
	}
	gnome_dialog_close (GNOME_DIALOG (dialog));
}



static void
satpref_qth_delete_cb      (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Delete" button. It deletes the selected location
	   from the list, provided there are more than one
	   entries.
	*/
	GList *selection = (GTK_CLIST (clist))->selection;
	
	if (selection) {
		gtk_clist_remove (GTK_CLIST (clist),
				  GPOINTER_TO_UINT (selection->data));
		satpref_changed ();
	}
}


static void
satpref_qth_edit_cb        (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Edit" button. It sets the global "edit" flag to
	   TRUE, and calls the satpref_qth_add_cb function
	*/
	GtkWidget *dialog;
	gint yesno;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *locbut,*wxbut;
	GtkTooltips *tips;
	GtkObject *adj1,*adj2,*adj3;
	GList *selection = (GTK_CLIST (clist))->selection;  /* Needed if we are in edit mode */
	gint new=0;
	gchar *buff,*cbuf[] = {" "};

	if (selection) {
		new = GPOINTER_TO_UINT (selection->data);
		
		/* create a new table */
		table = gtk_table_new (10, 4, FALSE);
		gtk_table_set_row_spacings (GTK_TABLE (table), 5);
		gtk_table_set_col_spacings (GTK_TABLE (table), 5);

		/* Name / Call */
		label = gtk_label_new (_("Name/Call:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 2, 0, 1);
		qth_name = gtk_entry_new ();
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, qth_name,
				      _("Enter the name of your ground station or your callsign, "\
					"if you happen to be a radio amateur. This field is optional."),
				      _("This name will be used on various occasions like prints or saved images."));
		gtk_table_attach_defaults (GTK_TABLE (table), qth_name, 2, 4, 0, 1);
	
		if (gtk_clist_get_text (GTK_CLIST (clist), new, 1, cbuf))
			gtk_entry_set_text (GTK_ENTRY (qth_name), cbuf[0]);

		/* Description */
		label = gtk_label_new (_("Description:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 2, 1, 2);
		qth_desc = gtk_entry_new ();
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, qth_desc,
				      _("You can enter a description of your ground station here, if you wish."),
				      _("You can enter a description of your ground station here, if you wish."));
		gtk_table_attach_defaults (GTK_TABLE (table), qth_desc, 2, 4, 1, 2);

		if (gtk_clist_get_text (GTK_CLIST (clist), new, 2, cbuf))
			gtk_entry_set_text (GTK_ENTRY (qth_desc), cbuf[0]);

		/* separator */
		gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 4, 2, 3);
	
		/* Latitude */
		label = gtk_label_new (_("Latitude:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 3, 4);
		adj1 = gtk_adjustment_new (0.0, -90, 90, 0.1, 1.0, 1);
		latsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj1), 0.1, 4);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (latsp), TRUE);
		gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (latsp),
						   GTK_UPDATE_IF_VALID);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, latsp,
				      _("The latitude of your ground station in decimal degrees North (South = negative)."),
				      _("The latitude of your ground station in decimal degrees North (South = negative)."));
		gtk_table_attach (GTK_TABLE (table), latsp, 2, 3, 3, 4, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

		if (gtk_clist_get_text (GTK_CLIST (clist), new, 5, cbuf))
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), g_strtod (cbuf[0], NULL));

		/* Longitude */
		label = gtk_label_new (_("Longitude:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 4, 5);
		adj2 = gtk_adjustment_new (0.0, -180.0, 180.0, 0.1, 1.0, 1);
		lonsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj2), 0.1, 4);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (lonsp), TRUE);
		gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (lonsp),
						   GTK_UPDATE_IF_VALID);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, lonsp,
				      _("The longitude of your ground station in decimal degrees West (East = negative)."),
				      _("The longitude of your ground station in decimal degrees West (East = negative)."));
		gtk_table_attach (GTK_TABLE (table), lonsp, 2, 3, 4, 5, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

		gtk_table_attach_defaults (GTK_TABLE (table), gtk_hseparator_new (), 0, 6, 6, 7);

		if (gtk_clist_get_text (GTK_CLIST (clist), new, 6, cbuf))
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), g_strtod (cbuf[0], NULL));

		/* Location */
		label = gtk_label_new (_("Location:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 7, 8);
		qth_loc = gtk_entry_new ();
		gtk_entry_set_max_length (GTK_ENTRY (qth_loc), 100);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, qth_loc,
				      _("The location of your ground station (City, Country)."),
				      _("The location of your ground station (City, Country)."));
		gtk_table_attach_defaults (GTK_TABLE (table), qth_loc, 2, 3, 7, 8);

		if (gtk_clist_get_text (GTK_CLIST (clist), new, 3, cbuf))
			gtk_entry_set_text (GTK_ENTRY (qth_loc), cbuf[0]);

		/* button */
		locbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_INDEX),
					      _("Select Location"));
		gtk_signal_connect (GTK_OBJECT (locbut), "clicked",
				    GTK_SIGNAL_FUNC (satpref_qth_locbut_cb), NULL);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, locbut,
				      _("Click to select a location from a list of more than 2500 "\
					"predefined locations around the world."),
				      _("Click to select a location from a list of more than 2500 "\
					"predefined locations around the world."));
		gtk_table_attach_defaults (GTK_TABLE (table), locbut, 3, 4, 7, 8);

		/* altitude */
		label = gtk_label_new (_("Altitude:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 8, 9);
		adj3 = gtk_adjustment_new (0, -1000, 9000, 5, 100, 1);
		altsp = gtk_spin_button_new (GTK_ADJUSTMENT (adj3), 1, 0);
		gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (altsp), TRUE);
		gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (altsp),
						   GTK_UPDATE_IF_VALID);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, altsp,
				      _("The altitude above sea level in meters."),
				      _("The altitude above sea level in meters."));
		gtk_table_attach (GTK_TABLE (table), altsp, 2, 3, 8, 9, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

		if (gtk_clist_get_text (GTK_CLIST (clist), new, 7, cbuf))
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (altsp), g_strtod (cbuf[0], NULL));

		/* weather station */
		label = gtk_label_new (_("Weather:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 9, 10);
		wxsta = gtk_entry_new ();
		gtk_entry_set_max_length (GTK_ENTRY (wxsta), 4);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, wxsta,
				      _("Nearest weather station. This is a four letter combination "\
					"and is used to obtain weather information from the internet."),
				      _("Nearest weather station. This is a four letter combination "\
					"and is used to obtain weather information from the internet."));
		gtk_table_attach_defaults (GTK_TABLE (table), wxsta, 2, 3, 9, 10);

		if (gtk_clist_get_text (GTK_CLIST (clist), new, 8, cbuf))
			gtk_entry_set_text (GTK_ENTRY (wxsta), cbuf[0]);

		/* button */
		wxbut = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_INDEX),
					     _("Select Station"));
		gtk_signal_connect (GTK_OBJECT (wxbut), "clicked",
				    GTK_SIGNAL_FUNC (satpref_qth_wxbut_cb), NULL);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, wxbut,
				      _("Click to select a weather station from a list of more than 2500 "\
					"predefined stations around the world. You will probably find one "\
					"near your location."),
				      _("Click to select a weather station from a list of more than 2500 "\
					"predefined stations around the world. You will probably find one "\
					"near your location."));
		gtk_table_attach_defaults (GTK_TABLE (table), wxbut, 3, 4, 9, 10);

#if defined (HAVE_HAMLIB)
		/* QRA locator */
		label = gtk_label_new (_("Locator:"));
		gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
		gtk_table_attach_defaults (GTK_TABLE (table), label, 1, 2, 5, 6);
		qra_loc = gtk_entry_new ();
		gtk_signal_connect (GTK_OBJECT (qra_loc), "changed",
				    GTK_SIGNAL_FUNC (satpref_qth_qra_changed_cb),
				    NULL);
		gtk_entry_set_max_length (GTK_ENTRY (qra_loc), 6);
		gtk_widget_set_sensitive (qra_loc, FALSE);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, qra_loc,
				      _("You can enter your Mainhead grid locator here."),
				      _("You can enter your Mainhead grid locator here."));
		gtk_table_attach (GTK_TABLE (table), qra_loc, 2, 3, 5, 6, GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);

		/* Add radio buttons */
		radio1 = gtk_radio_button_new (NULL);
		gtk_signal_connect (GTK_OBJECT (radio1), "toggled",
				    GTK_SIGNAL_FUNC (satpref_qth_latlon_toggled_cb),
				    locbut);
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, radio1,
				      _("Select this option if you wish to specify your location "\
					"using geographical coordinates"),
				      _("Select this option if you wish to specify your location "\
					"using geographical coordinates"));
		radio2 = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON (radio1));
		tips = gtk_tooltips_new ();
		gtk_tooltips_set_tip (tips, radio2,
				      _("Select this option if you wish to specify your location "\
					"using the Mainhead locator system"),
				      _("Select this option if you wish to specify your location "\
					"using the Mainhead locator system"));
		gtk_table_attach (GTK_TABLE (table), radio1, 0, 1, 3, 4, GTK_SHRINK, GTK_SHRINK, 0, 0);
		gtk_table_attach (GTK_TABLE (table), radio2, 0, 1, 5, 6, GTK_SHRINK, GTK_SHRINK, 0, 0);

		/* read in existing QRA locator */
		if (gtk_clist_get_text (GTK_CLIST (clist), new, 4, cbuf))
			gtk_entry_set_text (GTK_ENTRY (qra_loc), cbuf[0]);

		/* connect changed signals to calc functions */
		gtk_signal_connect (adj1, "value-changed",
				    GTK_SIGNAL_FUNC (satpref_qth_latlon_cb),
				    NULL);
		gtk_signal_connect (adj2, "value-changed",
				    GTK_SIGNAL_FUNC (satpref_qth_latlon_cb),
				    NULL);


#endif
		gtk_widget_show_all (table);

		/* create dialog */
		dialog = gnome_dialog_new (_("Add Location"),
					   GNOME_STOCK_BUTTON_OK,
					   GNOME_STOCK_BUTTON_CANCEL,
					   NULL);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);

		yesno = gnome_dialog_run (GNOME_DIALOG (dialog));


		if (!yesno) {
			/* apply changes */
			gtk_clist_set_text (GTK_CLIST (clist), new, 1, gtk_entry_get_text (GTK_ENTRY (qth_name)));
			gtk_clist_set_text (GTK_CLIST (clist), new, 2, gtk_entry_get_text (GTK_ENTRY (qth_desc)));
			gtk_clist_set_text (GTK_CLIST (clist), new, 3, gtk_entry_get_text (GTK_ENTRY (qth_loc)));
			gtk_clist_set_text (GTK_CLIST (clist), new, 4, gtk_entry_get_text (GTK_ENTRY (qra_loc)));
			buff = g_strdup_printf ("%.4f", gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (latsp)));
			gtk_clist_set_text (GTK_CLIST (clist), new, 5, buff);
			g_free (buff);
			buff = g_strdup_printf ("%.4f", gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (lonsp)));
			gtk_clist_set_text (GTK_CLIST (clist), new, 6, buff);
			g_free (buff);
			buff = g_strdup_printf ("%d", gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (altsp)));
			gtk_clist_set_text (GTK_CLIST (clist), new, 7, buff);
			g_free (buff);
			gtk_clist_set_text (GTK_CLIST (clist), new, 8, gtk_entry_get_text (GTK_ENTRY (wxsta)));

			/* trigger changed signal */
			satpref_changed ();
		}
		gnome_dialog_close (GNOME_DIALOG (dialog));
	}
	else {
		buff = g_strconcat (__FUNCTION__, ": ", _("Can't edit NULL selection"), NULL);
		satlog_log (SAT_LOG_CRITICAL, buff);
		g_free (buff);
	}	

}


static void 
satpref_qth_default_cb     (GtkWidget *button, gpointer clist)
{
	/* This function is called when the user clicks on the
	   "Default" button. It makes the selected radio the
	   default radio.
	*/
	GList *selection = (GTK_CLIST (clist)->selection);
	gint row,i;

	if (selection) {
		row = GPOINTER_TO_INT (selection->data);
		for (i=0; i < (GTK_CLIST (clist)->rows); i++)
			gtk_clist_set_text (GTK_CLIST (clist), i, 9,
					    (i == row) ? "YES" : " ");
		/* Disable "Delete" button, so we can't delete default location
		   N/A anymore: We allow deleting of default but only if more than
		   one QTH is available.
		*/
/*		gtk_widget_set_sensitive (delb, FALSE);*/
		/* trigger signal */
		satpref_changed ();
	}

	satpref_changed ();
}


static void
satpref_qth_locbut_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the
	   "Select" location button.
	*/
	GtkWidget *tree,*dialog,*swin;
	gint yesno;

	/* create the tree and hide WX station column */
	tree = satloc_tree_create ();
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_WX, FALSE);

	/* put tree into scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), tree);
	gtk_widget_show_all (swin);

	/* create a dialog window */
	dialog = gnome_dialog_new (_("Select a location"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gtk_widget_set (dialog, "width", 400, "height", 300, NULL);
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);

	/* OK is disabled until a selection is made */
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);

	/* conect signal for row selection */
	gtk_signal_connect (GTK_OBJECT (tree), "tree-select-row",
			    GTK_SIGNAL_FUNC (satpref_qth_loctree_select_cb),
			    dialog);
	gtk_signal_connect (GTK_OBJECT (tree), "tree-unselect-row",
			    GTK_SIGNAL_FUNC (satpref_qth_loctree_unselect_cb),
			    dialog);

	/* Fianlly, run the dialog */
	yesno = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	if (!yesno) {
		gtk_entry_set_text (GTK_ENTRY (qth_loc), selection.location);
		gtk_entry_set_text (GTK_ENTRY (wxsta), selection.wx);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), selection.lat);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), selection.lon);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (altsp), selection.alt);
	}
}


static void
satpref_qth_loctree_select_cb      (GtkCTree *ctree,
				    GList *node,
				    gint column,
				    gpointer dialog)
{
	/* This function is called when the user selects a row in
	   the locations tree. It updates the current selection
	   variables and enables the OK button in the dialog.
	*/
	GtkCTreeNode *par;
	gchar *buff1,*buff2;

	g_return_if_fail (dialog != NULL);

	if (selection.location)
		g_free (selection.location);
	if (selection.wx)
		g_free (selection.wx);

	/* Get parent node containing the country */
	par = GTK_CTREE_NODE (GTK_CTREE_ROW (GTK_CTREE_NODE (node))->parent);
	gtk_ctree_node_get_pixtext (ctree, par, 
				    SATLOC_TREE_COL_LOC,
				    &buff1,
				    0,
				    NULL, NULL);
	/* location */
	gtk_ctree_node_get_pixtext (ctree, 
				    GTK_CTREE_NODE (node),
				    SATLOC_TREE_COL_LOC,
				    &buff2,
				    0,
				    NULL,
				    NULL);
	
	selection.location = g_strdup_printf ("%s, %s", buff2, buff1);

	/* Latitude */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_LAT, &buff2);
	selection.lat = g_strtod (buff2, NULL);

	/* Longitude */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_LON, &buff2);
	selection.lon = g_strtod (buff2, NULL);

	/* Altitude */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_ALT, &buff2);
	selection.alt = (gint) g_strtod (buff2, NULL);

	/* weather station */
	gtk_ctree_node_get_text (ctree, GTK_CTREE_NODE (node),
				 SATLOC_TREE_COL_WX, &buff2);
	selection.wx = g_strdup (buff2);

	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, TRUE);
}


static void
satpref_qth_loctree_unselect_cb    (GtkCTree *ctree,
				    GList *node,
				    gint column,
				    gpointer dialog)
{
	/* This function is called when the user unselects a row in
	   the locations tree. It disables the OK button in the dialog.
	*/

	g_return_if_fail (dialog);
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);
}


static void
satpref_qth_wxbut_cb (GtkWidget *button, gpointer data)
{
	/* This function is called when the user clicks on the
	   "Select" nearest weather station button.
	*/
	GtkWidget *tree,*dialog,*swin;
	gint yesno;

	/* create the tree and hide unused columns */
	tree = satloc_tree_create ();
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_LAT, FALSE);
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_LON, FALSE);
	gtk_clist_set_column_visibility (GTK_CLIST (tree), SATLOC_TREE_COL_ALT, FALSE);

	/* put tree into scrolled window */
	swin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (swin),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (swin), tree);
	gtk_widget_show_all (swin);

	/* create a dialog window */
	dialog = gnome_dialog_new (_("Select a weather station"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gtk_widget_set (dialog, "width", 400, "height", 300, NULL);
	gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);

	/* OK is disabled until a selection is made */
	gnome_dialog_set_sensitive (GNOME_DIALOG (dialog), 0, FALSE);

	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), swin, TRUE, TRUE, 0);

	/* recycle functions from "select location" dialog */
	gtk_signal_connect (GTK_OBJECT (tree), "tree-select-row",
			    GTK_SIGNAL_FUNC (satpref_qth_loctree_select_cb),
			    dialog);
	gtk_signal_connect (GTK_OBJECT (tree), "tree-unselect-row",
			    GTK_SIGNAL_FUNC (satpref_qth_loctree_unselect_cb),
			    dialog);

	/* Fianlly, run the dialog */
	yesno = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	if (!yesno) {
		gtk_entry_set_text (GTK_ENTRY (wxsta), selection.wx);
	}
}




/* THe rest is only included if hamlib 1.2.0 or later is used **/
#if defined (HAVE_HAMLIB)

static void
satpref_qth_latlon_cb              (GtkAdjustment *adj, gpointer data)
{
	/* This function is called when either the latitude or the longitude spin
	   button is changed. It updates the locator entry.
	*/
	gchar locator[2*QRA_PAIR_COUNT+1];

	/* Check whether user is typing into lat/lon entries or the change comes from
	   automatic conversion (this can be checked by reading the status of the
	   longitude-latitude radio button).
	*/
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio1))) {

		/* Try to convert input */
		if (longlat2locator (-gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (lonsp)),
				     gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (latsp)),
				     locator, QRA_PAIR_COUNT) == RIG_OK) {

			/* Conversion is OK, update locator field */
			gtk_entry_set_text (GTK_ENTRY (qra_loc), g_strdup (locator));
		}
		else {
			/* Clear locator field and disable "Next" button
			   because user is typing some junk.
			   This could be avoided by using spinbuttons! */
			gtk_entry_set_text (GTK_ENTRY (qra_loc), "");
		}
	}
}


static void
satpref_qth_latlon_toggled_cb (GtkToggleButton *button, gpointer locbut)
{
	/* This function is called when the LAT/LON radio button
	   is toggled. It checks whether the button is pressed in
	   or not, and performs the corresponding actions.
	*/

	if (gtk_toggle_button_get_active (button)) {
		/* User selected LAT/LON */
		gtk_widget_set_sensitive (qra_loc, FALSE);
		gtk_widget_set_sensitive (latsp, TRUE);
		gtk_widget_set_sensitive (lonsp, TRUE);
		gtk_widget_set_sensitive (GTK_WIDGET (locbut), TRUE);
	}
	else {
		/* User selected QRA */
		gtk_widget_set_sensitive (qra_loc, TRUE);
		gtk_widget_set_sensitive (latsp, FALSE);
		gtk_widget_set_sensitive (lonsp, FALSE);
		gtk_widget_set_sensitive (GTK_WIDGET (locbut), FALSE);
	}

}

static void
satpref_qth_qra_changed_cb    (GtkWidget *qra, gpointer data)
{
	/* This function is called when the text in the Locator
	   entry is changed.
	*/

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio2))) {
		gdouble lat,lon;

		if (!locator2longlat (&lon, &lat, gtk_entry_get_text (GTK_ENTRY (qra)))) {
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (latsp), lat);
			gtk_spin_button_set_value (GTK_SPIN_BUTTON (lonsp), -lon);
		}
	}
}

#endif
