/*
 * GooCanvas. Copyright (C) 2005 Damon Chaplin.
 * Released under the GNU LGPL license. See COPYING for details.
 *
 * goocanvasview.h - the main canvas widget.
 */
#ifndef __GOO_CANVAS_VIEW_H__
#define __GOO_CANVAS_VIEW_H__

#include <gtk/gtk.h>
#include "goocanvasmodel.h"

G_BEGIN_DECLS


#define GOO_TYPE_CANVAS_VIEW            (goo_canvas_view_get_type ())
#define GOO_CANVAS_VIEW(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), GOO_TYPE_CANVAS_VIEW, GooCanvasView))
#define GOO_CANVAS_VIEW_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), GOO_TYPE_CANVAS_VIEW, GooCanvasViewClass))
#define GOO_IS_CANVAS_VIEW(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GOO_TYPE_CANVAS_VIEW))
#define GOO_IS_CANVAS_VIEW_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GOO_TYPE_CANVAS_VIEW))
#define GOO_CANVAS_VIEW_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), GOO_TYPE_CANVAS_VIEW, GooCanvasViewClass))


typedef struct _GooCanvasViewClass  GooCanvasViewClass;

/**
 * GooCanvasView
 *
 * The #GooCanvasView-struct struct contains private data only.
 */
struct _GooCanvasView
{
  GtkContainer container;

  /* The model we are viewing. */
  GooCanvasModel *model;

  /* The view corresponding to the root item of the model. */
  GooCanvasItemView *root_view;

  /* The bounds of the canvas, in canvas units (not pixels). */
  GooCanvasBounds bounds;

  /* The scale/zoom factor of the view, in pixels per device unit. */
  gdouble scale;

  /* Where the canvas is anchored (where it is displayed when it is smaller
     than the entire window). */
  GtkAnchorType anchor;

  /* Idle handler ID, for processing updates. */
  guint idle_id;

  /* This is TRUE if some view in the canvas needs an update. */
  guint need_update : 1;

  /* The view that the mouse is over. */
  GooCanvasItemView *pointer_item_view;

  /* The item view that has the pointer grab, or NULL. */
  GooCanvasItemView *pointer_grab_item_view;

  /* This is the item view that the grab was started from. When the grab ends
     we synthesize enter/leave notify events from this view. */
  GooCanvasItemView *pointer_grab_initial_item_view;

  /* This is the mouse button that started an implicit pointer grab.
     When the same button is released the implicit grab ends. */
  guint pointer_grab_button;

  /* The item view that has the keyboard focus, or NULL. */
  GooCanvasItemView *focused_item_view;

  /* The item view that has the keyboard grab, or NULL. */
  GooCanvasItemView *keyboard_grab_item_view;

  /* The synthesized event used for sending enter-notify and leave-notify
     events to item views. */
  GdkEventCrossing crossing_event;

  /* The main canvas window, which gets scrolled around. */
  GdkWindow *canvas_window;

  /* The offsets of the canvas within the canvas window, in pixels. These are
     used when the canvas is smaller than the window size and the anchor is not
     NORTH_WEST. */
  gint canvas_x_offset;
  gint canvas_y_offset;

  /* The adjustments used for scrolling. */
  GtkAdjustment *hadjustment;
  GtkAdjustment *vadjustment;

  /* Freezes any movement of the canvas window, until thawed. This is used
     when we need to set both adjustments and don't want it to scroll twice. */
  gint freeze_count;

  /* A window temporarily mapped above the canvas to stop X from scrolling
     the contents unnecessarily (i.e. when we zoom in/out). */
  GdkWindow *tmp_window;

  /* A hash table mapping canvas items to canvas views. */
  GHashTable *item_to_view;
};

struct _GooCanvasViewClass
{
  GtkContainerClass parent_class;

  void  (*set_scroll_adjustments)	   (GooCanvasView     *view,
					    GtkAdjustment     *hadjustment,
					    GtkAdjustment     *vadjustment);

  /* Virtual methods. */
  GooCanvasItemView* (* create_item_view)  (GooCanvasView     *view,
					    GooCanvasItem     *item,
					    GooCanvasItemView *parent_view);

  /* Signals. */
  void               (* item_view_created) (GooCanvasView     *view,
					    GooCanvasItemView *item_view,
					    GooCanvasItem     *item);
};


GType         goo_canvas_view_get_type       (void) G_GNUC_CONST;
GtkWidget*    goo_canvas_view_new            (void);

GooCanvasModel* goo_canvas_view_get_model    (GooCanvasView     *view);
void          goo_canvas_view_set_model	     (GooCanvasView     *view,
					      GooCanvasModel    *model);

GooCanvasItemView* goo_canvas_view_get_root_view (GooCanvasView     *view);

GooCanvasItemView* goo_canvas_view_get_item_view    (GooCanvasView     *view,
                                                     GooCanvasItem     *item);
GooCanvasItemView* goo_canvas_view_get_item_view_at (GooCanvasView     *view,
						     gdouble            x,
						     gdouble            y,
						     gboolean           is_pointer_event);

gdouble       goo_canvas_view_get_scale	     (GooCanvasView     *view);
void          goo_canvas_view_set_scale	     (GooCanvasView     *view,
					      gdouble            pixels_per_unit);

void          goo_canvas_view_get_bounds     (GooCanvasView     *view,
					      gdouble           *left,
					      gdouble           *top,
					      gdouble           *right,
					      gdouble           *bottom);
void          goo_canvas_view_set_bounds     (GooCanvasView     *view,
					      gdouble            left,
					      gdouble            top,
					      gdouble            right,
					      gdouble            bottom);

void          goo_canvas_view_scroll_to	     (GooCanvasView     *view,
					      gdouble            left,
					      gdouble            top);

void          goo_canvas_view_grab_focus     (GooCanvasView     *canvas_view,
					      GooCanvasItemView *item_view);

void          goo_canvas_view_render	     (GooCanvasView     *view,
					      cairo_t           *cr,
					      GooCanvasBounds   *bounds,
					      gdouble            scale);

/*
 * Coordinate conversion.
 */
void   goo_canvas_view_convert_to_pixels       (GooCanvasView     *canvas_view,
						gdouble           *x,
						gdouble           *y);
void   goo_canvas_view_convert_from_pixels     (GooCanvasView     *canvas_view,
						gdouble           *x,
						gdouble           *y);

void   goo_canvas_view_convert_to_item_space   (GooCanvasView     *canvas_view,
						GooCanvasItemView *item_view,
						gdouble           *x,
						gdouble           *y);
void   goo_canvas_view_convert_from_item_space (GooCanvasView     *canvas_view,
						GooCanvasItemView *item_view,
						gdouble           *x,
						gdouble           *y);


/*
 * Pointer/keyboard grabbing & ungrabbing.
 */
GdkGrabStatus goo_canvas_view_pointer_grab   (GooCanvasView     *canvas_view,
					      GooCanvasItemView *item_view,
					      GdkEventMask       event_mask,
					      GdkCursor         *cursor,
					      guint32            time);
void          goo_canvas_view_pointer_ungrab (GooCanvasView     *canvas_view,
					      GooCanvasItemView *item_view,
					      guint32            time);
GdkGrabStatus goo_canvas_view_keyboard_grab  (GooCanvasView     *canvas_view,
					      GooCanvasItemView *item_view,
					      gboolean           owner_events,
					      guint32            time);
void          goo_canvas_view_keyboard_ungrab (GooCanvasView     *canvas_view,
					       GooCanvasItemView *item_view,
					       guint32            time);


/*
 * Internal functions, mainly for canvas subclasses and item implementations.
 */
GooCanvasItemView* goo_canvas_view_create_item_view (GooCanvasView     *view,
						     GooCanvasItem     *item,
						     GooCanvasItemView *parent_view);
void          goo_canvas_view_unregister_item_view  (GooCanvasView     *view,
						     GooCanvasItem     *item);
void	      goo_canvas_view_update	     (GooCanvasView     *view);
void          goo_canvas_view_request_update (GooCanvasView     *view);
void          goo_canvas_view_request_redraw (GooCanvasView     *view,
					      GooCanvasBounds   *bounds);


G_END_DECLS

#endif /* __GOO_CANVAS_VIEW_H__ */
