/* $Id: rgpsp_freebsd.cc,v 1.2 2000/09/23 16:32:59 bergo Exp $ */

/*

    GPS/RGPSP - Graphical Process Statistics / Remote GPS Poller
    Copyright (C) 1999-2000 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>

#include <kvm.h>
#include <limits.h>

#include <osreldate.h>
#include <sys/conf.h>

#if __FreeBSD_version < 400000
#include <sys/rlist.h>
#endif

#include <sys/dkstat.h>
#include <sys/sysctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/proc.h>

#if __FreeBSD_version > 399999
#include <vm/vm.h>
#include <vm/pmap.h>
#include <freebsd4_vmmap.h>
#endif

#include <sys/user.h>
#include <sys/vmmeter.h>
#include <nlist.h>
#include <vm/vm.h>
#include <vm/vm_map.h>
#include <vm/vm_param.h>

static char *native_id="FreeBSD(i386) 2/3/4";

void poller_routine();
char *native_description();
char *tb[32];
char *really_big_buffer;

int lastpid,nprocesses;
char buffer[1024];
unsigned long mem_used,mem_free,mem_shd,mem_buf,swp_used,swp_free;
float cpu_usage;

kvm_t *kt;

/* local */

void actual_poll(int detail);

void poll_memory();
void poll_cpu();
void count_processes();
void poll_process_list();
void poll_detailed_list();
char * power_of_two_suffix(int expon);

/* FreeBSD's shiny swap info */
int swapmode(unsigned long *retavail, unsigned long *retfree);
void time_to_gps(time_t v,char *dest);

void poller_routine() {
  unsigned char x;
  int i,goon;
 
  for(i=0;i<32;i++)
    if ((tb[i]=(char *)malloc(1024))==NULL)
      return;

  really_big_buffer=(char *)malloc(20<<10);
  if (really_big_buffer==NULL) {
    cerr << "** rgpsp ** unable to allocate memory.\n";
  }

  do {
    kt=kvm_open(NULL,NULL,NULL,O_RDONLY,NULL);
    if (kt==NULL)
	usleep(300000);
  } while(kt==NULL);

  actual_poll(0);

  for(goon=1;goon;) {
    while(read(0,&x,1)<1) usleep(50*1000);
    switch(x) {
    case ' ': actual_poll(0); break; /* normal */
    case  12: actual_poll(1); break; /* details */
    case 'h': actual_poll(2); break; /* cpu&mem header only */
    case 'q': goon=0; break;         /* bye bye */
    }
  }
  for(i=0;i<32;i++) free(tb[i]);
  free(really_big_buffer);
  kvm_close(kt);
  return;
}

char *native_description() {
  return(native_id);
}

void actual_poll(int detail) {
  char local[256];

  if (detail<2) {
    count_processes();
  } else {
    lastpid=0;
    nprocesses=0;
  }
  poll_memory();
  poll_cpu();

  sprintf(local,"last pid: %d  <freebsd><special> rgpsp poller ! ! !\n",lastpid);
  strcpy(really_big_buffer,local);

  sprintf(local,"%d processes: no info about the state\n",nprocesses);
  strcat(really_big_buffer,local);

  sprintf(local,
	  "CPU states: %.2f%% user, 0.0%% nice, 0.0%% system, %.2f%% idle\n",
	  100.0*cpu_usage,100.0-100.0*cpu_usage);
  strcat(really_big_buffer,local);

  sprintf(local,
	  "Memory: %uK used, %uK free, %uK shd, %uK bufcch  Swap: %uK used, %uK free\n\n",
	  mem_used>>10,mem_free>>10,mem_shd>>10,
	  mem_buf>>10,swp_used>>10,swp_free>>10);
  strcat(really_big_buffer,local);
  
  sprintf(local,"PID OWNER PRI NICE SIZE RSS STATE TIME WCPU CPU COMMAND OTHERS\n");
  strcat(really_big_buffer,local);

  switch(detail) { /* 0=normal 1=detail 2=header only */
  case 0: poll_process_list(); break;
  case 1: poll_detailed_list(); break;
  }

  printf("%s\n",really_big_buffer);
  fflush(stdout);
}

void count_processes() {
	struct kinfo_proc * kip;
	struct nlist mylist[2];
	int fscale; /* thank you BSD guys for documenting this one */
	int mv,mem_unit,nicety;
	int cand;

	int cnt,i,j;

	lastpid=nprocesses=0;

	kip=kvm_getprocs(kt,KERN_PROC_ALL,0,&cnt);
	if (kip==NULL)
		return;

	mylist[0].n_name="_fscale";
	mylist[1].n_name=NULL;
	kvm_nlist(kt,mylist);
	fscale=1;
	if (mylist[0].n_value)
		kvm_read(kt,mylist[0].n_value,&fscale,sizeof(fscale));
	
	nprocesses=cnt;
	for(i=0;i<cnt;i++) {
		cand=(int)(kip[i].kp_proc.p_pid);
		if (cand>lastpid)
			lastpid=cand;
	}
}

void poll_cpu() {
	static unsigned long prevTotal=0;
	static unsigned long prevUsed=0;

	#define N_CP_TIME	0
	static struct nlist nl[] = {
		{ "_cp_time" },{ "" }
	};

	long        cp_time[CPUSTATES];
	unsigned long      cpu[4]; /* user,nice,sys,idle */
	unsigned long      lused,ltotal;

	if (cpu_usage>1.0)
		cpu_usage=0.0;

	if (kvm_nlist(kt, nl) < 0 || nl[0].n_type == 0)
		return;

	if (kvm_read(kt, nl[N_CP_TIME].n_value,
		     (char *)&cp_time, sizeof(cp_time)) == sizeof(cp_time))
	{
		cpu[0] = cp_time[CP_USER];
		cpu[1] = cp_time[CP_NICE];
		cpu[2] = cp_time[CP_SYS];
		cpu[3] = cp_time[CP_IDLE];
		lused=(cpu[0]+cpu[1]+cpu[2]);
		ltotal=lused+cpu[3];
		lused-=prevUsed;
		ltotal-=prevTotal;
		prevUsed+=lused;
		prevTotal+=ltotal;
		cpu_usage=(float)(((double)lused)/((double)ltotal));
		if (cpu_usage>1.0)
			cpu_usage=1.0;
		if (cpu_usage<0.0)
			cpu_usage=0.0;
	}
}

void poll_memory() {
	static unsigned long cnt_offset;
	static struct nlist mylist[2];
	static struct vmmeter sum;
	long pagesize;

	memset(&sum,0,sizeof(sum));

	mylist[0].n_name="_cnt";
	mylist[1].n_name=NULL;

	kvm_nlist(kt,mylist);
	cnt_offset=mylist[0].n_value;
	kvm_read(kt,cnt_offset,&sum,sizeof(sum));
	
	pagesize=sum.v_page_size;
	mem_free=(sum.v_free_count+sum.v_inactive_count)*pagesize;
	mem_used=(sum.v_active_count+sum.v_wire_count)*pagesize;
	mem_shd=mem_buf=0;
	
	swapmode(&swp_used, &swp_free);
	swp_used=swp_used-swp_free;

	swp_used<<=10;
	swp_free<<=10;
}

int swapmode(unsigned long *retavail, unsigned long *retfree)
{
	kvm_t *kvmd;

static struct nlist nl[] = {
#define N_CNT		0
	{ "_cnt" },
#define N_BUFSPACE	1
	{ "_bufspace" },
#if __FreeBSD_version < 400000
#define VM_SWAPLIST	2
	{ "_swaplist" },
#define VM_SWDEVT	3
	{ "_swdevt" },
#define VM_NSWAP	4
	{ "_nswap" },
#define VM_NSWDEV	5
	{ "_nswdev" },
#define VM_DMMAX	6
	{ "_dmmax" },
#endif
	{ "" }
};

	int used, avail;
#if  __FreeBSD_version >= 400000
	struct kvm_swap kvmswap;
#else
	char *header;
	int hlen, nswap, nswdev, dmmax;
	int i, div, nfree, npfree;
	struct swdevt *sw;
	long blocksize, *perdev;
	u_long ptr;
	struct rlist head;
#  if __FreeBSD_version >= 220000
	struct rlisthdr swaplist;
#  else 
	struct rlist *swaplist;
#  endif
	struct rlist *swapptr;
#endif

	static int warning = 10;

	if (warning <= 0)
		{
		/* a single warning */
		if (!warning)
	    		{
			warning--;
			fprintf(stderr, "Too much errors, stop reading swap devices ...\n");
			(void)sleep(3);
			}
		return(0);
		}
	warning--;		/* decrease counter, see end of function */

	kvmd=kt;
	if (kvmd == NULL)
		return(0);

	if (kvm_nlist(kvmd,nl)<0) {
		return(0);
	}

#if  __FreeBSD_version >= 400000
	if (kvm_getswapinfo(kvmd, &kvmswap, 1, 0) < 0)
		{
		fprintf(stderr, "kvm_getswapinfo failed\n");
		return(0);
		}

	*retavail = avail = kvmswap.ksw_total;
	used = kvmswap.ksw_used;
	*retfree = kvmswap.ksw_total - used;
#else
	if (kvm_read(kvmd, nl[VM_NSWAP].n_value,
		     &nswap, sizeof(nswap)) != sizeof(nswap))
		return(0);
	if (!nswap)
		{
		fprintf(stderr, "No swap space available\n");
		return(0);
		}

	if (kvm_read(kvmd, nl[VM_NSWDEV].n_value,
		     &nswdev, sizeof(nswdev)) != sizeof(nswdev))
		return(0);
	if (kvm_read(kvmd, nl[VM_DMMAX].n_value,
		     &dmmax, sizeof(dmmax)) != sizeof(dmmax))
		return(0);
	if (kvm_read(kvmd, nl[VM_SWAPLIST].n_value,
		     &swaplist, sizeof(swaplist)) != sizeof(swaplist))
		return(0);

	if ((sw = (struct swdevt *)malloc(nswdev * sizeof(*sw))) == NULL ||
	    (perdev = (long *)malloc(nswdev * sizeof(*perdev))) == NULL)
		{
		perror("malloc");
		exit(1);
		}
	if (kvm_read(kvmd, nl[VM_SWDEVT].n_value,
		     &ptr, sizeof ptr) != sizeof ptr)
		return(0);
	if (kvm_read(kvmd, ptr,
		     sw, nswdev * sizeof(*sw)) != nswdev * sizeof(*sw))
		return(0);

	/* Count up swap space. */
	nfree = 0;
	memset(perdev, 0, nswdev * sizeof(*perdev));
#if  __FreeBSD_version >= 220000
	swapptr = swaplist.rlh_list;
	while (swapptr)
#else
	while (swaplist)
#endif
		{
		int	top, bottom, next_block;
#if  __FreeBSD_version >= 220000
		if (kvm_read(kvmd, (u_long)swapptr, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#else
		if (kvm_read(kvmd, (u_long)swaplist, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#endif

		top = head.rl_end;
		bottom = head.rl_start;

		nfree += top - bottom + 1;

		while (top / dmmax != bottom / dmmax)
			{
			next_block = ((bottom + dmmax) / dmmax);
			perdev[(bottom / dmmax) % nswdev] +=
				next_block * dmmax - bottom;
			bottom = next_block * dmmax;
			}
		perdev[(bottom / dmmax) % nswdev] +=
			top - bottom + 1;

#if  __FreeBSD_version >= 220000
		swapptr = head.rl_next;
#else
		swaplist = head.rl_next;
#endif
		}

	header = getbsize(&hlen, &blocksize);
	div = blocksize / 512;
	avail = npfree = 0;
	for (i = 0; i < nswdev; i++)
		{
		int xsize, xfree;

		xsize = sw[i].sw_nblks;
		xfree = perdev[i];
		used = xsize - xfree;
		npfree++;
		avail += xsize;
		}

	/* 
	 * If only one partition has been set up via swapon(8), we don't
	 * need to bother with totals.
	 */
	*retavail = avail / 2;
	*retfree = nfree / 2;
	used = avail - nfree;
	free(sw); free(perdev);
#endif /* __FreeBSD_version >= 400000 */

	/* increase counter, no errors occurs */
	warning++; 

	return  (int)(((double)used / (double)avail * 100.0) + 0.5);

}

void poll_process_list() {
	struct kinfo_proc * kip;
	struct passwd *pw;
	double cpuv;
	struct nlist mylist[2];
	int fscale; /* thank you BSD guys for documenting this one */
	int mv,mem_unit,nicety;
	time_t born;
	struct pstats mypstats;
	struct user *u_addr = (struct user *)USRSTACK;
	long mem[2];
	char **argv;

	int cnt,i,j,cpid;
	char b2[1024],local[768];

	kip=kvm_getprocs(kt,KERN_PROC_ALL,0,&cnt);
	if (kip==NULL)
		return;

	mylist[0].n_name="_fscale";
	mylist[1].n_name=NULL;
	kvm_nlist(kt,mylist);
	fscale=1;
	if (mylist[0].n_value)
		kvm_read(kt,mylist[0].n_value,&fscale,sizeof(fscale));
	
	for(i=0;i<cnt;i++) {
		cpid=(int)(kip[i].kp_proc.p_pid);

		sprintf(b2,"%s",kip[i].kp_proc.p_comm);
		strcpy(tb[1],b2);

		if (!(kip[i].kp_proc.p_flag&P_SYSTEM)) {
			argv=kvm_getargv(kt,&(kip[i]),512);
			if (argv==NULL)
				b2[0]=0;
			else
				for(b2[0]=0,j=0;argv[j]!=NULL;j++) {
					strcat(b2,argv[j]);
					if (argv[j+1])
						strcat(b2," ");
					if (strlen(b2)>500)
						break;
				}
		} else {
			strcpy(b2,kip[i].kp_proc.p_comm);
		}

		b2[511]=0;
		strcpy(tb[9],b2);

		pw=getpwuid(kip[i].kp_eproc.e_pcred.p_ruid);
		if (pw!=NULL)
			sprintf(b2,"%s",pw->pw_name);
		else
			sprintf(b2,"%d",kip[i].kp_eproc.e_pcred.p_ruid);
		strcpy(tb[0],b2);

		mem[0]=kip[i].kp_eproc.e_vm.vm_rssize;
		mem[1]=kip[i].kp_eproc.e_vm.vm_map.size;

		mem[0]*=(long)getpagesize();

		mem_unit=0;
  		while(mem[0]>(99UL<<10)) {
			++mem_unit;
			mem[0]>>=10;
		}
		sprintf(b2,"%lu",mem[0]);
		strcat(b2,power_of_two_suffix(mem_unit));
		strcpy(tb[8],b2);

		mem_unit=0;
  		while(mem[1]>(99UL<<10)) {
			++mem_unit;
			mem[1]>>=10;
		}
		sprintf(b2,"%lu",mem[1]);
		strcat(b2,power_of_two_suffix(mem_unit));
		strcpy(tb[6],b2);

		b2[1]=b2[2]=b2[3]=0;
		switch(kip[i].kp_proc.p_stat) {
			case SSTOP:
				b2[0]='T';
				break;
			case SSLEEP:
				if (kip[i].kp_proc.p_flag&P_SINTR)
					b2[0]='S';
				else
					b2[0]='D';
				break;
			case SRUN:
			case SIDL:
				b2[0]='R';
				break;
			case SZOMB:
				b2[0]='Z';
				break;
			default:
				b2[0]='U';
		}

		nicety=(int)(kip[i].kp_proc.p_nice-NZERO);
		if (mem[0]==0)
			b2[1]='W';
		
		strcpy(tb[7],b2);

		cpuv=(double)(kip[i].kp_proc.p_pctcpu);
		cpuv=cpuv*100.0/fscale;
		sprintf(b2,"%.2f",cpuv);
		strcpy(tb[5],b2);

		sprintf(b2,"%d",nicety);
		strcpy(tb[3],b2);

		mv=kip[i].kp_proc.p_priority-PZERO;
		sprintf(b2,"%d",mv);
		strcpy(tb[2],b2);

#if __FreeBSD_version < 400000
		if (kvm_uread(kt,&(kip[i].kp_proc),
				(unsigned long)&u_addr->u_stats,
				(char *)&mypstats, sizeof(mypstats))==sizeof(mypstats)) {
			born=mypstats.p_start.tv_sec;
			time_to_gps(born,b2);
		} else
			sprintf(b2,"<not available>");
#else
	sprintf(b2,"<not stated in 4.x>");
#endif

		strcpy(tb[4],b2);
		sprintf(local,"%d %s %s %s %s %s %s 0:00 %s%% %s%% %s !%s! %c%s%c\n",
	 			cpid,tb[0],tb[2],tb[3],tb[6],tb[8],
				tb[7],tb[5],tb[5],tb[1],tb[4],8,tb[9],8);
		strcat(really_big_buffer,local);
	}
}

/* process list

    0 = user
    1 = name
    2 = pri
    3 = nice
    4 = start
    5 = cpu
    6 = size
    7 = state
    8 = rss
    9 = cmdline

    sprintf(local,"%d %s %s %s %s %s %s 0:00 %s%% %s%% %s !%s! %c%s%c\n",
	    <pid>,tb[0],tb[2],tb[3],tb[6],tb[8],
	    tb[7],tb[5],tb[5],tb[1],tb[4],8,tb[9],8);
    strcat(really_big_buffer,local);

*/

char * power_of_two_suffix(int expon) {
  static char sarray[16]={'K',0,'M',0,'G',0,'T',0,'P',0,'<','!','>',0};
  int i;

  i=expon;
  if (i==0)
    return(&sarray[1]);
  i=((i%7)-1)<<1;
  return(&sarray[i]);
}

void poll_detailed_list() {
	char b2[1024],local[768];
	double cpuv;
	struct nlist mylist[2];
	int i,j,cnt,fscale;
	struct kinfo_proc * kip;
	struct passwd *pw;
	int mv,mem_unit,nicety;
	time_t born;
	struct pstats mypstats;
	struct user *u_addr = (struct user *)USRSTACK;
	struct session mysession;
	long mem[2],sv[2];
	char **argv,*ttname;
	int cpid;

	kip=kvm_getprocs(kt,KERN_PROC_ALL,0,&cnt);
	if ((kip==NULL)||(!cnt))
		return;

	mylist[0].n_name="_fscale";
	mylist[1].n_name=NULL;
	kvm_nlist(kt,mylist);
	fscale=1;
	if (mylist[0].n_value)
		kvm_read(kt,mylist[0].n_value,&fscale,sizeof(fscale));
	
	for(i=0;i<cnt;i++) {
		cpid=(int)(kip[i].kp_proc.p_pid);

	sprintf(b2,"%s",kip[i].kp_proc.p_comm);
	strcpy(tb[1],b2);

	if (!(kip[i].kp_proc.p_flag&P_SYSTEM)) {
		argv=kvm_getargv(kt,&(kip[i]),512);
		if (argv==NULL)
			b2[0]=0;
		else
			for(b2[0]=0,j=0;argv[j]!=NULL;j++) {
				strcat(b2,argv[j]);
				if (argv[j+1])
					strcat(b2," ");
				if (strlen(b2)>500)
					break;
			}
	} else {
		strcpy(b2,kip[i].kp_proc.p_comm);
	}

	b2[511]=0;
	strcpy(tb[9],b2);

	pw=getpwuid(kip[i].kp_eproc.e_pcred.p_ruid);
	if (pw!=NULL)
		sprintf(b2,"%s",pw->pw_name);
	else
		sprintf(b2,"%d",kip[i].kp_eproc.e_pcred.p_ruid);
	strcpy(tb[0],b2);

	mem[0]=kip[i].kp_eproc.e_vm.vm_rssize;
	mem[1]=kip[i].kp_eproc.e_vm.vm_map.size;

	mem[0]*=(long)getpagesize();

	mem_unit=0;
	while(mem[0]>(99UL<<10)) {
		++mem_unit;
		mem[0]>>=10;
	}
	sprintf(b2,"%lu",mem[0]);
	strcat(b2,power_of_two_suffix(mem_unit));
	strcpy(tb[8],b2);

	mem_unit=0;
	while(mem[1]>(99UL<<10)) {
		++mem_unit;
		mem[1]>>=10;
	}
	sprintf(b2,"%lu",mem[1]);
	strcat(b2,power_of_two_suffix(mem_unit));
	strcpy(tb[6],b2);

	b2[1]=b2[2]=b2[3]=0;
	switch(kip[i].kp_proc.p_stat) {
		case SSTOP:
			b2[0]='T';
			break;
		case SSLEEP:
			if (kip[i].kp_proc.p_flag&P_SINTR)
				b2[0]='S';
			else
				b2[0]='D';
			break;
		case SRUN:
		case SIDL:
			b2[0]='R';
			break;
		case SZOMB:
			b2[0]='Z';
			break;
		default:
			b2[0]='U';
	}
	nicety=(int)(kip[i].kp_proc.p_nice-NZERO);
	if (mem[0]==0)
		b2[1]='W';
	
	strcpy(tb[7],b2);

	sprintf(b2,"%d",nicety);
	strcpy(tb[3],b2);

	mv=kip[i].kp_proc.p_priority-PZERO;
	sprintf(b2,"%d",mv);
	strcpy(tb[2],b2);

	cpuv=(double)(kip[i].kp_proc.p_pctcpu);
	cpuv=cpuv*100.0/fscale;
	sprintf(b2,"%.2f",cpuv);
	strcpy(tb[5],b2);

#if __FreeBSD_version < 400000
	if (kvm_uread(kt,&(kip[i].kp_proc),
			(unsigned long)&u_addr->u_stats,
			(char *)&mypstats, sizeof(mypstats))==sizeof(mypstats)) {
		born=mypstats.p_start.tv_sec;
		time_to_gps(born,b2);
	} else
		sprintf(b2,"<not available>");
#else
	sprintf(b2,"<not stated in 4.x>");
#endif
	strcpy(tb[4],b2);

	sprintf(b2,"%d",kip[i].kp_eproc.e_ppid);
	strcpy(tb[10],b2);
	sprintf(b2,"%d",kip[i].kp_eproc.e_pgid);
	strcpy(tb[11],b2);
	sprintf(b2,"%d",kip[i].kp_eproc.e_tpgid);
	strcpy(tb[13],b2);

	if (kvm_read(kt,(unsigned long)kip[i].kp_eproc.e_sess,
			(char *)&mysession, sizeof(mysession))==sizeof(mysession)) {
		sprintf(b2,"%d",mysession.s_sid);
	} else
		sprintf(b2,"<not available>");

	strcpy(tb[12],b2);
#if __FreeBSD_version > 399999
#define WEIRDSPEC_XXXX 1
strcpy(b2,"not stated in FreeBSD 4.x");
#endif

#ifndef WEIRDSPEC_XXX
	sprintf(b2,"%d",mypstats.p_ru.ru_minflt);
#endif
	strcpy(tb[14],b2);

#ifndef WEIRDSPEC_XXX
	sprintf(b2,"%d",mypstats.p_ru.ru_majflt);
#endif
	strcpy(tb[16],b2);

#ifndef WEIRDSPEC_XXX
	sprintf(b2,"%d",mypstats.p_cru.ru_minflt);
#endif
	strcpy(tb[15],b2);

#ifndef WEIRDSPEC_XXX
	sprintf(b2,"%d",mypstats.p_cru.ru_majflt);
#endif
	strcpy(tb[17],b2);

#ifndef WEIRDSPEC_XXX
	sv[0]=mypstats.p_ru.ru_utime.tv_sec*100+mypstats.p_ru.ru_utime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
#endif
	strcpy(tb[18],b2);
#ifndef WEIRDSPEC_XXX
	sv[0]=mypstats.p_ru.ru_stime.tv_sec*100+mypstats.p_ru.ru_stime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
#endif
	strcpy(tb[20],b2);

#ifndef WEIRDSPEC_XXX
	sv[0]=mypstats.p_cru.ru_utime.tv_sec*100+mypstats.p_cru.ru_utime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
#endif
	strcpy(tb[19],b2);
#ifndef WEIRDSPEC_XXX
	sv[0]=mypstats.p_cru.ru_stime.tv_sec*100+mypstats.p_cru.ru_stime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
#endif
	strcpy(tb[21],b2);

    /*
      10 PPID
      11 PGID
      12 SID
      13 TPGID
      14 MINFLT
      15 CMINFLT
      16 MAJFLT
      17 CMAJFLT

      18 UTIME
      19 CUTIME
      20 STIME
      21 CSTIME

     */

    sprintf(local,"%d %s %s %s %s %s %s 0:00 %s%% %s%% %s ",
	    cpid,tb[0],tb[2],tb[3],tb[6],tb[8],
	    tb[7],tb[5],tb[5],tb[1]);
    strcat(really_big_buffer,local);
    /* won't work if the long name contains backspaces. But if it does,
       you deserve it! */
    sprintf(local,"!%s! %c%s%c %s %s %s %s %s %s %s %s %s %s %s %s\n",
	    tb[4],8,tb[9],8,tb[10],tb[11],tb[12],tb[13],
	    tb[14],tb[15],tb[16],tb[17],
	    tb[18],tb[19],tb[20],tb[21]);
    strcat(really_big_buffer,local);
    }
}

void time_to_gps(time_t v,char *dest) {
  char a[64],b[64];
  strcpy(b,ctime(&v));
  memset(a,0,64);
  memcpy(a,b+11,8);
  strcat(a," ");
  memcpy(a+9,b,11);
  memcpy(a+20,b+20,4);
  strcpy(dest,a);
}
