#include "mainwindow.h"

#include <QToolBar>
#include <qmenubar.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qstatusbar.h>
#include <qtooltip.h>

#if QT_VERSION >= 0x040000
#  include "httpwindow.h"
#endif

#include "viewer.h"
#include "copyLabel.h"
#include "clickableLabel.h"
#include "gpsshogi/revision.h"
#ifdef USE_TOKYO_CABINET
#  include "gpsshogi/recorddb/facade.h"
#endif
MainWindow::MainWindow(QWidget *parent, const char *name)
  : QMainWindow(parent, name)
{
  viewer = new Viewer(this);
  setCentralWidget(viewer);
  resize(QSize(1000, 600));
  for (int i = 0; i < qApp->argc(); i++)
  {
    if (strcmp("-csa", qApp->argv()[i]) == 0)
    {
      if (i + 1 < qApp->argc() && qApp->argv()[i + 1][0] != '-')
      {
	std::string fileName(qApp->argv()[i + 1]);
	viewer->open(fileName);
      }
      if (i + 2 < qApp->argc() && qApp->argv()[i + 2][0] != '-')
      {
	bool ok;
	int index = QString(qApp->argv()[i + 2]).toInt(&ok);
	if (ok)
	  viewer->moveTo(index - 1);
      }
    }
    else if (strcmp("-kisen", qApp->argv()[i]) == 0)
    {
      if (i + 2 < qApp->argc() && qApp->argv()[i + 2][0] != '-')
      {
	std::string fileName(qApp->argv()[i + 1]);
	bool ok;
	int index = QString(qApp->argv()[i + 2]).toInt(&ok);
	if (ok)
	  viewer->open(fileName, index);
      }
    }
    else if (strcmp("-kif", qApp->argv()[i]) == 0)
    {
      if (i + 1 < qApp->argc() && qApp->argv()[i + 1][0] != '-')
      {
	std::string fileName(qApp->argv()[i + 1]);
	viewer->openKakinoki(fileName);
      }
    }
    else if (strcmp("-usi", qApp->argv()[i]) == 0)
    {
      if (i + 1 < qApp->argc() && qApp->argv()[i + 1][0] != '-')
      {
	std::string fileName(qApp->argv()[i + 1]);
	viewer->openUsi(fileName);
      }
    }
#if QT_VERSION >= 0x040000
    else if (strcmp("-url", qApp->argv()[i]) == 0)
    {
      if (i + 1 < qApp->argc() && qApp->argv()[i + 1][0] != '-')
      {
	HttpWindow::setLastUrl(qApp->argv()[i + 1]);
	viewer->reloadUrl();
      }
    }
#endif
  }

  QMenu *fileMenu = menuBar()->addMenu(tr("&File"));
  QAction *openAction = new QAction(tr("&Open"), this);
  openAction->setShortcut(tr("Ctrl+o"));
  connect(openAction, SIGNAL(activated()), viewer, SLOT(open()));
  fileMenu->addAction(openAction);

  QAction *nextAction = new QAction(tr("Open &Next"), this);
  connect(nextAction, SIGNAL(activated()), viewer, SLOT(nextFile()));
  nextAction->addTo(fileMenu);

  QAction *prevAction = new QAction(tr("Open &Previous"), this);
  connect(prevAction, SIGNAL(activated()), viewer, SLOT(prevFile()));
  prevAction->addTo(fileMenu);

  QAction *urlAction = new QAction(tr("Open &Url"), this);
  urlAction->setShortcut(tr("Ctrl+u"));
  connect(urlAction, SIGNAL(activated()), viewer, SLOT(openUrl()));
  urlAction->addTo(fileMenu);

  QAction *reloadUrlAction = new QAction(tr("&Reload Url"), this);
  reloadUrlAction->setShortcut(tr("Ctrl+r"));
  connect(reloadUrlAction, SIGNAL(activated()), viewer, SLOT(reloadUrl()));
  reloadUrlAction->addTo(fileMenu);

  QAction *hirateAction = new QAction(tr("&Hirate"), this);
  connect(hirateAction, SIGNAL(activated()), viewer, SLOT(hirate()));
  hirateAction->addTo(fileMenu);

  QAction *networkAction = new QAction(tr("Open Network &Connection"), this);
  connect(networkAction, SIGNAL(activated()), viewer, SLOT(network()));
  networkAction->addTo(fileMenu);

  QAction *viewAction = new QAction(tr("&View"), this);
  connect(viewAction, SIGNAL(activated()), viewer, SLOT(view()));
  viewAction->addTo(fileMenu);

  QAction *viewInNewTabAction = new QAction(tr("View In New &Tab"), this);
  connect(viewInNewTabAction, SIGNAL(activated()), viewer, SLOT(viewInNewTab()));
  viewInNewTabAction->addTo(fileMenu);

  QAction *watchFileAction = new QAction(tr("&Watch File"), this);
  watchFileAction->setToggleAction(true);
  connect(watchFileAction, SIGNAL(toggled(bool)), viewer, SLOT(watchFile(bool)));
  watchFileAction->addTo(fileMenu);

  QAction *saveToAction = new QAction(tr("Save &Moves To Now"), this);
  connect(saveToAction, SIGNAL(activated()), viewer, SLOT(saveMovesToCurrent()));
  saveToAction->addTo(fileMenu);

  QAction *exportAction = new QAction(tr("&Export"), this);
  connect(exportAction, SIGNAL(activated()), viewer, SLOT(exportCurrent()));
  exportAction->addTo(fileMenu);

  QAction *exitAction = new QAction(tr("&Quit"), this);
  exitAction->setShortcut(tr("Ctrl+q"));
  connect(exitAction, SIGNAL(activated()), qApp, SLOT(quit()));
  exitAction->addTo(fileMenu);

  QMenu *editMenu = new QMenu(this);
  QAction *copyAction = new QAction(tr("&Copy"), this);
  copyAction->setShortcut(tr("Ctrl+c"));
  connect(copyAction, SIGNAL(activated()), viewer, SLOT(copy()));
  copyAction->addTo(editMenu);

  QAction *copyBoardAndMovesAction =
    new QAction(tr("Copy &Board and Moves"), this);
  connect(copyBoardAndMovesAction, SIGNAL(activated()),
	  viewer, SLOT(copyBoardAndMoves()));
  copyBoardAndMovesAction->addTo(editMenu);

  QAction *copyUsiAction = new QAction(tr("Copy &USI"), this);
  connect(copyUsiAction, SIGNAL(activated()), viewer, SLOT(copyUsi()));
  copyUsiAction->addTo(editMenu);

  QAction *copyBoardAndMovesUsiAction =
    new QAction(tr("Copy Board and Moves USI"), this);
  connect(copyBoardAndMovesUsiAction, SIGNAL(activated()),
	  viewer, SLOT(copyBoardAndMovesUsi()));
  copyBoardAndMovesUsiAction->addTo(editMenu);

  QAction *editStateAction = new QAction(tr("Edit B&oard"), this);
  connect(editStateAction, SIGNAL(activated()),
	  viewer, SLOT(editState()));
  editStateAction->addTo(editMenu);

  QMenu *goMenu = new QMenu(this);

  QAction *initialAction = new QAction(tr("&Initial State"), this);
  connect(initialAction, SIGNAL(activated()), viewer, SLOT(toInitialState()));
  initialAction->setIconSet(loadPixmap("beginning.png"));
  initialAction->addTo(goMenu);

  QAction *forwardAction = new QAction(tr("&Forward"), this);
  forwardAction->setShortcut(tr("Ctrl+f"));
  forwardAction->setIconSet(loadPixmap("forward.png"));
  connect(forwardAction, SIGNAL(activated()), viewer, SLOT(forward()));
  forwardAction->addTo(goMenu);

  QAction *backAction = new QAction(tr("&Backward"), this);
  backAction->setShortcut(tr("Ctrl+b"));
  backAction->setIconSet(loadPixmap("backward.png"));
  connect(backAction, SIGNAL(activated()), viewer, SLOT(backward()));
  backAction->addTo(goMenu);

  QAction *lastAction = new QAction(tr("&Last"), this);
  lastAction->setIconSet(loadPixmap("end.png"));
  connect(lastAction, SIGNAL(activated()), viewer, SLOT(toLastState()));
  lastAction->addTo(goMenu);

  QMenu *analysisMenu = new QMenu(this);

  QAction *analyzeAction = new QAction(tr("&Analyze"), this);
  connect(analyzeAction, SIGNAL(activated()), viewer, SLOT(analyze()));
  analyzeAction->addTo(analysisMenu);

  QAction *analyzeInNewTabAction = new QAction(tr("Analyze In New &Tab"), this);
  connect(analyzeInNewTabAction, SIGNAL(activated()), viewer,
	  SLOT(analyzeInNewTab()));
  analyzeInNewTabAction->addTo(analysisMenu);
#ifndef OSL_PUBLIC_RELEASE
  QAction *analyzeProblemsAction = new QAction(tr("Analy&ze Problems"), this);
  connect(analyzeProblemsAction, SIGNAL(activated()), viewer, SLOT(analyzeProblems()));
  analyzeProblemsAction->addTo(analysisMenu);
#endif
  QAction *checkmateAction = new QAction(tr("&Checkmate Search"), this);
  connect(checkmateAction, SIGNAL(activated()), viewer, SLOT(checkmateSearch()));
  checkmateAction->addTo(analysisMenu);

  QAction *altCheckmateAction = new QAction(tr("C&heckmate Search (alt)"), this);
  connect(altCheckmateAction, SIGNAL(activated()), viewer, SLOT(altCheckmateSearch()));
  altCheckmateAction->addTo(analysisMenu);

#ifdef ENABLE_NTESUKI
  QAction *ntesukiAction = new QAction(tr("&Ntesuki Search"), this);
  connect(ntesukiAction, SIGNAL(activated()), viewer, SLOT(ntesukiSearch()));
  ntesukiAction->addTo(analysisMenu);
#endif
#ifndef OSL_PUBLIC_RELEASE
  QAction *quiescenceHalfAction = new QAction(tr("Q&uiescence Search Half Depth"), this);
  connect(quiescenceHalfAction, SIGNAL(activated()), viewer, SLOT(quiescenceSearchHalf()));
  quiescenceHalfAction->addTo(analysisMenu);

  QAction *piecePairAction = new QAction(tr("Show &PiecePair Dialog"), this);
  connect(piecePairAction, SIGNAL(activated()), viewer, SLOT(piecePairDialog()));
  piecePairAction->addTo(analysisMenu);

  QAction *moveGenerateAction = new QAction(tr("Show Generated &Moves Dialog"), this);
  connect(moveGenerateAction, SIGNAL(activated()), viewer, SLOT(moveGenerateDialog()));
  moveGenerateAction->addTo(analysisMenu);

  QAction *kifuAnalyzeAction = new QAction(tr("Analyze &Kifu"), this);
  connect(kifuAnalyzeAction, SIGNAL(activated()), viewer, SLOT(kifuAnalyze()));
  kifuAnalyzeAction->addTo(analysisMenu);

  QAction *evalDebugAction = new QAction(tr("Analyze &Evaluation Function (Progress)"),
					 this);
  connect(evalDebugAction, SIGNAL(activated()), viewer, SLOT(evalDebug()));
  evalDebugAction->addTo(analysisMenu);

  QAction *testEvalDebugAction = new QAction(tr("Analyze Evaluation Function (Test)"),
					 this);
  connect(testEvalDebugAction, SIGNAL(activated()), viewer,
	  SLOT(testEvalDebug()));
  testEvalDebugAction->addTo(analysisMenu);
#endif
  analyzeOnlineAction = new QAction(tr("Analyze &Online"), this);
  connect(analyzeOnlineAction, SIGNAL(toggled(bool)), viewer,
	  SLOT(setAnalyzeOnline(bool)));
  analyzeOnlineAction->setToggleAction(true);
  analyzeOnlineAction->addTo(analysisMenu);

  QMenu *viewMenu = new QMenu(this);

  flipAction = new QAction(tr("&Flip View"), this);
  connect(flipAction, SIGNAL(toggled(bool)), viewer,
	  SLOT(setOrientation(bool)));
  flipAction->setToggleAction(true);
  flipAction->addTo(viewMenu);
  effectAction = new QAction(tr("Show &Effect"), this);
  connect(effectAction, SIGNAL(toggled(bool)), viewer,
	  SLOT(enableEffect(bool)));
  effectAction->setToggleAction(true);
  effectAction->addTo(viewMenu);
  QAction *evalGraphAction = new QAction(tr("Evaluation Graph"), this);
  evalGraphAction->addTo(viewMenu);
  connect(evalGraphAction, SIGNAL(activated()), viewer, SLOT(openEvalGraph()));  
  lastMoveAction = new QAction(tr("Highlight &Last Move"), this);
  connect(lastMoveAction, SIGNAL(toggled(bool)), viewer,
	  SLOT(highlightLastMove(bool)));
  lastMoveAction->setToggleAction(true);
  lastMoveAction->addTo(viewMenu);
  lastMoveAction->toggle();

  menuBar()->insertItem(tr("&File"), fileMenu);
  menuBar()->insertItem(tr("&Edit"), editMenu);
  menuBar()->insertItem(tr("&Go"), goMenu);
  menuBar()->insertItem(tr("&Analysis"), analysisMenu);
  menuBar()->insertItem(tr("&View"), viewMenu);

  QToolBar *toolbar = addToolBar(tr("Go"));
  toolbar->addAction(loadPixmap("beginning.png"),
		     "&Initial State",
		     viewer, SLOT(toInitialState()));
  toolbar->addAction(loadPixmap("backward.png"),
		     tr("&Backward"),
		     viewer, SLOT(backward()));
  toolbar->addAction(loadPixmap("forward.png"),
		     tr("&Forward"),
		     viewer, SLOT(forward()));
  toolbar->addAction(loadPixmap("end.png"),
		     tr("&Last"),
		     viewer, SLOT(toLastState()));

  moveLabel = new QLabel(QString::fromUtf8(" 1234手 "), statusBar());
  moveLabel->setAlignment(Qt::AlignHCenter);
  moveLabel->setMinimumSize(moveLabel->sizeHint());
  statusBar()->addWidget(moveLabel);

  turnLabel = new QLabel(QString::fromUtf8(" 先手 "), statusBar());
  turnLabel->setAlignment(Qt::AlignHCenter);
  turnLabel->setMinimumSize(turnLabel->sizeHint());
  statusBar()->addWidget(turnLabel);

  evalLabel = new ClickableLabel(QString::fromUtf8("評価値: -1234 (-1234 -1234 -1234)"),
				 statusBar());
  evalLabel->setToolTip(QString::fromUtf8("評価値: 合計 (序盤 中盤 終盤"
					  ")"));
  evalLabel->setAlignment(Qt::AlignHCenter);
  evalLabel->setMinimumSize(evalLabel->sizeHint());
  statusBar()->addWidget(evalLabel);
  connect(evalLabel, SIGNAL(clicked()), viewer, SLOT(evalDebug()));

  progressLabel = new CopyRateLabel(QString::fromUtf8("進行度: 12 (12 12) (-8 -8 -8 -8)"),
				    statusBar());
  progressLabel->setAlignment(Qt::AlignHCenter);
  progressLabel->setMinimumSize(progressLabel->sizeHint());
  statusBar()->addWidget(progressLabel);

  threatmateLabel = new CopyRateLabel(QString::fromUtf8("詰めろ確率: 1234"),
				      statusBar());
  threatmateLabel->setAlignment(Qt::AlignHCenter);
  threatmateLabel->setMinimumSize(threatmateLabel->sizeHint());
  statusBar()->addWidget(threatmateLabel);

  filenameLabel = new CopyLabel(QString::fromUtf8("12345678901234567890"),
				statusBar());
  filenameLabel->setAlignment(Qt::AlignHCenter);
  filenameLabel->setMinimumSize(filenameLabel->sizeHint());
  statusBar()->addWidget(filenameLabel);

#ifdef USE_TOKYO_CABINET
  proDBLabel = new CopyRateLabel(QString::fromUtf8("プロ 123-123 55.5%"), 
				 statusBar());
  proDBLabel->setToolTip(QString::fromUtf8("先手勝 後手勝 先手勝率"));
  blackGpsDBLabel = new CopyRateLabel(QString::fromUtf8("GPS 12-12 55.5%"), 
				      statusBar());
  blackGpsDBLabel->setToolTip(QString::fromUtf8("GPS-Bonanza 先手勝 後手勝 先手勝率"));
  whiteGpsDBLabel = new CopyRateLabel(QString::fromUtf8("12-12 55.5%"), 
				      statusBar());
  whiteGpsDBLabel->setToolTip(QString::fromUtf8("Bonanza-GPS 先手勝 後手勝 先手勝率"));
  proDBLabel->setColor(QColor(255, 50, 50), QColor(50, 50, 255));
  blackGpsDBLabel->setColor(QColor(255, 50, 50), QColor(50, 50, 255));
  whiteGpsDBLabel->setColor(QColor(50, 50, 255), QColor(255, 50, 50));
  statusBar()->addWidget(proDBLabel);
  statusBar()->addWidget(blackGpsDBLabel);
  statusBar()->addWidget(whiteGpsDBLabel);
#endif

  std::string gpsname = "gpsshogi ";
#ifdef OSL_SMP
  gpsname += "(smp) ";
#endif
  gpsname += gpsshogi::gpsshogi_revision;
  QLabel *revisionLabel = new CopyLabel(QString::fromStdString(gpsname), statusBar());
  revisionLabel->setAlignment(Qt::AlignHCenter);
  revisionLabel->setMinimumSize(revisionLabel->sizeHint());
  statusBar()->addWidget(revisionLabel);

  connect(viewer, SIGNAL(statusChanged()),
	  this, SLOT(updateStatusBar()));
  connect(viewer, SIGNAL(orientationChanged(bool)),
	  this, SLOT(updateFlipButton(bool)));
  connect(viewer, SIGNAL(effectChanged(bool)),
	  this, SLOT(updateEffectButton(bool)));
  connect(viewer, SIGNAL(analyzeOnlineDisabled()),
	  this, SLOT(turnOffAnalyzeOnlineButton()));
  updateStatusBar();
}

void MainWindow::updateStatusBar()
{
  moveLabel->setText(QString::fromUtf8(" %1手 ").arg(viewer->moveCount()));
  if (viewer->turn() == osl::BLACK)
    turnLabel->setText(QString::fromUtf8("先手"));
  else
    turnLabel->setText(QString::fromUtf8("後手"));

  const viewer::EvalInfo eval = viewer->evalState();
  evalLabel->setText(QString::fromUtf8("評価値: %1 (%2 %3 %4)")
		     .arg(eval.total).arg(eval.opening).arg(eval.mid)
		     .arg(eval.ending));
  evalLabel->setRate(eval.total / 3000.0);
  osl::stl::vector<int> progress;
  viewer->progressOfState(progress);
  progressLabel->setText(QString::fromUtf8("進行度: %1 (%2 %3) (%4 %5 %6 %7)")
			 .arg(progress[0]).arg(progress[1]).arg(progress[2])
			 .arg(progress[3]).arg(progress[4]).arg(progress[5]).arg(progress[6]));
  progressLabel->setRate(progress[0]/16.0);
  const std::pair<double,double> threatmate_probability = viewer->checkmateProbability();
  threatmateLabel->setText(QString::fromUtf8("詰めろ確率: %1 %2")
			   .arg(threatmate_probability.first*100,0,'g',3)
			   .arg(threatmate_probability.second*100,0,'g',3));
  threatmateLabel->setRate(threatmate_probability.first);
  filenameLabel->setText(viewer->getFilename());

#ifdef USE_TOKYO_CABINET
  int win, loss, gps_win, gps_loss, bonanza_win, bonanza_loss;
  gpsshogi::recorddb::query(viewer->getState(), win, loss, gps_win, gps_loss, bonanza_win, bonanza_loss);
  double rate = (win+loss) ? (1.0*win/(win+loss)) : 0.5;
  proDBLabel->setText(QString::fromUtf8("プロ %1-%2 %3%") 
		      .arg(win).arg(loss).arg(100.0*rate,0,'g',3));
  proDBLabel->setRate(rate*2-1.0);
  rate = (gps_win+gps_loss) ? (1.0*gps_win/(gps_win+gps_loss)) : 0.5;
  blackGpsDBLabel->setText(QString::fromUtf8("GPS %1-%2 %3%")
			   .arg(gps_win).arg(gps_loss).arg(100.0*rate,0,'g',3));
  blackGpsDBLabel->setRate(rate*2-1.0);
  rate = (bonanza_win+bonanza_loss) ? (1.0*bonanza_win/(bonanza_win+bonanza_loss)) : 0.5;
  whiteGpsDBLabel->setText(QString::fromUtf8("%1-%2 %3%") 
			 .arg(bonanza_win).arg(bonanza_loss).arg(100.0*rate,0,'g',3));
  whiteGpsDBLabel->setRate(rate*2-1.0);
#endif
}

void MainWindow::updateFlipButton(bool sente)
{
  flipAction->setOn(!sente);
}

void MainWindow::updateEffectButton(bool on)
{
  effectAction->setOn(on);
}

void MainWindow::turnOffAnalyzeOnlineButton()
{
  analyzeOnlineAction->setOn(false);
}

QPixmap MainWindow::loadPixmap(const QString &name)
{
  return QPixmap(":/images/" + name);
}
