#include "ntesukiViewer.h"
#include "gpsshogi/gui/util.h"
#include "gpsshogi/gui/board.h"
#include "ntesukisearchdialog.h"
#include "osl/apply_move/applyMove.h"
#include "osl/ntesuki/ntesukiResult.h"

#include <qglobal.h>
#if QT_VERSION >= 0x040000
#include <Q3ProgressDialog>
#else
#include <qprogressdialog.h>
#define Q3ProgressDialog QProgressDialog
#endif
#include <qmessagebox.h>
#include <qlayout.h>
#include <qstring.h>
#include <qtooltip.h>
#include <qthread.h>
#include <qspinbox.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qapplication.h>
#include <qsettings.h>
#include <sstream>

using gpsshogi::gui::Util;

using namespace osl;
using namespace osl::ntesuki;

#define GET_VALUE(r,i) (r->getValueSlow(BLACK, i))
#define GET_VALUE_OP(r,i) (r->getValueSlow(WHITE, i))

/* ======================================================================
 * NtesukiMoveItem
 */

static QString
proof_str(const NtesukiRecord *record,
	  int i)
{
  if (!record)
  {
    return QString("N/A");
  }
  int proof = GET_VALUE(record,i).proof();
  if (proof > NtesukiResult::PROOF_LIMIT)
  {
    return QString("INF");
  }
  else
  {
    //return QString::number(proof);
    return QString("%1").arg(proof, 10);
  }
}

static QString
disproof_str(const NtesukiRecord *record,
	     int i)
{
  if (!record)
  {
    return QString("N/A");
  }
  int disproof = GET_VALUE(record, i).disproof();
  if (disproof > NtesukiResult::DISPROOF_LIMIT)
  {
    return QString("INF");
  }
  else
  {
    //return QString::number(disproof);
    return QString("%1").arg(disproof, 10);
  }
}


class NtesukiMoveItem : public MoveTreeItem
{
public:
  bool moves_expanded, is_pass;
  const NtesukiRecord *record;

  NtesukiMoveItem(MoveTree *parent, Move move,
		  QString text, QString text2,
		  const NtesukiRecord *record) :
    MoveTreeItem(parent, move, text,
		 proof_str(record, 1),
		 disproof_str(record, 1),
		 proof_str(record, 0),
		 disproof_str(record, 0),
		 text2,
		 record ?
		 QString("%1").arg(record->getChildCount(), 10) 
		 : "0"),
    moves_expanded(false), record(record)
  {
    is_pass = (text2 == "PASS");
  }
  NtesukiMoveItem(MoveTreeItem *parent, Move move,
		  QString text, QString text2,
		  const NtesukiRecord *record) :
    MoveTreeItem(parent, move, text,
		 proof_str(record, 1),
		 disproof_str(record, 1),
		 proof_str(record, 0),
		 disproof_str(record, 0),
		 text2,
		 record ?
		 QString("%1").arg(record->getChildCount(), 10) 
		 : "0"),
    moves_expanded(false), record(record)
  {
    is_pass = (text2 == "PASS");
  }

  bool isBestMove() const
  {
    if (record)
    {
      for (size_t i = 0; i < NtesukiRecord::SIZE; ++i)
      {
	if (GET_VALUE(record, i).isCheckmateSuccess())
	{
	  return true;
	}
      }
    }
    return false;
  }

  bool isBySimulation() const
  {
    return record->isBySimulation();
  }

  void paintCell(QPainter *p, const QColorGroup &cg,
		 int column, int width, int align)
  {
    if (column == 0)
    {
      QColorGroup colorGroup(cg);
      QColor bestColor(colorGroup.text());


      if (record == NULL)
      {
	bestColor = QColor("gray");
      }
      else
      {
	int g = 0;
#if 0
	if (ntesuki_move)
	{
	  if (ntesuki_move->isImmediateCheckmate())
	  {
	    g = 255;
	  }
	  else if(ntesuki_move->isBySimulation())
	  {
	    g = 127;
	  }
	}
#else
	if(record->isBySimulation())
	{
	  g = 127;
	}

#endif
	for (size_t i = 0; i < NtesukiRecord::SIZE; ++i)
	{
	  NtesukiResult result = GET_VALUE(record, i);

	  if (result.isCheckmateSuccess())
	  {
	    int b = 255 - i * 64;
	    if (b < 64) b = 64;
	    bestColor.setRgb(0,g,b);
	    break;
	  }
	  else if (result.isCheckmateFail())
	  {
	    int r = i * 64 + 128;
	    if (r > 255) r = 255;
	    bestColor.setRgb(r,g,0);
	  }
	}
      }

      colorGroup.setColor(QColorGroup::Text, bestColor);
      Q3ListViewItem::paintCell(p, colorGroup, column, width, align);
    }
    else
      Q3ListViewItem::paintCell(p, cg, column, width, align);
  }

  int compare(Q3ListViewItem *i, int col, bool ascending) const
  {
    if (is_pass)
      return -1;
    else if (((NtesukiMoveItem *)i)->is_pass)
      return 1;
    else if (record == NULL)
      return 1;
    else if (((NtesukiMoveItem *)i)->record == NULL)
      return -1;
    else if (isBySimulation())
      return 1;
    else if (((NtesukiMoveItem *)i)->isBySimulation())
      return -1;
    else if (isBestMove())
      return -1;
    else if (((NtesukiMoveItem *)i)->isBestMove())
      return 1;

    return MoveTreeItem::compare(i, col, ascending);
  }
};

#if QT_VERSION < 0x040000
/* ======================================================================
 * NtesukiTip
 */
class NtesukiTip : public QToolTip
{
public:
  NtesukiTip(QWidget *parent, Q3ListView *w) :
    QToolTip(parent), listView(w)
  {}
protected:
  virtual void maybeTip(const QPoint &pos)
  {
    NtesukiMoveItem *item = (NtesukiMoveItem *)listView->itemAt(pos);
    if (item)
    {
      QRect r = listView->itemRect((Q3ListViewItem *)item);
      QString s;
      if (item->record)
      {
	for (unsigned int i = 0; i < NtesukiRecord::SIZE; ++i)
	{
	  std::ostringstream oss;
	  oss << i << " "
	      << GET_VALUE_OP(item->record, i)
	      << "\n";
	  s.append(oss.str());
	}
      }
      else
      {
	s.append("not read");
      }
      tip(r, s);
    }
  }
private:
  Q3ListView *listView;
};
#endif

/* ======================================================================
 * NtesukiMoveViewer
 * -- the tree of moves
 */
NtesukiMoveViewer::
NtesukiMoveViewer(QWidget *parent,
		  const char *name)
  : MoveTree(parent, name), searcher(NULL)
{
  mg = new movegen_t;
  setRootIsDecorated(true);

#if QT_VERSION < 0x040000
  tip = new NtesukiTip(viewport(), this);
#endif

#if QT_VERSION >= 0x040000
  connect(this, SIGNAL(expanded(Q3ListViewItem *)),
	  this, SLOT(expandNode(Q3ListViewItem *)));
#else
  connect(this, SIGNAL(expanded(QListViewItem *)),
	  this, SLOT(expandNode(QListViewItem *)));
#endif

  addColumn("Move");
  addColumn("1 pn");
  addColumn("1 dn");
  addColumn("0 pn");
  addColumn("0 dn");
  addColumn("Flags");
  addColumn("Subtree");
}

NtesukiMoveViewer::~NtesukiMoveViewer()
{
#if QT_VERSION < 0x040000
  delete tip;
#endif
  delete searcher;
}

static QString
ntesukiMoveToString(const NtesukiMove& move)
{
  //TODO might want to use QTextStream or something
  std::ostringstream oss;
  move.flagsToStream(oss);
  const std::string move_str = oss.str();
  QString ret(move_str.c_str());
  return ret;
}
#if 0
static state::SimpleState
get_state_for_item(const state::SimpleState& initial_state,
		   NtesukiMoveItem *item)
{
  state::SimpleState state = initial_state;
  std::vector<osl::Move> moves;
  
  while (item)
  {
    moves.push_back(item->getMove());
    item = (NtesukiMoveItem*)(item->parent());
  }
}
#endif
void NtesukiMoveViewer::
addItem(NtesukiMoveItem *parent)
{
  const NtesukiRecord *record = parent->record;
  if(!record) return;


  NtesukiMoveItem *item = parent;
  osl::stl::vector<osl::Move> ms;
  for (; item; item = (NtesukiMoveItem *)item->parent())
  {
    ms.push_back(item->getMove());
  }
  NumEffectState nstate(*state);
  for (int i = ms.size() - 1; i >= 0; i--)
  {
    if (ms[i].isInvalid())
      break;
    osl::ApplyMoveOfTurn::doMove(nstate, ms[i]);
  }

  NtesukiMoveList moves;
  mg->generateSlow(nstate.turn(), nstate, moves);

  typedef NtesukiMoveList::const_iterator move_it_t;
  for (move_it_t move_it = moves.begin(); move_it != moves.end(); move_it++)
  {
    const NtesukiMove& move = *move_it;
    const NtesukiRecord *record_child = table->find(record->key.newHashWithMove(move.getMove()));
    
    NtesukiMoveItem *item =
      new NtesukiMoveItem(parent,
			  move.getMove(),
			  Util::moveToString(move.getMove()),
			  ntesukiMoveToString(move),
			  record_child);

    if (record_child)
    {
      item->setExpandable(true);
    }
  }
}

void NtesukiMoveViewer::
expandNode(Q3ListViewItem *qitem)
{
  NtesukiMoveItem *item = (NtesukiMoveItem *)qitem;
  if (item->moves_expanded) return;
  addItem(item);
  item->moves_expanded = true;
}

class NtesukiSearchThread : public QThread
{
public:
  NtesukiSearchThread(NtesukiMoveViewer::searcher_t *searcher,
		      Player player,
		      int readNodeLimit,
		      int &ntesukiNum)
    : searcher(searcher), player(player), readNodeLimit(readNodeLimit),
      ntesukiNum(ntesukiNum)
  {
  }
  virtual void run()
  {
    try
    {
      ntesukiNum = searcher->searchSlow(player, readNodeLimit);
    }
    catch (TableFull e)
    {
      ntesukiNum = -3;
    }
  }
private:
  NtesukiMoveViewer::searcher_t *searcher;
  Player player;
  const int readNodeLimit;
  int& ntesukiNum;
};

struct NtesukiSearchCanceled : public std::runtime_error
{
  NtesukiSearchCanceled()
    : std::runtime_error("Search Canceled")
  {
  }
};

static void
get_settings(NtesukiSearchDialog& dialog)
{
  QSettings settings;
  dialog.tableSizeLimitBox->setValue(settings.value("viewer/ntesuki/table", 10000).toInt());
  dialog.nodeLimitBox->setValue(settings.value("viewer/ntesuki/read", 10000000).toInt());
  dialog.searchTimeBox->setValue(settings.value("viewer/ntesuki/time", 60).toInt());

  dialog.NMF->setCheckState((Qt::CheckState)settings.value("viewer/ntesuki/NMF", Qt::Checked).toInt());
  dialog.PPT_ID->setCheckState((Qt::CheckState)settings.value("viewer/ntesuki/PPT_ID", Qt::Checked).toInt());
  dialog.INTERPOSE->setCheckState((Qt::CheckState)settings.value("viewer/ntesuki/INTERPOSE", Qt::Checked).toInt());
  dialog.NOPROMOTE->setCheckState((Qt::CheckState)settings.value("viewer/ntesuki/NOPROMOTE", Qt::Checked).toInt());
  dialog.DOMINANCE->setCheckState((Qt::CheckState)settings.value("viewer/ntesuki/DOMINANCE", Qt::Checked).toInt());
  dialog.FD->setValue(settings.value("viewer/ntesuki/FD", 0).toInt());

  dialog.psScheme->setCurrentItem(settings.value("viewer/ntesuki/psScheme", 0).toInt());
  dialog.iwScheme->setCurrentItem(settings.value("viewer/ntesuki/iwScheme", 2).toInt());
  dialog.isScheme->setCurrentItem(settings.value("viewer/ntesuki/isScheme", 0).toInt());

  dialog.cEstimates->setValue(settings.value("viewer/ntesuki/cEstimates", 2).toInt());
  dialog.hEstimates->setValue(settings.value("viewer/ntesuki/hEstimates", 0).toInt());
  dialog.gcRatio->setValue(settings.value("viewer/ntesuki/gcRatio", 30).toInt());
  dialog.gcMaxCount->setValue(settings.value("viewer/ntesuki/gcMaxCount", 10).toInt());
}

static void
set_settings(const NtesukiSearchDialog& dialog)
{
  QSettings settings;
  settings.setValue("viewer/ntesuki/table", dialog.tableSizeLimitBox->value());
  settings.setValue("viewer/ntesuki/read", dialog.nodeLimitBox->value());
  settings.setValue("viewer/ntesuki/time", dialog.searchTimeBox->value());

  settings.setValue("viewer/ntesuki/NMF", (int)dialog.NMF->checkState());
  settings.setValue("viewer/ntesuki/PPT_ID", (int)dialog.PPT_ID->checkState());
  settings.setValue("viewer/ntesuki/INTERPOSE", (int)dialog.INTERPOSE->checkState());
  settings.setValue("viewer/ntesuki/NOPROMOTE", (int)dialog.NOPROMOTE->checkState());
  settings.setValue("viewer/ntesuki/DOMINANCE", (int)dialog.DOMINANCE->checkState());
  settings.setValue("viewer/ntesuki/FD", dialog.FD->value());

  settings.setValue("viewer/ntesuki/psScheme", dialog.psScheme->currentItem());
  settings.setValue("viewer/ntesuki/iwScheme", dialog.iwScheme->currentItem());
  settings.setValue("viewer/ntesuki/isScheme", dialog.isScheme->currentItem());

  settings.setValue("viewer/ntesuki/cEstimates", dialog.cEstimates->value());
  settings.setValue("viewer/ntesuki/hEstimates", dialog.hEstimates->value());
  settings.setValue("viewer/ntesuki/gcRatio", dialog.gcRatio->value());
  settings.setValue("viewer/ntesuki/gcMaxCount", dialog.gcMaxCount->value());
}

bool NtesukiMoveViewer::
analyze(const state::SimpleState &sstate)
{
  clear();

  NtesukiSearchDialog dialog(this);
  get_settings(dialog);
  int result = dialog.exec();

  if (result != QDialog::Accepted)
  {
    return false;
  }
  set_settings(dialog);

  unsigned int tableNodeLimit = dialog.tableSizeLimitBox->value();
  unsigned int readNodeLimit  = dialog.nodeLimitBox->value();
  int time_limit = dialog.searchTimeBox->value();

  searcher_t::delay_non_pass = dialog.NMF->isChecked();
  searcher_t::ptt_invalid_defense = dialog.PPT_ID->isChecked();
  searcher_t::delay_interpose = dialog.INTERPOSE->isChecked();
  searcher_t::delay_nopromote = dialog.NOPROMOTE->isChecked();
  searcher_t::delay_non_attack = dialog.NONATTACK->isChecked();
  NtesukiRecord::use_dominance = dialog.DOMINANCE->isChecked();
  NtesukiRecord::max_for_split = dialog.MAX_FOR_SPLIT->isChecked();
  NtesukiRecord::fixed_search_depth = dialog.FD->value();


  NtesukiRecord::PSScheme psScheme =
    (NtesukiRecord::PSScheme)dialog.psScheme->currentItem();
  NtesukiRecord::IWScheme iwScheme =
    (NtesukiRecord::IWScheme)dialog.iwScheme->currentItem();
  NtesukiRecord::ISScheme isScheme =
    (NtesukiRecord::ISScheme)dialog.isScheme->currentItem();
  int cEstimates = dialog.cEstimates->value();
  int hEstimates = dialog.hEstimates->value();
  double gcRatio = dialog.gcRatio->value();
  NtesukiTable::Table::largeGCCount = dialog.gcMaxCount->value();

  state::NumEffectState state(sstate);
  // if searching for non-turn player, call state.changeTurn here.
  volatile int stop_flag = 0;
  if (searcher) delete searcher;

  searcher = new searcher_t (state, mg, tableNodeLimit, &stop_flag, true,
			     2 /* Threat Limit */,
			     iwScheme /* IWScheme */,
			     psScheme /* Dual Lambda */,
			     isScheme /* Inversion Scheme */,
			     cEstimates /* Tsumero Cost */,
			     hEstimates /* Tsumero Estimates */,
			     (static_cast<double>(gcRatio) / 100.0) /* The table size after GC */
			     );
  int ntesukiNum;

  NtesukiSearchThread thread(searcher,
			     state.turn(),
			     readNodeLimit,
			     ntesukiNum);
  try
  {
    thread.start();

    Q3ProgressDialog progress("Searching...", "Abort Search", time_limit,
			      this, 0, true);
    
    for (int i = 0; i < time_limit; i++)
    {
      timespec rq = { 1, 0 }, rm;
      nanosleep(&rq, &rm);
      progress.setProgress(i);
      qApp->processEvents();
      
      if (progress.wasCanceled())
      {
	stop_flag = true;
	thread.wait();
	progress.setProgress(time_limit);
	throw(NtesukiSearchCanceled());
	break;
      }
#if QT_VERSION >= 0x040000
      if (thread.isFinished())
	break;
#else
      if (thread.finished())
	break;
#endif
    }
    
    if (!stop_flag) stop_flag = true;
    thread.wait();
    progress.setProgress(time_limit);

    QString finish_message;
    if (ntesukiNum == 0)
    {
      finish_message = tr("Tsumi Found: %1 nodes read")
	.arg(searcher->getNodeCount());
    }
    else if (ntesukiNum > 0)
    {
      finish_message = tr("%1tesuki Found: %2 nodes read")
	.arg(ntesukiNum)
	.arg(searcher->getNodeCount());
    }
    else if (ntesukiNum == -1)
    {
      finish_message = tr("No ntesuki Found: %1 nodes read")
	.arg(searcher->getNodeCount());
    }
    else if (ntesukiNum == -2)
    {
      finish_message = tr("Read Limit Reached: %1 nodes read")
	.arg(searcher->getNodeCount());
    }
    else if (ntesukiNum == -3)
    {
      finish_message = tr("Table full: %1 nodes read")
	.arg(searcher->getNodeCount());
    }
    else
    {
      finish_message = tr("Unknown value %1 returned")
	.arg(ntesukiNum);
    }
    
    QMessageBox::information(this, "Ntesuki State", finish_message);
  }
  catch (NtesukiSearchCanceled e)
  {
    QMessageBox::information(this, "Ntesuki State", 
			     "Search canceled");
  }
  catch (std::runtime_error& e)
  {
    QMessageBox::information(this, "Ntesuki State", 
			     QString("failed ") + e.what());
    return false;
  }
  

  hash::HashKey key = hash::HashKey::calcHash(sstate);
  const NtesukiTable& t
    = searcher->getTable();
  table = &t;
  const NtesukiRecord* record
    = table->find(key);
  assert(record);


  /* Adding items
   */
  NumEffectState nstate(sstate);
  
  NtesukiMoveList moves;
  mg->generateSlow(nstate.turn(), nstate, moves);

  typedef NtesukiMoveList::const_iterator move_it_t;
  for (move_it_t move_it = moves.begin(); move_it != moves.end(); move_it++)
  {
    const NtesukiMove& move = *move_it;
    const NtesukiRecord *record_child = table->find(record->key.newHashWithMove(move.getMove()));
    //XXX
    NtesukiMoveItem *item =
      new NtesukiMoveItem(this,
			  move.getMove(),
			  Util::moveToString(move.getMove()),
			  ntesukiMoveToString(move),
			  record_child);
    if (record_child)
    {
      item->setExpandable(true);
    }
  }
  
  return true;
}

/* ======================================================================
 * NtesukiViewer
 *  the viewer, with the board on the left and the tree on the right
 */
NtesukiViewer::
NtesukiViewer(QWidget *parent, const char *name)
  : BoardAndListTabChild(parent, name)
{
  moveTree = new NtesukiMoveViewer(this);
  ((NtesukiMoveViewer*)moveTree)->setState(&initialState);

  QHBoxLayout *mainLayout = new QHBoxLayout(this);
  mainLayout->addWidget(board);
  mainLayout->addWidget(moveTree);

  init();
}

bool NtesukiViewer::
analyze(const state::SimpleState &state)
{
  initialState = state;
  board->setState(initialState);
  return ((NtesukiMoveViewer *)moveTree)->analyze(state);
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
