#!/usr/bin/env python

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import pygtk, gtk, os, re, pango, time, gobject, tempfile
import gquilt_utils, gquilt_gtk, gquilt_icons, gquilt_pfuns
import gquilt_const, gquilt_tool, gquilt_quilt, gquilt_mq

backend = {}
missing_backend = {}

def _add_backend(newifce):
    global backend
    if newifce.is_available():
        backend[newifce.name] = newifce
    else:
        missing_backend[newifce.name] = newifce

_add_backend(gquilt_quilt.interface())
_add_backend(gquilt_mq.interface())

null_backend = gquilt_tool.null_interface()

def report_backend_requirements(parent=None):
    msg = "No back ends are available. At least one of:" + os.linesep
    for key in missing_backend.keys():
        msg += "\t" + missing_backend[key].requires() + os.linesep
    msg += "must be installed/available for \"gquilt\" to do.anything useful."
    gquilt_gtk.info_dialog(msg, parent=parent).inform()

def avail_backends():
    return backend.keys()

def playground_type(dir=None):
    for b in backend.keys():
        if backend[b].is_playground(dir):
            return b
    return None

def set_ifce(dir=None):
    global ifce
    pgt = playground_type(dir)
    if pgt is None:
        ifce = null_backend
    else:
        ifce = backend[pgt]

def choose_backend():
    bel = avail_backends()
    if len(bel) == 0:
        report_backend_requirements()
        return False, ""
    elif len(bel) == 1:
        return True, bel[0]
    sfld = gquilt_gtk.select_from_list_dialog(list=bel, title="Choose back end")
    ok, req_backend = sfld.make_selection()
    return ok, req_backend

ifce = null_backend

def _path_relative_to_playground(path):
    return gquilt_utils.path_relative_to_dir(os.getcwd(), path)

class gquilt_patch_list(gquilt_gtk.text_list):
    def __init__(self, console=None, tooltips=None):
        gquilt_gtk.text_list.__init__(self, tooltips=tooltips)
        self.console = console
        self.patch_files = None
        self.playground = None
        self.append_unique_selection_menu_item("D_escription", "View/edit patch description", gtk.STOCK_EDIT, self._edit_patch_description)
        self.append_unique_selection_menu_item("_Files", "Show the files modified by the selected patch", gtk.STOCK_FILE, self._display_patch_files)
        self.append_unique_selection_menu_item("D_iff", "Displa the diff for the selected patch", gquilt_icons.diff, self._display_patch_diff)
        self.append_conditional_menu_item("_Meld", "Display diff for selected patch using \"meld\"", gquilt_icons.meld, self._display_patch_with_meld, self._meld_and_full_patch_ok)
        self.append_unique_selection_menu_item("Re_name", "Rename the selected patch", None, self._rename_patch)
        self.append_unique_selection_menu_item("_Delete", "Delete the selected patch from the series", gtk.STOCK_DELETE, self._delete_patch)
        self.append_unique_selection_menu_item("_Refresh", "Refresh the selected patch", gtk.STOCK_REFRESH, self._refresh_selected_patch)
        self.append_applied_menu_item("_Pop to", "Pop to the selected patch", gquilt_icons.pop, self._pop_to_selected_patch)
        self.append_unapplied_menu_item("P_ush to", "Push to the selected patch", gquilt_icons.push, self._push_to_selected_patch)
        self.append_conditional_menu_item("F_old", "Fold the selected patch into the top patch", gquilt_icons.fold, self._fold_selected_patch, self._foldable_patch)
        self.append_conditional_menu_item("Fo_ld to", "Fold patches up to the selected patch into the top patch", gquilt_icons.fold, self._fold_to_selected_patch, self._foldable_patch)
        self.append_conditional_menu_item("Finis_h to", "Move patches up to the selected patch into the underlying SCM", gquilt_icons.finish, self._finish_to_selected_patch, self._finishable_patch)
        self.append_unapplied_menu_item("Dupli_cate", "Duplicate the selected patch behind the top patch", gtk.STOCK_COPY, self._duplicate_selected_patch)
        self.append_conditional_menu_item("In_terdiff", "Place the \"interdiff\" of the top and selected patch behind the top patch", gtk.STOCK_PASTE, self._interdiff_selected_patch, self._interdiff_ok)
        self.update_contents()
    def set_console(self, c):
        self.console = c
    def set_patch_files(self, pf):
        self.patch_files = pf
    def update_patch_files(self):
        if self.patch_files:
            self.patch_files.update_files()
    def set_playground(self, pf):
        self.playground = pf
    def update_playground(self):
        if self.playground:
            self.playground.update()
    def _update_all_busy(self):
        self._show_busy()
        self.update_contents()
        self.update_patch_files()
        self.update_playground()
        self._unshow_busy()
    def _decorated_line_entry(self, patch, status):
        if status == gquilt_const.APPLIED:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.applied_ok)
        elif status == gquilt_const.TOP_PATCH:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.top_ok)
        elif status == gquilt_const.APPLIED_NEEDS_REFRESH:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.applied_not_ok)
        elif status == gquilt_const.TOP_PATCH_NEEDS_REFRESH:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.top_not_ok)
        elif status == gquilt_const.NOT_APPLIED:
            return (patch, pango.STYLE_ITALIC, "dark grey", None)
        else:
            return (patch, pango.STYLE_ITALIC, "magenta", None)
    def update_contents(self):
        res, (sn, series), err = ifce.get_series()
        if res != gquilt_const.OK:
            gquilt_gtk.info_dialog(err, parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        self.store.clear()
        for patch, status in series:
            self.store.append(self._decorated_line_entry(patch, status))
        col0 = self.view.get_column(0)
        if not sn:
            col0.set_title(ifce.name + " Patch Series")
        else:
            col0.set_title(ifce.name + " Patch Series: " + sn)
    def update_contents_busy(self):
        self._show_busy()
        self.update_contents()
        self._unshow_busy()
    def _unique_and_applied(self, arg=None):
        sel = self.view.get_selection()
        sel_sz = sel.count_selected_rows()
        if sel_sz == 1:
            model, iter = self.view.get_selection().get_selected()
            if iter is None:
                return False
            else:
                return model.get_value(iter, 3) != None
        else:
            return False
    def append_applied_menu_item(self, label, tooltip, stock_id, action, arg=None):
        self.append_conditional_menu_item(label, tooltip, stock_id, action, self._unique_and_applied, arg)
    def _unique_and_unapplied(self, arg=None):
        sel = self.view.get_selection()
        sel_sz = sel.count_selected_rows()
        if sel_sz == 1:
            model, iter = self.view.get_selection().get_selected()
            if iter is None:
                return False
            else:
                return model.get_value(iter, 3) is None
        else:
            return False
    def append_unapplied_menu_item(self, label, tooltip, stock_id, action, arg=None):
        self.append_conditional_menu_item(label, tooltip, stock_id, action, self._unique_and_unapplied, arg)
    # functions to implement menu functions
    def _edit_patch_description(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            gquilt_descr_dialog(patch, self.console).display()
    def _display_patch_files(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            dialog = gquilt_patch_file_dialog(patch, self.tooltips)
            dialog.display()
    def _display_patch_diff(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            gquilt_diff_dialog(patch).display()
    def _meld_and_full_patch_ok(self, pl):
        if gquilt_utils.which("meld") is None:
            return False
        if not self._unique_and_applied():
            return False
        return ifce.count_ok_meld(0)
    def _display_patch_with_meld(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            res, files, se = ifce.get_patch_files(patch, False)
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([files, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                return
            ifce.display_files_diff_in_viewer("meld", files, patch)
    def _delete_patch(self, mi):
        patch = self.get_selected_text()
        if patch != None and gquilt_gtk.info_ok_dialog('Confirm delete "' + patch + '" patch?', parent=gquilt_gtk.get_gtk_window(self)).ask():
            is_top = patch == ifce.top_patch()
            self._show_busy()
            res, so, se = ifce.do_delete_patch(self.console, patch)
            if is_top:
                self.update_patch_files()
            self._unshow_busy()
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
        self.update_contents_busy()
    def _refresh_selected_patch(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            refresh_patch(self, patch)
    def _rename_patch(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            dialog = gquilt_gtk.text_entry_dialog("Rename patch:", parent=gquilt_gtk.get_gtk_window(self))
            dialog.set_text(patch)
            res, newname = dialog.read_text()
            if res:
                self._show_busy()
                res, so, se = ifce.do_rename_patch(self.console, patch, newname)
                self._unshow_busy()
                if res != gquilt_const.OK:
                    gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            self.update_contents_busy()
    def pop_patch(self, patch=None):
        while True:
            self._show_busy()
            res, op, err = ifce.do_pop_to_patch(self.console, patch)
            self._unshow_busy()
            if res != gquilt_const.OK:
                if res == gquilt_const.ERROR_REFRESH:
                    if gquilt_gtk.info_refresh_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                        refresh_patch(self)
                        continue
                    else:
                        return False
                elif res == gquilt_const.ERROR_FORCE_OR_REFRESH:
                    ans = gquilt_gtk.info_force_or_refresh_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).ask()
                    if ans == gquilt_gtk.REFRESH_AND_RETRY:
                        refresh_patch(self)
                        continue
                    elif ans == gquilt_gtk.FORCE:
                        self._show_busy()
                        res, op, err = ifce.do_pop_patch(self.console, force=True)
                        self._unshow_busy()
                    else:
                        return False
                if res != gquilt_const.OK:
                    # if there's still a problem let the user know
                    if self.patch_files:
                        self.patch_files.update_files_busy()
                    self.update_contents()
                    gquilt_gtk.info_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                    return False
            return True
    def pop_to_patch(self, patch):
        while ifce.top_patch() != patch:
            if not self.pop_patch(patch):
                break
        self._update_all_busy()
    def pop_top_patch(self, widget=None):
        self.pop_patch()
        self._update_all_busy()
    def _pop_to_selected_patch(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            self.pop_to_patch(patch)
    def push_patch(self, patch=None):
        while True:
            self._show_busy()
            res, op, err = ifce.do_push_to_patch(self.console, patch)
            self._unshow_busy()
            if res != gquilt_const.OK:
                if res == gquilt_const.ERROR_REFRESH:
                    if gquilt_gtk.info_refresh_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                        self.patch_files.refresh_patch()
                        continue
                    else:
                        return False
                elif res == gquilt_const.ERROR_FORCE:
                    if gquilt_gtk.info_force_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                        self._show_busy()
                        res, op, err = ifce.do_push_patch(self.console, force=True)
                        self._unshow_busy()
                    else:
                        return False
                elif res == gquilt_const.ERROR_FORCE_OR_REFRESH:
                    ans = gquilt_gtk.info_force_or_refresh_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).ask()
                    if ans == gquilt_gtk.REFRESH_AND_RETRY:
                        self.patch_files.refresh_patch()
                        continue
                    elif ans == gquilt_gtk.FORCE:
                        self._show_busy()
                        res, op, err = ifce.do_push_patch(self.console, force=True)
                        self._unshow_busy()
                    else:
                        return False
                if res != gquilt_const.OK:
                    # if there's still a problem let the user know
                    if self.patch_files:
                        self.patch_files.update_files_busy()
                    self.update_contents()
                    gquilt_gtk.info_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                    return False
            return True
    def push_to_patch(self, patch):
        while ifce.top_patch() != patch:
            if not self.push_patch(patch):
                break
        self._update_all_busy()
    def push_next_patch(self, widget=None):
        self.push_patch()
        self._update_all_busy()
    def _push_to_selected_patch(self, mi):
        patch = self.get_selected_text()
        if patch != None:
            self.push_to_patch(patch)
    def _foldable_patch(self, pl):
        if not ifce.top_patch():
            return False
        return self._unique_and_unapplied()
    def _interdiff_ok(self, pl):
        return self._foldable_patch(pl) and (gquilt_utils.which("interdiff") != None)
    def _fold_selected_patch(self, mi):
        patch = self.get_selected_text()
        if patch == None:
            return
        self._show_busy()
        res, so, se = ifce.do_merge_patch(self.console, patch)
        self._unshow_busy()
        if res != gquilt_const.OK:
            gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
        self._update_all_busy()
    def _fold_to_selected_patch(self, mi):
        patch = self.get_selected_text()
        if patch == None:
            return
        more_to_do = True
        while more_to_do:
            next = ifce.next_patch()
            if not next:
                break
            self._show_busy()
            res, so, se = ifce.do_merge_patch(self.console, next)
            self._unshow_busy()
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                more_to_do = False
            self.update_contents_busy()
            more_to_do = more_to_do and patch != next
        self._update_all_busy()
    def _finishable_patch(self, pl):
        if not ifce.has_finish_patch() or not ifce.top_patch():
            return False
        return self._unique_and_applied()
    def _finish_to_selected_patch(self, mi):
        patch = self.get_selected_text()
        if patch == None:
            return
        more_to_do = True
        while more_to_do:
            next = ifce.base_patch()
            if not next:
                break
            while True:
                self._show_busy()
                ok = ifce.get_description_is_finish_ready(next)
                self._unshow_busy()
                if ok:
                    break
                msg = os.linesep.join(
                    ['"%s" has an empty description.' % next,
                     "Do you wish to:",
                     "\tcancel,",
                     "\tedit the description and retry, or",
                     "\tforce the finish operation?"
                    ])
                ans = gquilt_gtk.info_force_or_edit_dialog(msg, parent=gquilt_gtk.get_gtk_window(self)).ask()
                if ans == gtk.RESPONSE_CANCEL:
                    return
                elif ans == gquilt_gtk.FORCE:
                    break
                dialog = gquilt_descr_dialog(next, self.console)
                dialog.run()
                dialog.destroy()
            self._show_busy()
            res, so, se = ifce.do_finish_patch(self.console, next)
            self._unshow_busy()
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                more_to_do = False
            self.update_contents_busy()
            more_to_do = more_to_do and patch != next
        self._update_all_busy()
    def _duplicate_selected_patch(self, mi):
        patch = self.get_selected_text()
        res, orig_descr, se = ifce.get_patch_description(patch)
        if res != 0:
            orig_descr=""
        pfname = ifce.patch_file_name(patch)
        dialog = new_patch_dialog(
                      title="Duplicate Specification",
                      nameprompt="Enter duplicate patch name:",
                      descr=orig_descr)
        res, name, descr = dialog.read_name_and_description()
        if res:
            self._show_busy()
            res, so, se = ifce.do_import_patch(self.console, pfname, patchname=name)
            self._unshow_busy()
            if res == gquilt_const.ERROR_FORCE:
                if gquilt_gtk.info_force_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                    res, so, err = ifce.do_import_patch(self.console, pfname, patchname=name, force=True)
                else:
                    return
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            elif descr != orig_descr:
                res, so, se = ifce.do_set_patch_description(self.console, name, descr)
                if res != 0:
                    gquilt_gtk.info_dialog(se, parent=gquilt_gtk.get_gtk_window(self)).inform()
            self.update_contents()
    def _interdiff_selected_patch(self, mi):
        patch = self.get_selected_text()
        res, orig_descr, se = ifce.get_patch_description(patch)
        if res != 0:
            orig_descr=""
        pfname = ifce.patch_file_name(patch)
        dialog = new_patch_dialog(
                      title="Interdiff Patch Specification",
                      nameprompt="Enter interdiff patch name:",
                      descr=orig_descr)
        res, name, descr = dialog.read_name_and_description()
        if res:
            self._show_busy()
            topfname = ifce.patch_file_name(ifce.top_patch())
            res, diff, se = gquilt_utils.run_cmd("interdiff %s %s" % (topfname, pfname))
            tempfname = tempfile.mktemp()
            tf = open(tempfname, 'w')
            tf.write(descr + os.linesep)
            tf.write(diff)
            tf.close()
            res, so, se = ifce.do_import_patch(self.console, tempfname, patchname=name)
            self._unshow_busy()
            if res == gquilt_const.ERROR_FORCE:
                if gquilt_gtk.info_force_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                    res, so, err = ifce.do_import_patch(self.console, tempfname, patchname=name, force=True)
                else:
                    return
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            os.remove(tempfname)
            self.update_contents()

class gquilt_playground_tree(gquilt_gtk.dir_file_tree):
    def __init__(self, title="Playground Files", tooltips=None, console=None, update_views=None):
        gquilt_gtk.dir_file_tree.__init__(self, title, tooltips=tooltips)
        self.console = console
        self.patch_files = None
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.append_conditional_menu_item("_Add", "Add the selected files to the top patch", gtk.STOCK_ADD, self._add_playground_selection_to_patch, self._selection_and_has_add_file)
        self.append_selection_menu_item("_Copy", "Make copies of the selected files in the top patch", gtk.STOCK_COPY, self._copy_playground_selection_to_top_patch)
        self.append_selection_menu_item("_Move/Rename", "Move/rename the selected files in the top patch", gtk.STOCK_DND, self._move_playground_selection_to_top_patch)
        self.append_selection_menu_item("_Edit", "Edit the selected files after adding them to the top patch", gtk.STOCK_EDIT, self._edit_playground_selection_in_top_patch)
        self.append_selection_menu_item("_Peruse", "Peruse the selected files", gtk.STOCK_FILE, self._peruse_playground_selection)
        self.append_menu_item("_New", "Add a new file to the top patch", gtk.STOCK_NEW, self._add_new_file_to_patch)
        self._update_views = update_views
    def _selection_and_has_add_file(self, mi):
        if not ifce.has_add_files():
            return False
        return self.view.get_selection().count_selected_rows() > 0
    def set_patch_files(self, pf):
        self.patch_files = pf
    def update_busy(self):
        self._show_busy()
        self.update()
        self._unshow_busy()
    def update_all_busy(self):
        self._show_busy()
        self.update()
        if self.patch_files:
            self.patch_files.update_files()
        self._unshow_busy()
    def _wait_for_child_timeout(self, pid):
        rpid, status = os.waitpid(pid, os.WNOHANG)
        return rpid != pid
    def _clean_up_after_child(self, pid):
        gobject.timeout_add(2000, self._wait_for_child_timeout, pid)
    def peruse_selected_files(self):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        pid = gquilt_utils.peruse_files(files)
        self._clean_up_after_child(pid)
    def edit_selected_files(self):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        if not ifce.has_add_files():
            pid = gquilt_utils.edit_files(files)
            self._clean_up_after_child(pid)
            return
        filelist = []
        dirs = ""
        for f in files:
            if os.path.isdir(f):
                dirs += os.linesep + f
            else:
                filelist.append(f)
        if dirs != "":
            emsg = os.linesep.join(["The directories:", dirs, "",
                                    "have been removed from selection"])
            if not gquilt_gtk.info_ok_dialog(emsg, parent=gquilt_gtk.get_gtk_window(self)).ask():
                return
        if len(filelist) > 0:
            res, so, se = ifce.get_patch_files(None, False)
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                return
            added_files = []
            for f in filelist:
                if not f in so:
                    added_files.append(f)
            if len(added_files) > 0:
                res, so, se = ifce.do_add_files_to_patch(self.console, added_files)
                if res == gquilt_const.OK:
                    if self._update_views is not None:
                        self._update_views()
                    pid = gquilt_utils.edit_files(filelist)
                    self._clean_up_after_child(pid)
                else:
                    gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
    def _handle_double_click(self):
        self.edit_selected_files()
    def _add_playground_selection_to_patch(self, mi):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        filelist = []
        dirs = ""
        for f in files:
            if os.path.isdir(f):
                dirs += os.linesep + f
            else:
                filelist.append(f)
        if dirs != "":
            emsg = os.linesep.join(["The directories:", dirs, "",
                                    "have been removed from addition"])
            if not gquilt_gtk.info_ok_dialog(emsg, parent=gquilt_gtk.get_gtk_window(self)).ask():
                return
        if len(filelist) > 0:
            self._show_busy()
            res, so, se = ifce.do_add_files_to_patch(self.console, filelist)
            self._unshow_busy()
            if res == gquilt_const.OK:
                self.update_all_busy()
            else:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
    def _add_new_file_to_patch(self, mi):
        self.patch_files.add_new_file(mi)
    def _copy_playground_selection_to_top_patch(self, mi):
        self.patch_files.copy_files_to_top_patch(self.get_selected_files())
    def _move_playground_selection_to_top_patch(self, mi):
        self.patch_files.copy_files_to_top_patch(self.get_selected_files(), move=True)
    def _edit_playground_selection_in_top_patch(self, mi):
        self.edit_selected_files()
        return
    def _peruse_playground_selection(self, mi):
        self.peruse_selected_files()
        return

def refresh_patch(context, patch=None):
    context._show_busy()
    res, op, err = ifce.do_refresh_patch(context.console, patch)
    context._unshow_busy()
    if res == gquilt_const.ERROR_FORCE:
        if gquilt_gtk.info_force_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(context)).ask():
            res, op, err = ifce.do_refresh_patch(context.console, patch, force=True)
        else:
            return
    if res != gquilt_const.OK:
        gquilt_gtk.info_dialog(os.linesep.join([op, err]), parent=gquilt_gtk.get_gtk_window(context)).inform()

class patch_files_tree(gquilt_gtk.path_file_tree):
    def __init__(self, title='Files (in top patch)', patchname=None, tooltips=None):
        gquilt_gtk.path_file_tree.__init__(self, title)
        self.patchname = patchname
        if tooltips:
            self.set_tooltips(tooltips)
        self.playground = None
        self.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.append_menu_item("_Diff", "Display diff for selected files", gquilt_icons.diff, self._display_diff)
        self.append_conditional_menu_item("_Meld", "Display diff for selected files using \"meld\"", gquilt_icons.meld, self._display_diff_with_meld, self._meld_and_count_ok)
        self.reread_files()
    def set_playground(self, playground):
        self.playground = playground
    def _wait_for_child_timeout(self, pid):
        rpid, status = os.waitpid(pid, os.WNOHANG)
        return rpid != pid
    def _clean_up_after_child(self, pid):
        gobject.timeout_add(2000, self._wait_for_child_timeout, pid)
    def _read_files(self):
        res, lines, se = ifce.get_patch_files(self.patchname)
        if res != gquilt_const.OK:
            gquilt_gtk.info_dialog(os.linesep.join([lines, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        self._insert_lines(lines)
    def _insert_lines(self, lines):
        for line, status in lines:
            if status == gquilt_const.ADDED:
                fg = "#006600"
            elif status == gquilt_const.DELETED:
                fg = "#AA0000"
            else:
                fg = "black"
            found, iter = self.find_or_insert_file(line, foreground=fg)
            if not found:
                self.view.expand_to_path(self.store.get_path(iter))
            else:
                self.store.set_value(iter, 3, fg)
    def _handle_double_click(self):
        self.edit_selected_files()
    def reread_files(self):
        self.store.clear()
        self._read_files()
        self.view.expand_all()
    def update_files(self):
        res, lines, se = ifce.get_patch_files(self.patchname)
        if res != gquilt_const.OK:
            gquilt_gtk.info_dialog(os.linesep.join([lines, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        files = [x[0] for x in lines] 
        for f in self.store.get_file_paths():
            try:
                i = files.index(f)
            except:
                self.delete_file(f)
        self._insert_lines(lines)
    def update_files_busy(self):
        self._show_busy()
        self.update_files()
        self._unshow_busy()
    def update_playground_busy(self):
        if self.playground:
            self.playground.update_busy()
    def _display_diff(self, mi):
        files = self.get_selected_files()
        dialog = gquilt_diff_dialog(self.patchname, files)
        dialog.display()
    def _meld_and_count_ok(self, mi):
        if gquilt_utils.which("meld") is None:
            return False
        return ifce.count_ok_meld(self.view.get_selection().count_selected_rows())
    def _display_diff_with_meld(self, mi):
        files = self.get_selected_files()
        ifce.display_files_diff_in_viewer("meld", files, self.patchname)

class mutable_patch_files_tree(patch_files_tree):
    def __init__(self, title='Files (in top patch)', patchname=None, tooltips=None):
        patch_files_tree.__init__(self, title, patchname, tooltips)
        self.append_menu_item("_New", "Add a new file to the patch", gtk.STOCK_NEW, self.add_new_file)
        self.append_selection_menu_item("_Copy", "Make copies of the selected files", gtk.STOCK_COPY, self._copy_selection)
        self.append_selection_menu_item("_Move/Rename", "Move/rename the selected files in the patch", gtk.STOCK_DND, self._move_selection)
        self.append_menu_item("_Edit", "Edit selected files in the top patch", gtk.STOCK_EDIT, self.edit_selected_files)
        self.append_selection_menu_item("Re_solve", "Resolve problems with selected files in the top patch", gtk.STOCK_EDIT, self.resolve_selected_files)
        self.append_selection_menu_item("_Remove", "Remove the selected files from the top patch", gtk.STOCK_REMOVE, self._remove_selection_from_patch)
        self.append_selection_menu_item("Re_vert", "Revert the selected files to their state before the last refresh", gtk.STOCK_UNDO, self._revert_selection_in_patch)
        self.append_selection_menu_item("_Delete", "Delete the selected files", gtk.STOCK_DELETE, self._delete_selection)
        self.reread_files()
    def _popup_ok(self):
        ok = ifce.top_patch() != ""
        if not ok:
            gquilt_gtk.info_dialog("No patches applied", parent=gquilt_gtk.get_gtk_window(self)).inform()
        return ok
    def edit_selected_files(self, widget=None):
        files = self.get_selected_files()
        if len(files) == 0:
            res, files, se = ifce.get_patch_files(None, False)
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(se, parent=gquilt_gtk.get_gtk_window(self)).inform()
                return
        pid = gquilt_utils.edit_files(files)
        self._clean_up_after_child(pid)
    def resolve_selected_files(self, widget=None):
        files = self.get_selected_files()
        if len(files) == 0:
            res, files, se = ifce.get_patch_files(None, False)
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(se, parent=gquilt_gtk.get_gtk_window(self)).inform()
                return
        files_plus_rej = []
        for f in files:
            files_plus_rej.append(f)
            rejf = f + os.extsep + "rej"
            if os.path.exists(rejf):
                files_plus_rej.append(rejf)
        pid = gquilt_utils.edit_files(files_plus_rej)
        self._clean_up_after_child(pid)
    def add_new_file(self, mi):
        ok, name = gquilt_gtk.read_new_file_name()
        if not ok:
            return
        if os.path.isdir(name):
            gquilt_gtk.info_dialog("Cannot add directories to patch", parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        lname = _path_relative_to_playground(name)
        if lname == None:
            gquilt_gtk.info_dialog("File is outside playground", parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        if not os.path.exists(lname):
            created = True
            if self.console:
                self.console.exec_cmd("touch " + lname)
            else:
                gquilt_utils.run_cmd("touch " + lname)
        else:
            created = False
        self._show_busy()
        res, so, se = ifce.do_add_files_to_patch(self.console, [lname])
        self._unshow_busy()
        if res == gquilt_const.OK:
            self.update_files_busy()
            self.update_playground_busy()
        else:
            if created:
                try:
                    os.remove(lname)
                except:
                    pass
            gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
    def copy_files_to_top_patch(self, files, move=False):
        if len(files) == 0:
            return
        elif len(files) == 1:
            sugg = os.path.basename(files[0])
            startdir = os.path.join(os.getcwd(), os.path.dirname(files[0]))
            ok, dest = gquilt_gtk.read_new_file_name("Destination", startdir=startdir, sugg=sugg)
        else:
            ok, dest = gquilt_gtk.read_directory_name("Destination", ok_button_label=gtk.STOCK_OK)
        if not ok:
            return
        ldest = _path_relative_to_playground(dest)
        if ldest == None:
            gquilt_gtk.info_dialog("Destination is outside playground", parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        for file in files:
            self._show_busy()
            if move:
                res, so, se = ifce.do_move_file(self.console, file, ldest)
            else:
                res, so, se = ifce.do_copy_file(self.console, file, ldest)
            self._unshow_busy()
            if res == gquilt_const.ERROR_FORCE:
                if gquilt_gtk.info_force_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                    self._show_busy()
                    res, so, se = ifce.do_copy_file(self.console, file, ldest, True)
                    self._unshow_busy()
            elif res != gquilt_const.OK:
                if not gquilt_gtk.info_ok_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                    break
        self.update_files_busy()
        self.update_playground_busy()
    def _copy_selection(self, mi):
        self.copy_files_to_top_patch(self.get_selected_files())
    def _move_selection(self, mi):
        self.copy_files_to_top_patch(self.get_selected_files(), move=True)
    def _remove_selection_from_patch(self, mi):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = os.linesep.join([files[0], "",
                                   "Confirm remove selected file from patch?"])
        else:
            emsg = os.linesep.join(files + ["", "Confirm remove selected files from patch?"])
        if not gquilt_gtk.info_ok_dialog(emsg, parent=gquilt_gtk.get_gtk_window(self)).ask():
            return
        self._show_busy()
        res, so, se = ifce.do_remove_files_from_patch(self.console, files, self.patchname)
        self._unshow_busy()
        if res == gquilt_const.OK:
            self.update_files_busy()
            self.update_playground_busy()
        else:
            gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
    def _revert_selection_in_patch(self, mi):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = os.linesep.join([files[0], "",
                                   "Confirm revert changes to selected file?"])
        else:
            emsg = os.linesep.join(files + ["", "Confirm revert changes to selected files?"])
        if not gquilt_gtk.info_ok_dialog(emsg, parent=gquilt_gtk.get_gtk_window(self)).ask():
            return
        self._show_busy()
        res, so, se = ifce.do_revert_files_in_patch(self.console, files, self.patchname)
        self._unshow_busy()
        if res == gquilt_const.OK:
            self.update_files_busy()
            self.update_playground_busy()
        else:
            gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
    def update_patch(self):
        self.update_files_busy()
        self.update_playground_busy()
    def refresh_patch(self):
        refresh_patch(self, self.patchname)
        self.update_patch()
    def _delete_selection(self, mi):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = os.linesep.join([files[0], "",
                                   "Confirm delete selected file?"])
        else:
            emsg = os.linesep.join(files + ["", "Confirm delete selected files?"])
        if not gquilt_gtk.info_ok_dialog(emsg, parent=gquilt_gtk.get_gtk_window(self)).ask():
            return
        for file in files:
            try:
                os.remove(file)
            except OSError, (errno, error_message):
                emsg = os.linesep.join([file + ": " + error_message, "Continue?"])
                if gquilt_gtk.info_ok_dialog(emsg, parent=gquilt_gtk.get_gtk_window(self)).ask():
                    continue
        self.update_patch()

class gquilt_patch_file_dialog(gquilt_gtk.update_dialog):
    def __init__(self, patchname, tooltips=None):
        gquilt_gtk.update_dialog.__init__(self, "Patch Files: " + patchname)
        self.file_tree = patch_files_tree('Files for "' + patchname + '" patch', patchname, tooltips=tooltips)
        self.file_tree.set_size_request(240, 320)
        self.file_tree.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.vbox.pack_start(self.file_tree)
    def do_update(self):
        self.file_tree.update_files()

class gquilt_diff_dialog(gquilt_gtk.update_save_dialog):
    def __init__(self, patchname, files=[]):
        if patchname is None:
            self.patch = ifce.top_patch()
        else:
            self.patch = patchname
        title = 'gquilt: "' + self.patch + '" diff' + " ".join(files)
        gquilt_gtk.update_save_dialog.__init__(self, title)
        self.text = gquilt_gtk.scrolled_diff_text()
        x, y = self.text.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.text.set_size_request(x, y)
        self.action_area.pack_end(self.text.tws_display, expand=False, fill=False)
        self.action_area.reorder_child(self.text.tws_display, 0)
        self.vbox.pack_start(self.text)
        self.files = files
        self.do_update()
    def _editable_diff(self):
        return self.files == [] and not ifce.is_patch_applied(self.patch)
    def do_update(self):
        res, op, err = ifce.get_diff(self.files, self.patch)
        self.text.set_contents(op)
        if self.filename or self._editable_diff():
            self.text.view.set_editable(True)
            self.text.view.set_cursor_visible(True)
            self.set_response_sensitive(2, True)
        else:
            self.text.view.set_editable(False)
            self.text.view.set_cursor_visible(False)
            self.set_response_sensitive(2, False)
        if res != 0:
            gquilt_gtk.info_dialog(err, parent=gquilt_gtk.get_gtk_window(self)).inform()
    def do_write_to_file(self, filename):
        if not filename and self._editable_diff():
            text = self.text.text.get_text(self.text.text.get_start_iter(), self.text.text.get_end_iter())
            res = gquilt_pfuns.set_patch_diff_lines(ifce.patch_file_name(self.patch), text.splitlines())
            if not res:
                gquilt_gtk.info_dialog("Save modified \"%s\" patch failed" % self.patch, parent=gquilt_gtk.get_gtk_window(self)).inform()
            return True
        return self.text.save_to_file(filename)

class gquilt_combined_diff_dialog(gquilt_gtk.update_save_dialog):
    def __init__(self, start_patch=None, end_patch=None):
        self.start_patch = start_patch
        self.end_patch = end_patch
        title = 'gquilt: combined diff'
        if start_patch is None:
            if end_patch is None:
                title += ' (all applied patches)'
        else:
            title += ' from patch "' + end_patch + '"'
        if end_patch is not None:
            title += ' to patch "' + end_patch + '"'
        gquilt_gtk.update_save_dialog.__init__(self, title)
        self.text = gquilt_gtk.scrolled_diff_text(title)
        x, y = self.text.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.text.set_size_request(x, y)
        self.vbox.pack_start(self.text)
        self.do_update()
    def do_update(self):
        res, op, err = ifce.get_combined_diff(self.start_patch, self.end_patch)
        self.text.set_contents(op)
        if res != 0:
            gquilt_gtk.info_dialog(err, parent=gquilt_gtk.get_gtk_window(self)).inform()
            self.destroy()
    def do_write_to_file(self, filename):
        return self.text.save_to_file(filename)

class gquilt_descr_dialog(gtk.Dialog):
    def __init__(self, patchname, console=None):
        gtk.Dialog.__init__(self, "Description: " + patchname, None,
            gtk.DIALOG_DESTROY_WITH_PARENT,
            ("Sign _Off", 3, "_Ack", 4, "Auth_or", 5, "_Update", 1, gtk.STOCK_SAVE, 2,
             gtk.STOCK_CLOSE, 6))
        self.connect("response", self._process_response)
        #self.set_response_sensitive(2, False)
        self.view = gtk.TextView()
        self.text = self.view.get_buffer()
        x, y = self.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.view.set_size_request(x, y)
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.view)
        sw.show_all()
        self.vbox.pack_start(sw)
        self.view.set_cursor_visible(True)
        self.view.set_editable(True)
        self.patchname = patchname
        self.console = console
        self._do_update()
    def _do_update(self):
        if self.text.get_modified() and not gquilt_gtk.info_ok_dialog("Contents modified.  Really update?", parent=gquilt_gtk.get_gtk_window(self)).ask():
            return
        res, so, se = ifce.get_patch_description(self.patchname)
        if res != 0:
            gquilt_gtk.info_dialog(se, parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        self.text.set_text(so)
        self.text.set_modified(False)
    def _do_save(self):
        text = self.text.get_text(self.text.get_start_iter(), self.text.get_end_iter())
        res, so, se = ifce.do_set_patch_description(self.console, self.patchname, text)
        if res != 0:
            gquilt_gtk.info_dialog(se, parent=gquilt_gtk.get_gtk_window(self)).inform()
            return
        self.text.set_modified(False)
    def _process_response(self, dialog, response):
        if response == 1:
            self._do_update()
        elif response == 2:
            self._do_save()
        elif response == 3:
            self.text.insert_at_cursor("Signed-off-by: %s\n" % ifce.get_author_name_and_email())
        elif response == 4:
            self.text.insert_at_cursor("Acked-by: %s\n" % ifce.get_author_name_and_email())
        elif response == 5:
            self.text.insert_at_cursor("Author: %s\n" % ifce.get_author_name_and_email())
        elif response == 6:
            self._do_close()
        else:
            pass
    def _do_close(self):
        if self.text.get_modified() and not gquilt_gtk.info_ok_dialog("Contents modified.  Really close?", parent=gquilt_gtk.get_gtk_window(self)).ask():
            return
        self.destroy()
    def display(self):
        self.show_all()

class new_patch_dialog(gtk.Dialog):
    def __init__(self, title="New Patch Specification",
                 nameprompt="Enter new patch name:",
                 descr="", parent=None):
        gtk.Dialog.__init__(self, title, parent,
                            gtk.DIALOG_DESTROY_WITH_PARENT,
                            ("Sign _Off", 1, "_Ack", 2, "Auth_or", 3,
                             gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                             gtk.STOCK_OK, gtk.RESPONSE_OK)
                           )
        self.connect("response", self._process_response)
        if parent is None:
            self.set_position(gtk.WIN_POS_MOUSE)
        else:
            self.set_position(gtk.WIN_POS_CENTER_ON_PARENT)
        hbox = gtk.HBox()
        self.vbox.pack_start(hbox)
        hbox.show()
        self.prompt = gtk.Label(nameprompt)
        hbox.pack_start(self.prompt, expand=False)
        self.prompt.show()
        self.te = gtk.Entry()
        self.te.connect("activate", self._enter)
        self.te.show()
        hbox.add(self.te)
        hbox = gtk.HBox()
        self.vbox.pack_start(hbox)
        hbox.show()
        self.descr_label = gtk.Label("Description:")
        hbox.pack_start(self.descr_label, expand=False)
        self.descr_label.show()
        self.view = gtk.TextView()
        self.text = self.view.get_buffer()
        self.text.set_text(descr)
        x, y = self.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.view.set_size_request(x, y)
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.view)
        sw.show_all()
        self.vbox.pack_start(sw)
        self.view.set_cursor_visible(True)
        self.view.set_editable(True)
    def _enter(self, obj):
        if len(obj.get_text()) > 0:
            self.response(gtk.RESPONSE_OK)
        else:
            self.response(gtk.RESPONSE_CANCEL)
    def _process_response(self, dialog, response):
        if response == 1:
            self.emit_stop_by_name("response")
            self.text.insert_at_cursor("Signed-off-by: %s\n" % ifce.get_author_name_and_email())
        elif response == 2:
            self.emit_stop_by_name("response")
            self.text.insert_at_cursor("Acked-by: %s\n" % ifce.get_author_name_and_email())
        elif response == 3:
            self.emit_stop_by_name("response")
            self.text.insert_at_cursor("Author: %s\n" % ifce.get_author_name_and_email())
        else:
            pass
    def read_name_and_description(self):
        res = self.run()
        name = re.sub('[^A-Za-z0-9./\-]', '_', self.te.get_text())
        descr = self.text.get_text(self.text.get_start_iter(), self.text.get_end_iter())
        self.destroy()
        return (res == gtk.RESPONSE_OK and len(name) > 0, name, descr)

class gquilt(gtk.Window):
    def _create_menu_item(self, label, stock_id, action, tooltip=None, arg=None):
        mi = gtk.ImageMenuItem(label)
        if stock_id is not None:
            im = gtk.Image()
            im.set_from_stock(stock_id, gtk.ICON_SIZE_MENU)
            mi.set_image(im)
        mi.show()
        if arg == None:
            mi.connect("activate", action)
        else:
            mi.connect("activate", action, arg)
        if tooltip != None:
            self.tooltips.set_tip(mi, tooltip)
        return mi
    def _create_menu_bar(self):
        mb = gtk.MenuBar()
        pgm = gtk.Menu()
        pgm.append(self._create_menu_item(gtk.STOCK_NEW, gtk.STOCK_NEW, self._new_playground, "Create a new playground"))
        pgm.append(self._create_menu_item(gtk.STOCK_OPEN, gtk.STOCK_OPEN, self._open_playground, "Select a different playground"))
        pgm.append(gtk.SeparatorMenuItem())
        pgm.append(self._create_menu_item(gtk.STOCK_QUIT, gtk.STOCK_QUIT, self._quit, "Quit this program"))
        pgmi = gtk.MenuItem("_Playground")
        pgmi.set_submenu(pgm)
        mb.append(pgmi)
        actm = gtk.Menu()
        actm.append(self._create_menu_item("Pop  All", gquilt_icons.pop, self._pop_all_patches, "Remove all applied patches"))
        actm.append(self._create_menu_item("Push  All", gquilt_icons.push, self._push_all_patches, "Apply all patches in series"))
        actm.append(self._create_menu_item("Combined Diff", gquilt_icons.diff, self._display_combined_diff, "Get a combined diff for all applied patches"))
        actm.append(self._create_menu_item("Import Patch Series", gquilt_icons.import_patch, self._import_patch_series, "Import a series of quilt/mq patches"))
        actmi = gtk.MenuItem("_Actions")
        actmi.set_submenu(actm)
        mb.append(actmi)
        vctm = gtk.Menu()
        vctm.append(self._create_menu_item("Update  All", None, self._update_views, "Update all views"))
        vctm.append(self._create_menu_item("Update  Playground", None, self._update_playground_view, "Update playground files view"))
        vctm.append(self._create_menu_item("Update  Files", None, self._update_files_view, "Update patch files view"))
        vctm.append(self._create_menu_item("Update  Patches", None, self._update_patches_view, "Update patch list view"))
        vctmi = gtk.MenuItem("_Views")
        vctmi.set_submenu(vctm)
        mb.append(vctmi)
        mb.show_all()
        return mb
    def _tb_button(self, label, tooltip, stock_id, callback):
        im = gtk.Image()
        im.set_from_stock(stock_id, gtk.ICON_SIZE_LARGE_TOOLBAR)
        tbb = gtk.ToolButton(im, label)
        tbb.set_expand(False)
        tbb.connect("clicked", callback)
        if tooltip is not None:
            tbb.set_tooltip(self.tooltips, tooltip)
        return tbb
    def _create_tool_bar(self):
        tb = gtk.Toolbar()
        tb.set_orientation(gtk.ORIENTATION_HORIZONTAL)
        tb.set_style(gtk.TOOLBAR_BOTH)
        tb.insert(self._tb_button("Refresh", "Refresh the top patch", gtk.STOCK_REFRESH, self._refresh_top_patch), -1)
        tb.insert(self._tb_button("Push", "Push the next patch in the series", gquilt_icons.push, self.patches.push_next_patch), -1)
        tb.insert(self._tb_button("Pop", "Pop the top patch", gquilt_icons.pop, self.patches.pop_top_patch), -1)
        tb.insert(self._tb_button("New", "Start a new patch", gtk.STOCK_ADD, self._new_patch), -1)
        tb.insert(self._tb_button("Import", "Import an external patch", gquilt_icons.import_patch, self._import_patch), -1)
        tb.insert(self._tb_button("Fold", "Merge an external patch into the top patch", gquilt_icons.fold, self._fold_patch_file), -1)
        self.cmd_line = gquilt_gtk.EntryWithHistory()
        self.cmd_line.connect("activate", self._exec_typed_cmd)
        tbcl = gtk.ToolItem()
        tbcl.set_tooltip(self.tooltips, "Type in a back end command for execution")
        tbcl.set_expand(True)
        tbcl.add(self.cmd_line)
        tb.insert(tbcl, -1)
        return tb
    def _create_console(self):
        self.console = gquilt_gtk.console()
        self.console.set_size_request(720, 160)
    def _create_playground_file_tree(self):
        self.playground_files = gquilt_playground_tree(tooltips=self.tooltips, console=self.console, update_views=self._update_views)
        self.playground_files.set_size_request(240, 320)
    def _create_patch_file_tree(self):
        self.patch_files = mutable_patch_files_tree(tooltips=self.tooltips)
        self.patch_files.set_size_request(240, 320)
    def _create_patch_list(self):
        self.patches = gquilt_patch_list(tooltips=self.tooltips)
        self.patches.set_size_request(240, 320)
    def __init__(self):
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        try:
            self.set_icon_from_file(os.environ['GQUILT_ICON'])
        except:
            pass
        self.last_import_dir = None
        self.connect("destroy", self._quit)
        self.set_title("gquilt: " + os.getcwd())
        self.tooltips = gtk.Tooltips()
        self._create_console()
        self._create_patch_file_tree()
        self._create_patch_list()
        self._create_playground_file_tree()
        vbox = gtk.VBox()
        self.add(vbox)
        vbox.pack_start(self._create_menu_bar(), False)
        vbox.pack_start(self._create_tool_bar(), False)
        vpane = gtk.VPaned()
        vbox.pack_start(vpane)
        hpane_outer = gtk.HPaned()
        hpane_inner = gtk.HPaned()
        self.patch_files.set_console(self.console)
        self.patch_files.set_playground(self.playground_files)
        self.patches.set_console(self.console)
        self.patches.set_playground(self.playground_files)
        self.patches.set_patch_files(self.patch_files)
        self.playground_files.set_patch_files(self.patch_files)
        hpane_outer.add1(self.playground_files)
        hpane_outer.add2(hpane_inner)
        hpane_inner.add1(self.patch_files)
        hpane_inner.add2(self.patches)
        vpane.add1(hpane_outer)
        vpane.add2(self.console)
        self.show_all()
        self.show()
        self._set_playground(os.getcwd())
        self.set_focus(self.cmd_line)
    def _quit(self, widget):
        gtk.main_quit()
    def _add_playground_to_title(self):
        self.set_title("gquilt: " + os.getcwd())
    def _show_busy(self):
        self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        while gtk.events_pending():
            gtk.main_iteration()
    def _unshow_busy(self):
        self.window.set_cursor(None)
    def _update_patches_view(self, widget=None):
        self._show_busy()
        self.patches.update_contents()
        self._unshow_busy()
    def _update_files_view(self, widget=None):
        self.patch_files.update_files_busy()
    def _update_playground_view(self, widget=None):
        self.playground_files.update_busy()
    def _update_views(self, widget=None):
        self._show_busy()
        self.patches.update_contents()
        self.patch_files.update_files()
        self.playground_files.update()
        self._unshow_busy()
    def _display_combined_diff(self, widget):
        dialog = gquilt_combined_diff_dialog()
        dialog.display()
    def _set_playground(self, newpg):
        global backend
        os.chdir(newpg)
        newpgtype = playground_type()
        if newpgtype:
            newpg = backend[newpgtype].get_playground_root()
            os.chdir(newpg)
        elif len(backend) > 0:
            msg = os.linesep.join(["Directory %s is not repository." % newpg,
                                   "Do you wish to create one there?"])
            if gquilt_gtk.info_yes_no_dialog(msg, self).ask():
                ok, req_backend = choose_backend()
                if ok:
                    res, so, se = backend[req_backend].new_playground(newpg)
                    if res != gquilt_const.OK:
                        gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
        else:
            report_backend_requirements(gquilt_gtk.get_gtk_window(self))
        set_ifce()
        self._show_busy()
        self.playground_files.set_root(newpg)
        self.patches.update_contents()
        self.patch_files.reread_files()
        self._add_playground_to_title()
        self.console.log_entry_bold("playground: " + newpg)
        self._unshow_busy()
    def _new_playground(self, widget):
        ok, req_backend = choose_backend()
        if not ok:
            return
        ok, newpg = gquilt_gtk.read_directory_name("Select/create playground ..")
        if ok:
            res, so, se = backend[req_backend].new_playground(newpg)
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                return
            self._set_playground(newpg)
    def _open_playground(self, widget):
        ok, newpg = gquilt_gtk.read_directory_name("Select playground ..")
        if ok:
            self._set_playground(newpg)
    def _import_patch(self, widget):
        ok, name = gquilt_gtk.read_file_name("Select patch to import ..", startdir=self.last_import_dir)
        if ok:
            self.last_import_dir = os.path.dirname(name)
            self._show_busy()
            res, so, se = ifce.do_import_patch(self.console, name)
            self._unshow_busy()
            if res == gquilt_const.ERROR_FORCE:
                if gquilt_gtk.info_force_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                    res, so, err = ifce.do_import_patch(self.console, name, force=True)
                else:
                    return
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            self.patches.update_contents()
    def _import_patch_series(self, widget=None):
        ok, dirname = gquilt_gtk.read_directory_name("Select directory to import patch series from ..", startdir=self.last_import_dir)
        if ok:
            self.last_import_dir = dirname
            sfname = os.path.join(dirname, "series")
            if not os.path.exists(sfname):
                gquilt_gtk.info_dialog("Series file not found", parent=gquilt_gtk.get_gtk_window(self)).inform()
                return
            sf = open(sfname, 'r')
            series = sf.readlines()
            series.reverse()
            sf.close()
            for name_raw in series:
                name = name_raw.strip()
                if name == "" or name[0] == "#":
                    continue
                pfname = os.path.join(dirname, name)
                self._show_busy()
                res, so, se = ifce.do_import_patch(self.console, pfname)
                self._unshow_busy()
                if res == gquilt_const.ERROR_FORCE:
                    ans = gquilt_gtk.info_force_or_skip_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask()
                    if ans == gquilt_gtk.FORCE:
                        res, so, err = ifce.do_import_patch(self.console, pfname, force=True)
                    elif ans == gquilt_gtk.SKIP:
                        continue
                    else:
                        return
                if res != gquilt_const.OK:
                    gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
                self.patches.update_contents()
    def _fold_patch_file(self, widget):
        ok, name = gquilt_gtk.read_file_name("Select patch to merge ..", self.last_import_dir)
        if ok:
            self.last_import_dir = os.path.dirname(name)
            self._show_busy()
            res, so, se = ifce.do_merge_patch_file(self.console, name)
            self._unshow_busy()
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            self._update_views()
    def _new_patch(self, widget):
        dialog = new_patch_dialog()
        res, name, descr = dialog.read_name_and_description()
        if res:
            while True:
                self._show_busy()
                res, so, se = ifce.do_create_new_patch(self.console, name)
                self._unshow_busy()
                if res == gquilt_const.ERROR_REFRESH:
                    if gquilt_gtk.info_refresh_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                        self.patch_files.refresh_patch()
                        continue
                    else:
                        return
                elif res == gquilt_const.ERROR_FORCE:
                    if gquilt_gtk.info_force_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask():
                        self._show_busy()
                        res, op, err = ifce.do_create_new_patch(self.console, name, force=True)
                        self._unshow_busy()
                        break
                    else:
                        return
                elif res == gquilt_const.ERROR_FORCE_OR_REFRESH:
                    ans = gquilt_gtk.info_force_or_refresh_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).ask()
                    if ans == gquilt_gtk.REFRESH_AND_RETRY:
                        self.patch_files.refresh_patch()
                        continue
                    elif ans == gquilt_gtk.FORCE:
                        self._show_busy()
                        res, op, err = ifce.do_create_new_patch(self.console, name, force=True)
                        self._unshow_busy()
                        break
                    else:
                        return
                else:
                    break
            if res != gquilt_const.OK:
                gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
            elif descr != "":
                res, so, se = ifce.do_set_patch_description(self.console, name, descr)
                if res != 0:
                    gquilt_gtk.info_dialog(se, parent=gquilt_gtk.get_gtk_window(self)).inform()
            self._update_views()
    def _refresh_top_patch(self, widget=None):
        self.patch_files.refresh_patch()
    def _pop_all_patches(self, widget):
        self.patches.pop_to_patch("")
    def _push_all_patches(self, widget):
        self.patches.push_to_patch(ifce.last_patch_in_series())
    def _exec_typed_cmd(self, entry):
        self._show_busy()
        text = entry.get_text_and_clear_to_history()
        if text:
            res, so, se = ifce.do_exec_tool_cmd(self.console, text)
        else:
            res = gquilt_const.OK
        self._unshow_busy()
        if res != gquilt_const.OK:
            gquilt_gtk.info_dialog(os.linesep.join([so, se]), parent=gquilt_gtk.get_gtk_window(self)).inform()
        self._update_views()

if __name__ == "__main__":
    if len(backend) > 0 and playground_type() is None:
        ok, newpg = gquilt_gtk.read_directory_name("Select playground ..")
        if ok:
            os.chdir(newpg)
    else:
        ok = True
    if ok:
        app = gquilt()
        gtk.main()
