# -*- coding: utf-8 -*-
#
# Gramps - a GTK+/GNOME based genealogy program
#
# Copyright (C) 2003-2004  Donald N. Allingham
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

# $Id: rel_da.py,v 1.1.2.1 2004/08/02 20:00:57 rshura Exp $

# Written by Alex Roitman, largely based on Relationship.py by Don Allingham
# and on valuable input from Lars Kr. Lundin

#-------------------------------------------------------------------------
#
# GRAMPS modules
#
#-------------------------------------------------------------------------

import RelLib
from gettext import gettext as _

#-------------------------------------------------------------------------
#
# Danish-specific definitions of relationships
#
#-------------------------------------------------------------------------

_parents_level = [ "forældre", "bedsteforældre", "oldeforældre",
"tipoldeforældre", "tiptipoldeforældre" , "tiptiptipoldeforældre" ]

#-------------------------------------------------------------------------
#
#
#
#-------------------------------------------------------------------------
def get_parents(level):
    if level>len(_parents_level)-1:
        #return "fjern forfader"
        #Instead of "remote ancestors" using "tip (level) oldeforældre" here.
        return "tip (%d) oldeforældre" % level
    else:
        return _parents_level[level]

def pair_up(rel_list):
    result = []
    item = ""
    for word in rel_list[:]:
        if not word:
            continue
        if item:
            if word == 'søster':
                item = item[0:-1]
                word = 'ster'
            elif word == 'sønne':
                word = 'søn'
            result.append(item + word)
            item = ""
        else:
            item = word
    if item:
        result.append(item)
    gen_result = [ item + 's' for item in result[0:-1] ]
    return ' '.join(gen_result+result[-1:])

def get_direct_ancestor(person,rel_string):
    result = []
    for ix in range(len(rel_string)):
        if rel_string[ix] == 'f':
            result.append('far')
        else:
            result.append('mor')
    return pair_up(result)

def get_direct_descendant(person,rel_string):
    result = []
    for ix in range(len(rel_string)-2,-1,-1):
        if rel_string[ix] == 'f':
            result.append('sønne')
        else:
            result.append('datter')
    if person.getGender() == RelLib.Person.male:
        result.append('søn')
    else:
        result.append('datter')
    return pair_up(result)

def get_two_way_rel(person,first_rel_string,second_rel_string):
    result = []
    for ix in range(len(second_rel_string)-1):
        if second_rel_string[ix] == 'f':
            result.append('far')
        else:
            result.append('mor')
    if len(first_rel_string)>1:
        if first_rel_string[-2] == 'f':
            result.append('bror')
        else:
            result.append('søster')
        for ix in range(len(first_rel_string)-3,-1,-1):
            if first_rel_string[ix] == 'f':
                result.append('sønne')
            else:
                result.append('datter')
        if person.getGender() == RelLib.Person.male:
            result.append('søn')
        else:
            result.append('datter')
    else:
        if person.getGender() == RelLib.Person.male:
            result.append('bror')
        else:
            result.append('søster')
    return pair_up(result)
    
def is_spouse(orig,other):
    for f in orig.getFamilyList():
        if other == f.getFather() or other == f.getMother():
            return 1
    return 0

def apply_filter(person,rel_str,plist,pmap):
    if person == None:
        return
    plist.append(person)
    pmap[person.getId()] = rel_str
    
    family = person.getMainParents()
    if family:
        apply_filter(family.getFather(),rel_str+'f',plist,pmap)
        apply_filter(family.getMother(),rel_str+'m',plist,pmap)

def get_relationship(orig_person,other_person):
    """
    Returns a string representing the relationshp between the two people,
    along with a list of common ancestors (typically father,mother) 
    
    Special cases: relation strings "", "undefined" and "spouse".
    """

    firstMap = {}
    firstList = []
    secondMap = {}
    secondList = []
    common = []
    rank = 9999999
    
    if orig_person == None:
        return ("undefined",[])
    
    firstName = orig_person.getPrimaryName().getRegularName()
    secondName = other_person.getPrimaryName().getRegularName()
    
    if orig_person == other_person:
        return ('', [])
    if is_spouse(orig_person,other_person):
        if other_person.getGender() == RelLib.Person.male:
            return ("ægtefælle",[])
        else:
            return ("hustru",[])

    try:
        apply_filter(orig_person,'',firstList,firstMap)
        apply_filter(other_person,'',secondList,secondMap)
    except RuntimeError,msg:
        # Danish: 'Fandt en slægtskabsløkke'
        # Should be already translated in po file.
        return (_("Relationship loop detected"),None)
    
    for person in firstList:
        if person in secondList:
            new_rank = len(firstMap[person.getId()])
            if new_rank < rank:
                rank = new_rank
                common = [ person ]
            elif new_rank == rank:
                common.append(person)

    if common:
        person = common[0]
        secondRel = firstMap[person.getId()]
        firstRel = secondMap[person.getId()]
    else:
        return ("",[])

    if not firstRel:
        if not secondRel:
            return ('',common)
        else:
            return (get_direct_ancestor(other_person,secondRel),common)
    elif not secondRel:
        return (get_direct_descendant(other_person,firstRel),common)
    else:
        return (get_two_way_rel(other_person,firstRel,secondRel),common)


def get_grandparents_string(orig_person,other_person):
    """
    returns a string representing the relationship between the two people,
    along with a list of common ancestors (typically father,mother) 
    """
    firstMap = {}
    firstList = []
    secondMap = {}
    secondList = []
    common = []
    rank = 9999999

    if orig_person == None:
        return ("undefined",[])

    if orig_person == other_person:
        return ('', [])

    try:
        apply_filter(orig_person,'',firstList,firstMap)
        apply_filter(other_person,'',secondList,secondMap)
    except RuntimeError,msg:
        # FIXME: Danish: 'Fandt en slægtskabsløkke'
        return (_("Relationship loop detected"),None)
    
    for person in firstList:
        if person in secondList:
            new_rank = len(firstMap[person.getId()])
            if new_rank < rank:
                rank = new_rank
                common = [ person ]
            elif new_rank == rank:
                common.append(person)

    if common:
        person = common[0]
        secondRel = firstMap[person.getId()]
        firstRel = secondMap[person.getId()]
    else:
        return ('', [])

    if not firstRel:
        if not secondRel:
            return ('',common)
        else:
            return (get_parents(len(secondRel)),common)
    else:
        return None

#-------------------------------------------------------------------------
#
# Register this function with the Plugins system 
#
#-------------------------------------------------------------------------
from Plugins import register_relcalc, register_parents

register_relcalc(get_relationship,
    ["da","DA","da_DK","danish","Danish","da_DK.UTF8","da_DK@euro","da_DK.UTF8@euro",
            "dansk","Dansk", "da_DK.UTF-8", "da_DK.utf-8", "da_DK.utf8"])

register_parents(get_grandparents_string,
    ["da","DA","da_DK","danish","Danish","da_DK.UTF8","da_DK@euro","da_DK.UTF8@euro",
            "dansk","Dansk", "da_DK.UTF-8", "da_DK.utf-8", "da_DK.utf8"])
