// -*- c++ -*-
//------------------------------------------------------------------------------
//                              CardDeck.cpp
//------------------------------------------------------------------------------
// $Id: CardDeck.cpp,v 1.18 2005/11/13 02:50:48 vlg Exp $
//
//  Copyright (c) 2004 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Apr 24 2004
//------------------------------------------------------------------------------

#include <sstream>
#include <algorithm>

#include "CardDeck.h"
#include "Granule-main.h"
#include "Intern.h"

static const char* box_names [] = 
{ 
	N_("Card Box 1"), N_("Card Box 2"),
	N_("Card Box 3"), N_("Card Box 4"), 
	N_("Card Box 5") 
};

void
CardDeck::
set_name (int idx_)
{
	m_cdname = box_names [idx_];
}

int
CardDeck::
get_index () const
{
	for (int idx = 0; idx < CARD_BOX_SIZE; idx++) {
		if (m_cdname == box_names [idx]) {
			return (idx);
		}
	}
	Assure_exit (false);
}

bool
CardDeck::
erase (VCard* vcard_)
{
	trace_with_mask("CardDeck::erase(VCard)",GUITRACE|DECK);

	cardlist_iterator p = m_cards.begin ();
	while (p != m_cards.end ()) {
		if ((*p) == vcard_) {
			erase (p);
		}
		p++;
	}
}

void 
CardDeck::
play_card (VDeck::cardlist_iterator& iter_)
{
	CardRef* cref = dynamic_cast<CardRef*>(*iter_);
	Assure_exit (cref);
	cref->get_deck ().play_card (iter_);
}

void 
CardDeck::
remove_duplicates ()
{
	trace_with_mask("CardDeck::remove_duplicates",GUITRACE);

	CardRef* cr_p;
	CardRef* cr_q;

	std::sort (m_cards.begin (), m_cards.end (), Compare_CardRefs ());

	cardlist_iterator p = m_cards.begin ();
	cardlist_iterator q = p + 1;
	while (q != m_cards.end ()) {
		cr_p = dynamic_cast<CardRef*> (*p);
		cr_q = dynamic_cast<CardRef*> (*q);
		Assure_exit (cr_p && cr_q);
		if (cr_p->get_card ().get_id () == cr_q->get_card ().get_id ()) {
			delete (*q);
			m_cards.erase (q);
			q = p;
		}
		else {
			p = q;
		}
		q++;
	}
}

/** 
 * Report the ratio of cards visited vs. total and percentage-wise.
 * The size of the deck is a constant determined at the moment when
 * DeckPlayer starts playing it. It can change when a card is moved
 * to the upper or lower box. It can also remain constant when
 * the card is sent to the end of the deck for repetition (fringe
 * cases). The important concept is that we want to show the user
 * how many cards he has seen so far and how many remains to be seen
 * at least once.
 * We, therefore, consider initial Deck size and count our advance
 * through the Deck (first sweep) ourselves.
 */
void
CardDeck::
get_progress (float& pct_done_, std::string& msg_, int /* idx_ */)
{
	trace_with_mask("CardDeck::get_progress",GUITRACE);
	std::ostringstream os;

	DL((GRAPP,"m_init_size = %d; m_progress_idx = %d\n",
		m_init_size, m_progress_idx));

	if (m_init_size == 0) {
		pct_done_ = 0;
		msg_ = "";
		return;
	}
	if (m_progress_idx == m_init_size) {
		pct_done_ = 1.0;
		os << m_progress_idx << G_DIR_SEPARATOR_S << m_init_size;
		return;
	}
	m_progress_idx++;
	os << (m_progress_idx) << G_DIR_SEPARATOR_S << m_init_size;
	msg_ = os.str ();

	if (m_cards.size () <= 1) {
		pct_done_ = 0;
	}
	else {
		pct_done_ = (m_progress_idx) * 1.0 / m_init_size;
	}

}

void
CardDeck::
clear ()
{
	trace_with_mask("CardDeck::clear",GUITRACE);
	
	cardlist_iterator p = m_cards.begin ();
	while (p != m_cards.end ()) {
		delete (*p);
		p++;
	}
	m_cards.clear ();
}

void
CardDeck::
shuffle_subsets ()
{
	trace_with_mask("CardDeck::shuffle_subsets",GUITRACE);

	if (m_cards.size () <= 1) {
		return;
	}
	/** Algorithm
	 */
	bool inset = false;			// are we inside the subset?
	cardlist_iterator first;	
	cardlist_iterator second;	
	first = second = m_cards.begin ();

	while (second + 1 != m_cards.end ()) {
		second++;
		if ((**first) == (**second)) {
			inset = true;
			continue;
		}
		else {
			if (inset) {
				random_shuffle (first, second);
				inset = false;
			}
			first = second;
		}
	}
	if (inset) {
		random_shuffle (first, second+1);
	}
}
