// -*- c++ -*-
//------------------------------------------------------------------------------
//                              CardDeck.h
//------------------------------------------------------------------------------
// $Id: CardDeck.h,v 1.16 2005/11/13 02:50:48 vlg Exp $
//
//  Copyright (c) 2004 by Vladislav Grinchenko
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//
// Date: Apr 24 2004
//------------------------------------------------------------------------------
#ifndef CARD_DECK_H
#define CARD_DECK_H

#include "CardRef.h"

class VCard;

/** To remove duplicates we compare Card References by the IDs of 
	the cards they point to.
*/
class Compare_CardRefs {
public:
	int operator () (VCard* const& vc1_, VCard* const& vc2_)
	{
		CardRef* cr1 = dynamic_cast<CardRef*> (vc1_);
		CardRef* cr2 = dynamic_cast<CardRef*> (vc2_);
		Assure_exit (cr1 && cr2);
		return (cr1->get_card ().get_id () < cr2->get_card ().get_id ());
	}
};

class Compare_ExpTime_CardRefs {
public:
	int operator () (VCard* const& vc1_, VCard* const& vc2_)
	{
		return ((*vc1_) < (*vc2_));
	}
};

/** 
 * CardDeck represents each of the Card Boxes on the left of the MainWindow.
 * Each CardDeck holds a list of virtual cards, CardRef, that points
 * to the real Card in the real Deck.
 *
 * Because it is derived from VDeck, CardDeck can be played by DeckViewer.
 */

class CardDeck : public VDeck
{
public:
	CardDeck ();
	virtual ~CardDeck ();

	virtual cardlist_const_iterator begin() const { return m_cards.begin(); }
	virtual cardlist_const_iterator end() const { return m_cards.end (); }

	virtual cardlist_iterator begin () { return m_cards.begin (); }
	virtual cardlist_iterator end   () { return m_cards.end (); }

	virtual size_t size  () const { return m_cards.size (); }
	virtual bool   empty () const { return m_cards.empty (); }

	virtual void push_back (VCard* card_);

	/** Erase card from the VCard list only. Calling this method leaves 
		the Deck intact.
	*/
	virtual void erase (cardlist_iterator iter_);

	/** Erase card from the VCard list only. Calling this method leaves
		the Deck intact.
	 */
	virtual bool erase (VCard* card_);
		
	/** Remove all card references
	 */
	void clear ();

	/** Sort Card References based on their expiration time.
		Those with oldest expiration dates go to the front of the list.
	*/
	void sort_by_expiration_time ();

	virtual void play_card (VDeck::cardlist_iterator& iter_);

	virtual const string&  get_name () const { return m_cdname; }

	virtual int  get_index () const;
	virtual void get_progress (float& pct_done_, std::string& msg_, int idx_);
	virtual void reset_progress ();

	void set_name (int idx_);
	void remove_duplicates (); 

	/** Shuffle all subsets with equal expiration 
		timestamp, preserving the general expiration order.
	*/
	void shuffle_subsets ();

private:
	string         m_cdname;	// Number of the card box this deck represents
	cardlist_type  m_cards;

    /// The size at the moment when DeckPlayer starts playing the deck.
	u_int          m_init_size;	

	/// How many cards has already been played by the DeckPlayer so far.
	u_int          m_progress_idx;
	
};

inline
CardDeck::
CardDeck () : m_init_size (0), m_progress_idx (0)
{
	trace_with_mask("CardDeck::CardDeck",GUITRACE|DECK);
}

inline
CardDeck::
~CardDeck ()
{
	trace_with_mask("CardDeck::~CardDeck",GUITRACE|DECK);
}

inline void 
CardDeck::
push_back (VCard* card_) 
{ 
	m_cards.push_back (card_); 
}

inline void
CardDeck::
reset_progress () 
{ 
	trace_with_mask("CardDeck::reset_progress",GUITRACE|DECK);
	m_init_size = size ();
	m_progress_idx = 0;
}

inline void 
CardDeck::
erase (cardlist_iterator iter_) 
{ 
	trace_with_mask("CardDeck::erase(iter)",GUITRACE|DECK);
	m_cards.erase (iter_); 
}

inline void 
CardDeck::
sort_by_expiration_time () 
{ 
	std::sort (m_cards.begin (), m_cards.end (), Compare_ExpTime_CardRefs ());
}

#endif /* CARD_DECK_H */
