// -*- c++ -*-
//------------------------------------------------------------------------------
// $Id: DeckPlayer.h,v 1.49 2006/10/06 02:04:47 vlg Exp $
//------------------------------------------------------------------------------
//                            DeckPlayer.h
//------------------------------------------------------------------------------
//  Copyright (c) 2004-2005 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Fri Feb  6 23:37:11 EST 2004
//
//------------------------------------------------------------------------------
#ifndef DECK_PLAYER_H
#define DECK_PLAYER_H

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif

#include <gdkmm/pixbuf.h>
#include <gdkmm/image.h>
#include <gtkmm/messagedialog.h>
#include <gtkmm/notebook.h>
#include <gtkmm/label.h>
#include <gtkmm/textview.h>
#include <gtkmm/uimanager.h>
#include <gtkmm/table.h>

#ifdef IS_HILDON
#include <hildon-widgetsmm/appview.h>
#endif

#include "GrappConf.h"
#include "VDeck.h"

class DeckView;
class DeckPlayer;
class Gtk::ProgressBar;
class Gtk::EventBox;

/**
 * External events such as control buttons click
 * also cause switching between states.
 */
typedef enum {
	VC_START,					// Waiting for input
	VC_ARMED,					// User entered text into VC entry
	VC_VALIDATED				// Entry text validated
} VCState;

/**
 * Class VerifyControl encapsulates the verification element 
 * of the DeckPlayer dialog. This element is optionally shown
 * in the dialog between the notebook and the control toolbar.
 *
 */
class VerifyControl : public Gtk::Table
{
public:
	VerifyControl (DeckPlayer& parent_);

	void    set_focus_on_entry ();

	void    compare (const Glib::ustring& answer_);
	bool    matched () const { return m_was_equal; }
	void    clear   ();

	VCState get_state () const;
	void    set_state (VCState state_);

	void    on_entry_changed ();
	void    on_test_line_changed ();

	void    attach (Gtk::SpinButton* tl_) { m_test_line = tl_; }
	void    repaint () { m_entry->modify_font (CONFIG->get_input_font ()); }

private:
	static const int LABELS_SZ = 4;

	DeckPlayer&      m_parent;

	Gtk::Label*      m_report;		      // Correct, Incorrect
	Gtk::EventBox*   m_report_box;        // Box to hold report label
	Gtk::Entry*      m_entry;
	Gtk::Button*     m_button;	          // 'Check answer', 'Next'
	Gtk::Label*      m_label [LABELS_SZ];

	bool             m_was_equal;         // Keeps result of last compare() op.
	VCState          m_state;

	Gdk::Color       m_red;
	Gdk::Color       m_green;
	Gdk::Color       m_white;
	Gdk::Color       m_black;

	Gtk::SpinButton* m_test_line;         // Which line to compare for answer.
	int              m_line_num;          // Number of line to test
};

/**=============================================================================
 *                              Class DeckPlayer 
 **=============================================================================
 */
#ifdef IS_HILDON 
class DeckPlayer : public Hildon::AppView
#else
class DeckPlayer : public Gtk::Dialog
#endif
{
public:
	DeckPlayer (VDeck& vdeck_);

#ifdef IS_HILDON
	gint run ();
#endif

	/** Called in response to pressing <Close> button
	 */
	void on_close_clicked ();

	/** Called every time VBox that holds 
		the answer size of the card is resized.
	*/
	void answer_box_size_allocate_cb (Gtk::Allocation& size_);

	/** Called every time window is resized.
	*/
	void size_allocate_cb (Gtk::Allocation& size_);

	/** User pressed <Say It> button to play WAV audio, if available.
	 */
	void on_sayit_clicked ();

	/** Shuffle the cards in the deck
	 */
	void on_shuffle_clicked ();

	/** User pressed <View/Edit Deck> button to Add/Delete/Edit cards
	 */
	void on_edit_clicked ();

	/** User pressed <Edit Card> button to Edit current card
	 *
	 *  @param disable_cancel_ disables <Cancel> button if set to true.
	 */
	void on_vcard_edit_clicked (bool disable_cancel_ = false);

	// -*- Button Actions -*-
	void go_first_cb ();		// Go to the first card
	void go_prev_cb ();			// Go back one card
	void go_next_cb ();			// Go to the next card
	void go_last_cb ();			// Go to the last card
	void know_answer_cb ();		// I know the answer
	void no_clue_cb ();			// Have no clue about this one

	// VerifyControl is enabled and user clicked [Check] button
	void on_verify_clicked ();

	// -*- Radio Button Actions -*-
	void on_radio_selected (SideSelection selected_);

	// -*- Catch and process keyboard events -*-
	bool on_key_pressed (GdkEventKey* evt_);

	/** Get the deck pointer for comparison
	 */
	VDeck* get_deck () const { return &m_deck; }

	/** Reset iterator/count to reflect new card(s) added to the deck.
		Also, point iterator to the first card and show it. This method
		should be called each time the VDeck's count is modified (card is
		being moved or deleted).
	*/
	void reset ();

	/// Reset geometry from Configurator
	void update_geometry ();

	/** Take into account changed Preferences: adjust geometry
		and/or repaint the selected card.
	*/
	void repaint ();

	/** Clear the text fields of the old or stale text
	 */
	void clear_text ();

	/// Enable answer control buttons ("I know", "Not a clue")
	void enable_answer_controls ();

	/// Disable answer control buttons ("I know", "Not a clue")
	void disable_answer_controls ();

    /// Flip the Card
	void flip_card ();	

	/// Auto-pronounce word on next/flip action
	void auto_pronounce_card (SideSelection side_);

	SideSelection get_selected_size () const { return m_selected_side; }

	/** Remove all newlines and shrink all whitespace.
	*/
	static void shrink_to_single_line (Glib::ustring& txt_);

	/** Extend Stock Icons with application-specific ones.
	 */
	static Gtk::StockID SAY_IT;
	static Gtk::StockID THUMB_UP;
	static Gtk::StockID THUMB_DOWN;

private:
	void show_deck       ();	// Show first card of the loaded deck
	void switch_side     ();	// Show the front/back side of the card
	void adjust_count    ();	// Adjust count label: current/total
	void set_sensitivity ();    // De-sensetize inappropriate controls
	void set_win_name    ();    // Set tile name

	/** If deck is empty, paralize controls. Deck can be loaded empty,
	 *  or become empty once the last card is removed. The reverse
	 *  applies as soon as first card is added.
	 *  This only makes sense with real decks.
	 */
	void sensitize_controls ();

    /** Display the card pointed to by the iterator
	 */
	void show_card (bool with_adjust_count_ = true);	

	/** What should be considered an answer?
	 */
	Glib::ustring get_answer ();

	/** Adjust justification of the 'Answer' field depending on
	 *  wether it is a single line or multiline
	 */
	void set_answer_field (Glib::ustring answer_);

	/** Pango-markup tildas in the string
	 */
	static Glib::ustring markup_tildas (const Glib::ustring& src_);

	/** Check markup validity of the text fields of the card_
	 */
	bool markup_is_valid (VCard* card_);

	bool check_F6_key (gint keyval_);
	bool check_F7_key (gint keyval_);
	bool check_F8_key (gint keyval_);

#ifdef IS_HILDON
	Gtk::VBox* get_vbox () { return m_vbox; }
#endif

private:

#ifdef IS_HILDON
	bool              m_hildon_fullscreen;
#endif

	Gtk::Button*      m_close_button;
	Gtk::Button*      m_shuffle_button;
	Gtk::Button*      m_edit_button;         // View/Edit a Deck
	Gtk::Button*      m_vcard_edit_button;   // Edit a Card
	int               m_match_front_row; // Which row to select for answer
	Gtk::Label        m_count_label;
	SideSelection     m_selected_side;
	Gtk::Notebook*    m_notebook;
	Gtk::ProgressBar* m_play_progress;
	Gtk::Label        m_front;
	Gtk::Label        m_back;
	Gtk::Label        m_example;
	Gtk::SpinButton*  m_test_line_button;

	Gtk::VBox*        m_answer_box; /* VBox that holds the back of a card */
	int               m_answer_box_width; /* last resize width */

	VerifyControl*    m_verify_control;	// optional answer verification entry 

	Gtk::Widget*      m_toolbar_go_first;
	Gtk::Widget*      m_toolbar_go_prev;
	Gtk::Widget*      m_toolbar_go_next;
	Gtk::Widget*      m_toolbar_go_last;
	Gtk::Widget*      m_toolbar_know_it;
	Gtk::Widget*      m_toolbar_no_clue;

	Glib::RefPtr<Gdk::Pixbuf> m_playit_xpm;	// Sound icon
	Gtk::Image*       m_playit_icon;

	Gdk::Color        m_bg_white;
	Gdk::Color        m_bg_pink;
	Gdk::Color        m_bg_black;

	VDeck&            m_deck;
	VDeck::cardlist_iterator m_deck_iter; // Position in the Deck

	bool m_is_lesson_learned;	// Have user gone through the deck?
	bool m_is_real_deck;		// Is it a real Deck we are dealing with?

	Glib::RefPtr<Gtk::UIManager>   m_uimanager;
	Glib::RefPtr<Gtk::ActionGroup> m_actgroup;

	bool m_tab_activated;		// Leave Tab+Spacebar/Enter event to Dialog
	bool m_initialized;	        // Has the object been fully constructed?

#ifdef IS_HILDON
	Gtk::VBox*        m_vbox;
#endif
};

inline void
DeckPlayer::
flip_card ()
{
	m_notebook->set_current_page (m_notebook->get_current_page () ? 0 : 1);
	m_verify_control->set_focus_on_entry ();
}

inline void
DeckPlayer::
clear_text ()
{
	m_front.  set_text ("");
	m_back.   set_text ("");
	m_example.set_text ("");
}

/*------------------------------------------------------------
 * Madness I have to live with just for the sake of nokia 770
 *------------------------------------------------------------
 */

inline bool
DeckPlayer::
check_F6_key (gint keyval_)
{
#ifdef IS_HILDON
	if (keyval_ == GDK_F6) { // Full screen
		m_hildon_fullscreen != m_hildon_fullscreen;
		set_fullscreen (m_hildon_fullscreen);
		return true;
	}
#endif
	return false;
}

inline bool
DeckPlayer::
check_F7_key (gint keyval_)
{
#ifdef IS_HILDON
	return (keyval_ == GDK_F7);
#else
	return false;
#endif
}

inline bool
DeckPlayer::
check_F8_key (gint keyval_)
{
#ifdef IS_HILDON
	return (keyval_ == GDK_F8);
#else
	return false;
#endif
}

#endif /* DECK_PLAYER_H */
