/*
 * Grdc - GTK+/Gnome Remote Desktop Client
 * Copyright (C) 2009 - Vic Lee 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, 
 * Boston, MA 02111-1307, USA.
 */
 

#ifndef __GRDCSSH_H__
#define __GRDCSSH_H__

#include "config.h"

#ifdef HAVE_LIBSSH

#include <libssh/libssh.h>
#include <libssh/sftp.h>
#include <pthread.h>
#include "grdcfile.h"
#include "grdcinitdialog.h"

G_BEGIN_DECLS

/* ----------------- Some libssh wrappers to make it compatible with 0.2 ----------------- */
#if !(LIBSSH_VERSION_MAJOR >= 0 && LIBSSH_VERSION_MINOR >= 3)
#define sftp_closedir(d) sftp_dir_close(d)
#define channel_read_buffer channel_read
#define privatekey_free private_key_free
#define sftp_close sftp_file_close
#define sftp_unlink sftp_rm
#ifndef SSH_EOF
#define SSH_EOF SSH_ERROR
#endif
#endif

/* ----------------- SSH Base --------------- */

#define GRDC_SSH(a) ((GrdcSSH*)a)

typedef struct _GrdcSSH
{
    SSH_SESSION *session;
    gboolean authenticated;

    gchar *server;
    guint port;
    gchar *user;
    gint auth;
    gchar *password;
    gchar *pubkeyfile;
    gchar *privkeyfile;

    gchar *charset;
    gchar *error;

    pthread_mutex_t ssh_mutex;
} GrdcSSH;

gchar* grdc_ssh_identity_path (const gchar *id);

/* Auto-detect commonly used private key identities */
gchar* grdc_ssh_find_identity (void);

/* Initialize the ssh object */
gboolean grdc_ssh_init_from_file (GrdcSSH *ssh, GrdcFile *grdcfile);

/* Initialize the SSH session */
gboolean grdc_ssh_init_session (GrdcSSH *ssh);

/* Authenticate SSH session */
/* -1: Require password; 0: Failed; 1: Succeeded */
gint grdc_ssh_auth (GrdcSSH *ssh, const gchar *password);

/* -1: Cancelled; 0: Failed; 1: Succeeded */
gint grdc_ssh_auth_gui (GrdcSSH *ssh, GrdcInitDialog *dialog, gboolean threaded);

/* Error handling */
#define grdc_ssh_has_error(ssh) (((GrdcSSH*)ssh)->error!=NULL)
void grdc_ssh_set_error (GrdcSSH *ssh, const gchar *fmt);
#define grdc_ssh_set_application_error(ssh,msg) ((GrdcSSH*)ssh)->error = g_strdup (msg);

/* Converts a string to/from UTF-8, or simply duplicate it if no conversion */
gchar* grdc_ssh_convert (GrdcSSH *ssh, const gchar *from);
gchar* grdc_ssh_unconvert (GrdcSSH *ssh, const gchar *from);

void grdc_ssh_free (GrdcSSH *ssh);

/* ------------------- SSH Tunnel ---------------------- */

typedef struct _GrdcSSHTunnel
{
    GrdcSSH ssh;

    CHANNEL *channel;

    gint tunnel_server_sock;
    gint tunnel_client_sock;

    pthread_t thread;
    gboolean running;

    gchar *dest_host;
    gint dest_port;

    gchar *buffer;
    gint buffer_len;
} GrdcSSHTunnel;

/* Create a new SSH Tunnel session and connects to the SSH server */
GrdcSSHTunnel* grdc_ssh_tunnel_new_from_file (GrdcFile *grdcfile);

/* Start the tunnel. A new thread will be started and listening on a local port.
 * dest: The host:port of the remote destination
 * local_port: The listening local port for the tunnel
 */
gboolean grdc_ssh_tunnel_start (GrdcSSHTunnel *tunnel, const gchar *dest, gint local_port);

/* Tells if the tunnel is terminated after start */
gboolean grdc_ssh_tunnel_terminated (GrdcSSHTunnel *tunnel);

/* Free the tunnel */
void grdc_ssh_tunnel_free (GrdcSSHTunnel *tunnel);

/*----------------------- SFTP ------------------------*/

typedef struct _GrdcSFTP
{
    GrdcSSH ssh;

    SFTP_SESSION *sftp_sess;
} GrdcSFTP;

/* Create a new SFTP session object from GrdcFile */
GrdcSFTP* grdc_sftp_new_from_file (GrdcFile *grdcfile);

/* Create a new SFTP session object from existing SSH session */
GrdcSFTP* grdc_sftp_new_from_ssh (GrdcSSH *ssh);

/* open the SFTP session, assuming the session already authenticated */
gboolean grdc_sftp_open (GrdcSFTP *sftp);

/* Free the SFTP session */
void grdc_sftp_free (GrdcSFTP *sftp);

/*----------------------- SSH Terminal ------------------------*/

typedef struct _GrdcSSHTerminal
{
    GrdcSSH ssh;

    gint master;
    gint slave;
    pthread_t thread;
    GtkWidget *window;
    gboolean closed;
} GrdcSSHTerminal;

/* Create a new SSH Terminal session object from existing SSH session */
GrdcSSHTerminal* grdc_ssh_terminal_new_from_ssh (GrdcSSH *ssh);

/* open the SSH Terminal (init -> auth -> term) */
gboolean grdc_ssh_terminal_open (GrdcSSHTerminal *term);

G_END_DECLS

#else

#define GrdcSSH void
#define GrdcSSHTunnel void
#define GrdcSFTP void
#define GrdcTerminal void

#endif /* HAVE_LIBSSH */

#endif  /* __GRDCSSH_H__  */

