/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

/***************************************************************************
 *   Jean Baptiste Filippi - 01.11.2005                                                        *   Enrico Fucile
 *                                                                         *
 ***************************************************************************/
#include "grib_api_internal.h"
#define NUMBER(x) (sizeof(x)/sizeof(x[0]))

#define TINY_BUFFER 20

GRIB_INLINE static int grib_inline_strncmp(unsigned char* a,
                                           unsigned char* b,int n) {
  unsigned char* end;
  if (*a != *b) return 1;
  end=a+n;
  while( a!=end && *(a) == *(b)) {a++;b++;}
  return a!=end;
}

typedef int (grib_data_validate_proc) (grib_context* c, unsigned char* buffer, size_t* prod_len);

typedef int (grib_data_read_message_size_from_file_proc) (const grib_context *ctx, void* file, size_t* read, unsigned char* buffer, size_t* prod_len);

typedef int  (grib_data_read_message_size_proc) (const grib_context *ctx, unsigned char* data,
                                                  size_t data_len, size_t* prod_len);

typedef struct grib_magic{
  unsigned long    magic;
  char*   val;
  grib_data_read_message_size_from_file_proc* get_size_from_file;
  grib_data_read_message_size_proc* get_size;
  grib_data_validate_proc* validate;
} grib_magic;

#if 0
static int grib_search_message_end(grib_context *ctx, unsigned char* msg, size_t* prod_len);
#endif

static int grib_read_size_from_file_BUDG(const grib_context *c,  void* file, size_t* read, unsigned char* buffer, size_t* prod_len)
{
  *prod_len = 6000;
  return GRIB_SUCCESS;
}

static int grib_read_size_BUDG(const grib_context *c,  unsigned char* data,
                               size_t data_len,size_t* prod_len)
{
  *prod_len = 6000;
  return GRIB_SUCCESS;
}

static int grib_validate_GRIB(grib_context* ctx,unsigned char* buffer, size_t* prod_len)
{
  unsigned long v7777  = 0x37373737;
  unsigned long magic_buff=0;
  unsigned char* mybuf;
  int ret=GRIB_SUCCESS;

  mybuf =buffer + (*prod_len)-4;
  magic_buff=grib_decode_unsigned_byte_long(mybuf,0,4) ;

  if( magic_buff != v7777){
    grib_context_log(ctx, GRIB_LOG_WARNING, "grib_validate_GRIB: wrong message length");
#if 0
    ret=grib_search_message_end(ctx, buffer, prod_len);
    grib_context_log(ctx, GRIB_LOG_WARNING, "grib_validate_GRIB: found 7777 message length=%ld ",*prod_len);
#endif
    ret=GRIB_WRONG_LENGTH;
  }
  return ret;
}

static int grib_validate_BUFR(grib_context* ctx,unsigned char* buffer, size_t* prod_len)
{
  unsigned long v7777  = 0x37373737;
  unsigned long magic_buff=0;
  unsigned char* mybuf;
  int ret=GRIB_SUCCESS;

  mybuf =buffer + (*prod_len)-4;
  magic_buff=grib_decode_unsigned_byte_long(mybuf,0,4) ;

  if( magic_buff != v7777){
    grib_context_log(ctx, GRIB_LOG_WARNING, "grib_validate_GRIB: wrong message length");

    ret=GRIB_WRONG_LENGTH;
  }
  return ret;
}

#if 0
static int grib_search_message_end(grib_context *ctx, unsigned char* msg, size_t* prod_len){
  size_t i=0;
  unsigned char* mybuf=msg;
  int ret=GRIB_7777_NOT_FOUND;
  size_t end_search;

  end_search=*prod_len-3;

  for (i=0;i < end_search; i++) {
    if (*mybuf != '7' && (*(mybuf+1)) == '7' && (*(mybuf+2)) == '7' && (*(mybuf+3)) == '7') {
        ret=GRIB_SUCCESS;
        *prod_len=i+5;
        break;
    }
    mybuf++;
  }

  return ret;
}
#endif

static int grib_read_size_GRIB(const grib_context *c,  unsigned char* data,
                               size_t data_len,  size_t* prod_len)
{
  int ed = 0;
  char* end="7777";

  if (data_len < 20) return GRIB_INVALID_MESSAGE;

  ed = grib_decode_unsigned_byte_long(data,7,1);

  if(ed < 1 || ed > 2)
  {
    grib_context_log(c,GRIB_LOG_ERROR,"Cannot get edition number %d",ed);
    return GRIB_INVALID_MESSAGE;
  }


  if(ed == 1)
  {
    *prod_len = grib_decode_unsigned_byte_long(data,4,3);

    if(*prod_len <= data_len &&
         !grib_inline_strncmp(data+(*prod_len)-4,(unsigned char*)end,4))
      return GRIB_SUCCESS;

    if ((*prod_len & 0x800000))
      *prod_len= (*prod_len & 0x7fffff)*120-120;

    if(*prod_len <= data_len &&
         !grib_inline_strncmp(data+(*prod_len)-4,(unsigned char*)end,4))
      return GRIB_SUCCESS;

    return GRIB_7777_NOT_FOUND;
  }

  if(ed == 2)
    *prod_len = grib_decode_unsigned_byte_long(data,8,8);

  return GRIB_SUCCESS;
}

static int grib_read_size_from_file_GRIB(const grib_context *c,  void* file, size_t* read, unsigned char* buffer, size_t* prod_len)
{

  int ed = 0;

  if (*prod_len < 20) return GRIB_BUFFER_TOO_SMALL;


  ed = grib_decode_unsigned_byte_long(buffer,7,1);

  if(ed < 1 || ed > 2)
  {
    grib_context_log(c,GRIB_LOG_ERROR,"Cannot get edition number %d",ed);
    return GRIB_INVALID_MESSAGE;
  }

  if(ed == 1)
  {
    *prod_len = grib_decode_unsigned_byte_long(buffer,4,3);

    if(*prod_len & 0x800000)
    {
      char tmp[121];

      off_t here   = grib_context_tell(c,file);
      size_t len  = *prod_len & 0x7fffff;

      /* grib_context_log(c,GRIB_LOG_ERROR,"Special treatment for large GRIBS %ld %ld",(long)*read,(long)*prod_len); */

    grib_context_seek(c,(off_t)(*prod_len - *read - 4),SEEK_CUR,file);
      /* fseek(file,*prod_len - *read - 4,SEEK_CUR); */
      if(grib_context_read(c,tmp,4,file) == 4)
      {
        if(strncmp(tmp,"7777",4) == 0)
        {
        grib_context_seek(c,here,SEEK_SET,file);
          /* fseek(file,here,SEEK_SET); */
          /* grib_context_log(c,GRIB_LOG_ERROR,"Genuine large GRIB"); */
          return GRIB_SUCCESS;
        }
      }

      len *= 120;
      len -= 120;

      /* grib_context_log(c,GRIB_LOG_ERROR,"Bad large GRIBS %ld %ld %ld",(long)*read,(long)len); */
    grib_context_seek(c,here,SEEK_SET,file);
      /* fseek(file,here,SEEK_SET); */
    grib_context_seek(c,(off_t)(len - *read - 4),SEEK_CUR,file);
      /* fseek(file,len - *read - 4,SEEK_CUR); */
      /* grib_context_log(c,GRIB_LOG_ERROR,"pos %ld",(long)grib_context_tell(c,file)); */

      memset(tmp,0,sizeof(tmp));
      if(grib_context_read(c,tmp,sizeof(tmp)-1,file) > 4)
      {
        int i;
        for(i = 0; i < sizeof(tmp) -4 ; i++)
          if(tmp[i] == '7' && tmp[i+1] == '7' && tmp[i+2] == '7' && tmp[i+3] == '7')
          {
            len += i;
          grib_context_seek(c,here,SEEK_SET,file);
            /* fseek(file,here,SEEK_SET); */
            /* grib_context_log(c,GRIB_LOG_ERROR,"large GRIB is %ld",len); */
            *prod_len = len;
            return GRIB_SUCCESS;
          }

        /* grib_context_log(c,GRIB_LOG_ERROR,"Bad..."); */
      grib_context_seek(c,here,SEEK_SET,file);
        /* fseek(file,here,SEEK_SET); */
        return GRIB_7777_NOT_FOUND;
      }
      else
      {
      grib_context_seek(c,here,SEEK_SET,file);
        /* fseek(file,here,SEEK_SET); */
        return GRIB_IO_PROBLEM;
      }

    }
  }

  if(ed == 2)
    *prod_len = grib_decode_unsigned_byte_long(buffer,8,8);

  return GRIB_SUCCESS;
}

static int grib_read_size_from_file_BUFR(const grib_context *c,  void* file, size_t* read, unsigned char* buffer, size_t* prod_len)
{

  if (*prod_len < 20) return GRIB_BUFFER_TOO_SMALL;

  *prod_len = grib_decode_unsigned_byte_long(buffer,4,3);

  return GRIB_SUCCESS;
}

static int grib_read_size_BUFR(const grib_context *c,  unsigned char* data,
                               size_t data_len,  size_t* prod_len)
{

  return GRIB_NOT_IMPLEMENTED;

}


static struct grib_magic table[] =
{
  { 0x47524942,"GRIB",&grib_read_size_from_file_GRIB,&grib_read_size_GRIB,&grib_validate_GRIB},
  { 0x42554447,"BUDG",&grib_read_size_from_file_BUDG,&grib_read_size_BUDG,&grib_validate_GRIB},
  { 0x42554652,"BUFR",&grib_read_size_from_file_BUFR,&grib_read_size_BUFR,&grib_validate_BUFR}
};

static int grib_fetch_from_file(grib_context *ctx, void* file, size_t* read, unsigned char* buffer, size_t* prod_len,  int *prod){

  size_t strap_len = TINY_BUFFER;
  size_t strap_pos = 0;
  unsigned char strap[TINY_BUFFER];
  unsigned char *strap_temp;

  size_t i = 0;
  size_t bi = 0;
  unsigned char* data;
  unsigned long product = 0;
  int fret = 0;

  grib_context *c = ctx;

  data = buffer;

  if(!ctx) c = grib_context_get_default();

  if(*prod_len < 4)
    return GRIB_BUFFER_TOO_SMALL;

  fret = grib_context_read(c,strap, strap_len, file);

  if ( (fret < strap_len) && grib_context_eof(c,file) )
    return GRIB_END_OF_FILE;

  Assert(fret <= 0 || fret >= 4);

  strap_temp = strap;
  while(fret >= 4){
    bi = 0;
    product    = grib_decode_unsigned_byte_long(strap_temp,bi,4);

    for(i = 0; i < NUMBER(table) ; i++)
      if(product  == table[i].magic) {
        strap_temp     = strap+strap_pos;
        c->message_file_offset=grib_context_tell(c,file)+strap_pos-TINY_BUFFER;
        memcpy        (buffer,strap_temp,strap_len-strap_pos);
        *read         = strap_len-strap_pos;
        *prod         = i;

        if(*read < *prod_len)
        {
          fret = grib_context_read(c,buffer+*read,*prod_len-*read,file);

          if(fret <0)
            return GRIB_IO_PROBLEM;

          *read += fret;
        }

        return        GRIB_SUCCESS;
      }

    if(strap_pos++ < (strap_len-4))strap_temp++;
    else{
      strap_temp = strap+strap_len-4;
      memcpy(strap,strap_temp,4);
      strap_temp = strap+4;
      fret = grib_context_read(c, strap_temp, strap_len-4, file);
      strap_temp = strap;
      strap_pos=0;
    }
  }
  return GRIB_END_OF_FILE;

}

static unsigned char* grib_fetch(unsigned char* data, size_t* data_len,int *itable) {
  unsigned char* message=data;
  unsigned char* end=data+ *data_len;

  while (message != end) {
    for(*itable = 0; *itable < NUMBER(table) ; (*itable)++)
      if (!grib_inline_strncmp(message,(unsigned char*)table[*itable].val,4))
                     return message;

    message++;
    (*data_len)--;
  }

  return NULL;
}

int grib_read_any_alloc(grib_context *c, unsigned char** data,size_t* data_len,
                        unsigned char** message, size_t* message_len){
  int err=0;
  int itable = 0;
  unsigned char* p=NULL;
  size_t len=*data_len;

  if (len==0) return GRIB_END;

  p=grib_fetch(*data,data_len,&itable);

  if (p==NULL) return GRIB_END;

  if((err = (*table[itable].get_size)(c, p,*data_len,message_len)) != GRIB_SUCCESS) {
    grib_context_log(c, GRIB_LOG_ERROR,
         "grib_read_any: cannot get size (%s)",grib_get_error_message(err));
    return err;
  }

  *message = (unsigned char*)grib_context_malloc(c,*message_len);

  if(!*message) {
    grib_context_log(c, GRIB_LOG_ERROR, "grib_read_any_alloc: cannot allocate message (size = %ld)",*message_len);
     return GRIB_OUT_OF_MEMORY;
  }

  memcpy(*message,p,*message_len);

  if((err = (*table[itable].validate)(c,*message, message_len)) != GRIB_SUCCESS) {
    grib_context_free(c,*message);
    grib_context_log(c,GRIB_LOG_ERROR,
          "grib_read_any_alloc: cannot validate message (%s)",grib_get_error_message(err));
    *message = NULL;
  } else {
    *data_len = len- (p-(*data)) - *message_len;
    *data=p+*message_len;
  }

  return err;
}


int grib_read_any(grib_context *c, unsigned char** data,size_t* data_len,
                        unsigned char** message, size_t* message_len){
  int err=0;
  int itable = 0;
  unsigned char* p=NULL;
  size_t len=*data_len;

  if (len==0) return GRIB_END;

  p=grib_fetch(*data,data_len,&itable);

  if (p==NULL) return GRIB_END;

  if((err = (*table[itable].get_size)(c, p,*data_len,message_len)) != GRIB_SUCCESS) {
    grib_context_log(c, GRIB_LOG_ERROR,
         "grib_read_any: cannot get size (%s)",grib_get_error_message(err));
    return err;
  }

  *message = p;

  if((err = (*table[itable].validate)(c,*message, message_len)) != GRIB_SUCCESS) {
    grib_context_free(c,*message);
    grib_context_log(c,GRIB_LOG_ERROR,
          "grib_read_any_alloc: cannot validate message (%s)",grib_get_error_message(err));
    *message = NULL;
  } else {
    *data_len = len- (p-(*data)) - *message_len;
    *data=p+*message_len;
  }

  return err;
}

int grib_read_any_from_file  (grib_context *ctx, void* file, unsigned char* mesg , size_t* size){
  int err  = 0;
  int itable = 0;
  size_t read = 0;
  size_t buf_len = 0;
  size_t msg_len = 0;
  unsigned char buffer[TINY_BUFFER];

  buf_len=sizeof(buffer);

  if(!ctx) ctx =grib_context_get_default();


  if((err = grib_fetch_from_file(ctx, file, &read, buffer,  &buf_len , &itable)) != GRIB_SUCCESS)
  {
    if (err != GRIB_END_OF_FILE)
      grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file: cannot get message (%s)",grib_get_error_message(err));
    return err;
  }

  msg_len=buf_len;

  if((err = (*table[itable].get_size_from_file)(ctx, file,  &read, buffer,  &msg_len)) != GRIB_SUCCESS)
  {
    grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file: cannot message get size (%s)",grib_get_error_message(err));
    return err;
  }


  /* We must read as much as possible */
  if(msg_len > *size)
  {
    if(read >= *size)
    {
      memcpy(mesg,buffer,*size);
    }
    else
    {
      memcpy(mesg,buffer,read);
      mesg += read;

      if(grib_context_read(ctx, mesg, *size-read, file) != (*size-read))
      {
        grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file: cannot read partial message");
        return GRIB_IO_PROBLEM;
      }

      mesg -= read;
    }

    /* We should now position at the end of the message */

    {
      char buf[4096];
      long n = msg_len-*size;

      while(n > 0)
      {
        long m = n > sizeof(buf) ? sizeof(buf) : n;
        long p = grib_context_read(ctx,buf,m,file);
        if(p <= 0)
        {
          grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file: cannot seek at end of partial message");
          return GRIB_IO_PROBLEM;
        }

        n -= p;
      }

    }


    *size = msg_len;
    return GRIB_BUFFER_TOO_SMALL;
  }

  *size = msg_len;

  memcpy(mesg,buffer,read);

  Assert(read <= buf_len);

  if(msg_len > read)
  {
    long r = grib_context_read(ctx, mesg + read, msg_len-read, file);
    if(r != (msg_len-read))
    {
        grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file: cannot read whole message");
        return GRIB_IO_PROBLEM;
    }
  }

  return  (*table[itable].validate)(ctx,mesg, size);
}


int grib_read_any_from_file_alloc(grib_context *ctx, void* file,unsigned char** mesg, size_t* msg_len){
  int err  = 0;
  int itable = 0;
  size_t read = 0;
  size_t buf_len = 0;
  off_t offset=0,end=0;
  unsigned char buffer[TINY_BUFFER];
  struct stat finfo;
  off_t fsize;

  fstat(fileno((FILE*)file),&finfo);
  fsize=finfo.st_size;

  buf_len=sizeof(buffer);

  if(!ctx) ctx =grib_context_get_default();

  if((err = grib_fetch_from_file(ctx, file, &read, buffer,  &buf_len , &itable)) != GRIB_SUCCESS) {
    if (err != GRIB_END_OF_FILE)
      grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file_alloc: cannot get message (%s)",grib_get_error_message(err));
    return err;
  }


  *msg_len=buf_len;
  if((err = (*table[itable].get_size_from_file)(ctx, file,  &read, buffer,  msg_len)) != GRIB_SUCCESS ) {
    grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file_alloc: cannot get message size (%s)",grib_get_error_message(err));
    return err;
  }

  /* check if the file is shorter then the required buf_len*/
  end=offset+*msg_len-read;
  if (fsize != 0 && end > fsize ) {
    err=GRIB_WRONG_LENGTH;
    if (ctx->no_fail_on_wrong_length)  {
      grib_context_seek( ctx,ctx->message_file_offset+4,SEEK_SET,file);
      return grib_read_any_from_file_alloc(ctx,file,mesg,msg_len);
    }
    return err;
  }

  *mesg = grib_context_malloc(ctx,*msg_len);

  if(!*mesg) {
    grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file_alloc: cannot allocate message (size = %ld)",*msg_len);
     return GRIB_OUT_OF_MEMORY;
  }

  memcpy(*mesg,buffer,read);

  *mesg += read;

  if(grib_context_read(ctx, *mesg, *msg_len-read, file) != (*msg_len-read)) {
    grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file_alloc: cannot read whole message");
    *mesg -= read;
    grib_context_free(ctx,*mesg);
    *mesg = NULL;
    return GRIB_IO_PROBLEM;
  }

  *mesg -= read;

  if((err = (*table[itable].validate)(ctx,*mesg, msg_len)) != GRIB_SUCCESS) {
    grib_context_free(ctx,*mesg);
    if (err==GRIB_WRONG_LENGTH && ctx->no_fail_on_wrong_length)  {
        grib_context_seek( ctx,ctx->message_file_offset+4,SEEK_SET,file);
        return grib_read_any_from_file_alloc(ctx,file,mesg,msg_len);
    }
    grib_context_log(ctx, GRIB_LOG_ERROR, "grib_read_any_from_file_alloc: cannot validate message (%s)",
      grib_get_error_message(err));
    *mesg = NULL;
  }

  return err;
}

int grib_count_in_file(grib_context* c, FILE* f,int* n) {
   grib_handle* h;
   int error=0;
   *n=0;
   while ((h=grib_handle_new_from_file(c,f,&error))!=NULL) {
     (*n)++;
     grib_handle_delete(h);
   }
   rewind(f);
   return error==GRIB_END_OF_FILE ? 0 : error ;
}


