/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

/*
 * C Implementation: grib_tools
 *
 * Author: Enrico Fucile <enrico.fucile@ecmwf.int>
 *
 *
 */

#include "grib_tools.h"
#if HAVE_LIBJASPER
#include "jasper/jasper.h"
#endif

GRIB_INLINE static int grib_inline_strcmp(const char* a,const char* b) {
  if (*a != *b) return 1;
  while((*a!=0 && *b!=0) &&  *(a) == *(b) ) {a++;b++;}
  return (*a==0 && *b==0) ? 0 : 1;
}

static void grib_print_header(grib_runtime_options* options,grib_handle* h);
static void grib_tools_set_print_keys(grib_runtime_options* options,grib_handle* h,const char* ns);
static int grib_tool_with_orderby();
static int grib_tool_without_orderby();

int grib_tool(int argc, char **argv)
{

  grib_runtime_options options={
              0,         /* verbose       */
              0,         /* fail          */
              0,         /* skip          */
              12,        /* default_print_width */
              0,         /* print_header */
              0,         /* has_print_keys  */
              0,         /* print_mars_keys */
              0,         /* print_number */
              1,         /* print_statistics */
              0,         /* requested_print_keys */
              {{0,},},   /* grib_values print_keys[MAX_KEYS] */
              0,         /* print_keys_count  */
              0,         /* strict            */
              0,         /* multi_support     */
              0,         /* set_values_count  */
              {{0,},},   /* grib_values set_values[MAX_KEYS] */
              {{0,},},   /* grib_values constraints[MAX_KEYS] */
              0,         /* constraints_count */
              {{0,},},   /* grib_values compare[MAX_KEYS] */
              0,         /* compare_count */
              0,         /* handle_count      */
              0,         /* filter_handle_count */
              0,         /* file_count     */
              0,         /* grib_tools_file infile_extra */
              0,         /* grib_tools_file current_infile */
              0,         /* grib_tools_file infile */
              0,         /*grib_tools_file outfile */
              0,         /* grib_action action */
              0,         /* grib_rule rules */
              0,         /* int dump_flags; */
              0,         /* char* dump_mode; */
              0,         /* repack    */
              0,         /* error    */
              0,          /* gts    */
              0,          /* orderby    */
              0,          /* latlon    */
              {0,},
              {0,},
              {0,},
              {0,},
              {0,},
                4,
                0,
                -1,
              {0,},
                 0,       /* index */
                 0,       /* index_on */
                 0,        /* constant */
                 0         /* dump_filename*/

  };

  if (getenv("DOXYGEN_USAGE") && argc==1 ) usage_doxygen();

  grib_get_runtime_options(argc,argv,&options);

  grib_tool_before_getopt(&options);

  grib_process_runtime_options(argc,argv,&options);

#if HAVE_LIBJASPER
  if (getenv("JASPER_DEBUG")) jas_setdbglevel(atoi(getenv("JASPER_DEBUG")));
#endif


  grib_tool_init(&options);
  if (options.dump_filename) {
    dump_file= fopen(options.dump_filename,"w");
    if(!dump_file) {
      perror(options.dump_filename);
      exit(1);
    }
  } else {
    dump_file=stdout;
  }

  if (options.orderby)
    return grib_tool_with_orderby(&options);
  else
    return grib_tool_without_orderby(&options);

  if (options.dump_filename) fclose(dump_file);
}

static int grib_tool_with_orderby(grib_runtime_options* options) {
  int err=0;
  grib_failed *failed=NULL,*p=NULL;
  grib_handle* h=NULL;
  grib_context* c=NULL;
  grib_tools_file* infile=NULL;
  char** filenames;
  size_t files_count=0;
  grib_fieldset* set=NULL;
  int i=0;

  c=grib_context_get_default();

  infile=options->infile;
  files_count=0;
  while(infile) {files_count++;infile=infile->next;}

  filenames=(char**)grib_context_malloc(c,files_count*sizeof(char*));

  infile=options->infile;
  for (i=0;i<files_count;i++) {filenames[i]=infile->name;infile=infile->next;}

  if (grib_options_on("7")) c->no_fail_on_wrong_length=1;

  set=grib_fieldset_new_from_files(0,filenames,files_count,0,0,0,options->orderby,&err);

  options->handle_count=0;
  while((h = grib_fieldset_next_handle(set,&err))
       != NULL || err != GRIB_SUCCESS ) {
    options->handle_count++;
    options->error=err;

    if (!h) {
      fprintf(dump_file,"null handle\n");
      if (options->fail || err==GRIB_WRONG_LENGTH) GRIB_CHECK(err,0);

      failed=(grib_failed*)malloc(sizeof(grib_failed));
      failed->count=infile->handle_count;
      failed->error=err;
      failed->next=NULL;

      if (!infile->failed) {
        infile->failed=failed;
      } else {
        p=infile->failed;
        while (p->next) p=p->next;
        p->next=failed;
      }

      continue;
    }

    grib_print_header(options,h);

    grib_skip_check(options,h);

    if (options->skip && options->strict) {
      grib_tool_skip_handle(options,h);
      continue;
    }

    grib_tool_new_handle_action(options,h);

    grib_tool_print_key_values(options,h);

    grib_handle_delete(h);

  }


  grib_tool_finalise_action(options);

  return 0;
}

static int grib_tool_without_orderby(grib_runtime_options* options) {
  int err=0;
  grib_failed *failed=NULL,*p=NULL;
  grib_handle* h=NULL;
  grib_context* c=NULL;
  grib_tools_file* infile=NULL;

  c=grib_context_get_default();
  options->file_count=0;
  options->handle_count=0;
  options->filter_handle_count=0;
  options->current_infile=options->infile;
  infile=options->infile;

  if (grib_options_on("7")) c->no_fail_on_wrong_length=1;

  while (infile!=NULL && infile->name!=NULL) {

    if (options->print_statistics && options->verbose) fprintf(dump_file,"%s\n",infile->name);
    infile->file = fopen(infile->name,"r");
    if(!infile->file) {
      perror(infile->name);
      exit(1);
    }
    options->file_count++;
    infile->handle_count=0;
    infile->filter_handle_count=0;

    grib_tool_new_file_action(options);


    while((h = grib_handle_new_from_file(c,infile->file,&err))
         != NULL || err != GRIB_SUCCESS ) {
      infile->handle_count++;
      options->handle_count++;
      options->error=err;

      if (!h) {
        fprintf(dump_file,"null handle\n");
        if (options->fail || err==GRIB_WRONG_LENGTH) GRIB_CHECK(err,0);

        failed=(grib_failed*)malloc(sizeof(grib_failed));
        failed->count=infile->handle_count;
        failed->error=err;
        failed->next=NULL;

        if (!infile->failed) {
          infile->failed=failed;
        } else {
          p=infile->failed;
          while (p->next) p=p->next;
          p->next=failed;
        }

        continue;
      }

      grib_print_header(options,h);

      grib_skip_check(options,h);

      if (options->skip && options->strict) {
        grib_tool_skip_handle(options,h);
        continue;
      }

      grib_tool_new_handle_action(options,h);

      grib_print_key_values(options,h);

      grib_handle_delete(h);

    }

    grib_print_file_statistics(options,infile);

    if (infile->file) fclose(infile->file);

    if (infile->handle_count==0) {
      fprintf(dump_file,"no grib messages found in %s\n", infile->name);
      if (options->fail) exit(1);
    }

    infile=infile->next;
    options->current_infile=infile;

  }

  grib_print_full_statistics(options);

  grib_tool_finalise_action(options);

  return 0;
}

static void grib_print_header(grib_runtime_options* options,grib_handle* h) {

  size_t strlenkey=0;
  int width;
  if (!options->print_keys || options->handle_count != 1 )
    return;

  if (options->print_mars_keys) {
    grib_tools_set_print_keys(options,h,"mars");
  } else {
    grib_tools_set_print_keys(options,h,"ls");
  }

  if (options->print_keys
      && options->verbose
      && options->print_header) {
        int j=0;
        for (j=0;j<options->print_keys_count;j++) {
          strlenkey=strlen(options->print_keys[j].name);
          width= strlenkey < options->default_print_width  ?
            options->default_print_width+2 : strlenkey+2;
          if (options->default_print_width < 0)
            width=strlenkey+1;
          fprintf(dump_file,"%-*s",(int)width,options->print_keys[j].name);
        }
        if (options->latlon) {
          if (options->latlon_mode==4) {
            fprintf(dump_file,"       value1 ");
            fprintf(dump_file," value2 ");
            fprintf(dump_file," value3 ");
            fprintf(dump_file," value4 ");
         } else fprintf(dump_file," value ");
        }
        if (options->index_on) {
           fprintf(dump_file,"        value(%d) ",(int)options->index);
        }
        fprintf(dump_file,"\n");
   }
}


static void grib_tools_set_print_keys(grib_runtime_options* options, grib_handle* h, const char* ns) {
    int i=0;
    grib_keys_iterator* kiter=NULL;

    Assert(ns);

    if (options->requested_print_keys && !grib_options_on("P:"))
      return;

    for (i=options->requested_print_keys;i<options->print_keys_count;i++) {
      free((char*)options->print_keys[i].name);
      options->print_keys[i].name=NULL;
    }

    kiter=grib_keys_iterator_new(h,0,(char*)ns);
    if (!kiter) {
      fprintf(dump_file,"ERROR: Unable to create keys iterator\n");
      exit(1);
    }

    options->print_keys_count=options->requested_print_keys;
    while(grib_keys_iterator_next(kiter))
    {
      const char* name = grib_keys_iterator_get_name(kiter);

      if (options->print_keys_count >= MAX_KEYS ) {
         fprintf(stderr,"ERROR: keys list too long (more than %d keys)\n",
           options->print_keys_count);
         exit(1);
      }
      options->print_keys[options->print_keys_count].name=strdup(name);

      options->print_keys[options->print_keys_count].type=GRIB_TYPE_STRING;
      options->print_keys_count++;
    }

    grib_keys_iterator_delete(kiter);
}

void grib_skip_check(grib_runtime_options* options,grib_handle* h) {
    int i,ret=0;
    double dvalue=0;
    long lvalue=0;
    char value[MAX_STRING_LEN]={0,};
    options->skip=0;
    for (i=0;i < options->constraints_count && options->skip==0;i++) {
      size_t len=MAX_STRING_LEN;
      switch (options->constraints[i].type) {
        case GRIB_TYPE_STRING:
          ret=grib_get_string( h,options->constraints[i].name,value,&len);
          if ( options->constraints[i].equal ) {
            if (grib_inline_strcmp(value,options->constraints[i].string_value)) options->skip=1;
          } else {
            if (!grib_inline_strcmp(value,options->constraints[i].string_value)) options->skip=1;
          }
          break;
        case GRIB_TYPE_DOUBLE:
          ret=grib_get_double( h,options->constraints[i].name,&dvalue);
          if ( options->constraints[i].equal ) {
            if (dvalue != options->constraints[i].double_value) options->skip=1;
          } else {
            if (dvalue == options->constraints[i].double_value) options->skip=1;
          }
          break;
        case GRIB_TYPE_LONG:
          ret=grib_get_long( h,options->constraints[i].name,&lvalue);
          if ( options->constraints[i].equal ) {
            if (lvalue != options->constraints[i].long_value) options->skip=1;
          } else {
            if (lvalue == options->constraints[i].long_value) options->skip=1;
          }
          break;
        case GRIB_TYPE_MISSING:
          lvalue=grib_is_missing( h,options->constraints[i].name,&ret);
          options->skip = (lvalue == options->constraints[i].equal) ? 0 : 1;
          break;
      }
      if (ret != GRIB_SUCCESS && options->fail) {
        grib_context_log(h->context,GRIB_LOG_ERROR,"unable to get \"%s\" (%s)",
                options->constraints[i].name,grib_get_error_message(ret));
        exit(ret);
      }
    }

    if (!options->skip) {
       options->filter_handle_count++;
       if (options->current_infile)
          options->current_infile->filter_handle_count++;
    }
}

void grib_print_key_values(grib_runtime_options* options,grib_handle* h) {
  int i=0;
  int ret=0,width=0;
  size_t strlenkey=0;
  double dvalue=0;
  long lvalue=0;
  char value[MAX_STRING_LEN];
  char* notfound="not found";

  if (!options->verbose) return;

  for (i=0;i<options->print_keys_count;i++) {
    size_t len=MAX_STRING_LEN;
    ret=GRIB_SUCCESS;

    if (grib_is_missing(h,options->print_keys[i].name,&ret) && ret==GRIB_SUCCESS)
      sprintf(value,"MISSING");
    else if ( ret == GRIB_SUCCESS )
      switch (options->print_keys[i].type) {
        case GRIB_TYPE_STRING:
          ret=grib_get_string( h,options->print_keys[i].name,value,&len);
          break;
        case GRIB_TYPE_DOUBLE:
          ret=grib_get_double( h,options->print_keys[i].name,&dvalue);
          sprintf(value,"%20.0f",dvalue);
          break;
        case GRIB_TYPE_LONG:
          ret=grib_get_long( h,options->print_keys[i].name,&lvalue);
          sprintf(value,"%d",(int)lvalue);
          break;
        default:
          fprintf(dump_file,"invalid format option for %s\n",options->print_keys[i].name);
          exit(1);
      }
    if (ret != GRIB_SUCCESS) {
      if (options->fail) GRIB_CHECK(ret,options->print_keys[i].name);
        if (ret == GRIB_NOT_FOUND) strcpy(value,notfound);
        else {
          fprintf(dump_file,"%s %s\n",grib_get_error_message(ret),options->print_keys[i].name);
          exit(ret);
        }
    }
    strlenkey=strlen(options->print_keys[i].name);
    if (options->default_print_width < 0) {
      width=strlen(value);
      if (options->print_keys_count != 1) width++;
    }
    else if ( strlenkey < options->default_print_width )
      width= options->default_print_width+2;
    else
      width= strlenkey+2;
    fprintf(dump_file,"%-*s",(int)width,value);
  }

  if (options->latlon) {
    double min;
    if (options->latlon_idx<0) {
      min=options->distances[0];
      options->latlon_idx=0;
      i=0;
      for (i=1;i<4;i++) {
        if (min>options->distances[i]) {
          min=options->distances[i];
          options->latlon_idx=i;
        }
      }
    }
    if (options->latlon_mode==4){
      int i=0;
      for (i=0;i<4;i++) fprintf(dump_file,"%.3e ",options->values[i]);
    } else if (options->latlon_mode==1) {
      fprintf(dump_file,"%.3e ",options->values[options->latlon_idx]);
    }
  }
  if (options->index_on) {
    double v=0;
    if (grib_get_double_element(h,"values",options->index,&v) == GRIB_SUCCESS)
      fprintf(dump_file,"%.3e ",v);
  }
  fprintf(dump_file,"\n");

}


void grib_print_file_statistics(grib_runtime_options* options,grib_tools_file* file) {
  grib_failed* failed=NULL;

  Assert(file);

  failed=file->failed;

  if (!options->print_statistics || !options->verbose) return;

  fprintf(dump_file,"%d of %d grib messages in %s\n\n",
      file->filter_handle_count,
      file->handle_count,
      file->name);
  if (!failed) return;
/*
  fprintf(dump_file,"Following bad grib messages found in %s\n",
          file->name);
  fprintf(dump_file,"N      Error\n");
  while (failed){
    fprintf(dump_file,"%-*d    %s\n",
        7,failed->count,
        grib_get_error_message(failed->error));
    failed=failed->next;
  }
  fprintf(dump_file,"\n");
*/
}

void grib_print_full_statistics(grib_runtime_options* options) {
  if (options->print_statistics && options->verbose)
    fprintf(dump_file,"%d of %d total grib messages in %d files\n",
        options->filter_handle_count,options->handle_count,options->file_count);
}

void grib_tools_write_message(grib_runtime_options* options, grib_handle* h) {
  const void *buffer; size_t size;
  grib_file* of=NULL;
  int err=0;
  int ioerr=0;
  char filename[1024]={0,};
  Assert(options->outfile!=NULL && options->outfile->name!=NULL);

  if (options->error == GRIB_WRONG_LENGTH) return;

  if ((err=grib_get_message(h,&buffer,&size))!= GRIB_SUCCESS) {
    grib_context_log(h->context,GRIB_LOG_ERROR,"unable to get binary message\n");
    exit(err);
  }

  err = grib_recompose_name(h,NULL,options->outfile->name,filename,0);

  of=grib_file_open(filename,"w",&err);

  if (!of || !of->handle) {
    ioerr=errno;
    grib_context_log(h->context,(GRIB_LOG_ERROR)|(GRIB_LOG_PERROR),
                     "unable to open file %s\n",filename);
    exit(GRIB_IO_PROBLEM);
  }

  if (options->gts && h->gts_header)
    fwrite(h->gts_header,1,h->gts_header_len,of->handle);

  if(fwrite(buffer,1,size,of->handle) != size) {
    ioerr=errno;
    grib_context_log(h->context,(GRIB_LOG_ERROR)|(GRIB_LOG_PERROR),
                     "Error writing to %s",filename);
    exit(GRIB_IO_PROBLEM);
  }

  if (options->gts && h->gts_header) {
    char gts_trailer[4]={'\x0D','\x0D','\x0A','\x03'};
    fwrite(gts_trailer,1,4,of->handle);
  }

  grib_file_close(filename,&err);

  if (err != GRIB_SUCCESS) {
    grib_context_log(h->context,GRIB_LOG_ERROR,"unable to write message\n");
    exit(err);
  }

  options->outfile->file=NULL;

#if 0
  if (!options->outfile->file)  {
    options->outfile->file = fopen(options->outfile->name,"w");
    if(!options->outfile->file) {
      perror(options->outfile->name);
      exit(1);
    }
  }
  GRIB_CHECK(grib_get_message(h,&buffer,&size),0);
  if (options->gts && h->gts_header)
    fwrite(h->gts_header,1,h->gts_header_len,options->outfile->file);

  if(fwrite(buffer,1,size,options->outfile->file) != size)
  {
    perror(options->outfile->name);
    exit(1);
  }

  if (options->gts && h->gts_header) {
    char gts_trailer[4]={'\x0D','\x0D','\x0A','\x03'};
    fwrite(gts_trailer,1,4,options->outfile->file);
  }
#endif
                                                                                                                        
}


