/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  grig-druid-loc.c:  Location/QTH module for the configuration druid.

  Copyright (C)  2001, 2002  Groundstation Software Suite Team.

  Authors: Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "grig-druid-loc.h"

#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_RIG_H)
#  include <hamlib/rig.h>
#endif

#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
#  include <hamlib/rotator.h>
#endif


extern GdkColor headbg; /* defd in grig-druid.c */
extern GConfClient *confclient;   /* main.c */

GtkWidget *callwdg,*locwdg,*descwdg,*locatorwdg;
GtkWidget *latwdg,*lonwdg,*radloc,*radll;


/* private function prototypes */
static void grig_druid_loc_locator_changed_cb (GtkEntry *, gpointer);
static void grig_druid_loc_latlon_changed_cb (GtkEntry *, gpointer);
static void grig_druid_loc_toggled_cb (GtkToggleButton *, gpointer);
static void grig_druid_loc_page_prepare_cb (GnomeDruidPage *, gpointer, gpointer);


GtkWidget *grig_druid_loc_create (GtkWidget *druid)
{
	/* This function creates a GnomeDruidPageStandard containing
	   all necessary widgets for configuring the locations. This
	   function should only be called from the main configuration
	   druid function.
	*/

	GtkWidget *page,*uframe,*lframe;
	GtkWidget *label,*upper,*lower;
	gint numlocs,defloc;
	gchar *key;

	page = gnome_druid_page_standard_new_with_vals (_("Default Location (QTH)"),
							NULL);
	gnome_druid_page_standard_set_bg_color (GNOME_DRUID_PAGE_STANDARD (page),
						&headbg);
	gnome_druid_page_standard_set_logo_bg_color (GNOME_DRUID_PAGE_STANDARD (page),
						     &headbg);

	gtk_container_set_border_width (GTK_CONTAINER (GNOME_DRUID_PAGE_STANDARD (page)->vbox), 5);

	/* create widgets - upper table */
	upper = gtk_table_new (2, 4, FALSE);

	label = gtk_label_new (_("Callsign"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0f, 0.5f);
	gtk_misc_set_padding (GTK_MISC (label), 5, 1);
	gtk_table_attach_defaults (GTK_TABLE (upper), label, 0, 1, 0, 1);

	callwdg = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (upper), callwdg, 1, 2, 0, 1);

	label = gtk_label_new (_("Location"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0f, 0.5f);
	gtk_misc_set_padding (GTK_MISC (label), 5, 1);
	gtk_table_attach_defaults (GTK_TABLE (upper), label, 2, 3, 0, 1);

	locwdg = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (upper), locwdg, 3, 4, 0, 1);

	label = gtk_label_new (_("Description"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0f, 0.5f);
	gtk_misc_set_padding (GTK_MISC (label), 5, 1);
	gtk_table_attach_defaults (GTK_TABLE (upper), label, 0, 1, 1, 2);

	descwdg = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (upper), descwdg, 1, 4, 1, 2);

	/* create nice frame */
	uframe = gtk_frame_new (_("Name and Description"));
	gtk_container_add (GTK_CONTAINER (uframe), upper);

	/* lower table */
	lower = gtk_table_new (3, 4, FALSE);

	radloc = gtk_radio_button_new (NULL);
	gtk_signal_connect (GTK_OBJECT (radloc), "toggled",
			    GTK_SIGNAL_FUNC (grig_druid_loc_toggled_cb),
			    GUINT_TO_POINTER (0));
	gtk_table_attach (GTK_TABLE (lower), radloc, 0, 1, 0, 1, 0, 0, 5, 0);

	label = gtk_label_new (_("QRA Locator"));
	gtk_table_attach (GTK_TABLE (lower), label, 1, 2, 0, 1, 0, 0, 3, 0);

	locatorwdg = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (locatorwdg), 6);
	gtk_table_attach (GTK_TABLE (lower), locatorwdg, 2, 3, 0, 1, 0, 0, 0, 0);
	/* Connect "changed" signal. Whenever the text is changed, check whether
	   we have a valid locator. If yes, activate the next button, otherwise
	   deactivate.
	 */
	gtk_signal_connect (GTK_OBJECT (locatorwdg), "changed",
			    GTK_SIGNAL_FUNC (grig_druid_loc_locator_changed_cb),
			    druid);

	label = gtk_label_new (_("(eg. LH95MA)"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.5f);
	gtk_misc_set_padding (GTK_MISC (label), 5, 1);
	gtk_table_attach_defaults (GTK_TABLE (lower), label, 3, 4, 0, 1);

	radll = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON (radloc));
	gtk_signal_connect (GTK_OBJECT (radll), "toggled",
			    GTK_SIGNAL_FUNC (grig_druid_loc_toggled_cb),
			    GUINT_TO_POINTER (1));
	gtk_table_attach (GTK_TABLE (lower), radll, 0, 1, 1, 2, 0, 0, 5, 0);

	label = gtk_label_new (_("Latitude"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (lower), label, 1, 2, 1, 2, 0, 0, 3, 0);

	latwdg = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (latwdg), 6);
	gtk_table_attach (GTK_TABLE (lower), latwdg, 2, 3, 1, 2, 0, 0, 0, 0);

	gtk_signal_connect (GTK_OBJECT (latwdg), "changed",
			    GTK_SIGNAL_FUNC (grig_druid_loc_latlon_changed_cb),
			    druid);

	label = gtk_label_new (_("dec. deg. North"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.5f);
	gtk_misc_set_padding (GTK_MISC (label), 5, 1);
	gtk_table_attach_defaults (GTK_TABLE (lower), label, 3, 4, 1, 2);

	label = gtk_label_new (_("Longitude"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (lower), label, 1, 2, 2, 3, 0, 0, 3, 0);

	lonwdg = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (lonwdg), 6);
	gtk_table_attach (GTK_TABLE (lower), lonwdg, 2, 3, 2, 3, 0, 0, 0, 0);

	gtk_signal_connect (GTK_OBJECT (lonwdg), "changed",
			    GTK_SIGNAL_FUNC (grig_druid_loc_latlon_changed_cb),
			    druid);

	label = gtk_label_new (_("dec. deg. West"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0f, 0.5f);
	gtk_misc_set_padding (GTK_MISC (label), 5, 1);
	gtk_table_attach_defaults (GTK_TABLE (lower), label, 3, 4, 2, 3);

	/* See if we've got some defaults. This might be the case if the druid
	   is run because the user has messed up the configuration.
	*/
	defloc = gconf_client_get_int (confclient,
				       GRIG_CONFIG_LOC_DIR "/default",
				       NULL);
	numlocs = gconf_client_get_int (confclient,
					GRIG_CONFIG_LOC_DIR "/number",
					NULL);
	key = g_strdup_printf ("%s/%i", GRIG_CONFIG_LOC_DIR, defloc);
	g_print ("\n\n %s + %i \n\n", key, numlocs);
	if (numlocs && gconf_client_dir_exists (confclient, key, NULL)) {
		g_free (key);
		/* call */
		key = g_strdup_printf ("%s/%i/Callsign", GRIG_CONFIG_LOC_DIR, defloc);
		gtk_entry_set_text (GTK_ENTRY (callwdg), gconf_client_get_string (confclient, key, NULL));
		g_free (key);
		/* Location */
		key = g_strdup_printf ("%s/%i/Location", GRIG_CONFIG_LOC_DIR, defloc);
		gtk_entry_set_text (GTK_ENTRY (locwdg), gconf_client_get_string (confclient, key, NULL));
		g_free (key);
		/* Description */
		key = g_strdup_printf ("%s/%i/Description", GRIG_CONFIG_LOC_DIR, defloc);
		gtk_entry_set_text (GTK_ENTRY (descwdg), gconf_client_get_string (confclient, key, NULL));
		g_free (key);
		/* QRA locator */
		key = g_strdup_printf ("%s/%i/QRA", GRIG_CONFIG_LOC_DIR, defloc);
		gtk_entry_set_text (GTK_ENTRY (locatorwdg), gconf_client_get_string (confclient, key, NULL));
	}
	g_free (key);

	/* create nice frame */
	lframe = gtk_frame_new (_("Geographical Location"));
	gtk_container_add (GTK_CONTAINER (lframe), lower);

	gtk_box_pack_start_defaults (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page)->vbox), uframe);
	gtk_box_pack_start_defaults (GTK_BOX (GNOME_DRUID_PAGE_STANDARD (page)->vbox), lframe);

	gtk_signal_connect (GTK_OBJECT (page), "prepare",
			    GTK_SIGNAL_FUNC (grig_druid_loc_page_prepare_cb),
			    locatorwdg);

	/* some hack to make the widgets deactivate */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radll), TRUE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radloc), TRUE);

	/* see whether we have some existing values */

	return page;
}


static void
grig_druid_loc_locator_changed_cb (GtkEntry *entry, gpointer druid)
{
	/* This function is called when the text in the QRA locator
	   widget is changed. It converts the locator to latitude and
	   longitude using the conversion functions supplied by
	   hamlib/locator.c
	   It also activates/deactivates the "Next" button of the
	   druid, depending on whether the locator is valid or not.

	   ****      ONLY WORKS WITH hamlib >= 1.1.3          **** 

	*/
#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
	gdouble lat,lon;
	gchar *buff;

	if (!locator2longlat (&lon, &lat, gtk_entry_get_text (entry))) {
		/* locator is valid, update lat/lon and activate "next".
		   Only update lat/lon if user is actually is typing into
		   locator (not autoupdate).
		*/
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radloc))) {
			buff = g_strdup_printf ("%.2f", lat);
			gtk_entry_set_text (GTK_ENTRY (latwdg), buff);
			g_free (buff);
			buff = g_strdup_printf ("%.2f", lon);
			gtk_entry_set_text (GTK_ENTRY (lonwdg), buff);
			g_free (buff);
		}
		gtk_widget_set_sensitive (GNOME_DRUID (druid)->next, TRUE);
		
	}
	else {
		/* invalid locator, deactivate next */
		gtk_widget_set_sensitive (GNOME_DRUID (druid)->next, FALSE);
	}
#endif
}


static void
grig_druid_loc_latlon_changed_cb (GtkEntry *entry, gpointer druid)
{
	/* This function is called when the user modifies either the latitude
	   or longitude entry. The contents are converted to a QRA locator and
	   if the conversion is all right, the "Next" button is activated.

	   ****      ONLY WORKS WITH hamlib >= 1.1.3          **** 

	*/
#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
 	gchar loc[6];	

	/* Check whether user is typing into lat/lon entries or the change comes from
	   automatic conversion.
	*/
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radll))) {
		longlat2locator (g_strtod (gtk_entry_get_text (GTK_ENTRY (lonwdg)), NULL),
				 g_strtod (gtk_entry_get_text (GTK_ENTRY (latwdg)), NULL),
				 loc);
		gtk_entry_set_text (GTK_ENTRY (locatorwdg), g_strdup (loc));
	}
#endif
}



static void
grig_druid_loc_page_prepare_cb (GnomeDruidPage *page, gpointer druid, gpointer entry)
{
	/* This function is called just before the user arrives on the
	   location page. It is used to disable the "Next" button.

	   ****   ONLY for hamlib >= 1.1.3     ****
	*/
#if defined (HAVE_LIBHAMLIB) && defined (HAVE_HAMLIB_ROTATOR_H)
	gdouble lat,lon;

	if (!locator2longlat (&lon, &lat, gtk_entry_get_text (GTK_ENTRY (entry))))
		gtk_widget_set_sensitive (GNOME_DRUID (druid)->next, TRUE);
	else
		gtk_widget_set_sensitive (GNOME_DRUID (druid)->next, FALSE);
#endif
} 


static void
grig_druid_loc_toggled_cb (GtkToggleButton *button, gpointer index)
{
	/* This function is called when the user selects one of
	   the radio buttons.
	*/

	/* check whether button was checked or un-checked */
	if (gtk_toggle_button_get_active (button)) {
		switch (GPOINTER_TO_UINT (index)) {
		case 0:
			gtk_widget_set_sensitive (locatorwdg, TRUE);
			gtk_widget_set_sensitive (latwdg, FALSE);
			gtk_widget_set_sensitive (lonwdg, FALSE);
			break;
		case 1:
			gtk_widget_set_sensitive (latwdg, TRUE);
			gtk_widget_set_sensitive (lonwdg, TRUE);
			gtk_widget_set_sensitive (locatorwdg, FALSE);
			break;
		default:
		}
	}
}





gint grig_druid_loc_cancel ()
{
	/* This function should be called by the main config druid
	   cancel callback. The function returns 0 if everything was
	   all right or 1 if an error has occured.
	*/



	return 0;
}


gint grig_druid_loc_finish ()
{
	/* This function should be called by the main config druid
	   finish callback function. The function returns 0 if
	   everything was all right or 1 if an error has occured.
	*/
	gboolean ok = TRUE;
	gchar *buff;

	/* default location is in /apps/grig/locations/0/ */
	ok = ok && gconf_client_set_int (confclient,
					 GRIG_CONFIG_LOC_DIR "/default",
					 0, NULL);
	/* number of locations */
	ok = ok && gconf_client_set_int (confclient,
					 GRIG_CONFIG_LOC_DIR "/number",
					 1, NULL);

	/* callsign */
	buff = g_strdup (gtk_entry_get_text (GTK_ENTRY (callwdg)));
	if (buff) {
		g_strup (buff);  /* want callsign in uppercase */
		ok = ok && gconf_client_set_string (confclient,
						    GRIG_CONFIG_LOC_DIR "/0/Callsign",
						    buff, NULL);
		g_free (buff);
	}
	/* location */
	if (gtk_entry_get_text (GTK_ENTRY (locwdg))) {
		ok = ok && gconf_client_set_string (confclient,
						    GRIG_CONFIG_LOC_DIR "/0/Location",
						    gtk_entry_get_text (GTK_ENTRY (locwdg)),
						    NULL);
	}
	/* description */
	if (gtk_entry_get_text (GTK_ENTRY (descwdg))) {
		ok = ok && gconf_client_set_string (confclient,
						    GRIG_CONFIG_LOC_DIR "/0/Description",
						    gtk_entry_get_text (GTK_ENTRY (descwdg)),
						    NULL);
	}
	/* QRA Locator */
	buff = g_strdup (gtk_entry_get_text (GTK_ENTRY (locatorwdg)));
	g_strup (buff);
	ok = ok && gconf_client_set_string (confclient,
					    GRIG_CONFIG_LOC_DIR "/0/QRA",
					    buff, NULL);
	g_free (buff);
	/* latitude */
	ok = ok && gconf_client_set_float (confclient,
					   GRIG_CONFIG_LOC_DIR "/0/Lat",
					   g_strtod (gtk_entry_get_text (GTK_ENTRY (latwdg)), NULL),
					   NULL);
	/* longitude */
	ok = ok && gconf_client_set_float (confclient,
					   GRIG_CONFIG_LOC_DIR "/0/Lon",
					   g_strtod (gtk_entry_get_text (GTK_ENTRY (lonwdg)), NULL),
					   NULL);


	return (gint) !ok;  /* shoot me! */
}

