/*
 * GroMove widget header
 *
 * GroMove draws pixmap on a window (typically root window).
 *
 * Roughly, there are three dimensional states,
 * [state] x [direction] x [frame].
 *
 * This module doesn't care about how to move and where to move.
 * The next position, the next state and the next direction are provided
 * by an external module via interfaces, such as gro_move_move(),
 * gro_move_change_gstat() and gro_move_change_direct(), respectively.
 *
 * On the other hand, frame(a.k.a. animation frames) is taken care of
 * by this module.
 * Frame is automatically advanced by this module, and it repeats them.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifndef __GRO_MOVE_H__
#define __GRO_MOVE_H__

#include <gdk/gdk.h>
#include <gtk/gtkobject.h>
#include <gtk/gtksignal.h>

#include "controller.h"

/** Library private **/
#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#define GRO_TYPE_MOVE				(gro_move_get_type())
#define GRO_MOVE(obj)				(GTK_CHECK_CAST((obj), GRO_TYPE_MOVE, GroMove))
#define GRO_MOVE_CLASS(klass)		(GTK_CHECK_CLASS_CAST((klass), GRO_TYPE_MOVE, GroMoveClass))
#define GRO_IS_MOVE(obj)			(GTK_CHECK_TYPE((obj), GRO_TYPE_MOVE))
#define GRO_IS_MOVE_CLASS(klass)	(GTK_CHECK_CLASS_TYPE((klass), GRO_TYPE_MOVE))


typedef struct _GroMoveClass GroMoveClass;

/* The number of state is arbitrary(>=0),
   but there are two pre-defined states. */
enum {
	GRO_STAT_DEAD  = -1,/* special stat. This stat never has its frames */
	GRO_STAT_WAKE = 0	/* default value, but it has no other meaning. */
};

/* Private stuff */	
typedef struct _GroMovePrivate GroMovePrivate;

/*
 * Note: Exported variables are read-only. They are modified through interfaces.
 */
struct _GroMove {
	GtkObject object;

	guint num_gstat;		/* Number of stats */
	gint cur_gstat;			/* Current stat */
	guint num_direct;		/* Number of directions */
	guint cur_direct;		/* Current direction */
	GdkRectangle cur_rect;	/* aka. current position */

	/* motion logic is out of this module.
	   This is managed by the external module.
	   GroMove doesn't care about its content. */
	gpointer move_env;
	
	GroMovePrivate *privat;
};

struct _GroMoveClass {
	GtkObjectClass parent_class;

	/* methods */
	void (*move)(GroMove *gmove, const GroController *controller, const GroVector *vec);
	void (*push)(GroMove *gmove, const GroController *controller, GdkEventButton *event);
};


/* Global function declarations */
extern GtkType gro_move_get_type(void);
extern GtkObject* gro_move_new(guint num_gstat, guint num_direct, guint pix_width, guint pix_height);
extern void gro_move_set_frames(GroMove *gmove, const GPtrArray *frames, gint gstat, guint direct, gboolean b_static_frames);
extern void gro_move_move(GroMove *gmove, const GroController *controller, const GroVector *vec);
extern void gro_move_push(GroMove *gmove, const GroController *controller, GdkEventButton *event);
extern void gro_move_change_gstat(GroMove *gmove, gint gstat);
extern void gro_move_change_direct(GroMove *gmove, guint direct);
extern gboolean gro_move_is_eof(const GroMove *gmove);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __GRO_MOVE_H__ */
