/*-----------------------------------------------------------------+
 |                                                                 |
 |  Copyright (C) 2002-2003 Grubconf                               |
 |                     http://grubconf.sourceforge.net/            | 
 |                                                                 |
 | This program is free software; you can redistribute it and/or   |
 | modify it under the terms of the GNU General Public License     |
 | as published by the Free Software Foundation; either version 2  |
 | of the License, or (at your option) any later version           |
 |                                                                 |
 | This program is distributed in the hope that it will be useful, |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of  |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   |
 | GNU General Public License for more details.                    |
 |                                                                 |
 | A copy of the GNU General Public License may be found in the    |
 | installation directory named "COPYING"                          |
 |                                                                 |
 +-----------------------------------------------------------------+
 */
/* 
 * This file manages the list of bootable OS's. It also handles
 * interaction with the os configuration dialog window as far as
 * resetting it, populating it, and retrieving data from it
 */

#include <string.h>
#include <stdio.h>
#include <include/grubconf_os_list.h>

/* private functions */
int os_list_clean_data_linux (void *data);


/* Creates a new entry in the os list of type t from the current dialog
 */
GtkTreeIter   *
os_list_new (enum os_type t)
{
	GtkWidget *txtOSTitle = lookup_widget (diaOS, "txtOSTitle");
	GtkWidget *optLinuxRootDev =
		lookup_widget (diaOS, "optLinuxRootDev");
	GtkWidget *optLinuxBootDev =
		lookup_widget (diaOS, "optLinuxBootDev");
	GtkWidget *txtOSLinuxKernel =
		lookup_widget (diaOS, "txtOSLinuxKernel");
	GtkWidget *txtOSLinuxParams =
		lookup_widget (diaOS, "txtOSLinuxParams");
	GtkWidget *tvOSLinuxExtra = lookup_widget (diaOS, "tvOSLinuxExtra");
	GtkTextBuffer *tbOSLinuxExtra =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSLinuxExtra));
	GtkWidget *optOSWinDev = lookup_widget (diaOS, "optOSWinDev");
	GtkWidget *tvOSOther = lookup_widget (diaOS, "tvOSOther");
	GtkTextBuffer *tbOSOther =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSOther));
	GtkTextIter start, end;
	char *title;

	struct os_list_item *new_item = calloc (1, sizeof (struct os_list_item));
	struct os_list_linux *tmp_linux;
	struct os_list_windows *tmp_windows;
	if (new_item == NULL)
	{
		return 0;
	}
	title =	gtk_editable_get_chars (GTK_EDITABLE (txtOSTitle), 0, -1);
	if (strlen(title) < 1) {
		return 0;
	}
	new_item->type = t;

	switch (t)
	{
	case LINUX:
		tmp_linux =
			(struct os_list_linux *)
			calloc (1, sizeof (struct os_list_linux));
		if (tmp_linux == NULL)
		{
			os_list_free (new_item);
			free (new_item);
			return 0;
		}

		tmp_linux->rdev_index = get_selected (optLinuxRootDev);
		tmp_linux->bdev_index = get_selected (optLinuxBootDev);

		tmp_linux->kernel =
			gtk_editable_get_chars (GTK_EDITABLE
						(txtOSLinuxKernel), 0, -1);
		tmp_linux->kernel_parms =
			gtk_editable_get_chars (GTK_EDITABLE
						(txtOSLinuxParams), 0, -1);
		new_item->data = tmp_linux;
		gtk_text_buffer_get_bounds (tbOSLinuxExtra, &start, &end);
		new_item->unknown = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (tbOSLinuxExtra),&start, &end, 0);
		break;
	case WINDOWS:
		tmp_windows =
			(struct os_list_windows *)
			calloc (1, sizeof (struct os_list_windows));
		if (tmp_windows == NULL)
		{
			os_list_free (new_item);
			free (new_item);
			return 0;
		}

		tmp_windows->dev_index = get_selected (optOSWinDev);

		new_item->data = tmp_windows;
		break;
	case OTHER:
		gtk_text_buffer_get_bounds (tbOSOther, &start, &end);
		new_item->unknown = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (tbOSOther),&start, &end, 0);
	}

	
	return os_list_put(title, FALSE, new_item);;
}

/* Allocates a new os_list_item of type linux and appends it
 * to the clist
 */
struct os_list_item *
os_list_new_linux (gboolean def, char *new_title,
		   int rd_index,
		   int bd_index,
		   char *new_kernel,
		   char *new_kernel_parms, char *new_unknown)
{
	struct os_list_item *new_item = malloc (sizeof (struct os_list_item));
	struct os_list_linux *tmp_linux;
	if (new_item == NULL)
	{
		return NULL;
	}
	new_item->type = LINUX;
	tmp_linux = calloc (1, sizeof (struct os_list_linux));
	if (tmp_linux == NULL) {
		return NULL;
	}
	tmp_linux->rdev_index = rd_index;
	tmp_linux->bdev_index = bd_index;
	tmp_linux->kernel = new_kernel;
	tmp_linux->kernel_parms = new_kernel_parms;
	new_item->data = tmp_linux;
	new_item->unknown = new_unknown;

	os_list_put(new_title, def, new_item);
	return new_item;
}

/* Allocates a new os_list_item of type windows and appends it
 * to the clist
 */
struct os_list_item *
os_list_new_windows (gboolean def, char *new_title,
		     int index,
		     char *new_unknown)
{
	struct os_list_item *new_item = calloc (1, sizeof (struct os_list_item));
	struct os_list_windows *tmp_windows;
	if (new_item == NULL)
	{
		return NULL;
	}
	new_item->type = WINDOWS;
	tmp_windows = calloc (1, sizeof (struct os_list_windows));
	if (tmp_windows == NULL) {
		return NULL;
	}
	tmp_windows->dev_index = index;
	new_item->data = tmp_windows;
	new_item->unknown = new_unknown;

	os_list_put(new_title, def, new_item);

	return new_item;
}

/* Allocates a new os_list_item of type other and appends it
 * to the clist
 */
struct os_list_item *
os_list_new_other (gboolean def, char *new_title, char *new_unknown)
{
	struct os_list_item *new_item = calloc (1, sizeof (struct os_list_item));
	if (new_item == NULL)
	{
		return NULL;
	}
	new_item->type = OTHER;
	new_item->unknown = new_unknown;

	os_list_put(new_title, def, new_item);
	return new_item;
}

/* Inserts new_item into the list of OS'
 */
GtkTreeIter   *
os_list_put(char *title, gboolean def, struct os_list_item *new_item) {
	
	GtkWidget *treeview1 = lookup_widget (winMain, "treeview1");
	GtkListStore *store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (treeview1)));
	
	GtkTreeIter   *iter = malloc (sizeof (GtkTreeIter));

	gtk_list_store_append (store, iter);  /* Acquire an iterator */

	gtk_list_store_set (store, iter,
	                    DEFAULT_COLUMN, def,
	                    TITLE_COLUMN, title,
	                    DATA_COLUMN, new_item,
	                    -1);
	return iter;
//	printf ("appended %s, new size=%i\n", new_item->title, os_list_size);
}	

/* sets a previously created os list item of type t at index
 */
int
os_list_set (enum os_type t)
{
	struct os_list_item *new_item;
	struct os_list_linux *tmp_linux;
	struct os_list_windows *tmp_windows;
	char *title;
	
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *selection = 
	      gtk_tree_view_get_selection ( GTK_TREE_VIEW (lookup_widget (winMain, "treeview1")) );
	
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return 0;
	
	gtk_tree_model_get (model, &iter, DATA_COLUMN, &new_item, -1);	
	
	GtkWidget *txtOSTitle = lookup_widget (diaOS, "txtOSTitle");
	GtkWidget *optLinuxRootDev =
		lookup_widget (diaOS, "optLinuxRootDev");
	GtkWidget *optLinuxBootDev =
		lookup_widget (diaOS, "optLinuxBootDev");
	GtkWidget *txtOSLinuxKernel =
		lookup_widget (diaOS, "txtOSLinuxKernel");
	GtkWidget *txtOSLinuxParams =
		lookup_widget (diaOS, "txtOSLinuxParams");
	GtkWidget *tvOSLinuxExtra = lookup_widget (diaOS, "tvOSLinuxExtra");
	GtkTextBuffer *tbOSLinuxExtra =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSLinuxExtra));
	GtkWidget *optOSWinDev = lookup_widget (diaOS, "optOSWinDev");
	GtkWidget *tvOSOther = lookup_widget (diaOS, "tvOSOther");
	GtkTextBuffer *tbOSOther =
		gtk_text_view_get_buffer (GTK_TEXT_VIEW (tvOSOther));
	GtkTextIter start, end;


	if (new_item == NULL)
	{
		return -1;
	}

	os_list_free (new_item);
	title =	gtk_editable_get_chars (GTK_EDITABLE (txtOSTitle), 0, -1);
	new_item->type = t;

	switch (t)
	{
	case LINUX:
		tmp_linux = calloc (1, sizeof (struct os_list_linux));
		if (tmp_linux == NULL)
		{
			return -1;
		}
		tmp_linux->rdev_index = get_selected (optLinuxRootDev);
		tmp_linux->bdev_index = get_selected (optLinuxBootDev);

		tmp_linux->kernel =
			gtk_editable_get_chars (GTK_EDITABLE
						(txtOSLinuxKernel), 0, -1);
		tmp_linux->kernel_parms =
			gtk_editable_get_chars (GTK_EDITABLE
						(txtOSLinuxParams), 0, -1);
		new_item->data = tmp_linux;
		gtk_text_buffer_get_bounds (tbOSLinuxExtra, &start, &end);
		new_item->unknown = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (tbOSLinuxExtra),&start, &end, 0);
		break;
	case WINDOWS:
		tmp_windows =
			(struct os_list_windows *)
			calloc (1, sizeof (struct os_list_windows));
		if (tmp_windows == NULL)
		{
			return -1;
		}

		tmp_windows->dev_index = get_selected (optOSWinDev);

		new_item->data = tmp_windows;
		break;
	case OTHER:
		gtk_text_buffer_get_bounds (tbOSOther, &start, &end);
		new_item->unknown = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (tbOSOther),&start, &end, 0);
	}
	
	gtk_list_store_set (GTK_LIST_STORE (model), &iter,
	                    TITLE_COLUMN, title,
	                    DATA_COLUMN, new_item,
	                    -1);
	return 0;
}

/* deletes selected os from lists
 */
int
os_list_delete_selected ()
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *selection = 
	      gtk_tree_view_get_selection ( GTK_TREE_VIEW (lookup_widget (winMain, "treeview1")) );
	
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return 0;

	os_list_delete (model, &iter);
	
	return 1;
}

/* deletes provided os from lists
 */
int
os_list_delete (GtkTreeModel *model, GtkTreeIter *iter)
{
	struct os_list_item *tmp;
	gboolean def;
	gboolean valid_iter;

	gtk_tree_model_get (model, iter, DEFAULT_COLUMN, &def, DATA_COLUMN, &tmp, -1);	

	if (tmp) {
		os_list_free (tmp);
		free (tmp);
	}

// to work around a stupid gtk version thing I need this
#if GTK_MINOR_VERSION > 0
	valid_iter = gtk_list_store_remove (GTK_LIST_STORE (model), iter);
#else
	printf ("warning: old gtk\n");
	gtk_list_store_remove (GTK_LIST_STORE (model), iter);
	if (iter && iter->stamp)
		valid_iter = TRUE;
	else 
		valid_iter = FALSE;
#endif	
	
	// now reset the default OS if needed
	if (def) {
		if (valid_iter) {
			gtk_list_store_set (GTK_LIST_STORE (model), iter,
			                    DEFAULT_COLUMN, TRUE,
			                    -1);
		} else {
			if (gtk_tree_model_get_iter_first (model, iter)) {
				gtk_list_store_set (GTK_LIST_STORE (model), iter,
									DEFAULT_COLUMN, TRUE,
									-1);
			}
		}
	}

	return 0;
}

/* cleans the internal list of OS's
 */
int
os_list_clean ()
{
	GtkTreeModel *model = gtk_tree_view_get_model ( GTK_TREE_VIEW (lookup_widget (winMain, "treeview1")) );
	GtkTreeIter iter;
	
	while (gtk_tree_model_get_iter_first (model, &iter))
	{
		os_list_delete (model, &iter);
	}
	return 0;
}

/* free's any dynamic data allocated to item
 */
int
os_list_free (struct os_list_item *item)
{
	if (item->data != NULL)
	{
		if (item->type == LINUX)
		{
			os_list_clean_data_linux (item->data);
		}
		else
		{
			g_free (item->data);
		}
		item->data = NULL;
	}
	if (item->unknown != NULL)
	{
		g_free (item->unknown);
		item->unknown = NULL;
	}
	return 0;
}

/* cleans Linux specific data
 */
int
os_list_clean_data_linux (void *data)
{
	struct os_list_linux *tmp_linux;
	if (data == NULL)
	{
		return -1;
	}
	tmp_linux = (struct os_list_linux *) data;
	if (tmp_linux->kernel != NULL)
	{
		g_free (tmp_linux->kernel);
	}
	if (tmp_linux->kernel_parms != NULL)
	{
		g_free (tmp_linux->kernel_parms);
	}
	return 0;
}
