/*-----------------------------------------------------------------+
 |                                                                 |
 |  Copyright (C) 2002-2003 Grubconf                               |
 |                     http://grubconf.sourceforge.net/            | 
 |                                                                 |
 | This program is free software; you can redistribute it and/or   |
 | modify it under the terms of the GNU General Public License     |
 | as published by the Free Software Foundation; either version 2  |
 | of the License, or (at your option) any later version           |
 |                                                                 |
 | This program is distributed in the hope that it will be useful, |
 | but WITHOUT ANY WARRANTY; without even the implied warranty of  |
 | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   |
 | GNU General Public License for more details.                    |
 |                                                                 |
 | A copy of the GNU General Public License may be found in the    |
 | installation directory named "COPYING"                          |
 |                                                                 |
 +-----------------------------------------------------------------+
 */
/*
 * The main entry point.
 */

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

#include <config.h>
#include <include/interface.h>
#include <include/support.h>
#include <include/grubconf_global.h>

#define GCONF_FNAME_KEY "/apps/Grubconf/filename"

void print_usage(void);

void print_usage() {
	printf("Usage: grubconf [options] [filename]\n");
	printf("       -h   --help    Display this usage information\n"
	       "       -c   --create  Force to create a new file\n"
	       "       -s   --save    Use the specified filename for future \n"
	       "                      grubconf sessions\n"
	       "       filename       Specifies the grub configuration filename,\n"
	       "                      if not specified \"/boot/grub/menu.lst\" will be used\n");
}

int
main (int argc, char *argv[])
{
	grubconf_fname = NULL;
	gboolean save_fname = FALSE;
	gboolean create_fname = FALSE;
	
	int next_option;
	const char* short_options ="hcs";
	const struct option long_options[] = {
		{"help",	0,	NULL,	'h'},
		{"create",	0,	NULL,	'c'},
		{"save",	0,	NULL,	's'},
		{NULL,		0,	NULL,	0}
	};
	
	do {
		next_option = getopt_long (argc, argv, short_options, 
			long_options, NULL);
		switch (next_option) {
		case 'h':
			print_usage();
			return -1;
		case 'c':
			create_fname = TRUE;
			break;
		case 's':
			save_fname = TRUE;
			break;
		case -1:
			break;
		case '?':
		default:
			print_usage();
			return -1;
		}
	} while (next_option != -1);
	
	if (optind < argc - 1) {
		printf("grubconf: too many command line options\n");
		print_usage();
		return -1;
	}
	
	gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
			    argc, argv,
			    GNOME_PARAM_APP_DATADIR, PACKAGE_DATA_DIR, NULL);

	
	/* test for rootness */
	if (getgid() != 0) {
		gtk_message (GTK_MESSAGE_ERROR, _("You must be root to run this program"));
		return -1;
	}

	if (optind == argc - 1) {
		grubconf_fname = malloc (strlen (argv[optind]) + 1);
		strcpy (grubconf_fname, argv[optind]);
	}
	
	// deal with Gconf here i guess ...
	GConfClient *client = gconf_client_get_default ();

	if (!client) {
		printf ("no client\n");
		return -1;
	}

	if (!grubconf_fname)  {
		grubconf_fname = gconf_client_get_string (client, GCONF_FNAME_KEY, NULL);
		if (!grubconf_fname) {
			grubconf_fname = "/boot/grub/menu.lst";
			gconf_client_set_string (client, GCONF_FNAME_KEY, grubconf_fname, NULL);
		}
	} else if (grubconf_fname[0] != '/') {
		// make sure filename is absolute
		char *cwd = malloc (100);
		getcwd(cwd, 100);
		strcat (cwd, "/");
		strcat (cwd, grubconf_fname);
		grubconf_fname = cwd;
	}

	if (create_fname) {
		// attempt to create a file
		int fd = open (grubconf_fname, O_WRONLY | O_CREAT);
		if (fd > 0)
			close (fd);
	}
	
	init_global ();
	
	if (!grubconf_fname) {
		gtk_message (GTK_MESSAGE_ERROR, _(
				(save_fname ? "Could not find configuration file, "
						"not using --save command line option" :
						"Could not find configuration file")));
		return -1;
	}
	
	if (save_fname) {
		gconf_client_set_string (client, GCONF_FNAME_KEY, grubconf_fname, NULL);
	}

	create_winMain ();
	gtk_widget_show (winMain);
	create_diaOS ();
	create_diaMore ();
	create_diaHelp ();	
	create_diaKernel();

	if (!load_from_file ()) // must be done after GUI is initialized
		return -1;

	gtk_main ();

	return (0);
}
