//*****************************************************************************
//                              CmdGnuCapDC.cpp                               *
//                             -----------------                              *
//  Started     : 29/01/2008                                                  *
//  Last Update : 22/02/2008                                                  *
//  Copyright   : (C) 2008 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "gnucap/commands/CmdGnuCapDC.hpp"

//*****************************************************************************
// Constructor.

CmdGnuCapDC::CmdGnuCapDC( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdGnuCapDC::~CmdGnuCapDC( )
{
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .DC Vin 0.00 100.00m BY 10.00m TE 27.00 BASIC
//        .DC Vin 0.00 100.00m TI 1.01 TE 27.00 BASIC
//        .DC Vin 0.00 100.00m DE 100 TE 27.00 BASIC
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapDC::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1;
  double             df1;
  long               li1;
  int                i1;

  // Clear the object attributes
  os1 = *this;
  bClear( );
  *this = os1;

  // Tokenize the command string
  ostk1.SetString( *this );
  i1 = ostk1.CountTokens( );
  if( i1!=6 && i1!=9 )                                 return( FALSE );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 3 ).Upper( );
  if( os1 != wxT(".DC") )                              return( FALSE );
  m_osName = wxT("DC");

  m_bIsOk = TRUE;

  // Extract the sweep source label
  m_osSource = ostk1.GetNextToken( );
  if( m_osSource.IsEmpty( ) ) m_bIsOk = FALSE;

  // Extract the start value
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStart = (float) df1;
  else                                                 m_bIsOk = FALSE;

  // Extract the stop voltage
  os1 = ostk1.GetNextToken( );
  if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStop = (float) df1;
  else                                                 m_bIsOk = FALSE;

  // Extract the sweep type: linear, log or steps per decade
  os1 = ostk1.GetNextToken( );
  os1.MakeUpper( );
  if(      os1 == wxT("BY") ) m_eScale = eSCALE_LIN;
  else if( os1 == wxT("TI") ) m_eScale = eSCALE_LOG;
  else if( os1 == wxT("DE") ) m_eScale = eSCALE_DEC;
  else                                                 m_bIsOk = FALSE;

  // Extract the step size/count
  os1 = ostk1.GetNextToken( );
  if( m_eScale <= eSCALE_LOG )
  {
    if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fStep = (float) df1;
    else                                             m_bIsOk = FALSE;
  }
  else
  {
    if( ConvertType::bStrToInt( os1, &li1 ) ) m_fStep = (float) li1;
    else                                             m_bIsOk = FALSE;
  }

  // Extract the analysis temperature
  os1 = ostk1.GetNextToken( );
  if( os1.Upper( ) == wxT("TE") )
  {
    os1 = ostk1.GetNextToken( );
    if( ConvertType::bStrToFlt( os1, &df1 ) ) m_fTempC = (float) df1;
    else                                               m_bIsOk = FALSE;
  }

  // Check that the last field is "BASIC"
  if( ostk1.GetNextToken( ).Upper( ) != wxT("BASIC") ) m_bIsOk = FALSE;

  return( m_bIsOk );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapDC::bFormat( void )
{
  wxString  osCmd, os1;

  m_bIsOk = FALSE;

  // Set the command name
  osCmd = wxT(".DC");
  if( m_osName.IsEmpty( ) ) m_osName = wxT("DC");

  // Set signal source
  osCmd << wxT(' ') << m_osSource;

  // Set sweep parameters
  if( ! ConvertType::bFltToStrEng( (double) m_fStart, os1 ) )   return( FALSE );
  osCmd << wxT(' ') << os1;
  if( ! ConvertType::bFltToStrEng( (double) m_fStop,  os1 ) )   return( FALSE );
  osCmd << wxT(' ') << os1;
  switch( m_eScale )
  {
    case eSCALE_LIN : osCmd << wxT(" BY"); break;
    case eSCALE_LOG : osCmd << wxT(" TI"); break;
    case eSCALE_DEC : osCmd << wxT(" DE"); break;
    default         :                                           return( FALSE );
  }
  if( m_eScale < eSCALE_DEC )
  {
    if( ! ConvertType::bFltToStrEng( (double) m_fStep,  os1 ) ) return( FALSE );
    osCmd << wxT(' ') << os1;
  }
  else osCmd << wxT(' ') << (int) lroundf( m_fStep );

  // Set analysis temperature
  if( ! ConvertType::bFltToStrEng( (double) m_fTempC, os1 ) )   return( FALSE );
  osCmd << wxT(" TE ") << os1;

  // Append format modifier
  osCmd << wxT(" BASIC");

  *this = osCmd;

  m_bIsOk = TRUE;

  return( m_bIsOk );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdGnuCapDC::bClear( void )
{
  CmdBase::bClear( );

  m_fStart = -FLT_MAX;
  m_fStop  = -FLT_MAX;
  m_fStep  = -FLT_MAX;

  m_osSource.Empty( );

  m_eScale = eSCALE_NONE;

  m_fTempC = -FLT_MAX;

  return( TRUE );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   psPrefix - A prefix to every line displayed (usually just spaces)

void  CmdGnuCapDC::Print( const char * psPrefix )
{
  CmdBase::Print( psPrefix );

  cout << psPrefix << "m_fStart   : " << m_fStart << '\n';
  cout << psPrefix << "m_fStop    : " << m_fStop << '\n';
  cout << psPrefix << "m_fStep    : " << m_fStep << '\n';

  cout << psPrefix << "m_osSource : " << m_osSource.mb_str( ) << '\n';

  cout << psPrefix << "m_eScale   : ";
  switch( m_eScale )
  {
    case eSCALE_LIN  : cout << "eSCALE_LIN\n";  break;
    case eSCALE_LOG  : cout << "eSCALE_LOG\n";  break;
    case eSCALE_DEC  : cout << "eSCALE_DEC\n";  break;
    case eSCALE_OCT  : cout << "eSCALE_OCT\n";  break;
    case eSCALE_NONE : cout << "eSCALE_NONE\n"; break;
  }

  cout << psPrefix << "m_fTempC   : " << m_fTempC << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdGnuCapDC Test Utility"
       << "\n    Version 1.01 (22/02/2008)\n";

  // Create a GNU-CAP DC command object
  CmdGnuCapDC  tCmd_DC;

  // Use the following command example to check the formatter and the parser :
  osCmd = wxT(".DC Vin 0.00 100.00m TI 10.00m TE 27.00 BASIC");

  // Set things up for a formatter test
  tCmd_DC.bClear( );
  tCmd_DC.m_fStart   = 0.0;
  tCmd_DC.m_fStop    = 100.0E-3;
  tCmd_DC.m_fStep    = 10.0E-3;
  tCmd_DC.m_osSource = wxT("Vin");
  tCmd_DC.m_eScale   = eSCALE_LOG;
  tCmd_DC.m_fTempC   = 27.0;
  cout << "\nRun Formatter    : " << ( tCmd_DC.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format  : " << ( tCmd_DC == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\ntCmd_DC Contents : " << tCmd_DC.mb_str( ) << '\n';

  // Set things up for a parser test
  tCmd_DC.bClear( );
  tCmd_DC = osCmd;
  cout << "\nRun Parser       : " << ( tCmd_DC.bParse( ) ? "OK" : "FAULT" );
  tCmd_DC.bFormat( );
  cout << "\nTest Cmd Format  : " << ( tCmd_DC == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command  : " << osCmd  .mb_str( );
  cout << "\ntCmd_DC Contents : " << tCmd_DC.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
