/**
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gnome.h>
#include <glade/glade.h>

#include "gstcontrol.h"

struct _GstControlPrivate {
	GladeXML            *xml;
	GladeXML            *volume_xml;
	GObject             *adjustment_volume;
	GObject             *adjustment_seek;
	gboolean             seek_in_progress;
	GstMediaPlayMode     display_mode;

	gint64 time_nanos;
	gint64 length_nanos;
	gchar *time_str;
};

enum {
	TOGGLE_PLAY,
	TOGGLE_PLAYLIST,
	TOGGLE_INFO,
	NEXT,
	PREVIOUS,
	STOP,
	FULLSCREEN,
	VOLUME_CHANGE,
	SEEK_CHANGE,
	DISPLAY_MODE_CHANGE,
	LAST_SIGNAL
};

static gint gst_control_signals [LAST_SIGNAL] = { 0 };

static void  gst_control_destroy   (GtkObject  *object);
static void  gst_control_finalize  (GObject    *object); 
static gint  gst_control_expose    (GtkWidget  *widget, GdkEventExpose *event);
static void  gst_control_event_check (GtkWidget* widget, GstControl *control);
static void  gst_control_hide_volume_popup(GstControl *control);
static int   gst_control_popup_keypress (GtkWidget *widget, GdkEventKey *event, 
		                         GstControl *control);
static int   gst_control_popup_button_release (GtkWidget *widget, GdkEventButton *event, 
		                               GstControl *control);
static gint  gst_control_text_width (GtkWidget *widget, const gchar *text);

GNOME_CLASS_BOILERPLATE (GstControl, gst_control, 
			 GtkVBox, GTK_TYPE_VBOX)



static void
volume_changed (GtkAdjustment *adjustment, GstControl *control)
{
	gfloat volume;
	GstControlPrivate  *priv;

	g_return_if_fail(GST_IS_CONTROL(control));
	
	priv = control->_priv;
	volume = gtk_adjustment_get_value(GTK_ADJUSTMENT(priv->adjustment_volume));
	g_signal_emit (control, gst_control_signals [VOLUME_CHANGE], 0, volume);
	
	gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_volume_max"));
	gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_volume_medium"));
	gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_volume_min"));
	gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_volume_zero"));

	if (volume > 0.75){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_volume_max"));
	}
	else if (volume <= 0.75 && volume > 0.5){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_volume_medium"));
	}
	else if (volume <= 0.5 && volume > 0.25){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_volume_min"));
	}
	else{
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_volume_zero"));
	}
}

static gboolean
seek_changed (GtkWidget *widget, GdkEventButton *event, GstControl *control)
{
	g_return_if_fail(GST_IS_CONTROL(control));
	g_signal_emit (control, gst_control_signals [SEEK_CHANGE], 0, 
	               gtk_adjustment_get_value(GTK_ADJUSTMENT(control->_priv->adjustment_seek)));
	control->_priv->seek_in_progress = FALSE;
	return FALSE;
}

static gboolean
seek_started (GtkWidget *widget, GdkEventButton *event, GstControl *control)
{
	g_return_val_if_fail (GST_IS_CONTROL (control), FALSE);
	control->_priv->seek_in_progress = TRUE;
	return FALSE;
}

static void
gst_control_class_init (GstControlClass *class)
{
	GtkWidgetClass  *widget_class = (GtkWidgetClass *) class;
	GtkObjectClass  *object_class = (GtkObjectClass *) class;
	GObjectClass    *gobject_class = (GObjectClass *) class;

	object_class = (GtkObjectClass *) class;

	object_class->destroy = gst_control_destroy;

	gobject_class->finalize = gst_control_finalize;

	widget_class->expose_event = gst_control_expose;

	gst_control_signals [TOGGLE_PLAY] =
		g_signal_new ("toggle_play",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, toggle_play),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [TOGGLE_PLAYLIST] =
		g_signal_new ("toggle_playlist",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, toggle_playlist),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [TOGGLE_INFO] =
		g_signal_new ("toggle_info",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, toggle_info),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [NEXT] =
		g_signal_new ("clicked_next",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, next),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [PREVIOUS] =
		g_signal_new ("clicked_previous",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, previous),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [STOP] =
		g_signal_new ("stop",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, stop),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [FULLSCREEN] =
		g_signal_new ("fullscreen_toggled",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, next),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gst_control_signals [VOLUME_CHANGE] =
		g_signal_new ("volume_change",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, volume_change),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__DOUBLE,
			      G_TYPE_NONE, 1,
			      G_TYPE_DOUBLE);

	gst_control_signals [SEEK_CHANGE] =
		g_signal_new ("seek_change",
			      G_TYPE_FROM_CLASS (gobject_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GstControlClass, seek_change),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__DOUBLE,
			      G_TYPE_NONE, 1,
			      G_TYPE_DOUBLE);

	class->toggle_play = NULL;
	class->volume_change = NULL;
}


static void
gst_control_instance_init (GstControl *control)
{
	GstControlPrivate  *priv;

	control->_priv = g_new0 (GstControlPrivate, 1);
	priv = control->_priv;

	/* seek slider */
	priv->adjustment_seek = G_OBJECT (gtk_adjustment_new (0.0, 0.0, 1.0, 0.1, 0.1, 0.01));
	priv->seek_in_progress = FALSE;

	/* volume slider */
	priv->adjustment_volume = G_OBJECT (gtk_adjustment_new (0.9, 0.0, 1.0, 0.1, 0.10, 0.10));
}

GtkWidget *
gst_control_new (GladeXML *xml, GladeXML *volume_xml)
{
	GstControl *control;
	GstControlPrivate  *priv;
	GtkWidget *widget;
	gint width;

	control = g_object_new (GST_TYPE_CONTROL, NULL);

	priv = control->_priv;

	priv->xml = xml;
	priv->volume_xml = volume_xml;
	glade_xml_signal_connect_data(priv->xml, "clicked_handler", G_CALLBACK (gst_control_event_check), control);

	gtk_box_pack_start (GTK_BOX (control), glade_xml_get_widget(priv->xml, "hbox_controller"), TRUE, TRUE, 0);

	/* seek slider */
	gtk_range_set_adjustment(GTK_RANGE(glade_xml_get_widget(priv->xml, "hscale_seek")), 
	                         GTK_ADJUSTMENT(priv->adjustment_seek));
	glade_xml_signal_connect_data(priv->xml, "seek_started_handler", G_CALLBACK (seek_started), control);
	glade_xml_signal_connect_data(priv->xml, "seek_changed_handler", G_CALLBACK (seek_changed), control);

	/* volume slider */
	g_return_val_if_fail(priv->volume_xml != NULL, NULL);
	widget = GTK_WIDGET(glade_xml_get_widget(priv->volume_xml, "vscale_volume"));
	gtk_range_set_adjustment(GTK_RANGE(widget), GTK_ADJUSTMENT(priv->adjustment_volume));
	glade_xml_signal_connect_data(priv->volume_xml, "volume_changed_handler", G_CALLBACK (volume_changed), control);

	widget = glade_xml_get_widget(priv->volume_xml, "window_volume_popup");
	g_signal_connect (G_OBJECT (widget), "key-press-event",
			  G_CALLBACK (gst_control_popup_keypress), control);
	widget = glade_xml_get_widget(priv->volume_xml, "vscale_volume");
	g_signal_connect (G_OBJECT (widget), "button-release-event",
			  G_CALLBACK (gst_control_popup_button_release), control);

	/* time display */
	widget = glade_xml_get_widget(priv->xml, "label_time");
	width = gst_control_text_width(widget, "00:00 / 00:00");

	gst_control_set_display_mode(control, GST_MEDIA_PLAY_NORMAL);
	return GTK_WIDGET (control);
}

static void
gst_control_destroy (GtkObject *object)
{
	GstControl *control;

	g_return_if_fail (GST_IS_CONTROL (object));

	control = GST_CONTROL (object);

	GNOME_CALL_PARENT (GTK_OBJECT_CLASS, destroy, (object));
}

static void
gst_control_finalize (GObject *object)
{
	GstControl *control;

	g_return_if_fail (GST_IS_CONTROL (object));

	control = GST_CONTROL (object);

	GNOME_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static gint
gst_control_expose (GtkWidget  *widget, GdkEventExpose *event)
{
	GstControl *control;
	g_return_val_if_fail (GST_IS_CONTROL (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	control = GST_CONTROL(widget);

	if (GTK_WIDGET_DRAWABLE (widget)) {
		return GNOME_CALL_PARENT_WITH_DEFAULT
			(GTK_WIDGET_CLASS, expose_event, (widget, event), FALSE);
	}

	return FALSE;
}


void
gst_control_set_display_mode  (GstControl *control, GstMediaPlayMode display_mode)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));

	priv = control->_priv;

	if (display_mode == GST_MEDIA_PLAY_NORMAL){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_fullscreen"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_unfullscreen"));
	}
	else if (display_mode == GST_MEDIA_PLAY_FULLSCREEN){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_unfullscreen"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_fullscreen"));
	}
	else {
	}

	priv->display_mode = display_mode;
}

void
gst_control_set_state  (GstControl *control, GstElementState old_state, GstElementState new_state)
{
	GstControlPrivate  *priv;
	GtkButton *play_button;
	g_return_if_fail (GST_IS_CONTROL (control));

	priv = control->_priv;
	play_button = GTK_BUTTON(glade_xml_get_widget(priv->xml, "button_play"));

	if (new_state == GST_STATE_PLAYING){
		gtk_widget_show(glade_xml_get_widget(priv->xml, "image_paused"));
		gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_playing"));
		return;
	}

	gtk_widget_hide(glade_xml_get_widget(priv->xml, "image_paused"));
	gtk_widget_show(glade_xml_get_widget(priv->xml, "image_playing"));
}

GtkAdjustment*
gst_control_get_volume_adjustment (GstControl *control)
{
	g_return_if_fail (GST_IS_CONTROL (control));
	return GTK_ADJUSTMENT(control->_priv->adjustment_volume);
}

void
gst_control_set_mute  (GstControl *control, gboolean mute)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));
	priv = control->_priv;
	if (mute){
	}
	else {
	}
}

void
gst_control_set_enable_seek (GstControl* control, gboolean enable)
{
	GstControlPrivate  *priv;
	g_return_if_fail (GST_IS_CONTROL (control));
	priv = control->_priv;
	gtk_widget_set_sensitive(glade_xml_get_widget(priv->xml, "hscale_seek"), enable);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(priv->adjustment_seek), 0.0);
}

void
gst_control_set_seek_pos (GstControl* control, gdouble seek_pos)
{
	g_return_if_fail (GST_IS_CONTROL (control));

	if (control->_priv->seek_in_progress == FALSE){
		gtk_adjustment_set_value(GTK_ADJUSTMENT(control->_priv->adjustment_seek), seek_pos);
	}
}

gchar*
gst_control_get_time_string(time_t seconds)
{
	struct tm *tm = gmtime (&seconds);
	gchar *time;

	time = g_malloc (256);
	if (seconds > 3600) {
		/* include the hours here */
		if (strftime (time, 256, _("%H:%M:%S"), tm) <= 0)
			strcpy (time, _("--:--:--"));
	} else {
		/* just minutes and seconds */
		if (strftime (time, 256, _("%M:%S"), tm) <= 0)
			strcpy (time, _("--:--"));
	}
	
	return time;
}

static void
gst_control_update_time_label(GstControl *control)
{
	gchar *time_str;
	GstControlPrivate *priv = control->_priv;
	gint time_seconds = (gint) (priv->time_nanos / GST_SECOND);
	gchar *time_text = gst_control_get_time_string ((time_t) time_seconds);
	gint length_seconds = (gint) (priv->length_nanos / GST_SECOND);

	if (priv->length_nanos <= 0LL){
		time_str = time_text;
	}
	else {
	    gchar *length_text = gst_control_get_time_string ((time_t) length_seconds);
		time_str = g_strdup_printf(_("%s / %s"), time_text, length_text); 
		g_free (length_text);
		g_free (time_text);		
	}

	gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(priv->xml, "label_time")), time_str);

	if (priv->time_str){
		g_free(priv->time_str);
	}
	priv->time_str = time_str;
}

void
gst_control_set_time(GstControl *control, gint64 time_nanos)
{
	g_return_if_fail(GST_IS_CONTROL(control));
	control->_priv->time_nanos = time_nanos;
	gst_control_update_time_label(control);

}

void
gst_control_set_length(GstControl *control, gint64 length_nanos)
{
	g_return_if_fail(GST_IS_CONTROL(control));
	control->_priv->length_nanos = length_nanos;
	gst_control_update_time_label(control);
}

static void
gst_control_show_volume_popup(GstControl *control)
{
	GdkGrabStatus pointer, keyboard;
	gfloat volume;
	GtkWidget *volume_slider, *volume_button;
	gint x, y;
	gint width, height;
	GtkWidget *popup;
	GstControlPrivate *priv;
	GtkAllocation allocation;

	g_return_if_fail(GST_IS_CONTROL(control));
	priv = control->_priv;

	volume = gtk_adjustment_get_value(GTK_ADJUSTMENT(priv->adjustment_volume));
	volume_slider = glade_xml_get_widget(priv->volume_xml, "vscale_volume");
	popup = glade_xml_get_widget(priv->volume_xml, "window_volume_popup");
	volume_button = glade_xml_get_widget(priv->xml, "button_volume");
	g_return_if_fail(GTK_IS_WIDGET(volume_button));

	gdk_window_get_origin (GTK_WIDGET(control)->window, &x, &y);
	allocation = volume_button->allocation;
	
	x += allocation.x + allocation.width;
	y += allocation.y + (allocation.height / 2);

	/* align the slider position with the toggle button to minimise mouse movement */
	y -= (gint)((1.0F - volume) * 100.0F);

	x = x < 0 ? 0 : x;
	y = y < 0 ? 0 : y;

	gtk_window_move (GTK_WINDOW (popup), x, y);
	gtk_widget_show(popup);

	allocation = popup->allocation;

	/* If it goes off the right of the screen, put it on the other side of the button */
	if (x + allocation.width > gdk_screen_width()){
		x -= allocation.width + volume_button->allocation.width;
		gtk_window_move (GTK_WINDOW (popup), x, y);
	}

	/* If it goes off the bottom of the screen, move it up */
	if (y + allocation.height > gdk_screen_height()){
		y = gdk_screen_height() - allocation.height;
		gtk_window_move (GTK_WINDOW (popup), x, y);
	}

	gtk_widget_grab_focus (volume_slider);
	gtk_grab_add (volume_slider);
	pointer = gdk_pointer_grab (volume_slider->window,
	                            TRUE,
	                            (GDK_BUTTON_PRESS_MASK
	                            | GDK_BUTTON_RELEASE_MASK
	                            | GDK_POINTER_MOTION_MASK),
	                            NULL, NULL, GDK_CURRENT_TIME);	

	keyboard = gdk_keyboard_grab (volume_slider->window,
				      TRUE,
				      GDK_CURRENT_TIME);

	if (keyboard != GDK_GRAB_SUCCESS || pointer != GDK_GRAB_SUCCESS) {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(volume_button), FALSE);
	}
}

static void
gst_control_hide_volume_popup(GstControl *control)
{
	GtkWidget *popup;
	GtkWidget *volume_slider;
	GstControlPrivate *priv;

	g_return_if_fail(GST_IS_CONTROL(control));
	priv = control->_priv;

	volume_slider = glade_xml_get_widget(priv->volume_xml, "vscale_volume");
	popup = glade_xml_get_widget(priv->volume_xml, "window_volume_popup");

	gtk_widget_hide(popup);
	gtk_grab_remove (volume_slider);
	gtk_widget_hide(popup);
	gdk_keyboard_ungrab (GDK_CURRENT_TIME);
	gdk_pointer_ungrab (GDK_CURRENT_TIME);
}

static void
gst_control_event_check (GtkWidget* widget, GstControl *control)
{
	GstControlPrivate *priv;

	g_return_if_fail(GST_IS_CONTROL(control));

	priv = control->_priv;

	if (widget == glade_xml_get_widget(priv->xml, "button_play")){
		g_signal_emit (control, gst_control_signals [TOGGLE_PLAY], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_stop")){
		g_signal_emit (control, gst_control_signals [STOP], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_playlist")){
		g_signal_emit (control, gst_control_signals [TOGGLE_PLAYLIST], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_media_info")){
		g_signal_emit (control, gst_control_signals [TOGGLE_INFO], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_next")){
		g_signal_emit (control, gst_control_signals [NEXT], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_previous")){
		g_signal_emit (control, gst_control_signals [PREVIOUS], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_fullscreen")){
		g_signal_emit (control, gst_control_signals [FULLSCREEN], 0);
		return;
	}
	if (widget == glade_xml_get_widget(priv->xml, "button_volume")){
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))){
			gst_control_show_volume_popup(control);
		}
		else{
			gst_control_hide_volume_popup(control);
		}
		return;
	}
}

static int
gst_control_popup_keypress (GtkWidget *widget, GdkEventKey *event, 
		                 GstControl *control)
{
	GtkWidget *volume_button;
	GstControlPrivate *priv;
  
	g_return_val_if_fail (GST_IS_CONTROL (control), 0);
	priv = control->_priv;

	volume_button = glade_xml_get_widget(priv->xml, "button_volume");

	/* any unhandled key event will make the volume popup close */
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(volume_button), FALSE);
	return TRUE;
}

static int
gst_control_popup_button_release (GtkWidget *widget, GdkEventButton *event, 
		                               GstControl *control)
{
	GtkWidget *volume_button;
	GstControlPrivate *priv;
	g_return_val_if_fail (GST_IS_CONTROL (control), 0);

	priv = control->_priv;

	if (event->button == 1) {

		volume_button = glade_xml_get_widget(priv->xml, "button_volume");

		/* any unhandled key event will make the volume popup close */
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(volume_button), FALSE);
	}
	return FALSE;
}

static gint
gst_control_text_width (GtkWidget *widget, const gchar *text)
{
	PangoLayout *layout;
	gint width, height;

	layout = gtk_widget_create_pango_layout(widget, text);
	pango_layout_get_pixel_size(layout, &width, &height);

	g_object_unref(G_OBJECT(layout));

	return width;
}
