/* GStreamer
 * Copyright (C) 1999,2000,2001,2002 Erik Walthinsen <<omega@temple-baptist.com>
 *                    2000,2001,2002 Wim Taymans <wtay@chello.be>
 *                              2002 Steve Baker <steve@stevebaker.org>
 *                              2003 Julien Moutte <julien@moutte.net>
 *
 * gstmediaplay.c: Media Player widget for Gst-Player
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <locale.h>
#include <config.h>
#include <gnome.h>
#include <glade/glade.h>
#include <gst/player/gstmediaplay.h>
#include <gst/player/gtk-playlist.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gdk/gdkx.h>
#include <gconf/gconf-client.h>

#include "egg-recent-model.h"
#include "egg-recent-view.h"
#include "egg-recent-view-gtk.h"

struct _GstPlayer
{
  GladeXML *xml;
  GtkWidget *win;
  GtkWidget *menubar;
  GtkWidget *mplay_vbox;

  GstMediaPlay *mplay;

  GConfClient *gc;

  char *last_location;

  gboolean embedded_mode;
  gboolean fullscreen_mode;
  
  /* recent file stuff */
  EggRecentModel *recent_model;
  EggRecentViewGtk *recent_view;
};

typedef struct _GstPlayer GstPlayer;

static int xid = 0;
static int width = 0;
static int height = 0;

struct poptOption options[] = {
  {"xid", 'x', POPT_ARG_INT, &xid, 0,
   N_("Xid of the window we will use for embedding"),
   N_("XID")},
  {"width", 'w', POPT_ARG_INT, &width, 0,
   N_("Specify the width available in embedding window"),
   N_("WIDTH")},
  {"height", 'h', POPT_ARG_INT, &height, 0,
   N_("Specify the height available in embedding window"),
   N_("HEIGHT")},
  {NULL, '\0', POPT_ARG_INCLUDE_TABLE, NULL, 0, "GStreamer", NULL},
  {NULL, '\0', 0, NULL, 0, NULL, NULL}
};

void
on_play_pause_activate (GtkWidget * widget, GstPlayer * player)
{
  GstElementState playstate;

  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));

  if (!gtk_playlist_get_current_mrl (GTK_PLAYLIST (player->mplay->playlist)))
    return;

  playstate = gst_media_play_get_state (player->mplay);

  if (playstate == GST_STATE_PLAYING)
    {
      gst_media_play_set_state (player->mplay, GST_STATE_PAUSED);
    }
  else
    {
      gst_media_play_set_state (player->mplay, GST_STATE_PLAYING);
    }
}

static void
on_error (GstMediaPlay * mplay, char *msg, GstPlayer * player)
{
  GtkWidget *parent, *error_dialog;

  if (player == NULL)
    parent = NULL;
  else
    parent = player->win;

  error_dialog =
    gtk_message_dialog_new (GTK_WINDOW (player->win),
			    GTK_DIALOG_MODAL,
			    GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "%s", msg);
  gtk_dialog_set_default_response (GTK_DIALOG (error_dialog),
				   GTK_RESPONSE_OK);
  g_signal_connect (G_OBJECT (error_dialog), "destroy", G_CALLBACK
		    (gtk_widget_destroy), error_dialog);
  g_signal_connect (G_OBJECT (error_dialog), "response", G_CALLBACK
		    (gtk_widget_destroy), error_dialog);
  gtk_window_set_modal (GTK_WINDOW (error_dialog), TRUE);

  gtk_widget_show (error_dialog);
}

static char *
create_full_path (const char *path)
{
  char *retval, *curdir, *curdir_withslash, *escaped;

  g_return_val_if_fail (path != NULL, NULL);

  if (strstr (path, "://") != NULL)
    return g_strdup (path);

  if (path[0] == '/')
    {
      escaped = gnome_vfs_escape_path_string (path);

      retval = g_strdup_printf ("file://%s", escaped);
      g_free (escaped);
      return retval;
    }

  curdir = g_get_current_dir ();
  curdir_withslash = g_strdup_printf ("file:///%s%s",
				      curdir, G_DIR_SEPARATOR_S);
  g_free (curdir);

  escaped = gnome_vfs_escape_path_string (path);
  retval = gnome_vfs_uri_make_full_from_relative (curdir_withslash, escaped);
  g_free (curdir_withslash);
  g_free (escaped);

  return retval;
}

static void
add_to_recent (GstPlayer * player, char *incoming_file)
{
  EggRecentItem *item;
  char *local;
  char *filename = NULL;

  filename = create_full_path (incoming_file);
  if (!filename)
    return;

  if (strstr (filename, "file:///") == NULL)
    return;

  //FIXME another bug in egg, it seems
  //to fuck up file:/// uri into file:////
  local = g_filename_from_uri (filename, NULL, NULL);
  if (!local)
    return;

  /* FIXME egg recent seems to unescape our
     pure and clean uri
     playing something with a # in the name
     won't get saved properly */
  item = egg_recent_item_new_from_uri (local);
  if (local)
    g_free (local);
  egg_recent_item_add_group (item, "Gst-Player");
  egg_recent_model_add_full (player->recent_model, item);

  if (filename)
    g_free (filename);
}

static void
current_location (GstMediaPlay * mplay, gchar * location,
		  gchar * location_short, GstPlayer * player)
{
  GtkRequisition requisition;
  gtk_window_set_title (GTK_WINDOW (player->win), location_short);
  add_to_recent (player, location);
  if (!player->embedded_mode)
    {
      gtk_widget_size_request (player->win, &requisition);
      gtk_window_resize (GTK_WINDOW (player->win), requisition.width,
			 requisition.height);
    }
}

void
on_open_activate (GtkWidget * widget, GstPlayer * player)
{
  GtkWidget *fs = NULL;
  gint response;

  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));

  fs = gtk_file_selection_new (_("Select files"));
  gtk_file_selection_set_select_multiple (GTK_FILE_SELECTION (fs), TRUE);
  if (player->last_location != NULL)
    {
      gtk_file_selection_set_filename (GTK_FILE_SELECTION (fs),
				       player->last_location);
      g_free (player->last_location);
      player->last_location = NULL;
    }

  response = gtk_dialog_run (GTK_DIALOG (fs));
  gtk_widget_hide (fs);

  if (response == GTK_RESPONSE_OK)
    {
      gchar **filenames;

      filenames = gtk_file_selection_get_selections (GTK_FILE_SELECTION (fs));

      if (filenames[0])
	{
	  char *tmp;
	  gint i;

	  gst_media_play_ignore_playlist_changes (player->mplay, TRUE);

	  gtk_playlist_clear (GTK_PLAYLIST (player->mplay->playlist));

	  for (i = 0; filenames[i] != NULL; i++)
	    gtk_playlist_add_mrl (GTK_PLAYLIST (player->mplay->playlist),
				  filenames[i], NULL);

	  tmp = g_path_get_dirname (filenames[0]);
	  player->last_location = g_strconcat (tmp, G_DIR_SEPARATOR_S, NULL);
	  g_free (tmp);
	}
      g_strfreev (filenames);

      gst_media_play_ignore_playlist_changes (player->mplay, FALSE);

      gst_media_play_set_location (player->mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (player->
								  mplay->
								  playlist)));

      on_play_pause_activate (widget, player);
    }

  gtk_widget_destroy (fs);
}

void
on_open_location_activate (GtkWidget * widget, GstPlayer * player)
{
  GtkWidget *dialog, *textbox;
  gint response;

  dialog = gnome_dialog_new (_("Open Location"),
			     GNOME_STOCK_BUTTON_CANCEL,
			     _("Enqueue"), GNOME_STOCK_BUTTON_OK, NULL);

  gnome_dialog_set_close (GNOME_DIALOG (dialog), TRUE);
  gnome_dialog_close_hides (GNOME_DIALOG (dialog), TRUE);
  gnome_dialog_set_default (GNOME_DIALOG (dialog), 2);
  gnome_dialog_set_parent (GNOME_DIALOG (dialog),
			   GTK_WINDOW (gtk_widget_get_toplevel
				       (player->win)));

  textbox = gtk_entry_new ();
  if (player->last_location)
    {
      g_message ("last_location is %s", player->last_location);
      gtk_entry_set_text (GTK_ENTRY (textbox), player->last_location);
      gtk_editable_select_region (GTK_EDITABLE (textbox), 0, -1);
    }
  gtk_entry_set_width_chars (GTK_ENTRY (textbox), 80);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
		      textbox, TRUE, TRUE, GNOME_PAD);
  gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
				GTK_EDITABLE (textbox));

  gtk_widget_show_all (dialog);

  response = gnome_dialog_run (GNOME_DIALOG (dialog));

  switch (response)
    {
    case 2:
      gtk_playlist_clear (GTK_PLAYLIST (player->mplay->playlist));
    case 1:
      gtk_playlist_add_mrl (GTK_PLAYLIST (player->mplay->playlist),
			    gtk_entry_get_text (GTK_ENTRY (textbox)), NULL);
      gst_media_play_set_location (player->mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (player->
								  mplay->
								  playlist)));
      if (player->last_location)
	g_free (player->last_location);
      player->last_location =
	g_strdup (gtk_entry_get_text (GTK_ENTRY (textbox)));
      break;
    default:
      break;
    }

  gtk_widget_destroy (dialog);
}

void
on_open_audio_cd_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_location (player->mplay, "cda://audiocd");
  gst_media_play_toggle_play_pause (widget, player->mplay);
}

void
on_open_webcam_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_location (player->mplay, "v4l://webcam");
  gst_media_play_toggle_play_pause (widget, player->mplay);
}

void
on_quit_activate (GtkWidget * widget, GstPlayer * player)
{
  gtk_main_quit ();
}

void
on_properties_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_show_media_info (widget, player->mplay);
}

void
on_preferences_activate (GtkWidget * widget, GstPlayer * player)
{

}

void
on_fullscreen_activate (GtkWidget * widget, GstPlayer * player)
{
  GstMediaPlayMode display_mode;
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));

  display_mode = gst_media_play_get_display_mode (player->mplay);
  if (display_mode == GST_MEDIA_PLAY_FULLSCREEN)
    {
      gtk_window_unfullscreen (GTK_WINDOW (player->win));
    }
  else
    {
      gtk_window_fullscreen (GTK_WINDOW (player->win));
    }
}

void
on_fullscreen_toggled (GstMediaPlay * mplay, GstMediaPlayMode display_mode,
		       GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  on_fullscreen_activate (player->win, player);
}

void
on_scale_to_1_2_activate (GtkWidget * widget, GstPlayer * player)
{
  GtkRequisition requisition;
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_video_scale (player->mplay, 0.5);
  if (!player->embedded_mode)
    {
      gtk_widget_size_request (player->win, &requisition);
      gtk_window_resize (GTK_WINDOW (player->win), requisition.width,
			 requisition.height);
    }
}

void
on_scale_to_1_1_activate (GtkWidget * widget, GstPlayer * player)
{
  GtkRequisition requisition;
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_video_scale (player->mplay, 1.0);
  if (!player->embedded_mode)
    {
      gtk_widget_size_request (player->win, &requisition);
      gtk_window_resize (GTK_WINDOW (player->win), requisition.width,
			 requisition.height);
    }
}

void
on_scale_to_2_1_activate (GtkWidget * widget, GstPlayer * player)
{
  GtkRequisition requisition;
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_video_scale (player->mplay, 2.0);
  if (!player->embedded_mode)
    {
      gtk_widget_size_request (player->win, &requisition);
      gtk_window_resize (GTK_WINDOW (player->win), requisition.width,
			 requisition.height);
    }
}

void
on_best_fit_window_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_video_scale_override (player->mplay, FALSE);
}

void
on_toggle_visualization_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_toggle_visualisation (player->mplay);
}

void
on_show_hide_playlist_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  if (GTK_WIDGET_VISIBLE (player->mplay->playlist))
    gtk_widget_hide (player->mplay->playlist);
  else
    gtk_widget_show (player->mplay->playlist);
}

void
on_next_media_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_next (NULL, player->mplay);
}

void
on_previous_media_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_previous (NULL, player->mplay);
}

void
on_volume_up_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_volume (player->mplay,
			     gst_media_play_get_volume (player->mplay) + 0.1);
}

void
on_volume_down_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_set_volume (player->mplay,
			     gst_media_play_get_volume (player->mplay) - 0.1);
}

void
on_mute_volume_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_toggle_mute (widget, player->mplay);
}

void
on_about_activate (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  gst_media_play_show_about (player->mplay);
}

void
on_visual_toggled (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    {
      gst_media_play_set_visual (player->mplay, gtk_widget_get_name (widget));
    }
}

void
on_effect_toggled (GtkWidget * widget, GstPlayer * player)
{
  g_return_if_fail (GST_IS_MEDIA_PLAY (player->mplay));
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    {
      gboolean res;
      res = gst_media_play_set_effect (player->mplay,
                                       gtk_widget_get_name (widget));
      if (!res)
        {
          on_error (player->mplay,
                    "Can not change effect after the first media has been loaded",
                    player);
        }
    }
}

/* Fullscreen management stuff */
gboolean
cb_configure (GtkWidget * widget, GdkEventConfigure * event,
	      GstPlayer * player)
{
  /* Window is now effectively fullscreen */
  if (player->fullscreen_mode)
    {
      gst_media_play_set_display_mode (player->mplay,
				       GST_MEDIA_PLAY_FULLSCREEN,
				       event->width, event->height);
      gtk_widget_hide (player->menubar);
    }
  return FALSE;
}

gboolean
cb_fullscreen (GtkWidget * widget, GdkEvent * event, GstPlayer * player)
{
  /* Window is now effectively fullscreen */
  if ((event->window_state.new_window_state == GDK_WINDOW_STATE_FULLSCREEN)
      || (event->window_state.new_window_state == 20))
    {
      player->fullscreen_mode = TRUE;
    }
  else
    {
      player->fullscreen_mode = FALSE;
      gst_media_play_set_display_mode (player->mplay,
				       GST_MEDIA_PLAY_NORMAL, 0, 0);
      gtk_widget_show (player->menubar);
    }
  return TRUE;
}

/* Key press events management */
gint
handle_keypress (GtkWidget * widget, GdkEventKey * event, GstPlayer * player)
{
  g_return_val_if_fail (widget != NULL, 0);
  g_return_val_if_fail (GST_IS_MEDIA_PLAY (player->mplay), 0);

  if ((event->keyval == GDK_Escape) || (event->keyval == GDK_f))
    {
      on_fullscreen_activate (widget, player);
      return TRUE;
    }
  else
    return gst_media_play_handle_keypress (widget, event, player->mplay);
}

void
gst_player_connect_func (const gchar * handler_name,
			 GObject * object,
			 const gchar * signal_name,
			 const gchar * signal_data,
			 GObject * connect_object,
			 gboolean after, gpointer user_data)
{
  GModule *module;
  gpointer symbol;

  module = g_module_open (NULL, 0);

  if ((module) && (g_module_symbol (module, handler_name, &symbol)))
    {
      g_signal_connect (G_OBJECT (object), signal_name,
			G_CALLBACK (symbol), user_data);
    }
}

static void
on_recent_file_activate (EggRecentViewGtk * view, EggRecentItem * item,
			 GstPlayer * player)
{
  gchar *uri;
  gchar *filename;
  gboolean toggle_play;

  uri = egg_recent_item_get_uri (item);

  filename = gnome_vfs_get_local_path_from_uri (uri);
  if (filename == NULL)
    {
      g_free (uri);
      return;
    }

  if (gtk_playlist_get_current_mrl (GTK_PLAYLIST (player->mplay->playlist)) ==
      NULL)
    toggle_play = TRUE;
  else
    toggle_play = FALSE;

  gtk_playlist_add_mrl (GTK_PLAYLIST (player->mplay->playlist), filename,
			NULL);
  egg_recent_model_add_full (player->recent_model, item);

  if (toggle_play)
    gst_media_play_toggle_play_pause (player->win, player->mplay);

  g_free (uri);
  g_free (filename);
}

static void
setup_recent (GstPlayer * player)
{
  GtkWidget *menu_item;
  GtkWidget *menu;

  menu_item = glade_xml_get_widget (player->xml, "file_menu");
  menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (menu_item));
  menu_item = glade_xml_get_widget (player->xml, "recent_separator");

  g_return_if_fail (menu != NULL);
  g_return_if_fail (menu_item != NULL);

  player->recent_model = egg_recent_model_new (EGG_RECENT_MODEL_SORT_MRU);

  /* it would be better if we just filtered by mime-type, but there
   * doesn't seem to be an easy way to figure out which mime-types we
   * can handle */
  egg_recent_model_set_filter_groups (player->recent_model, "Gst-Player",
				      NULL);

  player->recent_view = egg_recent_view_gtk_new (menu, menu_item);
  egg_recent_view_gtk_show_icons (EGG_RECENT_VIEW_GTK
				  (player->recent_view), FALSE);
  egg_recent_view_set_model (EGG_RECENT_VIEW (player->recent_view),
			     player->recent_model);
  egg_recent_view_gtk_set_trailing_sep (player->recent_view, TRUE);

  g_signal_connect (player->recent_view, "activate",
		    G_CALLBACK (on_recent_file_activate), player);
}

GList *
get_plugins_list_from_class (GstPlayer *player, const char *classname)
{
  GList *pool_registries = NULL, *registries = NULL, *found_plugins_list = NULL;
  GList *plugins = NULL, *features = NULL;

  g_return_val_if_fail (player != NULL, NULL);

  pool_registries = gst_registry_pool_list ();

  registries = pool_registries;

  while (registries)
    {
      GstRegistry *registry = GST_REGISTRY (registries->data);

      plugins = registry->plugins;

      while (plugins)
	{
	  GstPlugin *plugin = GST_PLUGIN (plugins->data);

	  features = gst_plugin_get_feature_list (plugin);

	  while (features)
	    {
	      GstPluginFeature *feature = GST_PLUGIN_FEATURE (features->data);

	      if (GST_IS_ELEMENT_FACTORY (feature))
		{
		  GstElementFactory *factory = GST_ELEMENT_FACTORY (feature);
		  if (g_strrstr (factory->details->klass, classname))
		    {
		      found_plugins_list = g_list_append (found_plugins_list,
							  (char *)
							  g_strdup
							  (GST_OBJECT_NAME
							   (factory)));
		    }
		}
	      features = g_list_next (features);
	    }

	  plugins = g_list_next (plugins);
	}

      registries = g_list_next (registries);
    }

  g_list_free (pool_registries);
  pool_registries = NULL;

  return found_plugins_list;
}

void
setup_visuals_list (GstPlayer * player)
{
  GtkWidget *menu, *menu_item;
  GList *vis_plugins_list, *p, *menu_children;
  GSList *group = NULL;
  int i;
  char *current_plugin_name = NULL;

  menu_item = glade_xml_get_widget (player->xml, "visualization_menu");
  menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (menu_item));
  menu_item = glade_xml_get_widget (player->xml, "visualization_separator");

  g_return_if_fail (menu != NULL);
  g_return_if_fail (menu_item != NULL);

  vis_plugins_list = get_plugins_list_from_class (player, "Visualization");
  current_plugin_name = gconf_client_get_string (player->gc,
						 "/system/gstreamer/default/visualisation",
						 NULL);

  menu_children = GTK_MENU_SHELL (menu)->children;

  p = menu_children;
  i = 1;

  while (p)
    {
      GtkWidget *local_menu_item = (GtkWidget *) p->data;
      if (local_menu_item == menu_item)
	break;
      p = p->next;
      i++;
    }

  while (vis_plugins_list)
    {
      GtkWidget *plugin_menu_item;
      const char *name = vis_plugins_list->data;
      plugin_menu_item = gtk_radio_menu_item_new_with_label (group, name);
      group =
	gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM
				       (plugin_menu_item));
      if ((current_plugin_name)
	  && (g_strcasecmp (current_plugin_name, name) == 0))
        {
	  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
                                          (plugin_menu_item), TRUE);
        }
      else
        {
          gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
                                          (plugin_menu_item), FALSE);
        }
      gtk_widget_set_name (plugin_menu_item, name);
      g_signal_connect (G_OBJECT (plugin_menu_item), "toggled",
			G_CALLBACK (on_visual_toggled), player);
      gtk_menu_shell_insert (GTK_MENU_SHELL (menu), plugin_menu_item, i);
      i++;
      vis_plugins_list = g_list_next (vis_plugins_list);
    }

  if (current_plugin_name)
    g_free (current_plugin_name);
}

void
setup_effects_list (GstPlayer * player)
{
  GtkWidget *menu, *menu_item, *first_menu_item;
  GList *effects_plugins_list, *p, *menu_children;
  GSList *group = NULL;
  int i;

  menu_item = glade_xml_get_widget (player->xml, "effects_menu");
  menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (menu_item));
  menu_item = glade_xml_get_widget (player->xml, "effects_separator");

  g_return_if_fail (menu != NULL);
  g_return_if_fail (menu_item != NULL);

  effects_plugins_list = get_plugins_list_from_class (player, "Filter/Video/Effect");
  
  menu_children = GTK_MENU_SHELL (menu)->children;

  p = menu_children;
  i = 1;

  while (p)
    {
      GtkWidget *local_menu_item = (GtkWidget *) p->data;
      if (local_menu_item == menu_item)
	break;
      p = p->next;
      i++;
    }
    
  first_menu_item = gtk_radio_menu_item_new_with_label (group, "None");
  gtk_widget_set_name (first_menu_item, "None");
  group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (first_menu_item));
  g_signal_connect (G_OBJECT (first_menu_item), "toggled",
                    G_CALLBACK (on_effect_toggled), player);
  gtk_menu_shell_insert (GTK_MENU_SHELL (menu), first_menu_item, i);
  i++;
    
  while (effects_plugins_list)
    {
      GtkWidget *plugin_menu_item;
      const char *name = effects_plugins_list->data;
      plugin_menu_item = gtk_radio_menu_item_new_with_label (group, name);
      group =
	gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM
				       (plugin_menu_item));
      gtk_widget_set_name (plugin_menu_item, name);
      g_signal_connect (G_OBJECT (plugin_menu_item), "toggled",
			G_CALLBACK (on_effect_toggled), player);
      gtk_menu_shell_insert (GTK_MENU_SHELL (menu), plugin_menu_item, i);
      i++;
      effects_plugins_list = g_list_next (effects_plugins_list);
    }
}

int
main (int argc, char **argv)
{
  GstPlayer *player;
  char *filename;
  gboolean play_on_startup = FALSE;
  GnomeProgram *p;
  poptContext context;
  const gchar **argvn;

  options[3].arg = (void *) gst_init_get_popt_table ();

  /* Using opt as default scheduler */
  gst_scheduler_factory_set_default_name ("opt");

  if (!(p = gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
				argc, argv,
				GNOME_PARAM_POPT_TABLE, options,
				GNOME_PARAM_HUMAN_READABLE_NAME,
				_("Gst-Player"), GNOME_PARAM_APP_DATADIR,
				DATADIR, NULL)))
    g_error ("gnome_program_init failed");

  /* i18n initialization */
  bindtextdomain (PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  textdomain (PACKAGE);

  glade_init ();
  gnome_vfs_init ();

  player = g_new0 (GstPlayer, 1);

  player->gc = gconf_client_get_default ();
  player->embedded_mode = FALSE;
  player->fullscreen_mode = FALSE;

  filename = gnome_program_locate_file (NULL,
					GNOME_FILE_DOMAIN_APP_DATADIR,
					"gst-player/ui/gst-player.glade",
					TRUE, NULL);

  player->xml = glade_xml_new (filename, NULL, NULL);

  glade_xml_signal_autoconnect_full (player->xml, gst_player_connect_func,
				     player);

  player->mplay = gst_media_play_new ();
  g_return_val_if_fail (GST_IS_MEDIA_PLAY (player->mplay), -1);

  player->mplay_vbox = glade_xml_get_widget (player->xml, "player_vbox");
  player->win = glade_xml_get_widget (player->xml, "gst-player");
  player->menubar = glade_xml_get_widget (player->xml, "player_menubar");

  if (xid)
    {
      //GTK_WINDOW(player->win)->type = GTK_WINDOW_POPUP;
      gtk_window_set_decorated (GTK_WINDOW (player->win), FALSE);

      gtk_widget_show (player->win);
      gtk_widget_hide (player->menubar);
    }
  else
    {
      setup_recent (player);
      setup_visuals_list (player);
      setup_effects_list (player);
      gtk_widget_show_all (player->win);
    }

  gtk_box_pack_end (GTK_BOX (player->mplay_vbox), GTK_WIDGET (player->mplay),
		    TRUE, TRUE, 0);

  gtk_widget_show (GTK_WIDGET (player->mplay));

  if (xid)
    {
      GdkWindow *new_parent;

      new_parent = gdk_window_foreign_new (xid);
      if (new_parent == NULL)
	{
	  g_warning ("lost parent window\n");
	}
      gdk_window_reparent (player->win->window, new_parent, 0, 0);
    }

  g_signal_connect (G_OBJECT (player->win), "window-state-event",
		    G_CALLBACK (cb_fullscreen), player);
  g_signal_connect (G_OBJECT (player->win), "configure-event",
		    G_CALLBACK (cb_configure), player);
  g_signal_connect (G_OBJECT (player->win), "delete-event",
		    G_CALLBACK (on_quit_activate), player);
  g_signal_connect (G_OBJECT (player->win), "key-press-event",
		    G_CALLBACK (handle_keypress), player);
  g_signal_connect (G_OBJECT (player->mplay), "display_mode_change",
		    G_CALLBACK (on_fullscreen_toggled), player);
  g_signal_connect (G_OBJECT (player->mplay), "current_location",
		    G_CALLBACK (current_location), player);
  g_signal_connect (G_OBJECT (player->mplay), "playback_error",
		    G_CALLBACK (on_error), player);


  g_object_get (p, "popt-context", &context, NULL);
  argvn = poptGetArgs (context);

  if (argvn)
    {
      gint i = 0;
      while (argvn[i] != NULL)
	{
	  gtk_playlist_add_mrl (GTK_PLAYLIST (player->mplay->playlist),
				argvn[i], NULL);
	  i++;
	}
      gst_media_play_set_location (player->mplay,
				   gtk_playlist_get_current_mrl (GTK_PLAYLIST
								 (player->
								  mplay->
								  playlist)));

      /* check if we should play on startup */
      if (!gst_media_gconf_get_boolean ("play-on-startup", &play_on_startup))
	play_on_startup = FALSE;
      if (play_on_startup)
	on_play_pause_activate (player->win, player);
    }

  gtk_main ();
  return 0;
}
