/*
 * strpool.c -- functions for managing a pool of strings
 *
 * Copyright 2001 Carl Worth
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Carl Worth not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  Carl Worth makes no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * CARL WORTH DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL CARL WORTH BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "strpool.h"

strpool_t *strpool_new(void)
{
    strpool_t *ret;
    ret = malloc(sizeof(strpool_t));
    if (ret == NULL) {
	fprintf(stderr, "Out of memory.\n");
	return NULL;
    }

    strpool_init(ret);
    return ret;
}

int strpool_init(strpool_t *pool)
{
    pool->size = 0;
    pool->num_strs = 0;
    pool->pool = NULL;
    return strpool_grow(pool, STRPOOL_CHUNK_SIZE);
}

int strpool_free(strpool_t *pool)
{
    int i;
    for (i=0; i < pool->num_strs; i++) {
	if (pool->pool[i])
	    free(pool->pool[i]);
    }
    pool->num_strs = 0;
    free(pool->pool);
    pool->size = 0;
    free(pool);
    return 0;
}

int strpool_grow(strpool_t *pool, size_t increment)
{
    int i;

    pool->pool = realloc(pool->pool, sizeof(char *) * (pool->size + increment));
    if (pool->pool == NULL) {
	fprintf(stderr, "Out of memory.\n");
	return -1;
    }

    for (i = pool->size; i < pool->size + increment; i++) {
	pool->pool[i] = NULL;
    }
    pool->size += increment;

    return 0;
}

char *strpool_add(strpool_t *pool, char *str)
{
    char *dup_str = strdup(str);

    while (pool->num_strs >= pool->size) {
	int err;
	err = strpool_grow(pool, STRPOOL_CHUNK_SIZE);
	if (err != 0)
	    return NULL;
    }

    pool->pool[pool->num_strs++] = dup_str;
    return dup_str;
}
