/*
 * Module for style(GtkStyle).
 * See "style.h" for the details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include "diff.h"
#include "gui.h"
#include "style.h"


/* rc file */
#define SYS_GTKDIFFRC	"gtkdiff/gtkdiffrc"	/* $datadir/SYS_GTKDIFFRC */
#define USER_GTKDIFFRC	".gtkdiffrc"

/* Private variables */
static gboolean b_found_rcfile;

/* Private function declarations */
static void create_default_styles(void);

/* Default styles */
static GtkStyle *range_style[MAX_NUM_COMPARE_FILES];
static GtkStyle *darea_style;
static GtkStyle *text_style;


/**
 * style_init:
 * At first, search for rc file in the current directory and home directory.
 * If found, set b_found_rcfile=TRUE not to use default styles.
 * If not found, create default styles to use them later.
 **/
void
style_init(void)
{
	char *homedir;
	char *sys_rcpath;
	char path[PATH_MAX];

	b_found_rcfile = TRUE;
	/* Current directory */
	if (access(USER_GTKDIFFRC, R_OK) == 0) {
		gtk_rc_parse(USER_GTKDIFFRC);
		return;
	}

	/* Home directory */
	homedir = getenv("HOME");
	if (homedir) {
		g_snprintf(path, sizeof(path), "%s/%s", homedir, USER_GTKDIFFRC);
		if (access(path, R_OK) == 0) {
			gtk_rc_parse(path);
			return;
		}
	}

	/* System directory */
	sys_rcpath = gnome_datadir_file(SYS_GTKDIFFRC);
	if (sys_rcpath) {
		gtk_rc_parse(sys_rcpath);
		g_free(sys_rcpath);
		return;
	}
	
	/* rc file is not found, so use hard-coded default styles. */
	b_found_rcfile = FALSE;
	create_default_styles();
}

void
style_set_text(GtkWidget *text)
{
	if (b_found_rcfile == TRUE)
		return;/* do nothing */

	gtk_widget_set_style(text, text_style);
}

void
style_set_drawingarea(GtkWidget *darea)
{
	if (b_found_rcfile == TRUE)
		return;/* do nothing */

	gtk_widget_set_style(darea, darea_style);
}

void
style_set_gdiff_range(GtkWidget *range, int n)
{
	if (b_found_rcfile == TRUE)
		return;/* do nothing */

	gtk_widget_set_style(range, range_style[n]);
}


/* ---The followings are private functions--- */
/**
 * create_default_styles:
 * Create default styles with hard-coded values.
 * Called when gtkdiff can't find its rc file.
 **/
static void
create_default_styles(void)
{
	/* XXX performance hit? Don't care. This function is rarely executed. */
#define DEF_FIXED_WIDTH_FONT	"-adobe-courier-medium-r-normal--14-140-75-75-m-90-iso8859-1,-*-*-medium-r-normal--*-*-*-*-m-*-*-*,-*-*-medium-r-normal--*-*-*-*-c-*-*-*"
	GtkStyle *defstyle;
	GdkColormap *cmap;
	GdkColor color_range_fg[MAX_NUM_COMPARE_FILES] = {
		{0, 0xffff, 0x7fff, 0x0000},
		{0, 0x0000, 0xffff, 0xffff},
		{0, 0x7fff, 0xffff, 0x0000}};
	GdkColor color_darea_fg = {0, 0x0000, 0x0000, 0xffff};
	GdkFont *old_font;
	int n;
	
	/* allocate colors */
	cmap = gdk_colormap_get_system();
	for (n = 0; n < MAX_NUM_COMPARE_FILES; n++) {
		gdk_colormap_alloc_color(cmap, &color_range_fg[n], FALSE, TRUE);
	}		
	
	defstyle = gtk_widget_get_default_style();
	for (n = 0; n < MAX_NUM_COMPARE_FILES; n++) {
		range_style[n] = gtk_style_copy(defstyle);
		range_style[n]->fg[GTK_STATE_NORMAL] = color_range_fg[n];
	}		
	darea_style = gtk_style_copy(defstyle);
	text_style = gtk_style_copy(defstyle);
	
    darea_style->fg[GTK_STATE_NORMAL] = color_darea_fg;

	/* font for text widget (prefer fixed width font) */
	old_font = text_style->font;
    text_style->font = gdk_fontset_load(DEF_FIXED_WIDTH_FONT);
	if (text_style->font)
		gdk_font_unref(old_font);
	else
		text_style->font = old_font;
}

