/*
|
|  Copyright 2005 Steve Wahl <steve at pro-ns dot net>
|
|  This file contains routines to create the iTunesSD file, as
|  used by the ipod shuffle.
|
|  Like itunesdb.c, it is derived from the perl script "mktunes.pl"
|  (part of the gnupod-tools collection) written by Adrian
|  Ulrich <pab at blinkenlights.ch>.
|
|  Small(?) portions derived from itunesdb.c, so Jorg Schuler probably
|  has some copyright ownership in this file as well.
|
|  The code contained in this file is free software; you can redistribute
|  it and/or modify it under the terms of the GNU Lesser General Public
|  License as published by the Free Software Foundation; either version
|  2.1 of the License, or (at your option) any later version.
|
|  This file is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
|  Lesser General Public License for more details.
|
|  You should have received a copy of the GNU Lesser General Public
|  License along with this code; if not, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
|
|  iTunes and iPod are trademarks of Apple
|
|  This product is not supported/written/published by Apple!
|
|  $Id: itunesdb.c,v 1.73 2004/12/13 15:57:05 jcsjcs Exp $
*/

/* notes:

  All software currently seems to write iTunesDB as well as iTunesSD
  on the iPod shuffle.  I assume that reading from the iTunesSD file
  is not necessary.  The iTunesStats file is different, but I leave
  that for another day.

  The iTunesSD file format is as follows (taken from WikiPodLinux, feb
  '05):

    Offset Field         Bytes   Value
     (hex)               (dec)

  iTunesSD header (occurs once, at beginning of file):

    00      num_songs     3       number of song entries in the file

    03      unknown       3       always(?) 0x010600
    06      header size   3       size of the header (0x12, 18 bytes)
    09      unknown       3       possibly zero padding

  iTunesSD song entry format (occurs once for each song)

   000      size of entry 3       always(?) 0x00022e (558 bytes)
   003      unk1          3       unknown, always(?) 0x5aa501
   006      starttime     3       Start Time, in 256 ms increments
                                  e.g. 60s = 0xea (234 dec)
   009      unk2          3       unknown (always 0?)
   00C      unk3          3       unknown, some relationship to starttime
   00F      stoptime      3       Stop Time, also in 256 ms increments.
                                  Zero means play to end of file.
   012      unk4          3       Unknown.
   015      unk5          3       Unknown, but associated with stoptime?
   018      volume        3       Volume - ranges from 0x00 (-100%) to 0x64
                                  (0%) to 0xc8 (100%)
   01B      file_type     3       0x01 = MP3, 0x02 = AAC, 0x04=WAV
   01E      unk6          3       unknown (always 0x200?)
   021      filename      522     filename of the song, padded at the end
                                  with 0's.  Note: forward slashes are used
                                  here, not colons like in the iTunesDB --
                                  for example,
                                  "/iPod_Control/Music/F00/Song.mp3"
   22B      shuffleflag   1       If this value is 0x00, the song will be
                                  skipped while the player is in shuffle
                                  mode.  Any other value will allow it to be
                                  played in both normal and shuffle modes.
                                  iTunes 4.7.1 sets this to 0 for audio books.
   22C      bookmarkflag  1       If this flag is 0x00, the song will not be
                                  bookmarkable (i.e. its playback position
                                  won't be saved when switching to a different
                                  song). Any other value wil make it
                                  Bookmarkable.  Unlike hard drive based iPods,
                                  all songs can be marked as bookmarkable,
                                  not just .m4b and .aa
   22D      unknownflag   1       unknown, always? 0x00.

All integers in the iTunesSD file are in BIG endian form...

*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "itunesdb.h"
#include "support.h"
#include "file.h"

#ifdef IS_GTKPOD
/* we're being linked with gtkpod */
#define itunesdb_warning(...) g_print(__VA_ARGS__)
#else
/* The following prints the error messages to the shell, converting
 * UTF8 to the current locale on the fly: */
#define itunesdb_warning(...) do { gchar *utf8=g_strdup_printf (__VA_ARGS__); gchar *loc=g_locale_from_utf8 (utf8, -1, NULL, NULL, NULL); fprintf (stderr, "%s", loc); g_free (loc); g_free (utf8);} while (FALSE)
#endif




/* Write "data", "n" bytes long to current position in file.
   Returns TRUE on success, FALSE otherwise */
static gboolean put_data (FILE *file, gchar *data, gint n)
{
	if (fwrite (data, 1, n, file) != n) return FALSE;
	return TRUE;
}


/* Write 1-byte integer "n" to "file".
   Returns TRUE on success, FALSE otherwise */
static gboolean put8int (FILE *file, guint8 n)
{
	return put_data (file, (gchar *)&n, 1);
}


/* Write 3-byte integer "n" to "file" in big endian order.
   Returns TRUE on success, FALSE otherwise */
static gboolean put24bint (FILE *file, guint32 n)
{
	gchar buf[3] ;
	buf[0] = (n >> 16) & 0xff ;
	buf[1] = (n >> 8)  & 0xff ;
	buf[2] = (n >> 0)  & 0xff ;
	return put_data (file, buf, 3);
}


gboolean itunessd_write_it(FILE *file)
{
	guint32 i;
	gunichar2 namebuf[261], *path16 ;
	gchar *path ;
	glong pathlen ;
	guint32 track_num = it_get_nr_of_tracks() ;
	Track *track ;

	put24bint(file, track_num) ;
	put24bint(file, 0x010600) ;
	put24bint(file, 0x12) ;	/* size of header */
	put24bint(file, 0x0) ;	/* padding? */
	put24bint(file, 0x0) ;
	put24bint(file, 0x0) ;

	for(i = 0 ; i < track_num ; i++)
	{
		if ( (track = it_get_track_by_nr(i)) == 0 )
		{
			g_warning("Invalid track Index!\n") ;
			break ;
		}
		put24bint(file, 0x00022e) ;
		put24bint(file, 0x5aa501) ;
		/* starttime is in 256 ms incr. for shuffle */
		put24bint(file, track->starttime / 256) ;
		put24bint(file, 0) ;
		put24bint(file, 0) ;
		put24bint(file, track->stoptime / 256) ;
		put24bint(file, 0) ;
		put24bint(file, 0) ;
		/* track->volume ranges from -255 to +255 */
		/* we want 0 - 200 */
		put24bint(file, ((track->volume + 255) * 201) / 511) ;

		/* XXX FIXME  The next one should be 0x01 for MP3,
		** 0x02 for AAC, and 0x04 for WAV, but I can't find
		** a suitable indicator within the track structure? */

		put24bint(file, 0x01) ;
		put24bint(file, 0x200) ;
		
		/* shuffle uses forward slash separator, not colon */
		path = g_strdup(track->ipod_path) ;
		itunesdb_convert_filename_ipod2fs(path) ;
		path16 = g_utf8_to_utf16(path, -1, NULL, &pathlen, NULL) ;
		
		g_free(path) ;
		if(path16 == NULL) {
			return FALSE ;
		}

		memset(namebuf, 0, sizeof(namebuf)) ;
		memcpy(namebuf, path16, pathlen * 2) ;

		g_free(path16) ;

		put_data(file, (gchar *)namebuf, 522) ;

		/* XXX FIXME: should depend on something, not hardcoded */
		put8int(file, 0x1) ; /* song eligible in shuffle mode */
		put8int(file, 0) ;
		put8int(file, 0) ;
	}
	return TRUE ;
}	


gboolean itunessd_write_to_file (const gchar *filename)
{
	FILE *file = NULL ;
	gboolean result = FALSE ;

#if ITUNESDB_DEBUG
	fprintf(stderr, "Writing to %s\n", filename) ;
#endif

	if ((file = fopen(filename, "w+")) == NULL)
	{
		itunesdb_warning(_("Could not open iTunesSD \"%s\" for writing.\n"),
				 filename) ;
		return FALSE ;
	}

	result = itunessd_write_it(file) ;
	fclose(file) ;

	return result ;
}

