/* Clipboard handling functionality */

#include <unistd.h>
#include <stdlib.h>
#include <gtk/gtk.h>

/* Uncomment for debugging */
// #define DEBUG

#include "main.h"
#include "clipboard.h"

/* currenthandler becomes the currentcontentprovider when the user
 * chooses to cut/copy something. In this case, we will send the 
 * provider a completion message in case it's content is actually being
 * used. The content string is provided to the completion callback as
 * a parameter */
static clipboard_handlerinfo currentcontentprovider;
static char *currentcontent = NULL;

static clipboard_handlerinfo currenthandler;

/* Auxiliary functions for handler management */

void clipboard_invalidate(clipboard_handlerinfo *handler)
{  
   handler->obtain  	= NULL;
   handler->paste 	= NULL;
   handler->completion 	= NULL;
   handler->data	= NULL;
};

int clipboard_matches(const clipboard_handlerinfo *info1, const clipboard_handlerinfo *info2)
{
   if ((!info1)||(!info2))
     return 0;
#define MATCH(field) if (info1->field!=info2->field) return 0
   MATCH(obtain);
   MATCH(paste);
   MATCH(completion);
   MATCH(data);
#undef MATCH   
   return 1;
};

void clipboard_copyentries(const clipboard_handlerinfo *source, clipboard_handlerinfo *destination)
{
   if ((!source)||(!destination))
     return;
#define COPY(field) destination->field = source->field
   COPY(obtain);
   COPY(paste);
   COPY(completion);
   COPY(data);
#undef COPY
};


void clipboard_setcurrenthandler(const clipboard_handlerinfo *handler)
{
   clipboard_copyentries(handler, &currenthandler);
};

/* Remove current clipboard handler if matching passed handler */
int  clipboard_removeifcurrent(const clipboard_handlerinfo *handler)
{
   int result = clipboard_matches(handler,&currenthandler);
   if (result)
     clipboard_invalidate(&currenthandler);
   if (clipboard_matches(handler,&currentcontentprovider))
     {
	result = 1;
	clipboard_invalidate(&currentcontentprovider);
     };
   return result;
};

/* The functions below can be used to trigger respectively a copy/paste/cut 
 * action using the current clipboard handler */
void clipboard_perform_obtain(ObtainContentMode mode)
{
   const char *newcontent = NULL;
   if (currentcontent)
     {
	/* the current content provider is about to lose focus
	 * so call the completion callback with success set to false */
	if (currentcontentprovider.completion)
	  currentcontentprovider.completion(currentcontentprovider.data,0,currentcontent);
	free(currentcontent);
	currentcontent = NULL;
     };
   clipboard_copyentries(&currenthandler, &currentcontentprovider);
   newcontent = currentcontentprovider.obtain(mode, currentcontentprovider.data);
   if (newcontent)
     {
	currentcontent = strdup(newcontent);
	gtk_selection_owner_set (window,
				 GDK_SELECTION_PRIMARY,
				 GDK_CURRENT_TIME);
     };
};

void clipboard_perform_cut()
{
   clipboard_perform_obtain(CutMode);
};

void clipboard_perform_copy()
{
   clipboard_perform_obtain(CopyMode);
};

void clipboard_selreceived(GtkWidget *w,
			   GtkSelectionData *selection_data,
			   gpointer data)
{
   if ((selection_data->type==GDK_SELECTION_TYPE_STRING)&&
       selection_data->data&&currenthandler.paste)
     {
	char *localcopy = (char*)malloc(selection_data->length+1);
	strncpy(localcopy, (const char*)selection_data->data,selection_data->length);
	currenthandler.paste(currenthandler.data, localcopy);
#ifdef DEBUG   
	printf("received string '%s'\n", localcopy);	
#endif     
	free(localcopy);
     };
};

void clipboard_selrequested(GtkWidget *w,
			    GtkSelectionData *selection_data,
			    guint info,
			    guint time_stamp,
			    gpointer data)
{
   if (currentcontent)
     {
	gtk_selection_data_set(selection_data,GDK_SELECTION_TYPE_STRING,
			       8, currentcontent, strlen(currentcontent)+1);
	if (currentcontentprovider.completion)
	  currentcontentprovider.completion(currentcontentprovider.data,1,currentcontent);
     };
};

void clipboard_perform_paste()
{
   gtk_selection_convert(window, GDK_SELECTION_PRIMARY, 
			 GDK_TARGET_STRING, GDK_CURRENT_TIME);
};

void clipboard_init()
{
   clipboard_invalidate(&currenthandler);
   clipboard_invalidate(&currentcontentprovider);
   
   gtk_signal_connect(GTK_OBJECT(window),"selection_received",
		      GTK_SIGNAL_FUNC(clipboard_selreceived),NULL);
   gtk_selection_add_target(window, GDK_SELECTION_PRIMARY,
			    GDK_SELECTION_TYPE_STRING,1);
   gtk_signal_connect(GTK_OBJECT(window),"selection_get",
		      GTK_SIGNAL_FUNC(clipboard_selrequested),NULL);
};
