/* Some functions to handle lists of files as given back on drag and drop
 * events or popup events (delete,rename,etc) */

#ifndef FILEMAN_H
#define FILEMAN_H

#include <glib.h>

#include "vfs.h"

/* callback functions to make fileman functions generic */
typedef int (*fileman_mkdircall)(char *dirname,char *path,gpointer data);
typedef int (*fileman_addfilecall)(char *filename,char *path,gpointer data);

typedef struct 
{
   fileman_mkdircall mkdir;
   fileman_addfilecall addfile;
   /* data pointer to be passed to those calls */
   gpointer data;
} fileman_fileitembackends;

extern fileman_fileitembackends fileman_createlinks;
extern fileman_fileitembackends fileman_makecopies;

/* default calls used by compatibility macros */
int fileman_defaultmkdir(char *dirname,char *path,gpointer data);
int fileman_defaultaddfile(char *filename,char *path,gpointer data);
int fileman_defaultcopyfile(char *filename,char *path,gpointer data);


/* the following function can be used to parse a selection as given back
 * by drag and drop and call filehandle(char*) for each separate file */
int fileman_selectionhandler(char *s,char *p,
			     int (*filehandle)(char *s,char *p,
					       fileman_mkdircall,
					       fileman_addfilecall,
					       gpointer data
					       ),
			     fileman_mkdircall,
			     fileman_addfilecall,
			     gpointer data
			     );

/* remove the specified single file item. If this is a directory,it will
 * be removed recursively. Parameters from char *p are unused and set to NULL */
int fileman_remove_item(char *s,
			char *p,
			fileman_mkdircall mkdir,
			fileman_addfilecall addfile,
			gpointer data
			);

/* remove a whole batch of files using the function above */
int fileman_remove_items(char *s);

/* add a file using the specified callback functions */
int fileman_addnormalfile_generic(char *f,
				  char *p,
				  fileman_mkdircall,
				  fileman_addfilecall,
				  gpointer data);
/* add a single normal file to the path specified by char *p.
 * this is done by creating a symbolic link to the source file in
 * the destination directory */
#define fileman_addnormalfile(file,path) \
   fileman_addnormalfile_generic(file,path, \
				 fileman_defaultmkdir, \
				 fileman_defaultaddfile, \
				 NULL)


/* add a directory with subdirs using the specified callback functions 
 * if contentof is TRUE,only the content of the directory,not the directory
 * itself is added to the destination */
int fileman_adddirectory_generic(char *sd,
				 char *p,
				 GList **linkcount,
				 int contentof,
				 fileman_mkdircall,
				 fileman_addfilecall,
				 gpointer data
				 );

/* add a single directory to the path specified by char *p
 * all the items of this directory will be added depending on wether theyre
 * directories themselves,in which case adddirectory will simply be called
 * recursively,or wether theyre just normal files,which will then be added
 * using addnormalfile() 
 * linkcount should be a pointer to a GList pointer initialized with NULL */
#define fileman_adddirectory(sd,p,linkcount) \
   fileman_adddirectory_generic(sd, \
				p, \
				linkcount, \
				0, \
				fileman_defaultmkdir, \
				fileman_defaultaddfile, \
				NULL )



/* Status becomes >0 if something went wrong */
typedef void (*fileman_additems_donecallback)(int status,gpointer data);
typedef struct
{
   /* called when done */
   fileman_additems_donecallback cb;
   gpointer data;
   /* item list to be processed */
   GList *items;
   
   /* counts the errors that occurred while processing the list */
   int errors;
   
   /* handlers installed by the caller */
   gpointer generic_userdata;
   GList *ihandlers;
} fileman_additems_state;
typedef void (*fileman_additems_continuecb)(fileman_additems_state*);
typedef void (*fileman_itemcallback)(char *item,
				     gpointer generic_userdata,
				     gpointer user_data,
				     /* this has to be called
				      * to notify the main handler that this
				      * item has been processed successfully */
				     fileman_additems_continuecb,
				     fileman_additems_state*);
/* the following function is a very generic approach towards implementing 
 * a system capable of adding *any* type of dnd content to a destination.
 * The caller to this function provides specialized callback functions
 * for each item to be handled thus keeping this alghorithm very simple */
void fileman_additemlist(GList *items,		      
			 fileman_additems_donecallback,
			 gpointer data,
			 /* user data to be passed to *all* item handlers */
			 gpointer generic_userdata,
			 int num_handlers,
			 ... /* char *type:           e.g. "file:"
			      * fileman_itemcallback: callback processing item
			      * gpointer data         user data passed to callback
			      */
			 );

/* add a single generic file item to the patch specified by char *p   
 * this can be either a normal file or a directory.
 * This function shouldn't be used directly but from an iterator
 * (say fileman_additemlist) */
void fileman_addfile_generic(char *f,     
			     gpointer generic_userdata,    
			     gpointer user_data,
			     fileman_additems_continuecb cb,
			     fileman_additems_state *cbdata);

typedef struct 
{
   GList *encodermapping;
   int dragtype;
} fileman_addstreamglobal;

/* used to copy the contents of a trackstream into a file     
 * within the specified path with the name of the stream as filename
 * This function should only be used from an iterator 
 * (say fileman_additemlist) */
void fileman_addstream(char*s,    
		       gpointer generic_userdata,     
		       gpointer user_data,      
		       fileman_additems_continuecb cb,       
		       fileman_additems_state *cbdata);

/* add a directory to a specified path,ask the user for a name
 * call "finished" when done. */
void fileman_mkdir_dialog(vfs_filesystem *vfs,			  
			  char *path,
			  void (*finished)(gpointer),
			  gpointer data);

/* display a dialog asking for a new name,rename file then */
void fileman_rename_dialog(vfs_filesystem *vfs,
			   char *oldfile,
			   void(*finished)(char *newname,gpointer),
			   gpointer data);  

/* returns TRUE if specified file exists */
int fileman_file_exist(char *filename);   


#endif // FILEMAN_H
