/* this file contains a primitive filesystem abstraction layer.
 * It provides functions to read and manipulate directories.
 * Thus, it isn't a proper filesystem, it's merely an abstraction of the
 * directory management functions of a filesystem.
 * It's purpose is to provide a common interface for both the virtualdir.c
 * constructs and the real filesystem functions as defined in dirlow.c
 * We'll stick to Posix Path rules here */

#ifndef VFS_H
# define VFS_H

# include <gtk/gtk.h>

#include "types.h"
# include "datacopydlg.h"

# define VFS_MAXFILENAMESIZE 255

/* this structure is returned by the filestat function,
 * regardless of the underlying filesystem architecture */
typedef struct
{
   // filename of the object
   char name[VFS_MAXFILENAMESIZE+1];
   // it's filesize
   unsigned long int filesize;
   // is the object a directory ?
   int isdir;
   // is the object a symlink ?
   int islink;
   // only valid when requested with the getfilelinks flag set
   int filelinks;
   // do we have to precache this object ?
   int precache;
   // true if a file is "pure virtual", coming from another session
   int is_reference;
}
vfs_direntry;

/* callbacks of this type can be used to stay informed about ongoing copy
 * processes etc.
 * status is 0 on success */
typedef void (*vfs_operationcomplete)(int status,gpointer data);

/* Type declarations for function pointers */
typedef int (*createdir_t)(gpointer fs,char *path);
typedef gtoaster_handle_t (*opendir_t)(gpointer fs,char *path);
typedef char *(*readdirentry_t)(gpointer fs,gtoaster_handle_t handle);
typedef void (*closedir_t)(gpointer fs,gtoaster_handle_t handle);
typedef vfs_direntry *(*filestat_t)(gpointer fs,const char *path,int getfilelinks);
typedef char *(*getlinkdestination_t)(gpointer fs,char *path);
typedef int  (*symlink_t)(gpointer fs,char *linkdest,char *linkname);
typedef void (*copy_t)(gpointer fs,char *destpath,
		       gpointer sourcefs,char *sourcepath,
		       datacopydlg_dlginfo *progress,
		       vfs_operationcomplete callback, gpointer data,
		       int link_requested,int copy_move
		       );
typedef char *(*createfile_getrealname_t)(gpointer fs,char *path);
typedef char *(*getrealname_t)(gpointer fs,char *path);
typedef int (*openfile_t)(gpointer fs,char *path);
typedef int (*del_t)(gpointer fs,char *path);
typedef int (*ren_t)(gpointer fs,char *path,char *newname);
typedef void (*destroy_t)(gpointer fs);

/* nothing anybody would have to worry about unless he wants to implement
 * his own filesystem ;-) */
typedef struct
{
   createdir_t          	createdir;
   opendir_t      		opendir;
   readdirentry_t 		readdirentry;
   closedir_t     		closedir;
   filestat_t     		filestat;
   getlinkdestination_t 	getlinkdestination;
   symlink_t            	symlink;
   copy_t         		copy;
   createfile_getrealname_t 	createfile_getrealname;
   getrealname_t  		getrealname;
   openfile_t     		openfile;
   del_t          		del;
   ren_t          		ren;
   destroy_t      		destroy;
   /* this pointer references additional informations stored by the
    * different filesystems */
   gpointer 			data;

   /* A list of error strings describing the errors that occurred while
    * using the filesystem so far */
   GList *errorstrings;
   /* Number of errors whose strings haven't been requested yet */
   int unprocessed_errors;
}
vfs_filesystem;

/* create a vfs filesystem structure to be used for consequent accesses */
vfs_filesystem *vfs_create(createdir_t    		createdir,
			   opendir_t      		opendir,
			   readdirentry_t 		readdirentry,
			   closedir_t     		closedir,
			   filestat_t     		filestat,
			   getlinkdestination_t 	getlinkdestination,
			   symlink_t            	symlink,
			   copy_t         		copy,
			   createfile_getrealname_t     createfile_getrealname,
			   getrealname_t  		getrealname,
			   openfile_t     		openfile,
			   del_t          		del,
			   ren_t          		ren,
			   destroy_t      		destroy,
			   gpointer data);

/* High-level filesystem access */


/* Get latest error message, returns NULL if none.
 * String is owned by the filesystem and will be freed together with it */
const char *vfs_getlasterrorstring(vfs_filesystem*);

/* add an error string to the list of errors.
 * This will make a copy of the string and add the copy to the error GList */
void vfs_adderrorstring(vfs_filesystem*,const char *msg);

/* Create a directory, return 0 on success */
int vfs_createdir (vfs_filesystem*,char* newdir);
/* Open a directory, returns directory handle or 0 on failure */
gtoaster_handle_t vfs_opendir (vfs_filesystem*,char* dir);
/* will return the full path of the next directory item or
 * NULL if the end of the directory has been reached.
 * caller frees string */
char *vfs_readdirentry(vfs_filesystem*,gtoaster_handle_t handle);
void vfs_closedir (vfs_filesystem*,gtoaster_handle_t handle);
/* this will retrieve the file statistics of a specific file.
 * caller frees structure.
 * returns NULL on failure */
vfs_direntry *vfs_filestat(vfs_filesystem*,const char *path,int getfilelinks);

/* retrieve the destination of a symbolic link.
 * Returns NULL on failure */
char *vfs_getlinkdestination(vfs_filesystem*,char *path);

/* create a symbolic link within the filesystem.
 * returns 0 on success */
int vfs_symlink(vfs_filesystem*,char*linkdest,char*linkname);

/* How to add files and streams.
 * One word about streams: those will be available through a
 * special "filesystem" managing tracks.
 * So once the special id stream: is received, this filesystem
 * will be used to get access to the data.
 * It's the destination filesystem's driver to determine whether
 * an object added to it has to be retrieved immediately
 * (mkisofs can't read from streams, it needs the files on the disc)
 * or if we can have them written away in realtime
 * (some yet to be invented really cool isofs generator ;-))
 *
 * The link_requested flag tells us that we should only link to the
 * source file and read it from it's original source when required
 * (that is when burning the cd).
 * This flag is usually true unless the user is pressing the right
 * mouse button.
 *
 * It's the destination fs's responsibility to decide on whether to honour
 * this flag. One helpful source of information to base this decision
 * on is the precache flag of the source object.
 * If it's set, we'll precache.
 * Regardless what happens, a filesystem driver has to make sure that
 * a file linked this way will *not* show the symbolic link flag
 * but has to behave as a perfectly ordinary file
 * The difference between this sort of "linking" and a symbolic link
 * within a filesystem is that the latter may not point "outside"
 * a filesystem
 *
 * No return value, will be set for the operationcomplete callback.
 * This operation is performed in the background.
 *
 * Note that destinationpath has to be the complete new filename */
void vfs_copy(vfs_filesystem*destinationfs,
	      char *destinationpath,
	      vfs_filesystem*sourcefs,
	      char *sourcepath,
	      datacopydlg_dlginfo *progress,
	      vfs_operationcomplete callback,
	      gpointer data,
	      int link_requested,
	      int copy_move);

/* create a file within the filesystem and return a filename that
 * can be used to write data into this file. The returned filename
 * is invalid, the file has to be created by the caller once it's
 * name has been obtained.
 * This function returns NULL if files can't be added to the filesystem
 * this way.
 * path has to be the complete desired name within the filesystem */
char *vfs_createfile_getrealname(vfs_filesystem*fs,
				 char *path);

/* return the corresponding filename in the standard unix tree if
 * possible. returns NULL if this file cannot be accessed using
 * normal unix methods. The only way to get access to the file
 * is to use the vfs open function in this case */
char *vfs_getrealname(vfs_filesystem*fs,
		      char *path);
/* we have to get our data from somewhere, unless we're using
 * the "normal" filesystem of the operating system as mkisofs does in
 * link mode.
 * Note that the return value is a normal file descriptor which can be
 * used to read up to filesize bytes from the file.
 * Never try to do anything unusual with it. It's for reading.
 * No seeking etc. is guaranteed to work.
 * returns -1 in case of an error.
 *
 * File can be closed using the normal close() function of the operating
 * system */
int vfs_openfile(vfs_filesystem*,char *path);

int vfs_deletefile(vfs_filesystem*,char *path);
int vfs_renamefile(vfs_filesystem*,char *path,char *newname);

void vfs_destroy(vfs_filesystem*);

/* Utility functions.
 * Those do not need an implementation for every filesystem.
 * Functions above will be used instead to accomplish their functionality */

int vfs_isdirectory(vfs_filesystem*,char *path);
int vfs_islink(vfs_filesystem*,char *path);
int vfs_isreference(vfs_filesystem*,const char *path);

/* filesize of a file */
unsigned long int vfs_filesize(vfs_filesystem *fs,char *path);

/* returns the number of sub directories of the path.
 *  0  if path is not valid
 *  1  if path is a file
 *  2  if path is an empty directory
 *  >2 num subdirs is (result-2) */
int vfs_numberofsubdirs(vfs_filesystem*,char *path);

/* returns a URI looking somewhat like
 * "gtoastervfs:0xdeadbeef,/this/is/a/path"
 * If it matches any of the standard filesystems and sources
 * (e.g. ftp,http,file,..), URI will be built according to the
 * respective notation */
char *vfs_buildvfsuri(vfs_filesystem*,char *path);

/* parse uri and return filesystem and path therein.
 * path must point to a databuffer sufficiently large to hold the
 * whole path. Just make it the size of uri to be sure.
 * returns NULL if uri is invalid.
 *
 * Note that this will also parse file:/xxxx uris.
 * In this case, the standard unixtree filesystem is returned */
vfs_filesystem *vfs_parseuri(char *uri,char *path);

/* Although this function takes (almost) the same parameters as vfs_copy(),
 * it's something completely different.
 * for a single file passed as an argument to this function,
 * it will do the same, though.
 * What this function does is to recursively add a whole directory of
 * files of a source filesystem to a destination filesystem.
 * All Subdirectories are processed recursively (well not really
 * as we have to cooperate with gtk_main(), this function is essentially
 * building up it's own stack, remaining within a single stackframe all the time
 *
 * Note that destinationpath has to be the complete new filename
 *
 * If progress thread is anything but -1, this thread will be reset
 * whenever the recursion is crossing a file boundary
 *
 * This function and the  vfs_remove_recursively function below
 * return a handle which can be used to cancel the current process */
gtoaster_handle_t vfs_copy_recursively(vfs_filesystem*destinationfs,
			 char *destinationpath,
			 vfs_filesystem*sourcefs,
			 char *sourcepath,
			 datacopydlg_dlginfo *progress,
			 int progress_thread,
			 vfs_operationcomplete callback,
			 gpointer data,
			 int link_requested,
			 int copy_move);

/* Filesystem is the destination filesystem */
void vfs_copy_recursively_cancel(gtoaster_handle_t handle);

gtoaster_handle_t vfs_remove_recursively(vfs_filesystem*fs,
			   char*path,
			   vfs_operationcomplete callback,
			   gpointer data,
			   int content_only);

/* Filesystem is the destination filesystem */
void vfs_remove_recursively_cancel(gtoaster_handle_t handle);

int vfs_remove_recursively_blocking(vfs_filesystem*fs,
				    char *path,
				    int content_only);


void vfs_install_autodelete(vfs_filesystem *fs,
			    const char *path,
			    vfs_operationcomplete *callback,
			    gpointer *data);
			    

#endif // VFS_H

