/* some sort of xml parser. Used to read/write compilations.
 *
 * This xml parser is optimized for it's designated purpose and does by
 * no means adhere to the standard. It will process all data on-the-fly
 * without unnecessarily wasting memory.
 * Also note that this parser will perform all of it's actions asynchronously
 * that is: all functions will return immediately and call a callback
 * once done */

#include <string.h>
#include <stdlib.h>

#include "xml.h"
#include "helpings.h"

/* Ignore cr and lf within XML */
const char xml_ignorechars[]=
{
   13,10
};

/* In between tags we ignore space and tab additionally to
 * cr and lf */
const char xml_ignoreoutside[]=
{
   13,10,20,9
};

/* Call the appropriate item handler, returns zero on success,
 * !=0 if no appropriate handler could be determined */
int xml_callhandler(xml_context_t *context,
		    const char *tag)
{
   xml_taghandler_t *handlers=(xml_taghandler_t*)context->handlers;
   int i;
   /* FIXME: If proper handler couldn't be found,
    * execute dummy handler and continue processing.
    * Report error nevertheless of course */
   int result=1;
   for (i=0;(handlers[i].tag!=NULL)&&result;++i)
     if (!strcasecmp(handlers[i].tag,tag))
       {
	  handlers[i].handler(context,
			      tag,
			      0,NULL,
			      handlers[i].data);
	  result=0;
       };
   return result;
};

/* will return a value >1 when done with current block.
 * 1 if everything was ok, 2 in case of a parse error */
int xml_parsenext(xml_context_t *context)
{
   int result=0;
   char *ignore=helpings_getterminatedstring(context->source,
					     '<',strlen(xml_ignoreoutside),
					     xml_ignoreoutside);
   /* only if no parse error occurred.
    * If ignore contains any characters, those are definitely out of place
    * here */
   if (!strlen(ignore))
     {
	/* Get our tag.
	 * FIXME: <xxx arg=yyy/> not yet implemented */
	char *tag=helpings_getterminatedstring(context->source,
					       '>',strlen(xml_ignorechars),
					       xml_ignorechars);
	if (*tag!='/') // opening tag
	  {
	     if (!strlen(tag))
	       // empty string, probably an indication for EOF.
	       // The step below will test whether 
	       // all tags are closed
	       result=1; // ok, seems we're done.
	     if (context->currenttag) // no tag may be open, either way
	       result=2;
	     if (!result)
	       {
		  context->currenttag=strdup(tag);
		  /* Everything ok. Found tag, call it's handler.
		   * If something went wrong, return parse error.
		   * Note that the handler itself will not return a result.
		   * It's the xml_callhandler() function that
		   * will generate an error if no handler could be found
		   * for the specified tag */
		  if (xml_callhandler(context,tag))
		    result=2;
	       };
	  }
	else
	  {
	     /* Ok, terminating tag. Make sure it matches our context */
	     if ((!context->parenttag)||strcasecmp(context->parenttag,tag))
	       result=2;
	     else
	       /* We're done and everything worked brilliantly */
	       result=1;
	  };
	free(tag);
     }
   else
     /* Garbage output found in unexpected places */
     result=2;
   free(ignore);
   return result;
};

/* Call this from xml_itemhandler_t to tell parser we're done
 * processing the current item */
void xml_doneitem(int status,xml_context_t *context)
{
   int finished=0; /* 0 - not yet finished
		    * 1 - finished with status OK
		    * 2 - finished with some kind of error */
   if (context->currenttag)
     {
	/* if we were called by an item handler it's time to free
	 * our currenttag now */
	free(context->currenttag);
	context->currenttag=NULL;
     };
   if (status)
     finished=2; /* Parse error in item handler */
   else
     finished=xml_parsenext(context); /* What's our status ? */

   if (finished)
     {
	((xml_donecallback_t)context->donecallback)(finished-1,context,context->data);
	/* We're done with the current context.
	 * Destroy it */
	if (context->parenttag)
	  free(context->parenttag);
	/* currenttag should be all closed by now.
	 * But we don't want to have memory leaking around, do we ? */
	if (context->currenttag)
	  free(context->currenttag);
	free(context);
     };
};

void xml_parse_internal(FILE *source,
			const char *ptag,
			const xml_taghandler_t *initial_handlers,
			xml_donecallback_t donecb,void*data)
{
   xml_context_t *context=(xml_context_t*)malloc(sizeof(xml_context_t));
   context->handlers=(void*)initial_handlers;
   context->source=source;
   context->currenttag=NULL;
   context->parenttag=((ptag)?strdup(ptag):NULL);
   context->donecallback=(void*)donecb;
   context->data=data;
   /* Use the xml_doneitem handler as called by the item handler.
    * Set status to 1 (error) if invalid FILE *source was passed */
   xml_doneitem((source==NULL),context);
};

/* This function will read from the FILE specified and
 * will process any xml compatible data found according to the
 * rules defined by passing a list of handlers.
 * Will call specified callback upon completion */
void xml_parse(FILE *source,
	       const xml_taghandler_t *initial_handlers,
	       xml_donecallback_t donecb,void*data)
{
   xml_parse_internal(source,NULL,initial_handlers,
		      donecb,data);
};

/* Same as above, only FILE will be taken from specified context.
 * Specified context will be left untouched otherwise */
void xml_parse_context(xml_context_t *source,
		       const xml_taghandler_t *initial_handlers,
		       xml_donecallback_t donecb,void*data)
{
   xml_parse_internal(source->source,
		      source->currenttag,
		      initial_handlers,
		      donecb,
		      data);
};

/* returns pointer to the contents of the current xml tag.
 * if length != NULL the length of the content area will be stored
 * in the int addressed by length.
 * Any content will be preprocessed to resolve any escape sequences
 * before it is returned (e.g. &amp; will be replaced by &)
 * If !=NULL, status will contain the result of the getcontent call
 *
 * Returned buffer is caller-owned */
char *xml_getcontent(xml_context_t *context,int *status,int *length)
{
   char *result=helpings_getterminatedstring(context->source,
					     '<',
					     strlen(xml_ignorechars),
					     xml_ignorechars);
   char *terminator=helpings_getterminatedstring(context->source,
						 '>',
						 strlen(xml_ignorechars),
						 xml_ignorechars);
   if (status)
     {
	if (((*terminator)!='/')||(!context->currenttag)||
	    strcasecmp(terminator+1,context->currenttag))
	  *status=1;
	else
	  *status=0;
     };
   free(terminator);
   if (length)
     *length=strlen(result);
   /* FIXME: Translate string according to the escape rules
    * defined by xml */
   return result;
};

