/* GTS - Library for the manipulation of triangulated surfaces
 * Copyright (C) 1999 Stphane Popinet
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include "config.h"
#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif /* HAVE_GETOPT_H */
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include "gts.h"

static void traverse_manifold (GtsTriangle * t, GtsSurface * s)
{
  if (g_slist_length (GTS_FACE (t)->surfaces) > 1)
    return;

  gts_surface_add_face (s, GTS_FACE (t));
  if (g_slist_length (t->e1->triangles) == 2) {
    if (t->e1->triangles->data != t)
      traverse_manifold (t->e1->triangles->data, s);
    else
      traverse_manifold (t->e1->triangles->next->data, s);
  }
  if (g_slist_length (t->e2->triangles) == 2) {
    if (t->e2->triangles->data != t)
      traverse_manifold (t->e2->triangles->data, s);
    else
      traverse_manifold (t->e2->triangles->next->data, s);
  }
  if (g_slist_length (t->e3->triangles) == 2) {
    if (t->e3->triangles->data != t)
      traverse_manifold (t->e3->triangles->data, s);
    else
      traverse_manifold (t->e3->triangles->next->data, s);
  }
}

static void non_manifold_edges (GtsEdge * e, gpointer * data)
{
  GtsSurface * s = data[0];
  GSList ** non_manifold = data[1];

  if (gts_edge_face_number (e, s) > 2) {
    GSList * i = e->triangles;

    while (i) {
      if (gts_face_has_parent_surface (i->data, s) &&
	  !g_slist_find (*non_manifold, i->data))
	*non_manifold = g_slist_prepend (*non_manifold, i->data);
      i = i->next;
    }
  }
}

static void traverse_boundary (GtsEdge * e, gpointer * data)
{
  GtsSurface * orig = data[0];
  gchar * fname = data[1];
  guint * components = data[2];
  gboolean * verbose = data[3];
  GtsFace * f = gts_edge_is_boundary (e, orig);

  if (f != NULL && g_slist_length (f->surfaces) == 1) {
    GtsSurface * s = gts_surface_new (gts_surface_class (),
				      gts_face_class (),
				      gts_edge_class (),
				      gts_vertex_class ());
    gchar name[80];
    FILE * fp;
    GSList * non_manifold = NULL, * i;
    gpointer data[2];

    data[0] = s;
    data[1] = &non_manifold;
    traverse_manifold (GTS_TRIANGLE (f), s);

    gts_surface_foreach_edge (s, (GtsFunc) non_manifold_edges, data);
    i = non_manifold;
    while (i) {
      gts_surface_remove_face (s, i->data);
      i = i->next;
    }
    g_slist_free (non_manifold);

    sprintf (name, "%s.%d.gts", fname, (*components)++);
    fp = fopen (name, "wt");
    if (fp == NULL) {
      fprintf (stderr, "split: cannot open file `%s'.\n", name);
      exit (1);
    }
    gts_surface_write (s, fp);
    fclose (fp);

    if (*verbose) {
      fprintf (stderr, "# Boundary component %d\n", *components - 1);
      gts_surface_print_stats (s, stderr);
    }
  }
}

static void traverse_remaining (GtsFace * f, gpointer * data)
{
  gchar * fname = data[1];
  guint * components = data[2];
  gboolean * verbose = data[3];

  if (g_slist_length (f->surfaces) == 1) {
    GtsSurface * s = gts_surface_new (gts_surface_class (),
				      gts_face_class (),
				      gts_edge_class (),
				      gts_vertex_class ());
    gchar name[80];
    FILE * fp;
    GSList * non_manifold = NULL, * i;
    gpointer data[2];

    data[0] = s;
    data[1] = &non_manifold;
    traverse_manifold (GTS_TRIANGLE (f), s);

    gts_surface_foreach_edge (s, (GtsFunc) non_manifold_edges, data);
    i = non_manifold;
    while (i) {
      gts_surface_remove_face (s, i->data);
      i = i->next;
    }
    g_slist_free (non_manifold);

    sprintf (name, "%s.%d.gts", fname, (*components)++);
    fp = fopen (name, "wt");
    if (fp == NULL) {
      fprintf (stderr, "split: cannot open file `%s'.\n", name);
      exit (1);
    }
    gts_surface_write (s, fp);
    fclose (fp);

    if (*verbose) {
      fprintf (stderr, "# Component %d\n", *components - 1);
      gts_surface_print_stats (s, stderr);
    }
  }
}

/* split - splits a surface into connected and manifold components */
int main (int argc, char * argv[])
{
  GtsSurface * s = NULL;
  int c = 0;
  GtsFile * fp;
  gboolean verbose = FALSE;
  gchar fname[80] = "component";
  guint components = 0;
  gpointer data[4];

  data[0] = s;
  data[1] = fname;
  data[2] = &components;
  data[3] = &verbose;

  /* parse options using getopt */
  while (c != EOF) {
#ifdef HAVE_GETOPT_LONG
    static struct option long_options[] = {
      {"help", no_argument, NULL, 'h'},
      {"verbose", no_argument, NULL, 'v'}
    };
    int option_index = 0;
    switch ((c = getopt_long (argc, argv, "hv", 
			      long_options, &option_index))) {
#else /* not HAVE_GETOPT_LONG */
    switch ((c = getopt (argc, argv, "hv"))) {
#endif /* not HAVE_GETOPT_LONG */
    case 'v': /* verbose */
      verbose = TRUE;
      break;
    case 'h': /* help */
      fprintf (stderr,
             "Usage: split [OPTION] [FNAME] < FILE\n"
	     "Splits the GTS surface into connected and manifold components.\n"
	     "FNAME is the base name of the components created (default is `component').\n"
	     "\n"
	     "  -v    --verbose  print statistics about the surface\n"
	     "  -h    --help     display this help and exit\n"
	     "\n"
	     "Reports bugs to %s\n",
	     GTS_MAINTAINER);
      return 0; /* success */
      break;
    case '?': /* wrong options */
      fprintf (stderr, "Try `split --help' for more information.\n");
      return 1; /* failure */
    }
  }

  if (optind < argc)
    strcpy (fname, argv[optind]);

  /* read surface in */
  s = gts_surface_new (gts_surface_class (),
		       gts_face_class (),
		       gts_edge_class (),
		       gts_vertex_class ());
  fp = gts_file_new (stdin);
  if (gts_surface_read (s, fp)) {
    fputs ("split: file on standard input is not a valid GTS file\n", 
	   stderr);
    fprintf (stderr, "stdin:%d:%d: %s\n", fp->line, fp->pos, fp->error);
    return 1; /* failure */
  }

  /* if verbose on print stats */
  if (verbose)
    gts_surface_print_stats (s, stderr);

  /* boundary components */
  gts_surface_foreach_edge (s, (GtsFunc) traverse_boundary, data);

  /* remaining components */
  gts_surface_foreach_face (s, (GtsFunc) traverse_remaining, data);

  return 0; /* success */
}
