#ifndef IFACE_H
#define IFACE_H

/*
 * Encapsulate access to a network interface
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

/* 
 * Provide the class and exceptions required to access interface data
 */

#include <time.h>
#include <netinet/in.h>
#include "Exception.h"

struct ifreq;

/// Exception raised when an error occurs accessing an interface
class IFaceException : public SystemException
{
public:
	IFaceException(int errorCode, const std::string& context) throw ()
		: SystemException(errorCode, context) {}

	virtual const char* type() const throw () { return "IFaceException"; }
};

/// Exception raised when an error occurs accessing the MII functionality of an
/// interface
class MIIException : public IFaceException
{
public:
	MIIException(int errorCode, const std::string& context) throw ()
		: IFaceException(errorCode, context) {}

	virtual const char* type() const throw () { return "MIIException"; }
};

/// Configuration data for a network interface
struct if_params
{
  short flags;
  unsigned int addr_flags;
  struct sockaddr_in addr;
  struct sockaddr_in dstaddr;
  struct sockaddr_in broadaddr;
  struct sockaddr_in netmask;
  struct sockaddr hwaddr;

  // Debugging function, not thread safe
  void print();
};

/// Access the informations on an interface
class IFace
{
protected:
	int _socket;
	struct ifreq* _ifr;
	bool _up;
	bool _run;
	bool _conn;
	bool _has_iface;
	bool _has_mii;

	void read_interface_configuration(struct if_params *ifp)
		throw (IFaceException);

	void write_interface_configuration(const struct if_params& ifp)
		throw (IFaceException);

	int IFace::mdio_read(int location) throw (MIIException);

public:
	/// Create an object to access informations of the interface whose name
	/// is in `name'
	IFace(const std::string& name) throw (SystemException, IFaceException, MIIException);
	~IFace() throw ();

	/// Tell if the interface was up at the time of the last update()
	bool up() const throw () { return _up; }
	/// Tell if the interface was running at the time of the last update()
	bool running() const throw () { return _run; }
	/// Tell if the interface was connected at the time of the last update()
	bool connected() const throw () { return _conn; }

	/// Tell if the interface exists
	bool has_iface() const throw () { return _has_iface; }

	/// Tell if the interface MII interface is working
	bool has_mii() const throw () { return _has_mii; }

	/// Get the interface name
	std::string name() const throw ();

	/// Update the interface status
	void update() throw (IFaceException, MIIException);

	/// Bring the interface up in broadcast mode
	/**
	 * Returns a struct if_params descriving the previous interface configuration
	 */
	if_params initBroadcast(int timeout) throw (IFaceException);

	/// Get the interface configuration
	/**
	 * Returns a struct if_params descriving the current interface configuration
	 */
	if_params getConfiguration() throw (IFaceException);

	/// Set the interface configuration
	/**
	 * Returns a struct if_params descriving the previous interface configuration
	 */
	if_params setConfiguration(const if_params& config) throw (IFaceException);
};

// vim:set ts=4 sw=4:
#endif
