#ifndef NETWATCHER_H
#define NETWATCHER_H

/*
 * Thread to capture packets from a network
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "Exception.h"
#include "NetBuffer.h"
#include "nettypes.h"
#include <string>

class pcapException : public ConsistencyCheckException
{
protected:
	std::string _pcap_errmsg;

public:
	pcapException(const std::string& pcap_errmsg, const std::string& context) throw ()
		: ConsistencyCheckException(context), _pcap_errmsg(pcap_errmsg) {}
	pcapException(const std::string& context) throw ()
		: ConsistencyCheckException(context), _pcap_errmsg() {}
	~pcapException() throw () {}

	virtual const char* type() const throw ()
	{
		return "pcapException";
	}

	virtual std::string desc() const throw ()
	{
		if (_pcap_errmsg.size())
			return _pcap_errmsg + " " + _context;
		else
			return "Unknown pcap error " + _context;
	}
};

class PacketListener
{
public:
	virtual ~PacketListener() {}
	virtual void handleARP(const NetBuffer& pkt) throw () {}
	virtual void handleDHCP(const NetBuffer& pkt) throw () {}
	virtual void handleICMP(const NetBuffer& pkt) throw () {}
	virtual void handleEthernet(const NetBuffer& pkt) throw () {}
};

/*
 * Injects ethernet packets to a given interface.
 *
 * Every method is thread-safe
 */
class NetWatcherImpl;

class NetWatcher
{
protected:
	NetWatcherImpl* impl;
	
public:
	NetWatcher(const std::string& iface) throw (SystemException, pcapException);
	NetWatcher(const NetWatcher& f) throw ();
	~NetWatcher() throw (SystemException);
	NetWatcher& operator=(const NetWatcher& f) throw (SystemException);

	struct ether_addr* getMACAddress() throw (pcapException);

	void addARPListener(PacketListener* pl) throw ();
	void addDHCPListener(PacketListener* pl) throw ();
	void addICMPListener(PacketListener* pl) throw ();
	void addEthernetListener(PacketListener* pl) throw ();

	void shutdown() throw ();
};

// vim:set ts=4 sw=4:
#endif
