# Gufw 10.04.4 - http://gufw.tuxfamily.org
# Copyright (C) 2008-2010 Marcos Alvarez Costales <marcos@softastur.org>
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.

import commands

from GufwLog import GufwLog
from Rule    import Rule


class Firewall:
    """Set or get the properties of Firewall"""
    def __init__(self):
        self.gufw_log = GufwLog()
        self.rule = Rule()
        
        self.status = self.get_ufw_status()
        self.incoming_policy = self.get_ufw_policy("incoming")
        self.outgoing_policy = self.get_ufw_policy("outgoing")
        self.ufw_logging = self.get_ufw_log()
        
    def get_ufw_status(self):
        ufw_status = commands.getstatusoutput("LANGUAGE=C ufw status")
        if ufw_status[1].find("Status: active") != -1:
            return "enable"
        else:
            return "disable"
    
    def get_ufw_policy(self, policy):
        if policy == "incoming":
            ufw_default_incoming = commands.getstatusoutput("grep DEFAULT_INPUT_POLICY /etc/default/ufw")
            if ufw_default_incoming[1].find("ACCEPT") != -1:
                return "allow"
            elif ufw_default_incoming[1].find("DROP") != -1:
                return "deny"
            elif ufw_default_incoming[1].find("REJECT") != -1:
                return "reject"
        
        elif policy == "outgoing":
            ufw_default_outgoing = commands.getstatusoutput("grep DEFAULT_OUTPUT_POLICY /etc/default/ufw")
            if ufw_default_outgoing[1].find("ACCEPT") != -1:
                return "allow"
            elif ufw_default_outgoing[1].find("DROP") != -1:
                return "deny"
            elif ufw_default_outgoing[1].find("REJECT") != -1:
                return "reject"
    
    def get_ufw_log(self):
        ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw status verbose")
        if ufw_cmd[1].find("Logging: on (high)") != -1:
            return "high"
        elif ufw_cmd[1].find("Logging: on (medium)") != -1:
            return "medium"
        elif ufw_cmd[1].find("Logging: on (low)") != -1:
            return "low"
        elif ufw_cmd[1].find("Logging: on (full)") != -1:
            return "full"
        else:
            return "off"
    
    def get_status(self):
        """Return actual status"""
        return self.status
    
    def set_status(self, status):
        """Set status to ufw [enable|disable]"""
        if status == "enable":
            cmd = "ufw enable"
            self.status = "enable"
        else:
            cmd = "ufw disable"
            self.status = "disable"
        
        ufw_cmd = commands.getstatusoutput(cmd)
        self.gufw_log.add_log(cmd)
    
    def get_default_policy(self, policy):
        """Get actual policy for incoming or outgoing traffic [allow|deny|reject]"""
        if policy == "incoming":
            return self.incoming_policy
        else:
            return self.outgoing_policy
            
    def set_default_policy(self, direction, policy):
        """Set policy [allow|deny|reject] for the direction [incoming|outgoing] specify"""
        if direction == "incoming":
            if policy == "allow":
                cmd = "ufw default allow incoming"
                self.incoming_policy = "allow"
            elif policy == "deny":
                cmd = "ufw default deny incoming"
                self.incoming_policy = "deny"
            elif policy == "reject":
                cmd = "ufw default reject incoming"
                self.incoming_policy = "reject"
                
        elif direction == "outgoing":
            if policy == "allow":
                cmd = "ufw default allow outgoing"
                self.outgoing_policy = "allow"
            elif policy == "deny":
                cmd = "ufw default deny outgoing"
                self.outgoing_policy = "deny"
            elif policy == "reject":
                cmd = "ufw default reject outgoing"
                self.outgoing_policy = "reject"
        
        ufw_cmd = commands.getstatusoutput(cmd)
        self.gufw_log.add_log(cmd)
    
    def reset_configuration(self):
        """Reset the configuration to initial status"""
        ufw_cmd = "ufw --force reset"
        cmd = commands.getstatusoutput(ufw_cmd)
        self.rule.refresh_rules()
        self.gufw_log.add_log(ufw_cmd)
    
    def get_ufw_logging(self):
        """Return the ufw logging"""
        return self.ufw_logging
        
    def refresh_ufw_logging(self):
        """Reload the ufw logging"""
        self.ufw_logging = self.get_ufw_log()
    
    def set_ufw_logging(self, logging):
        """Set the ufw logging"""
        if logging == "off":
            ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw logging off")
        elif logging == "on":
            ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw logging on")
        elif logging == "low":
            ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw logging low")
        elif logging == "medium":
            ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw logging medium")
        elif logging == "high":
            ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw logging high")
        elif logging == "full":
            ufw_cmd = commands.getstatusoutput("LANGUAGE=C ufw logging full")
        
        self.ufw_logging = self.get_ufw_log()
    
    def get_gufw_log(self):
        log_local, log_server = self.gufw_log.get_log()
        return log_local, log_server
    
    def remove_gufw_log(self):
        self.gufw_log.remove_log()
    
    def get_gufw_logging(self):
        return self.gufw_log.get_logging()
    
    def set_gufw_logging(self, logging):
        self.gufw_log.set_logging(logging)
    
    def add_rule(self, is_program, insert_number, action, direction, log, protocol, fromip, fromport, toip, toport):
        previus_rules = self.get_rules()
        ufw_result, cmd = self.rule.add_rule(is_program, insert_number, action, direction, log, protocol, fromip, fromport, toip, toport)
        self.gufw_log.add_log(cmd)
        if previus_rules != self.get_rules():
            return 0
        else:
            return 1
    
    def remove_rule(self, number_rule):
        ufw_result, cmd = self.rule.remove_rule(number_rule)
        self.gufw_log.add_log(cmd)
        return ufw_result
    
    def get_rules(self):
        return self.rule.get_rules()
        
