# Gufw 10.04.4 - http://gufw.tuxfamily.org
# Copyright (C) 2008-2010 Marcos Alvarez Costales <marcos@softastur.org>
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.

import commands


class Rule:
    """Get, insert and remove Rules"""
    def __init__(self):
        self.rules = self.get_ufw_rules()
    
    def get_ufw_rules(self):
        return_rules = []
        ufw_rules = commands.getstatusoutput("LANGUAGE=C ufw status verbose")
        rule_lines = ufw_rules[1].split("\n")
        
        for descomponent_rules in rule_lines:
            if descomponent_rules.find("ALLOW")  != -1 or \
               descomponent_rules.find("DENY")   != -1 or \
               descomponent_rules.find("LIMIT")  != -1 or \
               descomponent_rules.find("REJECT") != -1:
                return_rules.append(descomponent_rules)
        
        return return_rules
    
    def get_rules(self):
        """Get actual rules"""
        return self.rules
    
    def add_rule(self, is_program, insert_number, action, direction, log, protocol, fromip, fromport, toip, toport):
        """Add a rule"""
        # Component rule
        if is_program:
            rule = "ufw insert &insert &action &direction &log proto &protocol from &fromIP port &fromPort to &toIP port &toPort"
        else:
            rule = "ufw insert &insert &action &direction &log &toPort" 
        
        # Insert Number
        if insert_number != "0":
            rule = rule.replace("&insert", insert_number)
        else:
            rule = rule.replace("insert &insert ", "")
        
        # Action
        rule = rule.replace("&action", action)
        
        # Direction
        rule = rule.replace("&direction", direction)
        
        # Log
        if log != "log-default":
            rule = rule.replace("&log", log)
        else:
            rule = rule.replace("&log ", "")
        
        # Protocol
        if protocol != "both":
            rule = rule.replace("&protocol", protocol)
        else:
            rule = rule.replace(" proto &protocol ", " ")
            
        # FROM
        if fromip != "":
            rule = rule.replace("&fromIP", fromip)
        else:
            rule = rule.replace("&fromIP", "any")
            
        if fromport != "":
            rule = rule.replace("&fromPort", fromport)
        else:
            rule = rule.replace(" port &fromPort ", " ")
            
        # TO
        if toip != "":
            rule = rule.replace("&toIP", toip)
        else:
            rule = rule.replace("&toIP", "any")
            
        if toport != "":
            rule = rule.replace("&toPort", toport)
        else:
            rule = rule.replace(" port &toPort", "")
        
        # Run ufw command
        ufw_cmd = commands.getstatusoutput(rule)
        if ufw_cmd[0] == 0:
            self.rules = self.get_ufw_rules()
        return ufw_cmd[0], rule
    
    def remove_rule(self, number_rule):
        """Remove number rule"""
        command_rule = "ufw --force delete &number".replace("&number", number_rule)
        ufw_cmd = commands.getstatusoutput(command_rule)
        if ufw_cmd[0] == 0:
            self.rules = self.get_ufw_rules()
        return ufw_cmd[0], command_rule
        
    def refresh_rules(self):
        """Refresh rules in ufw"""
        self.rules = self.get_ufw_rules()
