# Gufw 13.10.2 - http://gufw.org
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.

import glob, os

import gettext
from gettext import gettext as _
gettext.textdomain('gufw')


class Basic():
    """Set or get the Firewall properties"""
    def __init__(self, backend):
        self.backend = backend
        
        self.status   = self.backend.get_status()
        self.incoming = self.backend.get_policy('incoming')
        self.outgoing = self.backend.get_policy('outgoing')
        
        self.all_profiles = self._read_all_profiles()
        self.profile = self._read_default_profile()
        self.backend.set_profile_values(self.profile, self.status, self.incoming, self.outgoing, self.get_rules())
        if not self.all_profiles:
            self.all_profiles = self._read_all_profiles()
        
    # PROFILE
    def get_profile(self):
        return self.profile
    
    def set_profile(self, profile):
        operation = []
        
        # Force show current rules
        if not self.status:
            self.backend.set_status(True)
        # Remove Gufw rules previous profile
        old_rules = self.get_rules()
        ind = len(old_rules)
        for old_row in reversed(old_rules):
            if old_row['command']: # It's a gufw rule
                result = self.backend.delete_rule(str(ind))
                line = _("Deleting previous Gufw rule") + ": " + str(ind) + ' ' + result[0] + ' > ' + result[1].replace('\n', ' | ')
                operation.append(line)
            ind -= 1
        
        # Set new profile
        self.profile = profile
        self.backend.set_config_value('Profile', profile)
        
        # New Profile
        (new_status, new_incoming, new_outgoing, new_rules) = self.backend.get_profile_values(profile)
        
        # Append Gufw rules new profile
        for new_row in new_rules:
            if new_row['command']: # It's a gufw rule
                result = self.backend.add_rule('', # Insert in order
                                               new_row['policy'],
                                               new_row['direction'],
                                               new_row['iface'], 
                                               new_row['logging'], 
                                               new_row['protocol'], 
                                               new_row['from_ip'], 
                                               new_row['from_port'], 
                                               new_row['to_ip'], 
                                               new_row['to_port'])
                line = _("Appending new Gufw rule") + ": " + result[0] + ' > ' + result[1].replace('\n', ' | ')
                operation.append(line)
        
        # New status
        self.status = new_status
        self.backend.set_status(self.status)
        self.incoming = new_incoming
        self.backend.set_policy('incoming', self.incoming)
        self.outgoing = new_outgoing
        self.backend.set_policy('outgoing', self.outgoing)
        
        return operation
    
    def get_all_profiles(self):
        return self.all_profiles
    
    def add_profile(self, profile):
        self.all_profiles.append(profile)
        rules = []
        self.backend.set_profile_values(profile, self.status, self.incoming, self.outgoing, rules) # Will create profile
    
    def delete_profile(self, profile):
        self.all_profiles.remove(profile)
        self.backend.delete_file_profile(profile)
    
    def rename_profile(self, old, new):
        if old == self.profile:
            return
        self.backend.rename_file_profile(old, new)
        self.all_profiles[self.all_profiles.index(old)] = new
    
    def import_profile(self, file):
        new_profile = os.path.basename(file)
        new_profile = os.path.splitext(new_profile)[0]
        (status, incoming, outgoing, rules) = self.backend.get_profile_values(file)
        self.add_profile(new_profile)
        self.backend.set_profile_values(new_profile, status, incoming, outgoing, rules)
        
    def export_profile(self, file):
        self.backend.export_profile(self.profile, file)
        
    
    # FIREWALL
    def get_status(self):
        return self.status
    
    def set_status(self, status):
        self.status = status
        self.backend.set_status(status)
        self.backend.set_profile_values(self.profile, status, self.incoming, self.outgoing, self.get_rules())
    
    def get_policy(self, policy):
        if policy == 'incoming':
            return self.incoming
        elif policy == 'outgoing':
            return self.outgoing
    
    def set_policy(self, policy, value):
        if policy == 'incoming':
            self.incoming = value
            self.backend.set_policy(policy, value)
        else:
            self.outgoing = value
            self.backend.set_policy(policy, value)
        
        self.backend.set_profile_values(self.profile, self.status, self.incoming, self.outgoing, self.get_rules())
    
    def reset(self):
        rules = []
        self.backend.set_profile_values(self.profile, self.status, self.incoming, self.outgoing, rules)
        self.backend.reset_fw()
    
    # RULES
    def get_rules(self):
        if self.backend.get_status():
            ufw_rules = self.backend.get_rules()
        else:
            self.backend.set_status(True)
            ufw_rules = self.backend.get_rules()
            self.backend.set_status(False)
        profile_rules = self._get_rules_profile()
        return self._compose_rules(ufw_rules, profile_rules)
    
    def add_rule(self, description, insert, policy, direction, iface, logging, proto, from_ip, from_port, to_ip, to_port):
        rules_before = self.get_rules()
        rules_profile_before = self._get_rules_profile()
        
        cmd = self.backend.add_rule(insert, policy, direction, iface, logging, proto, from_ip, from_port, to_ip, to_port)
        rules_after = self.get_rules()
        
        if len(rules_before) != len(rules_after):
            cmd.insert(0, True)
            self._regenerate_file_profile(rules_before, rules_profile_before, rules_after, description, cmd[1], policy, direction, iface, logging, proto, from_ip, from_port, to_ip, to_port)
        else:
            cmd.insert(0, False)
        return cmd # For logging
    
    def delete_rule(self, num):
        rules_before = self.get_rules()
        rules_profile_before = self._get_rules_profile()
        result = self.backend.delete_rule(str(num))
        rules_after = self.get_rules()
        self._regenerate_file_profile(rules_before, rules_profile_before, rules_after)
        return result # For logging
    
    # EXTRAS
    def _read_default_profile(self):
        value = self.backend.get_config_value('Profile')
        if value and value in self.all_profiles:
            return value
        
        # First Gufw run or wrong config file
        rules = []
        self.backend.set_profile_values(_("Public").replace(' ', '_'), True, 'reject',  'allow', rules)
        self.backend.set_profile_values(_("Office").replace(' ', '_'), True, 'deny',    'allow', rules)
        
        default_profile = _("Home").replace(' ', '_')
        self.backend.set_config_value('Profile', default_profile)
        
        return default_profile
    
    def _read_all_profiles(self):
        profiles = []
        files = glob.glob('/etc/gufw/*.profile')
        files.sort(key=lambda x: os.path.getctime(x)) # Sort by time and date
        
        for file in files:
            profiles.append(_(os.path.splitext(os.path.basename(file))[0]).replace(' ', '_')) # Filename without path and extension
        
        return profiles
    
    def _compose_rules(self, ufw_rules, profile_rules):
        rules = []
        for ufw_rule in ufw_rules:
            rule = {'ufw_rule'   : ufw_rule, # ufw rule
                    'description': '', # description
                    'command'    : '', # command
                    'policy'     : '', # policy
                    'direction'  : '', # direction
                    'protocol'   : '', # proto
                    'from_ip'    : '', # from_ip
                    'from_port'  : '', # from_port
                    'to_ip'      : '', # to_ip
                    'to_port'    : '', # to_port
                    'iface'      : '', # iface
                    'logging'    : ''} # logging
            
            for profile_rule in profile_rules:
                if ufw_rule == profile_rule['ufw_rule']:
                    rule = {'ufw_rule'    : profile_rule['ufw_rule'],    # ufw rule
                            'description' : profile_rule['description'], # description
                            'command'     : profile_rule['command'],     # command
                            'policy'      : profile_rule['policy'],      # policy
                            'direction'   : profile_rule['direction'],   # direction
                            'protocol'    : profile_rule['protocol'],    # proto
                            'from_ip'     : profile_rule['from_ip'],     # from_ip
                            'from_port'   : profile_rule['from_port'],   # from_port
                            'to_ip'       : profile_rule['to_ip'],       # to_ip
                            'to_port'     : profile_rule['to_port'],     # to_port
                            'iface'       : profile_rule['iface'],       # iface
                            'logging'     : profile_rule['logging']}     # logging
                    break
            
            rules.append(rule)
            
        return rules
    
    def _get_rules_profile(self):
        rules_profile = []
        (status, incoming, outgoing, rules) = self.backend.get_profile_values(self.profile)
        for rule in rules:
            conver_rules = {'ufw_rule'    : rule['ufw_rule'],                              
                            'description' : rule['description'],         
                            'command'     : rule['command'],               
                            'policy'      : rule['policy'],                
                            'direction'   : rule['direction'],             
                            'protocol'    : rule['protocol'],              
                            'from_ip'     : rule['from_ip'],               
                            'from_port'   : rule['from_port'],             
                            'to_ip'       : rule['to_ip'],                 
                            'to_port'     : rule['to_port'],               
                            'iface'       : rule['iface'],                 
                            'logging'     : rule['logging']}               
            rules_profile.append(conver_rules)
        
        return rules_profile
        
    def _regenerate_file_profile(self, ufw_before, profile_before, ufw_after, description='', cmd='', policy='', direction='', iface='', logging='', proto='', from_ip='', from_port='', to_ip='', to_port=''):
        """Here there are dragons!"""
        final_rules = []            
        # Operation ufw_before  ufw_after   profile_before  profile_after
        #     +         A         A+B             A               A+B   <-- A completed from profile_before | B completed from parameters
        #     -        A+B         A             A+B               A    
        #     x        A+B        A+C            A+B              A+C   <-- A completed from profile_before | C completed from parameters
        # We have here profile_before + ufw_after + parameters > Just complete!
        for ufw_rule in ufw_after:
            found = False
            for profile_rule in profile_before:
                if ufw_rule['ufw_rule'] == profile_rule['ufw_rule']: # Complete from previous profile
                    found = True
                    new_rule = {'ufw_rule'    : profile_rule['ufw_rule'],    
                                'description' : profile_rule['description'], 
                                'command'     : profile_rule['command'],     
                                'policy'      : profile_rule['policy'],      
                                'direction'   : profile_rule['direction'],   
                                'protocol'    : profile_rule['protocol'],    
                                'from_ip'     : profile_rule['from_ip'],     
                                'from_port'   : profile_rule['from_port'],   
                                'to_ip'       : profile_rule['to_ip'],       
                                'to_port'     : profile_rule['to_port'],     
                                'iface'       : profile_rule['iface'],       
                                'logging'     : profile_rule['logging']}     
            if not found:
                if ufw_before.count(ufw_rule) == 0: # New: Complete from parameters
                    new_rule = {'ufw_rule'    : ufw_rule['ufw_rule'],
                                'description' : description, 
                                'command'     : cmd,
                                'policy'      : policy,    
                                'direction'   : direction, 
                                'protocol'    : proto,    
                                'from_ip'     : from_ip,   
                                'from_port'   : from_port, 
                                'to_ip'       : to_ip,   
                                'to_port'     : to_port,   
                                'iface'       : iface,   
                                'logging'     : logging}   
                else: # Old ufw rule: Nothing
                    continue
                    
            final_rules.append(new_rule) # Just adding rules, a regenerate will be update the rules
        
        self.backend.set_profile_values(self.profile, self.status, self.incoming, self.outgoing, final_rules)

