# -*- coding: utf-8 -*-
# Gufw 13.10.2 - http://gufw.org
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# Gufw is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Gufw is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with Gufw; if not, see http://www.gnu.org/licenses for more
# information.


from gi.repository import Gtk, GObject
import os, glob, ConfigParser

from constants import *

import gettext
from gettext import gettext as _
gettext.textdomain('gufw')


class WinAdd:
    def __init__(self, fw_frontend, utils, winMain, main_add_btn):
        self.fw = fw_frontend
        self.utils = utils
        self.main_add_btn = main_add_btn
        
        self.apps = AppProfiles()
        
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("gufw")
        self.builder.add_from_file('/usr/share/gufw/ui/add.ui')
        self._set_objects_name()
        self._set_initial_values()
        self.win_add.set_transient_for(winMain)
        self.builder.connect_signals(self)
        self.win_add.show_all()
        self.warningbox.hide() # Not works setting from glade
    
    def _set_objects_name(self):
        self.win_add = self.builder.get_object('AddRule')
        self.tabs = self.builder.get_object('addTabs')
        
        self.preconfig_policy      = self.builder.get_object('preconfig_policy')
        self.preconfig_direction   = self.builder.get_object('preconfig_direction')
        self.preconfig_category    = self.builder.get_object('preconfig_category')
        self.preconfig_subcategory = self.builder.get_object('preconfig_subcategory')
        self.preconfig_app         = self.builder.get_object('preconfig_app')
        self.preconfig_description = self.builder.get_object('preconfig_description')
        self.img_description       = self.builder.get_object('img_description')
        self.warningbox            = self.builder.get_object('warningbox')
        self.lbl_inforule          = self.builder.get_object('lbl_preconfig_info')
        
        self.simple_rule_name = self.builder.get_object('simple_rule_name')
        self.simple_policy    = self.builder.get_object('simple_policy')
        self.simple_direction = self.builder.get_object('simple_direction')
        self.simple_protocol  = self.builder.get_object('simple_protocol')
        self.simple_port      = self.builder.get_object('simple_port')
        
        self.advanced_rule_name = self.builder.get_object('advanced_rule_name')
        self.advanced_insert    = self.builder.get_object('advanced_insert')
        self.advanced_policy    = self.builder.get_object('advanced_policy')
        self.advanced_direction = self.builder.get_object('advanced_direction')
        self.advanced_iface     = self.builder.get_object('advanced_iface')
        self.advanced_log       = self.builder.get_object('advanced_log')
        self.advanced_protocol  = self.builder.get_object('advanced_protocol')
        self.advanced_from_ip   = self.builder.get_object('advanced_from_ip')
        self.advanced_from_port = self.builder.get_object('advanced_from_port')
        self.advanced_to_ip     = self.builder.get_object('advanced_to_ip')
        self.advanced_to_port   = self.builder.get_object('advanced_to_port')

    def _set_initial_values(self):
        self._set_app_combobox()
        
        # Bug gnome #549478
        self.preconfig_policy.set_active(0)
        self.preconfig_direction.set_active(0)
        
        self.simple_policy.set_active(0)
        self.simple_direction.set_active(0)
        self.simple_protocol.set_active(0)
        
        self.advanced_policy.set_active(0)
        self.advanced_direction.set_active(0)
        self.advanced_iface.append_text(_("All Interfaces"))
        for ifaceName in self.fw.get_net_interfaces():
            self.advanced_iface.append_text(ifaceName)
        self.advanced_iface.set_active(0)
        self.advanced_log.set_active(0)
        self.advanced_protocol.set_active(0)
        
    def _set_app_combobox(self):
        categories = self.apps.get_categories()
        for cat in categories:
            self.preconfig_category.append_text(cat)
        self.preconfig_category.set_active(0)
        
        subcategories = self.apps.get_subcategories(self.preconfig_category.get_active_text())
        for subcat in subcategories:
            self.preconfig_subcategory.append_text(subcat)
        self.preconfig_subcategory.set_active(0)
        
        apps = self.apps.get_apps_cat_subcat(self.preconfig_category.get_active_text(), self.preconfig_subcategory.get_active_text())
        for app in apps:
            self.preconfig_app.append_text(app)
        self.preconfig_app.set_active(0)
        
        tooltip = self.apps.get_app(self.preconfig_app.get_active_text())[0] + '\n' + _("Ports") + ": " + self.apps.get_app(self.preconfig_app.get_active_text())[1]
        self.img_description.set_tooltip_text(tooltip)
    
    def on_btnAddClose_clicked(self, widget, data=None):
        self._win_close()
    
    def on_AddRule_delete_event(self, widget, data=None):
        self._win_close()
    
    def _win_close(self):
        self.main_add_btn.set_sensitive(True)
        self.win_add.destroy()
    
    def on_copy_from_IP_clicked(self, widget, data=None):
        self.advanced_from_ip.set_text(self.fw.get_internal_ip())
    
    def on_copy_to_IP_clicked(self, widget, data=None):
        self.advanced_to_ip.set_text(self.fw.get_internal_ip())
    
    def on_copy_simple_to_advanced_clicked(self, widget, data=None):
        self.advanced_rule_name.set_text(self.simple_rule_name.get_text())
        self.advanced_policy.set_active(self.simple_policy.get_active())
        self.advanced_direction.set_active(self.simple_direction.get_active())
        self.advanced_protocol.set_active(self.simple_protocol.get_active())
        self.advanced_to_port.set_text(self.simple_port.get_text())
    
        self.tabs.set_current_page(2)
    
    def on_copy_preconfig_to_advanced_clicked(self, widget, data=None):
        self.advanced_rule_name.set_text(self.preconfig_app.get_active_text())
        self.advanced_policy.set_active(self.preconfig_policy.get_active())
        self.advanced_direction.set_active(self.preconfig_direction.get_active())
        self.advanced_protocol.set_active(2)
        self.advanced_to_port.set_text(self.apps.get_app(self.preconfig_app.get_active_text())[1])
        self.advanced_protocol.set_sensitive(False) # Protocol not apply
        
        self.tabs.set_current_page(2)
    
    def on_advanced_from_port_changed(self, widget, data=None):
        if ('/' in self.advanced_from_port.get_text() or
            '/' in self.advanced_from_port.get_text()):
            self._set_from_port_sensitive(False)
        else:
            self._set_from_port_sensitive(True)
    
    def on_advanced_to_port_changed(self, widget, data=None):
        if ('/' in self.advanced_to_port.get_text() or
            '/' in self.advanced_to_port.get_text()):
            self._set_to_port_sensitive(False)
        else:
            self._set_to_port_sensitive(True)
    
    def _set_from_port_sensitive(self, value=True):
        self.advanced_protocol.set_sensitive(value)
        self.advanced_iface.set_sensitive(value)
        self.advanced_from_ip.set_sensitive(value)
        self.advanced_to_ip.set_sensitive(value)
        self.advanced_to_port.set_sensitive(value)
        if value:
            if self.advanced_from_ip.get_text() or self.advanced_to_ip.get_text():
                self.advanced_iface.set_sensitive(True)
            else:
                self.advanced_iface.set_sensitive(False)
    
    def _set_to_port_sensitive(self, value=True):
        self.advanced_protocol.set_sensitive(value)
        self.advanced_iface.set_sensitive(value)
        self.advanced_from_ip.set_sensitive(value)
        self.advanced_to_ip.set_sensitive(value)
        self.advanced_from_port.set_sensitive(value)
        if value:
            if self.advanced_from_ip.get_text() or self.advanced_to_ip.get_text():
                self.advanced_iface.set_sensitive(True)
            else:
                self.advanced_iface.set_sensitive(False)
    
    def on_advanced_from_ip_changed(self, widget, data=None):
        if self.advanced_from_ip.get_text() or self.advanced_to_ip.get_text():
            self.advanced_iface.set_sensitive(True)
        else:
            self.advanced_iface.set_sensitive(False)
    
    def on_advanced_to_ip_changed(self, widget, data=None):
        if self.advanced_from_ip.get_text() or self.advanced_to_ip.get_text():
            self.advanced_iface.set_sensitive(True)
        else:
            self.advanced_iface.set_sensitive(False)
    
    def on_preconfig_category_changed(self, widget, data=None):
        # Delete all subcategories
        self.preconfig_subcategory.remove_all()
        # Append new entries
        subcategories = self.apps.get_subcategories(self.preconfig_category.get_active_text())
        for subcat in subcategories:
            self.preconfig_subcategory.append_text(subcat)
        self.preconfig_subcategory.set_active(0)
    
    def on_preconfig_subcategory_changed(self, widget, data=None):
        # Delete all apps
        self.preconfig_app.remove_all()
        # Append new entries
        apps = self.apps.get_apps_cat_subcat(self.preconfig_category.get_active_text(), self.preconfig_subcategory.get_active_text())
        for app in apps:
            self.preconfig_app.append_text(app)
        self.preconfig_app.set_active(0)
    
    def on_preconfig_app_changed(self, widget, data=None):
        if self.preconfig_app.get_active_text() != None:
            tooltip = self.apps.get_app(self.preconfig_app.get_active_text())[0] + '\n' + _("Ports") + ": " + self.apps.get_app(self.preconfig_app.get_active_text())[1]
            self.img_description.set_tooltip_text(tooltip)
            if self.apps.get_app(self.preconfig_app.get_active_text())[3]:
                self.lbl_inforule.set_text(self.apps.get_app(self.preconfig_app.get_active_text())[3])
                self.warningbox.show()
            else:
                self.warningbox.hide()
    
    def on_btnAddRuleWin_clicked(self, widget, data=None):
        if not self.fw.get_status():
            self.utils.show_dialog(_("Error: Firewall is disabled"), _("The firewall has to be enabled first"))
            return
        
        if self.tabs.get_current_page() == 0:
            self._add_rule_preconfig()
        elif self.tabs.get_current_page() == 1:
            self._add_rule_simple()
        elif self.tabs.get_current_page() == 2:
            self._add_rule_advanced()
    
    def _add(self, profile='', name='', policy='', direction='', proto='', from_ip='', from_port='', to_ip='', to_port='', insert='', iface='', logging=''):
        flag_OK = True
        flag_Warning = False
        
        # Split possible ports > 135,139,445/tcp|137,138/udp
        for from_split in from_port.split('|'):
            for to_split in to_port.split('|'):
                if ('/tcp' in from_split or 
                    '/udp' in from_split or
                    '/tcp' in to_split   or
                    '/udp' in to_split):
                    proto = ''
                
                from_port = from_split
                to_port = to_split
                if direction == 'both':
                    cmd = self.fw.add_rule(name, insert, policy, 'in',  iface, logging, proto, from_ip, from_port, to_ip, to_port)
                    cmd = self.fw.add_rule(name, insert, policy, 'out', iface, logging, proto, from_ip, from_port, to_ip, to_port)
                else:
                    cmd = self.fw.add_rule(name, insert, policy, direction, iface, logging, proto, from_ip, from_port, to_ip, to_port)
                
                if cmd[0]:
                    flag_Warning = True
                    self.utils.add_to_log(cmd[1])
                else:
                    flag_OK = False
                    self.utils.add_to_log(_("Error running") + ": " + cmd[1] + ' > ' + cmd[2].replace('\n', ' | '))
        
        # OK
        if flag_OK and flag_Warning:
            self.utils.set_statusbar_msg(_("Rule(s) added"))
        # Some OK, some KO
        elif flag_OK and not flag_Warning:
            self.utils.set_statusbar_msg(_("Warning: Some rules added. Review the log"))
        # KO
        else:
            self.utils.set_statusbar_msg(_("Error: No rules added. Review the log"))
    
    def _add_rule_preconfig(self):
        self._add(self.fw.get_profile(),                               # profile
                 self.preconfig_app.get_active_text(),                 # name
                 NUM2POLICY[self.preconfig_policy.get_active()],       # policy
                 NUM2DIRECTION[self.preconfig_direction.get_active()], # direction
                 '',                                                   # proto
                 '',                                                   # from IP
                 '',                                                   # from port
                 '',                                                   # to IP
                 self.apps.get_app(self.preconfig_app.get_active_text())[1]) # to port[/proto][|port[/proto]]
        self.utils.print_rules(self.fw.get_rules())
    
    def _add_rule_simple(self):
        if not self.simple_port.get_text():
            self.utils.show_dialog(_("Insert Port"), _("You need to insert a port in the port field"))
            return
        
        if self.simple_port.get_text().upper() == 'PRISM':
            self.utils.show_dialog(_("Edward Snowden's Greatest Fear"), _('"Nothing Will Change"'))
            return
        
        self._add(self.fw.get_profile(),                            # profile
                 self.simple_rule_name.get_text(),                  # name
                 NUM2POLICY[self.simple_policy.get_active()],       # policy
                 NUM2DIRECTION[self.simple_direction.get_active()], # direction
                 NUM2PROTO[self.simple_protocol.get_active()],      # protocol
                 '',                                                # from IP
                 '',                                                # from port
                 '',                                                # to IP
                 self.simple_port.get_text())                       # to port
        self.utils.print_rules(self.fw.get_rules())
    
    def _add_rule_advanced(self):
        # Validations
        if not self.utils.validate_rule(self.advanced_from_ip.get_text(), self.advanced_from_port.get_text(), self.advanced_to_ip.get_text(), self.advanced_to_port.get_text()):
            return
        
        insert = self.advanced_insert.get_text()
        if insert == '0':
            insert = ''
        
        iface = ''
        if self.advanced_iface.get_sensitive() and self.advanced_iface.get_active_text() != _("All Interfaces"):
            iface = self.advanced_iface.get_active_text()
        
        to_ip = to_port = from_ip = from_port = ''
        if self.advanced_from_ip.get_sensitive():
            from_ip = self.advanced_from_ip.get_text()
        if self.advanced_from_port.get_sensitive():
            from_port = self.advanced_from_port.get_text()
        if self.advanced_to_ip.get_sensitive():
            to_ip = self.advanced_to_ip.get_text()
        if self.advanced_to_port.get_sensitive():
            to_port = self.advanced_to_port.get_text()
        
        self._add(self.fw.get_profile(),                             # profile
                 self.advanced_rule_name.get_text(),                  # name
                 NUM2POLICY[self.advanced_policy.get_active()],       # policy
                 NUM2DIRECTION[self.advanced_direction.get_active()], # direction
                 NUM2PROTO[self.advanced_protocol.get_active()],      # protocol
                 from_ip,                                              # from IP
                 from_port,                                            # from port
                 to_ip,                                                # to IP
                 to_port,                                              # to port
                 insert,                                               # insert number
                 iface,                                                # interface
                 NUM2LOGGING[self.advanced_log.get_active()])         # logging        
        self.utils.print_rules(self.fw.get_rules())


class AppProfiles:
    """Load app profiles for Add Window from config files"""
    def __init__(self):
        # Is translated?
        self.all_apps = self._load_from_files()
        self.all_categories = self._get_all_categories()
    
    def _load_from_files(self):
        # Get all service config files
        apps = {}
        os.chdir(DIR_PROFILES)
        cfg = ConfigParser.ConfigParser()  
        cfg.read(glob.glob('*.*'))
        
        for section in cfg.sections():
            title = description = ports = categories = warning = ''
            
            if cfg.has_option(section, "title"):
                title = cfg.get(section, "title")
            if cfg.has_option(section, "description"):
                description = cfg.get(section, "description")
            if cfg.has_option(section, "ports"):
                ports = cfg.get(section, "ports")
            if cfg.has_option(section, "categories"):
                categories = cfg.get(section, "categories")
            if cfg.has_option(section, "warning"):
                warning = cfg.get(section, "warning")
                
            if title and description and ports and categories:
                if warning != '':
                    apps[_(title)] = [_(description), ports, _(categories), _(warning)]
                else:
                    apps[_(title)] = [_(description), ports, _(categories), '']
        return apps
    
    def _get_all_categories(self):
        all_categ = []
        for app in self.all_apps:
            category = self.all_apps[app][2]
            if not all_categ.count(category):
                all_categ.append(category)
        return all_categ
    
    def get_categories(self):
        categ = []
        for cat in self.all_categories:
            current_cat = cat.split(';')[0]
            if not categ.count(current_cat):
                categ.append(current_cat)
        categ.sort()
        return categ
    
    def get_subcategories(self, category):
        subcateg = []
        for cat in self.all_categories:
            current_cat = cat.split(';')[0]
            if category == current_cat:
                try:
                    current_subcat = cat.split(';')[1]
                except:
                    current_subcat = ''
                if not subcateg.count(current_subcat) and current_subcat:
                    subcateg.append(current_subcat)
        subcateg.sort()
        subcateg.insert(0, _("All"))
        return subcateg
    
    def get_apps_cat_subcat(self, cat, subcat):
        apps = []
        for app in self.all_apps:
            current_cat = self.all_apps[app][2].split(';')[0]
            try:
                current_subcat = self.all_apps[app][2].split(';')[1]
            except:
                current_subcat = ''
            if ( ( cat == current_cat and subcat == _("All") ) or
                 ( cat == current_cat and subcat == current_subcat ) ):
                apps.append(app)
        apps.sort()
        return apps
    
    def get_app(self, app):
        return self.all_apps[app]

