/*
 * Copyright (C) 2002-2009
 * Emmanuel Saracco <esaracco@users.labs.libre-entreprise.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "utils.h"
#include "project.h"
#include "application.h"

typedef struct _UCCookie UCCookie;
struct _UCCookie
{
  gchar *name;
  gchar *value;
  gchar *expires_a;
  time_t expires_tm;
};

typedef struct _UCCookiesSearchInfos UCCookiesSearchInfos;
struct _UCCookiesSearchInfos
{
  const gchar *key;
  GList *value;
};

static void
uc_cookies_free_callback (gpointer key, gpointer value, gpointer user_data);
static GList *uc_cookies_value_get (const gchar * key);
static void uc_cookies_get_value (const gchar * cookie,
				  const gchar * name, gchar ** key,
				  gchar ** value);

static void
uc_cookies_free_callback (gpointer key, gpointer value, gpointer user_data)
{
  GList *items = NULL;
  GList *list = (GList *) value;

  if (list)
    {
      items = g_list_first (list);
      while (items != NULL)
	{
	  UCCookie *c = (UCCookie *) items->data;

	  items = g_list_next (items);

	  g_free (c->name), c->name = NULL;
	  g_free (c->value), c->value = NULL;
	  g_free (c->expires_a), c->expires_a = NULL;
	  g_free (c), c = NULL;
	}
      g_list_free (list), list = NULL;
    }
}

void
uc_cookies_free (void)
{
  GHashTable *cookies = uc_project_get_cookies ();

  g_hash_table_foreach (cookies, uc_cookies_free_callback, NULL);
  g_hash_table_destroy (cookies), cookies = NULL;

  uc_project_set_cookies (cookies);
}

static void
uc_cookies_value_get_cb (gpointer key, gpointer value, gpointer data)
{
  UCCookiesSearchInfos *infos = (UCCookiesSearchInfos *) data;
  if (infos->value)
    return;

  if (uc_utils_memcasecmp (infos->key, (gchar *) key))
    infos->value = (GList *) value;
}

static GList *
uc_cookies_value_get (const gchar * key)
{
  GHashTable *cookies = uc_project_get_cookies ();
  UCCookiesSearchInfos *infos = NULL;
  GList *value = NULL;

  infos = g_new0 (UCCookiesSearchInfos, 1);
  infos->key = key;
  infos->value = NULL;

  g_hash_table_foreach (cookies, uc_cookies_value_get_cb, infos);

  value = infos->value;
  g_free (infos), infos = NULL;

  return value;
}

gchar *
uc_cookies_get_header_field (gchar * path)
{
  GList *list = NULL;
  gchar *ret = NULL;
  time_t now = time (NULL);

  if ((list = uc_cookies_value_get (path)))
    {
      GList *item = NULL;
      GString *str = g_string_new ("Cookie:");

      item = g_list_first (list);
      while (item != NULL)
	{
	  UCCookie *c = (UCCookie *) item->data;

	  if (c->expires_tm > 0 && c->expires_tm <= now)
	    ;
	  else
	    g_string_append_printf (str, " %s=%s;", c->name, c->value);

	  item = g_list_next (item);
	}

      if (strchr (str->str, '='))
	{
	  g_string_erase (str, str->len - 1, 1);
	  g_string_append (str, "\r\n");
	  ret = str->str;
	  g_string_free (str, FALSE);
	}
      else
	g_string_free (str, TRUE);
    }

  if (!ret)
    ret = g_strdup ("");

  return ret;
}

/**
 * uc_cookies_get_value:
 *
 *
 * If @name is %NULL, we fill @key and @value. If @name is not %NULL we look
 * for @name and fill @value (@key is not filled)
 */
static void
uc_cookies_get_value (const gchar * cookie, const gchar * name,
		      gchar ** key, gchar ** value)
{
  gchar **fields = NULL;
  gchar **name_value = NULL;
  guint i = 0;

  fields = g_strsplit (cookie, ";", 0);
  if (!name)
    {
      g_strstrip (fields[0]);
      name_value = g_strsplit (fields[0], "=", 2);
      *key = g_strdup (name_value[0]);
      *value = g_strdup (name_value[1]);
      g_strfreev (name_value), name_value = NULL;
    }
  else
    {
      for (i = 0; fields[i] != NULL && *value == NULL; i++)
	{
	  g_strstrip (fields[i]);
	  name_value = g_strsplit (fields[i], "=", 2);
	  if (name_value && name_value[0] && name_value[1] &&
	      !strcasecmp (name_value[0], name))
	    *value = g_strdup (name_value[1]);
	  g_strfreev (name_value), name_value = NULL;
	}
    }
  g_strfreev (fields), fields = NULL;
}

void
uc_cookies_add (const UCLinkProperties * prop, const gchar * cook)
{
  gchar *key = NULL;
  gchar *value = NULL;
  gchar *expires = NULL;
  time_t expires_tm;
  time_t now = time (NULL);
  gchar *path = NULL;
  gchar *cookie = NULL;
  GList *list = NULL;
  GList *item = NULL;
  gboolean found = FALSE;
  gboolean bad_cookie = FALSE;
  GHashTable *cookies = uc_project_get_cookies ();

  /* add a default path "/" if there is no path specified 
   * FIXME: be really case insensitive */
  if (!strstr (cook, "ath=/") && !strstr (cook, "ATH=/"))
    cookie = g_strconcat (cook, "; path=/", NULL);
  else
    cookie = g_strdup (cook);

  /* get path */
  uc_cookies_get_value (cookie, "path", NULL, &path);

  /* get expiration */
  uc_cookies_get_value (cookie, "expires", NULL, &expires);

  /* get cookie */
  uc_cookies_get_value (cookie, NULL, &key, &value);

  if ((list = (GList *) g_hash_table_lookup (cookies, path)))
    {
      item = g_list_first (list);
      while (item != NULL && !found)
	{
	  UCCookie *c = (UCCookie *) item->data;

	  item = g_list_next (item);

	  if (!strcmp (c->name, key))
	    {
	      found = TRUE;

	      /* if value is empty, or cookie has expires */
	      if (!strlen (value) ||
		  (expires && (uc_utils_http_atotm (expires) <= now)))
		{
		  if (uc_project_get_cookies_warn_deleted ()
		      &&
		      !uc_application_cookie_warning_dialog_show
		      (prop->h_name, prop->path,
		       _("The following cookie will be <b>deleted</b>"), &key,
		       &value, &path, &expires, UC_COOKIES_ACTION_DELETE))
		    goto exit_refused;

		  g_free (c->name), c->name = NULL;
		  g_free (c->value), c->value = NULL;
		  g_free (c->expires_a), c->expires_a = NULL;
		  list = g_list_remove (list, c);
		  g_free (c), c = NULL;

		  g_hash_table_replace (cookies, g_strdup (path), list);

		  goto exit_refused;
		}
	      /* update cookie content */
	      else
		{
		  if (uc_project_get_cookies_warn_updated () &&
		      !uc_application_cookie_warning_dialog_show
		      (prop->h_name, prop->path,
		       _("The following cookie will be <b>updated</b>"), &key,
		       &value, &path, &expires, UC_COOKIES_ACTION_UPDATE))
		    goto exit_refused;

		  g_free (c->value), c->value = NULL;
		  c->value = value;
		  g_free (c->expires_a), c->expires_a = NULL;
		  c->expires_tm = 0;
		  if ((expires_tm = uc_utils_http_atotm (expires)) > 0)
		    {
		      c->expires_a = strdup (expires);
		      c->expires_tm = expires_tm;
		    }
		  else if (expires)
		    bad_cookie = TRUE;
		}
	    }
	}
    }

  expires_tm = uc_utils_http_atotm (expires);

  /* add cookie */
  if (!found && (!expires || (uc_utils_http_atotm (expires) > now)))
    {
      UCCookie *c = NULL;

      if (uc_project_get_cookies_warn_added () &&
	  !uc_application_cookie_warning_dialog_show (prop->h_name,
						      prop->path,
						      _
						      ("The following cookie will be <b>added</b>"),
						      &key, &value, &path,
						      &expires,
						      UC_COOKIES_ACTION_ADD))
	goto exit_refused;

      expires_tm = uc_utils_http_atotm (expires);

      /* if user inputed a expired date */
      if (expires && expires_tm <= now)
	{
	  if (expires_tm < 0)
	    bad_cookie = TRUE;
	  goto exit_refused;
	}

      c = g_new0 (UCCookie, 1);
      c->name = key;
      c->value = value;
      c->expires_a = NULL;
      c->expires_tm = 0;
      if (expires_tm > 0)
	{
	  c->expires_a = g_strdup (expires);
	  c->expires_tm = expires_tm;
	}

      list = g_list_prepend (list, c);
    }
  /* the cookie to add has already expired */
  else if (!found && (expires && expires_tm <= now))
    {
      if (expires_tm < 0)
	bad_cookie = TRUE;
      goto exit_refused;
    }
  /* the cookie was updated, so key (cookie name) remain the same */
  else
    g_free (key), key = NULL;

  g_hash_table_replace (cookies, path, list);

  goto exit_ok;

exit_refused:

  g_free (key), key = NULL;
  g_free (value), value = NULL;
  g_free (path), path = NULL;

exit_ok:

  if (bad_cookie)
    uc_application_dialog_show (_
				("The expiration date for this cookie was incorrect. It has been refused."),
				GTK_MESSAGE_WARNING);

  g_free (expires), expires = NULL;
  g_free (cookie), cookie = NULL;

  uc_project_set_cookies (cookies);
}
