/*******************************************************************************#
#           guvcview              http://guvcview.berlios.de                    #
#                                                                               #
#           Paulo Assis <pj.assis@gmail.com>                                    #
#                                                                               #
# This program is free software; you can redistribute it and/or modify          #
# it under the terms of the GNU General Public License as published by          #
# the Free Software Foundation; either version 2 of the License, or             #
# (at your option) any later version.                                           #
#                                                                               #
# This program is distributed in the hope that it will be useful,               #
# but WITHOUT ANY WARRANTY; without even the implied warranty of                #
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 #
# GNU General Public License for more details.                                  #
#                                                                               #
# You should have received a copy of the GNU General Public License             #
# along with this program; if not, write to the Free Software                   #
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA     #
#                                                                               #
********************************************************************************/

#include <glib/gprintf.h>
#include <string.h>
#include "audio_effects.h"
#include "ms_time.h"

/*--------------------------- sound callback ------------------------------*/
int 
recordCallback (const void *inputBuffer, void *outputBuffer,
	unsigned long framesPerBuffer,
	const PaStreamCallbackTimeInfo* timeInfo,
	PaStreamCallbackFlags statusFlags,
	void *userData )
{
	struct paRecordData *data = (struct paRecordData*)userData;
	const SAMPLE *rptr = (const SAMPLE*)inputBuffer;
	int i;
	
	data->framesPerBuffer = framesPerBuffer;
	int numSamples= data->framesPerBuffer * data->channels;

	/*set to zero on paComplete*/    
	data->streaming=1;

	if( inputBuffer == NULL )
	{
		for( i=0; i<numSamples; i++ )
		{
			data->recordedSamples[data->sampleIndex] = 0;/*silence*/
			data->sampleIndex++;
		}
	}
	else
	{
		for( i=0; i<numSamples; i++ )
		{
			data->recordedSamples[data->sampleIndex] = *rptr++;
			data->sampleIndex++;
		}
	}

	data->numSamples += numSamples;
	//if (data->numSamples > (data->maxIndex-(2*numSamples)))
	if(data->numSamples > data->tresh)
	{
		//primary buffer near limit (don't wait for overflow)
		//or video capture stopped
		//copy data to secondary buffer and restart primary buffer index
		//the buffer is only writen every 1sec or so, plenty of time for a read to complete,
		//anyway lock a mutex on the buffer just in case a read operation is still going on.
		// This is not a good idea as it may cause data loss
		//but since we sould never have to wait, it shouldn't be a problem.
		//printf("capvid: %d\n",data->capVid);
		g_mutex_lock( data->mutex );
			data->snd_numSamples = data->numSamples;
			data->snd_numBytes = data->numSamples*sizeof(SAMPLE);
			memcpy(data->vid_sndBuff, data->recordedSamples ,data->snd_numBytes);
			data->a_ts = ns_time();
			/*flags that secondary buffer as data (can be saved to file)*/
			data->audio_flag=1;
		g_mutex_unlock( data->mutex );
		data->sampleIndex=0;
		data->numSamples = 0;
	}

	if(data->capVid) return (paContinue); /*still capturing*/
	else 
	{
		/*recording stopped*/
		if(!(data->audio_flag) && data->streaming) 
		{
			/*need to copy remaining audio to secondary buffer*/
			g_mutex_lock( data->mutex);
				data->snd_numSamples = data->numSamples;
				data->snd_numBytes = data->numSamples*sizeof(SAMPLE);
				memcpy(data->vid_sndBuff, data->recordedSamples ,data->snd_numBytes);
				data->a_ts = ns_time();
				/*flags that secondary buffer as data (can be saved to file)*/
				data->audio_flag=1;
			g_mutex_unlock( data->mutex);
			data->sampleIndex=0;
			data->numSamples = 0;
		}
		data->streaming=0;
		return (paComplete);
	}
}

void
set_sound (struct GLOBAL *global, struct paRecordData* data) 
{
	int totalFrames;
	int MP2Frames=0;
	int numSamples;
	
	if(global->Sound_SampRateInd==0)
		global->Sound_SampRate=global->Sound_IndexDev[global->Sound_UseDev].samprate;/*using default*/
	
	if(global->Sound_NumChanInd==0) 
	{
		/*using default if channels <3 or stereo(2) otherwise*/
		global->Sound_NumChan=(global->Sound_IndexDev[global->Sound_UseDev].chan < 3) ? 
			global->Sound_IndexDev[global->Sound_UseDev].chan : 2;
	}
	
	data->samprate = global->Sound_SampRate;
	data->channels = global->Sound_NumChan;
	data->numsec = global->Sound_NumSec;
	data->MPEG_Frame_size = 1152; /*Layer 2 Mpeg Audio: 1 frame is 1152 samples*/
	
	/* setting maximum buffer size*/
	totalFrames = data->numsec * data->samprate;
	numSamples = totalFrames * data->channels;

	if(data->samprate < 32000)
		data->tresh = (data->MPEG_Frame_size * 3) * data->channels;
	else 
		data->tresh = (data->MPEG_Frame_size * 7) * data->channels;
	/*round to libtwolame Frames (1 Frame = 1152 samples)*/
	MP2Frames=numSamples / data->MPEG_Frame_size;
	numSamples=MP2Frames * data->MPEG_Frame_size;

	if(numSamples < (data->tresh + (4 * data->MPEG_Frame_size))) 
		numSamples = data->tresh + (4 * data->MPEG_Frame_size);

	data->input_type = PA_SAMPLE_TYPE;
	data->mp2Buff = NULL;
	
	data->snd_numBytes = numSamples * sizeof(SAMPLE);
	
	data->recordedSamples = g_new0(SAMPLE, numSamples);
	data->maxIndex = numSamples;
	data->sampleIndex = 0;
	
	data->audio_flag = 0;
	data->flush = 0;
	data->streaming = 0;
	data->stream = NULL;
	
	/*secondary shared buffer*/
	data->vid_sndBuff = g_new0(SAMPLE, numSamples);
	/*buffer for video PCM 16 bits*/
	data->vid_sndBuff1=NULL;
	/*set audio device to use*/
	data->inputParameters.device = global->Sound_IndexDev[global->Sound_UseDev].id; /* input device */
}

int
init_sound(struct paRecordData* data)
{
	PaError err = paNoError;

	switch(data->api)
	{
#ifdef PULSEAUDIO
		case PULSE:
			if(pulse_init_audio(data))
				goto error;
			break;
#endif
		case PORT:
		default:
			//err = Pa_Initialize();
			//if( err != paNoError ) goto error;
			if(data->stream)
			{
				if( !(Pa_IsStreamStopped( data->stream )))
				{
					Pa_AbortStream( data->stream );
					Pa_CloseStream( data->stream );
					data->stream = NULL;
				}
			}
				
			/* Record for a few seconds. */
			data->inputParameters.channelCount = data->channels;
			data->inputParameters.sampleFormat = PA_SAMPLE_TYPE;
			if (Pa_GetDeviceInfo( data->inputParameters.device ))
				data->inputParameters.suggestedLatency = Pa_GetDeviceInfo( data->inputParameters.device )->defaultHighInputLatency;
			else
				data->inputParameters.suggestedLatency = DEFAULT_LATENCY_DURATION/1000.0;
			data->inputParameters.hostApiSpecificStreamInfo = NULL; 
	
			/*---------------------------- Record some audio. ----------------------------- */
	
			err = Pa_OpenStream(
				&data->stream,
				&data->inputParameters,
				NULL,                  /* &outputParameters, */
				data->samprate,
				data->MPEG_Frame_size,            // buffer size = Mpeg frame size (1152 samples)
				//paFramesPerBufferUnspecified,       // buffer Size - set by portaudio
				//paClipOff | paDitherOff, 
				paNoFlag,      /* PaNoFlag - clip and dhiter*/
				recordCallback, /* sound callback */
				data ); /* callback userData */
	
			if( err != paNoError ) goto error;
	
			err = Pa_StartStream( data->stream );
			if( err != paNoError ) goto error; /*should close the stream if error ?*/
			break;
	}
	
	/*sound start time - used to sync with video*/
	data->snd_begintime = ns_time();

	return (0);
error:
	g_printerr("An error occured while starting portaudio\n" );
	g_printerr("Error number: %d\n", err );
	g_printerr("Error message: %s\n", Pa_GetErrorText( err ) ); 
	data->streaming=0;
	data->flush=0;
	if(data->api < 1)
	{
		if(data->stream) Pa_AbortStream( data->stream );
	}
	g_free( data->recordedSamples );
	data->recordedSamples=NULL;
	g_free(data->vid_sndBuff);
	data->vid_sndBuff=NULL;

	return(-1);
} 

int
close_sound (struct paRecordData *data) 
{
	int err =0;
	data->capVid = 0;
	/*make sure we stoped streaming */
	int stall = wait_ms( &data->streaming, FALSE, 10, 50 );
	if(!(stall)) 
	{
		g_printerr("WARNING:sound capture stall (still streaming(%d)) \n",
			data->streaming);
			data->streaming = 0;
	}
	/*stops and closes the audio stream*/
	if(data->stream)
	{
		if(Pa_IsStreamActive( data->stream ) > 0)
		{
			g_printf("Aborting audio stream\n");
			err = Pa_AbortStream( data->stream );
		}
		else
		{
			g_printf("Stoping audio stream\n");
			err = Pa_StopStream( data->stream );
		}
		if( err != paNoError ) goto error;
	}
	if(data->api < 1)
	{
		g_printf("Closing audio stream...\n");
		err = Pa_CloseStream( data->stream );
		if( err != paNoError ) goto error; 
	}
	data->stream = NULL;
	if(data->audio_flag) 
		g_printerr("Droped %i bytes of audio data\n", data->snd_numBytes);
	data->audio_flag=0;
	data->flush = 0;

	/*---------------------------------------------------------------------*/
	/*make sure no operations are performed on the buffers*/
	g_mutex_lock( data->mutex);
		/*free primary buffer*/
		g_free( data->recordedSamples  );
		data->recordedSamples=NULL;
		g_free(data->vid_sndBuff);
		data->vid_sndBuff = NULL;
	
		g_free(data->mp2Buff);
		data->mp2Buff = NULL;
		g_free(data->vid_sndBuff1);
		data->vid_sndBuff1 = NULL;
	g_mutex_unlock( data->mutex );
	
	return (0);
error:  
	g_printerr("An error occured while closing the portaudio stream\n" );
	g_printerr("Error number: %d\n", err );
	g_printerr("Error message: %s\n", Pa_GetErrorText( err ) );
	data->flush=0;
	data->audio_flag=0;
	data->streaming=0;
	g_mutex_lock( data->mutex);
		g_free( data->recordedSamples );
		data->recordedSamples=NULL;
		if(data->api < 1) 
		{
			Pa_CloseStream( data->stream );
		}
		data->stream = NULL;
		g_free(data->vid_sndBuff);
		data->vid_sndBuff = NULL;
		
		g_free(data->mp2Buff);
		data->mp2Buff = NULL;
		g_free(data->vid_sndBuff1);
		data->vid_sndBuff1 = NULL;
	g_mutex_unlock( data->mutex );
	return(-1);
}

/* saturate float samples to int16 limits*/
static gint16 clip_int16 (float in)
{
	in = (in < -32768) ? -32768 : (in > 32767) ? 32767 : in;
	
	return ((gint16) in);
}

void Float2Int16 (struct paRecordData* data)
{
	if (data->vid_sndBuff1 == NULL) 
		data->vid_sndBuff1 = g_new0(gint16, data->maxIndex);
	
	float res = 0.0;
	int samp = 0;
	for(samp=0; samp < data->snd_numSamples; samp++)
	{
		res = data->vid_sndBuff[samp] * 32768 + 385;
		/*clip*/
		data->vid_sndBuff1[samp] = clip_int16(res);
	}
}

