// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The DeviceList represents the central data structure of
//                  the application. Its contents are displayed in the main
//                  window.
// ****************************************************************************

#include <float.h>

#include <QString>
#include <QList>

#include "toolconstants.h"

#include "common.h"
#include "config.h"
#include "mainwindow.h"
#include "main.h"

// ---------------------------
//         t_Device
// ---------------------------

static const int DEVICE_MIN_RUNNING_SECONDS = 20;  // The minmum number of acquisition seconds before any estimation can be made

void t_Device::Initialise (void)
{
   static bool Initialised = false;

   if (!Initialised)
   {
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_SERNR_MATCH_MODEL_MISMATCH ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_SERNR_MATCH_LENGTH_MISMATCH))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_BAD_STATE                  ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_BAD_ABORTREASON            ))
      CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_DEVICE_NOT_CLEAN                  ))
      Initialised = true;
   }

//   this->SerialNumber      = QString();
//   this->LinuxDevice       = QString();
//   this->Model             = QString();
   this->Local                = false;
   this->SectorSize           = 0;
   this->SectorSizePhys       = 0;
   this->Size                 = 0;
   this->Removable            = false;

   this->State                = Idle;
//   this->Acquisition.Message       = QString();
   this->AbortRequest         = false;
   this->AbortReason          = None;
   this->DeleteAfterAbort     = false;
   this->pFifoRead            = NULL;
   this->pThreadRead          = NULL;
   this->pFileSrc             = NULL;
   this->CurrentReadPos       = 0;
//   this->BadSectors
//   this->BadSectorsVerify
   this->FallbackMode         = false;
   this->StartTimestamp       = QDateTime();
   this->StartTimestampVerify = QDateTime();
   this->StopTimestamp        = QDateTime();

   this->pThreadWrite         = NULL;
   this->CurrentWritePos      = 0;
   this->CurrentVerifyPos     = 0;
   this->pFifoWrite           = NULL;

   this->pThreadHash          = NULL;
   this->pFifoHashIn          = NULL;
   this->pFifoHashOut         = NULL;
   memset (&this->MD5Digest         , 0, sizeof(this->MD5Digest         ));
   memset (&this->MD5DigestVerify   , 0, sizeof(this->MD5DigestVerify   ));
   memset (&this->SHA256Digest      , 0, sizeof(this->SHA256Digest      ));
   memset (&this->SHA256DigestVerify, 0, sizeof(this->SHA256DigestVerify));

   this->pFifoCompressIn      = NULL;
   this->pFifoCompressOut     = NULL;
   this->FifoMaxBlocks        = 0;
   this->FifoBlockSize        = 0;

//   this->Acquisition.Path          = QString();
//   this->Acquisition.ImageFilename = QString();
//   this->Acquisition.InfoFilename  = QString();
   this->Acquisition.Format        = t_File::NotSet;
   this->Acquisition.CalcHashes    = false;
   this->Acquisition.VerifySource  = false;
//   this->Acquisition.CaseNumber    = QString();
//   this->Acquisition.EvidenceNumber= QString();
//   this->Acquisition.Examiner      = QString();
//   this->Acquisition.Description   = QString();
//   this->Acquisition.Notes         = QString();

   this->Info.SetDevice(this);

   this->PrevTimestamp     = QTime();
   this->PrevSpeed         = 0.0;
   this->Checked           = false;
}

void t_Device::InitialiseDeviceSpecific (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                                         quint64 SectorSize, quint64 SectorSizePhys, quint64 Size)
{                                        //lint !e578: Declaration of symbol 'Xxx' hides symbol 't_Device::Xxx'
   this->SerialNumber   = SerialNumber;
   this->LinuxDevice    = LinuxDevice;
//   this->LinuxDevice    = "/data/virtualbox/knoppicilin.iso";
   this->Model          = Model;

   this->SectorSize     = SectorSize;
   this->SectorSizePhys = SectorSizePhys;
   this->Size           = Size;
//   this->Size           = std::min (10737418240ULL, Size;  // For faster testing (we make guymager believe that the device is smaller)
//   this->Size           = std::min (1073741824ULL, Size);  // For faster testing (we make guymager believe that the device is smaller)
}

t_Device::t_Device (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                    quint64 SectorSize, quint64 SectorSizePhys, quint64 Size)
{                   //lint !e578: Declaration of symbol 'Xxx' hides symbol 't_Device::Xxx'
   Initialise ();
   InitialiseDeviceSpecific (SerialNumber, LinuxDevice, Model, SectorSize, SectorSizePhys, Size);
}

t_Device::t_Device (const QString &SerialNumber, const PedDevice *pPedDev)
{                   //lint !e578: Declaration of symbol 'Xxx' hides symbol 't_Device::Xxx'
   Initialise ();
   InitialiseDeviceSpecific (SerialNumber, pPedDev->path, pPedDev->model,
                                           (quint64) pPedDev->sector_size, (quint64) pPedDev->phys_sector_size,
                                           (quint64) pPedDev->length * (quint64) pPedDev->sector_size);
}

t_Device::~t_Device()
{
   if (pThreadRead   || pFifoRead    ||
       pThreadHash   || pFifoHashIn  ||
       pThreadWrite  || pFifoHashOut ||
       pFileSrc      || pFifoWrite )
      CHK_EXIT (ERROR_DEVICE_NOT_CLEAN)

   pThreadRead  = NULL; pFifoRead    = NULL;
   pThreadHash  = NULL; pFifoHashIn  = NULL;
   pThreadWrite = NULL; pFifoHashOut = NULL;
   pFileSrc     = NULL; pFifoWrite   = NULL;
}

bool t_Device::HasHashThread (void) const
{
   return Acquisition.CalcHashes && CONFIG (UseSeparateHashThread);
}

bool t_Device::HasCompressionThreads (void) const
{
   return (CONFIG (CompressionThreads) > 0) && ((Acquisition.Format == t_File::EWF) ||
                                                (Acquisition.Format == t_File::AFF));
}


//lint -save -e818 pDevice could have declared as pointing to const

QVariant t_Device::GetSerialNumber (t_pDevice pDevice)
{
   return pDevice->SerialNumber;
}

QVariant t_Device::GetLinuxDevice (t_pDevice pDevice)
{
   return pDevice->LinuxDevice;
}

QVariant t_Device::GetModel (t_pDevice pDevice)
{
   return pDevice->Model;
}

QVariant t_Device::GetState (t_pDevice pDevice)
{
   QString State;

   switch (pDevice->State)
   {
      case Idle        : if (pDevice->Local)
                             State = t_MainWindow::tr("Local device");
                         else State = t_MainWindow::tr("Idle");
                         break;
      case Acquire      : State = t_MainWindow::tr("Acquisition running" ); break;
      case Verify       : State = t_MainWindow::tr("Verification running"); break;
      case AcquirePaused: State = t_MainWindow::tr("Device disconnected, acquisition paused" ); break;
      case VerifyPaused : State = t_MainWindow::tr("Device disconnected, verification paused"); break;
      case Cleanup      : State = t_MainWindow::tr("Cleanup" ); break;
      case Finished     : if (pDevice->Acquisition.VerifySource)
                          {
                             if (HashMD5Match   (&pDevice->MD5Digest   , &pDevice->MD5DigestVerify   ) &&
                                 HashSHA256Match(&pDevice->SHA256Digest, &pDevice->SHA256DigestVerify))
                                  State = t_MainWindow::tr("Finished - hashes verified & ok"      );
                             else State = t_MainWindow::tr("Finished - hash verification mismatch");
                          }
                          else
                          {
                             State = t_MainWindow::tr("Finished");
                          }
                          break;
      case Aborted : switch (pDevice->AbortReason)
                     {
                        case t_Device::None                : State = t_MainWindow::tr("Aborted - Error: Reason is 'none'" ); break;
                        case t_Device::UserRequest         : State = t_MainWindow::tr("Aborted by user" );                   break;
                        case t_Device::ThreadWriteFileError: State = t_MainWindow::tr("Aborted - Image file write error" );  break;
                        case t_Device::ThreadReadFileError : State = t_MainWindow::tr("Aborted - Device read error" );       break;
                        default                            : CHK_EXIT (ERROR_DEVICE_BAD_ABORTREASON)
                     }
                     break;
      default      : CHK_EXIT (ERROR_DEVICE_BAD_STATE)
   }
   if (pDevice->State != Aborted)
   {
      QString Msg;
      CHK_EXIT (pDevice->GetMessage (Msg))
      if (!Msg.isEmpty())
         State += " - " + Msg;
   }

   return State;
}

QVariant t_Device::GetSectorSize (t_pDevice pDevice)
{
   return pDevice->SectorSize;
}

QVariant t_Device::GetSectorSizePhys (t_pDevice pDevice)
{
   return pDevice->SectorSizePhys;
}

QVariant t_Device::GetSize (t_pDevice pDevice)
{
   return pDevice->Size;
}


QVariant t_Device::GetSizeHumanFrac (t_pDevice pDevice, bool SI, int FracLen, int UnitThreshold)
{
   QString      SizeHuman;
   const char *pUnit;
   double       Sz;
   double       Divisor;

   if (SI)
        Divisor = 1000.0;
   else Divisor = 1024.0;

   if (UnitThreshold == AutoUnitThreshold)
      UnitThreshold = (int) Divisor;

   Sz = pDevice->Size;
   pUnit = "Byte";
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "kB" : "KiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "MB" : "MiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "GB" : "GiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "TB" : "TiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "PB" : "PiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "EB" : "EiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "ZB" : "ZiB"; }
   if (Sz >= UnitThreshold) { Sz = Sz / Divisor; pUnit = SI ? "YB" : "YiB"; }

   if (FracLen == AutoFracLen)
   {
      if      (Sz >= 100) FracLen = 0;
      else if (Sz >= 10 ) FracLen = 1;
      else                FracLen = 2;
   }
   SizeHuman = MainGetpNumberLocale()->toString (Sz, 'f', FracLen) + pUnit;

   return SizeHuman;
}

QVariant t_Device::GetSizeHuman (t_pDevice pDevice)
{
   return GetSizeHumanFrac (pDevice, true, 1);
}

QVariant t_Device::GetBadSectorCount (t_pDevice pDevice)
{
   if (!pDevice->StartTimestamp.isNull())
        return pDevice->GetBadSectorCount (false);
   else return "";
}

static void DeviceGetProgress (t_pDevice pDevice, quint64 *pCurrent=NULL, quint64 *pTotal=NULL)
{
   quint64 Current, Total;

   if (pDevice->Acquisition.VerifySource)
   {
      Total = 2 * pDevice->Size;
      if (pDevice->StartTimestampVerify.isNull())
           Current = pDevice->GetCurrentWritePos ();
      else Current = pDevice->GetCurrentVerifyPos() + pDevice->Size;
   }
   else
   {
      Total   = pDevice->Size;
      Current = pDevice->GetCurrentWritePos();
   }
   if (pTotal  ) *pTotal   = Total;
   if (pCurrent) *pCurrent = Current;
}

QVariant t_Device::GetProgress (t_pDevice pDevice)
{
   quint64 Current, Total;

   if (!pDevice->StartTimestamp.isNull())
   {
      DeviceGetProgress (pDevice, &Current, &Total);
      return (double) Current / Total;
   }

   return -DBL_MAX;
}

QVariant t_Device::GetCurrentSpeed (t_pDevice pDevice)
{
   QString Result;
   QTime   Now;
   double  MBs;
   double  Speed;
   int     MilliSeconds;
   quint64 CurrentPos;

   if ((pDevice->State == Acquire) ||  // Don't display anything if no acquisition is running
       (pDevice->State == Verify ))
   {
      Now = QTime::currentTime();
      DeviceGetProgress (pDevice, &CurrentPos);
      if (!pDevice->PrevTimestamp.isNull())
      {
         MilliSeconds = pDevice->PrevTimestamp.msecsTo (Now); // As this fn is called within 1s-interval, time_t would not be precise enough
         if (MilliSeconds > 0)
         {
            MBs   = (double) (CurrentPos - pDevice->PrevPos) / BYTES_PER_MEGABYTE;
            Speed = (1000.0*MBs) / MilliSeconds;
            pDevice->PrevSpeed = (pDevice->PrevSpeed + Speed) / 2.0;
         }
         Result = MainGetpNumberLocale()->toString (pDevice->PrevSpeed, 'f', 2);
      }
      pDevice->PrevTimestamp = Now;
      pDevice->PrevPos       = CurrentPos;
   }

   return Result;
}

QVariant t_Device::GetAverageSpeed (t_pDevice pDevice)
{
   QLocale Locale;
   QString Result;
   double  MBs;
   int     Seconds;
   quint64 CurrentPos;

   if (pDevice->StartTimestamp.isNull())
      return QString();

   if (pDevice->StopTimestamp.isNull()) // As long as the stop timestamp is Null, the acquisition is still running
        Seconds = pDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());
   else Seconds = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);

   if ((pDevice->StopTimestamp.isNull()) && (Seconds < DEVICE_MIN_RUNNING_SECONDS))
      return "--";

   DeviceGetProgress (pDevice, &CurrentPos);
   MBs = ((double) CurrentPos) / BYTES_PER_MEGABYTE;
   Result = Locale.toString (MBs/Seconds, 'f', 2);

   return Result;
}

QVariant t_Device::GetRemaining (t_pDevice pDevice)
{
   QString Result;
   char    Buff[10];
   int     TotalSeconds;
   time_t  Now;
   int     hh, mm, ss;
   quint64 Current, Total;

   if ((pDevice->State == Acquire) || // Don't display anything if no acquisition is running
       (pDevice->State == Verify ))
   {
      time (&Now);
      TotalSeconds = pDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());
      if (TotalSeconds < DEVICE_MIN_RUNNING_SECONDS)
      {
         Result = "--";
      }
      else
      {
         DeviceGetProgress (pDevice, &Current, &Total);
         ss  = (int) ((double)Total / Current * TotalSeconds); // Estimated total time
         ss -= TotalSeconds;                                    // Estimated remaining time
         hh = ss / SECONDS_PER_HOUR;   ss %= SECONDS_PER_HOUR;
         mm = ss / SECONDS_PER_MINUTE; ss %= SECONDS_PER_MINUTE;
         snprintf (Buff, sizeof(Buff), "%02d:%02d:%02d", hh, mm, ss);
         Result = Buff;
      }
   }

   return Result;
}

static inline int DeviceFifoUsage (t_pFifo pFifo)
{
   int Usage;
   CHK_EXIT (pFifo->Usage(Usage))
   return Usage;
}

QVariant t_Device::GetFifoStatus (t_pDevice pDevice)
{
   QString Result;

   if ((pDevice->State == Acquire) ||  // Don't display anything if no acquisition is running
       (pDevice->State == Verify ))
   {
      if (!pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 w") .arg (DeviceFifoUsage(pDevice->pFifoRead));
      }
      else if (!pDevice->HasHashThread() && pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 c %2 w") .arg (DeviceFifoUsage(pDevice->pFifoRead      ))
                                          .arg (DeviceFifoUsage(pDevice->pFifoCompressOut));
      }
      else if (pDevice->HasHashThread() && !pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 m %2 w") .arg (DeviceFifoUsage(pDevice->pFifoRead   ))
                                          .arg (DeviceFifoUsage(pDevice->pFifoHashOut));
      }
      else if (pDevice->HasHashThread() && pDevice->HasCompressionThreads())
      {
         Result = QString ("r %1 h %2 c %3 w") .arg (DeviceFifoUsage(pDevice->pFifoRead       ))
                                               .arg (DeviceFifoUsage(pDevice->pFifoHashOut    ))
                                               .arg (DeviceFifoUsage(pDevice->pFifoCompressOut));
      }
   }

   return Result;
}


//lint -restore


// ---------------------------
//        t_DeviceList
// ---------------------------

t_DeviceList::t_DeviceList(void)
   :QList<t_pDevice>()
{
   static bool Initialised = false;

   if (!Initialised)
   {
      Initialised = true;
      qRegisterMetaType<t_pDeviceList>("t_pDeviceList");
   }
}

t_DeviceList::~t_DeviceList()
{
   int i;

   for (i=0; i<count(); i++)
      delete at(i);
}

t_pDevice t_DeviceList::AppendNew (const QString &SerialNumber, const PedDevice *pPedDev)
{
   t_pDevice pDev;

   pDev = new t_Device (SerialNumber, pPedDev);
   append (pDev);

   return pDev;
}

t_pDevice t_DeviceList::AppendNew (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                                   quint64 SectorSize, quint64 SectorSizePhys, quint64 Size)
{
   t_pDevice pDev;

   pDev = new t_Device (SerialNumber, LinuxDevice, Model, SectorSize, SectorSizePhys, Size);
   append (pDev);

   return pDev;
}

//t_pDevice t_DeviceList::SearchLinuxDevice  (const QString &LinuxDevice)
//{
//   t_pDevice pDev;
//   int        i;
//
//   for (i=0; i<count(); i++)
//   {
//      pDev = at(i);
//      if (pDev->LinuxDevice == LinuxDevice)
//         return pDev;
//   }
//   return NULL;
//}
//
//t_pDevice t_DeviceList::SearchSerialNumber (const QString &SerialNumber)
//{
//   t_pDevice pDev;
//   int        i;
//
//   for (i=0; i<count(); i++)
//   {
//      pDev = at(i);
//      if (pDev->SerialNumber == SerialNumber)
//         return pDev;
//   }
//   return NULL;
//}

APIRET t_DeviceList::MatchDevice (t_pcDevice pDevCmp, t_pDevice &pDevMatch)
{
   bool Found = false;
   int  i;

   for (i=0; (i<count()) && !Found; i++)
   {
      pDevMatch = at(i);
      if (pDevMatch->SerialNumber.isEmpty())
      {
         Found = (pDevCmp->SerialNumber.isEmpty() && (pDevMatch->State != t_Device::AcquirePaused) &&
                                                     (pDevMatch->State != t_Device::VerifyPaused ) &&
                                                     (pDevMatch->Model == pDevCmp->Model         ) &&
                                                     (pDevMatch->Size  == pDevCmp->Size          ));
      }
      else
      {
//         Found = (pDevMatch->SerialNumber == pDevCmp->SerialNumber);
//         if (Found)
//         {                                                                                            // If the serial numbers match, the whole
//            if (pDevMatch->Model != pDevCmp->Model) return ERROR_DEVICE_SERNR_MATCH_MODEL_MISMATCH;   // rest must match as well. Otherwise,
//            if (pDevMatch->Size  != pDevCmp->Size ) return ERROR_DEVICE_SERNR_MATCH_LENGTH_MISMATCH;  // something very strange is going on...
//         }

// The comparision above cannot be used, because there are certain devices, that behave as if 2 devices aer connected when
// plugged in - with the same serial number!! Example: I once had a memory stick from Intuix, that reported 2 devices: The
// memory stick itself and a CD-Rom memory stick itself and a CD-Rom containing device drivers for some strange OS. Both
// devices had the same serial number. That's why the check now also includes the size and the model.

         Found = ((pDevMatch->SerialNumber == pDevCmp->SerialNumber) &&
                  (pDevMatch->Size         == pDevCmp->Size )        &&
                  (pDevMatch->Model        == pDevCmp->Model));
      }
   }

   if (!Found)
      pDevMatch = NULL;

   return NO_ERROR;
}

const char *t_Device::StateStr (void)
{
   const char *pStr;
   switch (State)
   {
      case Idle         : pStr = "Idle";            break;
      case Acquire      : pStr = "Acquire";         break;
      case AcquirePaused: pStr = "AcquirePaused";   break;
      case Verify       : pStr = "Verify";          break;
      case VerifyPaused : pStr = "VerifyPaused";    break;
      case Cleanup      : pStr = "Cleanup";         break;
      case Finished     : pStr = "Finished";        break;
      case Aborted      : pStr = "Aborte";          break;
      default           : pStr = "Unknown";
   }

   return pStr;
}

