// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************
//  Module:         The info field area
// ****************************************************************************

#include <QtGui>

#include "toolconstants.h"

#include "common.h"
#include "main.h"
#include "infofield.h"

class t_InfoFieldLocal
{
   public:
      QLabel *pLabelParams;
      QLabel *pLabelValues;
};

t_InfoField::t_InfoField ()
{
   CHK_EXIT (ERROR_INFOFIELD_CONSTRUCTOR_NOT_SUPPORTED)
} //lint !e1401 not initialised

t_InfoField::t_InfoField (QWidget *pParent)
   :QFrame (pParent)
{
   CHK_EXIT (TOOL_ERROR_REGISTER_CODE (ERROR_INFOFIELD_CONSTRUCTOR_NOT_SUPPORTED))

   setFrameStyle ((int)QFrame::Panel | (int)QFrame::Sunken);
   pOwn = new t_InfoFieldLocal;

   QHBoxLayout *pLayout = new QHBoxLayout (this);
   pOwn->pLabelParams = new QLabel (this);
   pOwn->pLabelValues = new QLabel (this);
   pLayout->addWidget (pOwn->pLabelParams);
   pLayout->addWidget (pOwn->pLabelValues);
   pLayout->addStretch();

   pOwn->pLabelParams->setText (         tr("Size")
                                + "\n" + tr("Sector size")
                                + "\n" + tr("Image file")
                                + "\n" + tr("Info file")
                                + "\n" + tr("Current speed")
                                + "\n" + tr("Started", "Start timestamp and running time")
                                + "\n" + tr("Hash calculation")
                                + "\n" + tr("Source verification"));
}

t_InfoField::~t_InfoField ()
{
   delete pOwn->pLabelParams;
   delete pOwn->pLabelValues;
   delete pOwn;
}

void t_InfoField::SlotShowInfo (t_pDevice pDevice)
{
   QString StrValue, StrSpeed, Format;
   quint64 Size;

   if (pDevice)
   {
      // Size
      // ----
      Size = t_Device::GetSize (pDevice).toULongLong();
      StrValue = MainGetpNumberLocale()->toString(Size) + " " + tr("bytes");
      StrValue += " ("  + t_Device::GetSizeHumanFrac (pDevice, false).toString();
      StrValue += " / " + t_Device::GetSizeHumanFrac (pDevice, true ).toString() + ")";

      // Sector size
      // -----------
      quint64 SectorSize     = t_Device::GetSectorSize    (pDevice).toULongLong();
      quint64 SectorSizePhys = t_Device::GetSectorSizePhys(pDevice).toULongLong();
      StrValue += "\n" + MainGetpNumberLocale()->toString(SectorSize);
      if (SectorSize != SectorSizePhys)
         StrValue += " / " + MainGetpNumberLocale()->toString(SectorSizePhys);

      // Image/Info file
      // ---------------
      if (pDevice->Acquisition.Format == t_File::NotSet)
      {
         StrValue += "\n\n";
      }
      else
      {
         CHK_EXIT (t_File::GetFormatExtension   (pDevice->Acquisition.Format, NULL, &Format))
         StrValue += "\n" + pDevice->Acquisition.ImagePath + pDevice->Acquisition.ImageFilename + QSTR_TO_PSZ(Format);
         StrValue += "\n" + pDevice->Acquisition.InfoPath  + pDevice->Acquisition.InfoFilename  + t_File::pExtensionInfo;
      }

      // Speed
      // -----
      StrSpeed = t_Device::GetCurrentSpeed  (pDevice).toString();
      if (!StrSpeed.isEmpty())
         StrSpeed += " MB/s";
      StrValue += "\n" + StrSpeed;

      // Running time
      // ------------
      if (pDevice->Acquisition.Format == t_File::NotSet)
      {
         StrValue += "\n";
      }
      else
      {
         int Hours, Minutes, Seconds;

         StrValue += "\n" + pDevice->StartTimestamp.toString ("d. MMMM hh:mm:ss");

         if ((pDevice->State == t_Device::Acquire) ||  // Don't display anything if no acquisition is running
             (pDevice->State == t_Device::Verify ))
              Seconds = pDevice->StartTimestamp.secsTo (QDateTime::currentDateTime());
         else Seconds = pDevice->StartTimestamp.secsTo (pDevice->StopTimestamp);
         Hours    = Seconds / SECONDS_PER_HOUR  ; Seconds -= Hours   * SECONDS_PER_HOUR;
         Minutes  = Seconds / SECONDS_PER_MINUTE; Seconds -= Minutes * SECONDS_PER_MINUTE;

         StrValue += QString(" (%1:%2:%3)") .arg(Hours,2,10,QChar('0')) .arg(Minutes,2,10,QChar('0')) .arg(Seconds,2,10,QChar('0'));
      }

      // Hash
      // ----
      if (pDevice->Acquisition.CalcHashes)
           StrValue += "\n" + tr("on" , "Display that hash calculation is on");
      else StrValue += "\n" + tr("off", "Display that hash calculation is off");

      // Source verification
      // -------------------
      if (pDevice->Acquisition.VerifySource)
           StrValue += "\n" + tr("on" , "Display that source verification is on");
      else StrValue += "\n" + tr("off", "Display that source verification is off");
   }
   pOwn->pLabelValues->setText (StrValue);
}

