/*  GWhere.
 *  Copyright (C) 2000  Sbastien LECACHEUR
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "../gwsupport.h"
#include "gwfileinfo.h"


#include <stdlib.h>
#include <string.h> /* strlen strncmp */
#include <time.h>
#include "tools/gwtools.h"


#if defined ( OS_WIN32)
	#define S_IFLNK 0120000
	#define S_IFSOCK 0140000
	#define S_ISUID 0004000
	#define S_ISGID 0002000
	#define S_ISVTX 0001000
	#define S_IRGRP 0000040
	#define S_IWGRP 0000020
	#define S_IXGRP 0000010
	#define S_IROTH 0000004
	#define S_IWOTH 0000002
	#define S_IXOTH 0000001
#endif


typedef struct file_info
{
	gchar *name;			/* File name						*/
	mode_t rights;			/* File rights						*/
	gchar *owner;			/* File owner						*/
	gchar *group;			/* File owner group					*/
	guint64 size;			/* File size						*/
	glong inode;			/* File inode						*/
	gulong cdate;			/* File creation date					*/
	gulong adate;			/* File access date					*/
	gulong mdate;			/* File modification date				*/
	struct category *category;	/* File category					*/
	gchar *description;		/* File description (or comment)			*/
} FileInfo;


FileInfo * file_info_new ( void)
{
	FileInfo *p = NULL;


#ifdef GW_DEBUG_DATA_COMPONENT
	static gint i = 0;


	i++;

	g_print ( "*** GW - %s (%d) :: %s() : %d calls\n", __FILE__, __LINE__, __PRETTY_FUNCTION__, i);
#endif

	if ( (p = (FileInfo*)g_malloc0 ( sizeof ( FileInfo))) != NULL )
	{
		p->name = NULL;
		p->rights = 0;
		p->owner = NULL;
		p->group = NULL;
		p->size = 0;
		p->inode = 0;
		p->cdate = 0;
		p->adate = 0;
		p->mdate = 0;
		p->category = NULL;
		p->description = NULL;
	}

	return p;
}


gint file_info_set_name ( FileInfo *p, gchar *name)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->name != NULL )
		{
			g_free ( p->name);
		}

		p->name = name;

		return 0;
	}

	return -1;
}


gint file_info_set_rights ( FileInfo *p, mode_t rights)
{
	if ( p != NULL )
	{
		p->rights = rights;

		return 0;
	}

	return -1;
}


gint file_info_set_rights_from_gchar ( FileInfo *p, gchar *rights)
{
	mode_t mode = 000000;


	if ( (p != NULL) && (rights != NULL) && strlen(rights)==10 )
	{
		switch (rights[0])
		{
			case 'b':	mode|=S_IFBLK;
					break;
			case 'l':	mode|=S_IFLNK;
					break;
			case 'd':	mode|=S_IFDIR;
					break;
			case 'c':	mode|=S_IFCHR;
					break;
			case 'p':	mode|=S_IFIFO;
					break;
			case 's':	mode|=S_IFSOCK;
					break;
			default:	break;
		}

		if ( rights[1]=='r')
		{
			mode|=S_IRUSR;
		}

		if ( rights[2]=='w')
		{
			mode|=S_IWUSR;
		}

		switch (rights[3])
		{
			case 'x':	mode|=S_IXUSR;
					break;
			case 's':	mode|=S_ISUID;
					mode|=S_IXUSR;
					break;
			case 'S':	mode|=S_ISUID;
					break;
			default:	break;
		}

		if ( rights[4]=='r')
		{
			mode|=S_IRGRP;
		}

		if ( rights[5]=='w')
		{
			mode|=S_IWGRP;
		}

		switch (rights[6])
		{
			case 'x':	mode|=S_IXGRP;
					break;
			case 's':	mode|=S_ISGID;
					mode|=S_IXGRP;
					break;
			case 'S':	mode|=S_ISGID;
					break;
			default:	break;
		}

		if ( rights[7]=='r')
		{
			mode|=S_IROTH;
		}

		if ( rights[8]=='w')
		{
			mode|=S_IWOTH;
		}

		switch (rights[9])
		{
			case 'x':	mode|=S_IXOTH;
					break;
			case 't':	mode|=S_ISVTX;
					mode|=S_IXOTH;
					break;
			case 'T':	mode|=S_ISVTX;
					break;
			default:	break;
		}

		p->rights = mode;

		return 0;
	}

	return -1;
}


gint file_info_set_owner ( FileInfo *p, gchar *owner)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->owner != NULL )
		{
			g_free ( p->owner);
		}

		p->owner = owner;

		return 0;
	}

	return -1;
}


gint file_info_set_group ( FileInfo *p, gchar *group)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->group != NULL )
		{
			g_free ( p->group);
		}

		p->group = group;

		return 0;
	}

	return -1;
}


gint file_info_set_size ( FileInfo *p, guint64 size)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		p->size = size;

		return 0;
	}

	return -1;
}


gint file_info_set_inode ( FileInfo *p, glong inode)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		p->inode = inode;

		return 0;
	}

	return -1;
}


gint file_info_set_cdate ( FileInfo *p, gulong cdate)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		p->cdate = cdate;

		return 0;
	}

	return -1;
}


gint file_info_set_adate ( FileInfo *p, gulong adate)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		p->adate = adate;

		return 0;
	}

	return -1;
}


gint file_info_set_mdate ( FileInfo *p, gulong mdate)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		p->mdate = mdate;

		return 0;
	}

	return -1;
}


gint file_info_set_category ( struct file_info *p, struct category *category)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->category != NULL )
		{
			/* Must decrese category count */
		}

		p->category = category;

		return 0;
	}

	return -1;
}


gint file_info_set_description ( FileInfo *p, gchar *description)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->description != NULL )
		{
			g_free ( p->description);
		}

		p->description = description;

		return 0;
	}

	return -1;
}


gchar * file_info_get_name ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->name != NULL )
		{
			return p->name;
		}
		else
		{
			return "";
		}
	}

	return NULL;
}


gboolean file_info_get_isdirectory ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		return ( (S_IFMT & p->rights) == S_IFDIR );
	}

	/* Ugly coder!!*/
	return FALSE;
}


gboolean file_info_get_is_real_directory ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL)
	{
		if ( file_info_get_isdirectory ( p) )
		{
			if ( file_info_get_name ( p) == NULL)
			{
				return FALSE;
			}

			if ( strlen ( file_info_get_name ( p )) == 1 )
			{
				if ( strncmp ( file_info_get_name ( p), ".", 1) == 0 )
				{
					return FALSE;
				}
			}
			else { if ( strlen ( file_info_get_name ( p )) == 2 )
			{
				if ( strncmp ( file_info_get_name ( p), "..", 2) == 0 )
				{
					return FALSE;
				}
			}}

			return TRUE;
		}
	}

	return FALSE;
}


mode_t file_info_get_rights ( FileInfo *p)
{
	if ( p != NULL )
	{
		return p->rights;
	}

	return -1;
}


gchar* file_info_get_rights_to_gchar ( struct file_info *p)
{
	gchar* rigths = NULL;


	if ( p != NULL )
	{
		rigths = g_strdup ( "----------");
#if defined ( OS_LINUX) || defined ( OS_CYGWIN)
			if ( S_ISLNK ( p->rights) )
			{
				rigths[0] = 'l';
			}
			else
			{
#endif
				switch ( S_IFMT & p->rights )
				{
					case S_IFBLK:	rigths[0] = 'b';

							break;

					case S_IFLNK:	rigths[0] = 'l';

							break;

					case S_IFDIR:	rigths[0] = 'd';

							break;

					case S_IFCHR:	rigths[0] = 'c';

							break;

					case S_IFIFO:	rigths[0] = 'p';

							break;

					case S_IFSOCK:	rigths[0] = 's';

							break;
				}
#if defined ( OS_LINUX) || defined ( OS_CYGWIN)
			}
#endif

			/* Owner rights */
			if (S_IRUSR & p->rights) rigths[1] = 'r';
			if (S_IWUSR & p->rights) rigths[2] = 'w';
			if (S_ISUID & p->rights)
			{
				if (S_IXUSR & p->rights)
				{
					rigths[3] = 's';
				}
				else
				{
					rigths[3] = 'S';
				}
			}
			else
			{
				if (S_IXUSR & p->rights)
				{
					rigths[3] = 'x';
				}
			}

			/* Group rights */
			if (S_IRGRP & p->rights) rigths[4] = 'r';
			if (S_IWGRP & p->rights) rigths[5] = 'w';
			if (S_ISGID & p->rights)
			{
				if (S_IXGRP & p->rights)
				{
					rigths[6] = 's';
				}
				else
				{
					rigths[6] = 'S';
				}
			}
			else
			{
				if (S_IXGRP & p->rights)
				{
					rigths[6] = 'x';
				}
			}

			/* Others rights */
			if (S_IROTH & p->rights) rigths[7] = 'r';
			if (S_IWOTH & p->rights) rigths[8] = 'w';
			if (S_ISVTX & p->rights)
			{
				if (S_IXOTH & p->rights)
				{
					rigths[9] = 't';
				}
				else
				{
					rigths[9] = 'T';
				}
			}
			else
			{
				if (S_IXOTH & p->rights)
				{
					rigths[9] = 'x';
				}
			}
	}

	return rigths;
}


gchar * file_info_get_owner ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->owner != NULL )
		{
			return p->owner;
		}
		else
		{
			return "";
		}
	}

	return NULL;
}


gchar * file_info_get_group ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->group != NULL )
		{
			return p->group;
		}
		else
		{
			return "";
		}
	}

	return NULL;
}


guint64 file_info_get_size ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		return p->size;
	}

	return 0;
}


glong file_info_get_inode ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		return p->inode;
	}

	return -1;
}


gulong file_info_get_cdate ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		return p->cdate;
	}

	return 0;
}


gulong file_info_get_adate ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		return p->adate;
	}

	return 0;
}


gulong file_info_get_mdate ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		return p->mdate;
	}

	return 0;
}


struct category * file_info_get_category ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->category != NULL )
		{
			return p->category;
		}
	}

	return NULL;
}


gchar * file_info_get_category_name ( struct file_info *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( file_info_get_category ( p) == NULL )
		{
			return "";
		}

		return category_get_name ( file_info_get_category ( p));
	}

	return NULL;
}


gint file_info_get_category_index ( struct file_info *p)
{
	gint index = 0;


#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( file_info_get_category ( p) != NULL )
		{
			if  ( (index = category_get_index ( file_info_get_category ( p))) != -1 )
			{
				return index;
			}
		}

	}

	return -1;
}


gchar * file_info_get_description ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( p->description != NULL )
		{
			return p->description;
		}
/**/		else
		{
			return "";
		}
/**/	}

	return NULL;
}


gchar * file_info_get_cdate_to_gchar ( FileInfo *p)
{
	struct tm *time = NULL;


#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( (time = localtime ( &(p->cdate))) != NULL )
		{
			return gw_tm_to_str_format ( time);
		}
	}

	return NULL;
}


gchar * file_info_get_adate_to_gchar ( FileInfo *p)
{
	struct tm *time = NULL;


#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( (time = localtime ( &(p->adate))) != NULL )
		{
			return gw_tm_to_str_format ( time);
		}
	}

	return NULL;
}


gchar * file_info_get_mdate_to_gchar ( FileInfo *p)
{
	struct tm *time = NULL;


#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s()\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

	if ( p != NULL )
	{
		if ( (time = localtime ( &(p->mdate))) != NULL )
		{
			return gw_tm_to_str_format ( time);
		}
	}

	return NULL;
}


gint file_info_free ( FileInfo *p)
{
#ifdef GW_DEBUG_DATA_COMPONENT
	static gint i = 0;


	i++;

	g_print ( "*** GW - %s (%d) :: %s() : %s %d calls\n", __FILE__, __LINE__, __PRETTY_FUNCTION__, file_info_get_name(p), i);
#endif

	if ( p != NULL )
	{
		file_info_set_name ( p, NULL);
		file_info_set_owner ( p, NULL);
		file_info_set_group ( p, NULL);
		file_info_set_description ( p, NULL);

		g_free ( p);

#ifdef GW_DEBUG_DATA_COMPONENT
	g_print ( "*** GW - %s (%d) :: %s() : done\n", __FILE__, __LINE__, __PRETTY_FUNCTION__);
#endif

		return 0;
	}

	return -1;
}
