/* functions related to selecting time zones */

#include <stdio.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <libintl.h>

#include "gworldclock.h"
#include "zones.h"
#include "misc.h"

#define _(A) gettext(A)


void DeleteZone( GtkWidget *w, gpointer clocklist )
{
   extern gint changed;
   GString *title, *msg;
   gchar *button[]={"OK"};
   GtkTreeModel *clocklistModel;
   GtkTreeIter iter;

   if ( gtk_tree_selection_get_selected( 
	   gtk_tree_view_get_selection( clocklist ),
	   &clocklistModel,
	   &iter ) )
   {
      gtk_list_store_remove( GTK_LIST_STORE(clocklistModel), &iter );
      changed = 1;
      resizeWindow( gtk_widget_get_toplevel( GTK_WIDGET(clocklist) ), 
		    clocklist );
   }
   else 
   {
      title = g_string_new(_("Delete Zone"));
      msg = g_string_new(_("No zone chosen for deleting."));
      showMessageDialog( msg->str, GTK_MESSAGE_WARNING );
      g_string_free(msg,TRUE);
      g_string_free(title,TRUE);
   }
}


/*  Save list of time zones to configfile */
gint SaveZones(GtkWidget *w, gpointer clocklist)
{
  FILE *cf;
  extern GString *configfile;
  extern gint changed;
  gint N,i;
  gchar *description, *timezone;
  GtkTreeModel *clocklistModel;
  GtkTreeIter iter;
  gboolean gotIter = FALSE;

  if ( !(cf=fopen(configfile->str,"w")) ) {
    return 0;
  }

  clocklistModel = gtk_tree_view_get_model( clocklist );
  gotIter =  gtk_tree_model_get_iter_first(clocklistModel, &iter);

  while( gotIter )
  {
     gtk_tree_model_get( clocklistModel, &iter,
			 TZ_NAME, &timezone,
			 TZ_DESCRIPTION, &description,
			 -1);

    /* only write description if there is one! */
    if(strlen(description)==0)
      fprintf(cf,"%s\n", timezone );
    else
       fprintf(cf,"%s    \"%s\"\n", timezone, description );

    g_free(timezone);
    g_free(description);
    gotIter = gtk_tree_model_iter_next( clocklistModel, &iter );
  }

  changed=0;
  return ( ! fclose( cf ) );
}

/* Handle "rows_reordered" signal, indicating the rows in the clock have been
   moved */
/* Note this callback function does not get  called for some reason,
   I don't know why.  A bug in GTK+ ? 
   You will have to save the reordered list by hand for the time being.
*/
void registerReorderedRows( GtkTreeModel* clocklistModel,
			     GtkTreePath *arg1,
			     GtkTreeIter *arg2,
			     gpointer new_order,
			     gpointer user_data)
{
  extern gint changed;

  /* mark clocklist as "changed" */
  changed = 1;
}

gint CodeInList(gchar *code, GSList *List)
{
  /* can't use g_slist_find, unfortunately, since the data in the list is both the
     country code and (unknown) country name
  */

  GSList *item;

  item = List;
  while (item) {
    if ( ! strcmp(code,((NameCodeType *)item->data)->code) )
      return TRUE;  /* found match */
    item = item->next;
  }
  return FALSE;
}

GSList* AddNameCodeEntry(gchar *code, gchar *name, GSList *List)
{
  NameCodeType *entry;
 
  entry = g_malloc0(sizeof(NameCodeType));
  if(!entry)
    g_print(_("Could not create list: %s"),g_strerror(errno));
  entry->name = g_strdup(name);
  entry->code = g_strdup(code);
  List = g_slist_append(List, (gpointer)entry);

  /* we don't free entry here do we?  It's on record and is only to be freed when
     the item is released from the list */
  return List;
}

/* the documentation is not too clear about allocating and free lists */
/* does g_slist_free deallocate all the links in the list, or just the first? */
/* I will assume it does the entire list */
void  ClearNameCodeList(GSList **List) 
{
  GSList *item;

  if(*List) {
    item = *List;
    while (item) {
      g_free( ((NameCodeType *)item->data)->name );
      g_free( ((NameCodeType *)item->data)->code );
      item = item->next;
    }
    g_slist_free(*List);
  }
  *List=NULL;
}

/* for given continent, find corresponding countries as identified in ZONE_TABLE
   and prepare list of country name using COUNTRY_TABLE
*/

GSList* FetchCountries(gchar *continent)
{
  FILE *fpc, *fpz;
  GString *title, *msg;
  gchar *button[]={"OK"};
  gchar line[500];
  gchar *codec, *codez, *name;
  GSList *Countries;
  
  if (strlen(continent)==0 )
    return NULL;

  if ( !(fpz=fopen(ZONE_TABLE,"r")) ) {  
    title = g_string_new(_("Read Zone Table"));
    msg = g_string_new(NULL);
    g_string_sprintf(msg,_(" Error reading zone table \"%s\": \n %s \nHow very sad.\n"),
		     ZONE_TABLE , g_strerror(errno) );
    showMessageDialog( msg->str, GTK_MESSAGE_ERROR );
    g_string_free(msg,TRUE);
    g_string_free(title,TRUE);
  }
  if ( !(fpc=fopen(COUNTRY_TABLE,"r")) ) {  
    title = g_string_new(_("Read Zone Table"));
    msg = g_string_new(NULL);
    g_string_sprintf(msg,_(" Error reading country table \"%s\": \n %s \nHow very sad.\n"),
		     COUNTRY_TABLE , g_strerror(errno) );
    showMessageDialog( msg->str, GTK_MESSAGE_ERROR );
    g_string_free(msg,TRUE);
    g_string_free(title,TRUE);
  }

  Countries=NULL;
  while(fgets(line,500,fpz)) {
    if (line[0] != '#') {
      
      /* check for continent in TZ value (third item on the line in ZONE_TABLE)
	 Also read country code at beginning of zone table entry.
	 Strictly this is only 2 characters, but I will allow for a whole string
	 (in my opinion 3 character would be more meaningful.  The standard sux */
      sscanf(line,"%as %*s %as",&codez,&name);
      if(strstr(name,continent)) {
	if(!CodeInList(codez,Countries)) {
	  g_free(name);
	  rewind(fpc);
	  while(fgets(line,500,fpc)) {
	    if (line[0] != '#') {

	      /* first, identify country */
	      if(sscanf(line,"%as",&codec)==1) {
		if (!strcmp(codez,codec)) {

		  /* then extract name as entire string to \0 after tab */
		  /* (first make sure \n in line is reset to \0) */
		  name = (gchar *) strchr(line,'\n');
		  *name = '\0';
		  name = (gchar *) strchr(line,'\t');
		  name++;

		  Countries = AddNameCodeEntry(codec,name,Countries);
		}
		g_free(codec);
	      }	
	    }
	  }  
	}
      }
      else
	g_free(name);
      g_free(codez);
    }
  }
  fclose(fpc);
  fclose(fpz);

  return Countries;
}



/* from given country code ("*country"), find list of regions in ZONE_TABLE */
/* input: country is the two-letter country code from ISO3166 */
GSList* FetchRegions(gchar *country)
{
  FILE *fp;
  GString *title, *msg;
  gchar *button[]={"OK"};
  gchar line[500];
  gchar *code, *TZvalue, *region, *ptr;
  GSList *Regions;
  
  if (strlen(country)==0 )
    return NULL;

  if ( !(fp=fopen(ZONE_TABLE,"r")) ) {  
    title = g_string_new(_("Read Zone Table"));
    msg = g_string_new(NULL);
    g_string_sprintf(msg,_(" Error reading zone table \"%s\": \n %s \nHow very sad.\n"),
		     ZONE_TABLE , g_strerror(errno) );
    showMessageDialog( msg->str, GTK_MESSAGE_ERROR );
    g_string_free(msg,TRUE);
    g_string_free(title,TRUE);
  }

  Regions=NULL;
  while(fgets(line,500,fp)) {
    if (line[0] != '#') {
      
      /* check for entries corresponding to country code value 
	 (first item on the line in ZONE_TABLE)
	 Get name of region from third item on the line. */
      /* alternatively we may want to get the description from the optional
	 fourth item, where available.  Worry about that some other time */
      sscanf(line,"%as %*s %as",&code,&TZvalue);

      if(!strcmp(code,country)) {

	/* region name is in format:  continent/region
	   Extract the region part from the continent */
	ptr = (gchar *) strchr(TZvalue,'/');
	if(ptr)
	  region = g_strdup((gchar*)(ptr+1));
	else
	  region = g_strdup(TZvalue);

	/* Some regions have an underscore '_' in place of space */
	/* convert these to a real space */
	while( (ptr=(gchar*)strchr(region,'_')) ) 
	  *ptr = ' ';

	if(!CodeInList(TZvalue,Regions))
	  Regions = AddNameCodeEntry(TZvalue,region,Regions);
	g_free(region);
      }
      g_free(TZvalue);
      g_free(code);
    }
  }
  fclose(fp);

  return Regions;
}

void UpdateCountries(GtkWidget *ContinentCList,
		     gint row,
		     gint column,
		     GdkEventButton *event,
		     gpointer ZoneData)
{
  gchar *continent, *ptr;
  GSList *Countries, *item;
  gchar *rowdata[1];
  gint newRow;
  GtkCList *clist;

  gtk_clist_get_text( (GtkCList *) ContinentCList, row, 0, &ptr);

  /* check value of continent here  */
  continent = g_strdup(ptr);
  /* change "Americas" to "America" by wiping out the 's' */
  if(!strcmp(continent,"Americas"))
    continent[7]='\0';

  /* similarly remove " Ocean" in "Pacific Ocean", etc
     by making the ' ' a null character */
  ptr = (gchar *) strchr(continent,' ');
  if(ptr)
    *ptr='\0';

  clist = (GtkCList *) ((AddZoneStruct*)ZoneData)->countryCList;
  Countries=FetchCountries(continent);
  gtk_clist_freeze( clist );
  gtk_clist_clear( clist ); 
  item = Countries;
  while(item) {
    rowdata[0]=g_strdup( ((NameCodeType *)item->data)->name);
    newRow=gtk_clist_append(clist, rowdata);
    gtk_clist_set_row_data( clist, newRow, 
			    g_strdup(((NameCodeType *)item->data)->code));
    g_free(rowdata[0]);
    item = item->next;
  }
  ClearNameCodeList(&Countries);
  g_free(continent);
  gtk_clist_select_row( clist,0,0); 
  gtk_clist_thaw( clist );

}

/*  Why does the scroll box get larger when you keep selecting a country?? */
void UpdateRegions(GtkWidget *CountryCList,
		   gint row,
		   gint column,
		   GdkEventButton *event,
		   gpointer ZoneData)
{
  gchar *country;
  GSList *Regions, *item;
  gchar *rowdata[1];
  gint newRow;
  GtkCList *clist;

  country = (gchar *) gtk_clist_get_row_data( (GtkCList *) CountryCList, row);

  clist = (GtkCList *) ((AddZoneStruct*)ZoneData)->regionCList;
  Regions=FetchRegions(country);
  gtk_clist_freeze( clist );
  gtk_clist_clear( clist ); 
  item = Regions;
  while(item) {
    rowdata[0]=g_strdup( ((NameCodeType *)item->data)->name);
    newRow=gtk_clist_append(clist, rowdata);
    gtk_clist_set_row_data( clist, 
			    newRow, g_strdup(((NameCodeType *)item->data)->code));
    g_free(rowdata[0]);
    item = item->next;
  }
  ClearNameCodeList(&Regions);
  gtk_clist_thaw( clist );
}

void SelectRegion(GtkWidget *RegionCList,
		  gint row,
		  gint column,
		  GdkEventButton *event,
		  gpointer ZoneData)
{
  gchar *description, *TZ;

  gtk_clist_get_text( (GtkCList *) RegionCList, row, 0, &description);
  TZ = (gchar *) gtk_clist_get_row_data( (GtkCList *) RegionCList, row);
  
  gtk_entry_set_text( (GtkEntry *)((AddZoneStruct*)ZoneData)->DescriptionEntry, 
		      description);
  gtk_entry_set_text( (GtkEntry *)((AddZoneStruct*)ZoneData)->TZEntry, TZ);
}

/* when left mouse button is double-clicked,
   send "key-pressed-event" to one of the Entry boxes 
   which will be handled by adding the given zone.
   We're assuming here that "select-row" preceded the double-click event */
gint ButtonPressedInRegionList(GtkWidget *regionlist, 
			       GdkEventButton *event, gpointer ZoneData)
{
  GdkEventKey *KeyEvent;
  gint return_val;
  
  if( (event->type==GDK_2BUTTON_PRESS) && (event->button==1)) {
    KeyEvent = (GdkEventKey*)g_malloc(sizeof(GdkEventKey));
    KeyEvent->keyval=GDK_Return;
    gtk_signal_emit_by_name( GTK_OBJECT( ((AddZoneStruct*)ZoneData)->TZEntry ),
			     "key-press-event", (GdkEvent*)KeyEvent, &return_val);
    g_free(KeyEvent);
    return TRUE;
  }
  return FALSE;
}



/* zones are selected according to the method used in tzselect:
   First the continent is chosen, then, if necessary, the country is chosen,
   with countries being identified from the two-letter code in the
   entries of  [/usr/share/zoneinfo/]zone.tab (and country names taken from 
   iso3166.tab)  Then the region (or city) of that country is identified, from 
   zone.tab.
*/

void  PrepareZoneNotes(GtkWidget **ZoneNotes, AddZoneStruct *Zone)
{
  GtkWidget *label;
  GtkWidget *scrolled_window;
  GtkWidget *continentCList, *countryCList, *regionCList;
  gchar *ContinentLabel=_("Continents");
  gchar *CountryLabel=_("Countries");
  gchar *RegionLabel=_("Regions");
  gchar *rowdata[1];
  gint row, i;

  *ZoneNotes = gtk_notebook_new ();
  gtk_widget_show(*ZoneNotes);

  label = gtk_label_new (ContinentLabel);


  /* Create a scrolled window to pack the CList widget into */
  /* can't decide if I want it or not, for continents at least */
  /*  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
      gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
      gtk_widget_show (scrolled_window);
  */

  /* list of continents */
  continentCList = gtk_clist_new_with_titles( 1, NULL);
  gtk_clist_column_titles_hide( (GtkCList *)continentCList );
  for(i=0;i<Ncontinents;i++) {
    rowdata[0]=g_strdup(continents[i]);
    row=gtk_clist_append( (GtkCList *) continentCList,rowdata);
    g_free(rowdata[0]);
  }
  /*  gtk_container_add(GTK_CONTAINER(scrolled_window), continentCList);*/
  gtk_widget_show (continentCList);
  gtk_notebook_append_page (GTK_NOTEBOOK (*ZoneNotes), continentCList, label);


  /* list of countries */
  label = gtk_label_new (CountryLabel);
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  /* the vertical scroll policy should really be AUTOMATIC, but there is a 
     problem that if you first view a list of countries not needing the 
     scrollbar, then change to another list that does need it, the scrollbar 
     does not appear.  That is the bar appears fixed according to its first
     view, without updating.  Changing the policy to ALWAYS does not help,
     the "apparent size" of the list seems fixed.
     So, fix by upgrading to GtkTreeView.
     This problem applies also to cities and regions below.
  */ 
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  gtk_widget_show (scrolled_window);

  countryCList = gtk_clist_new_with_titles( 1, NULL);
  Zone->countryCList = countryCList;
  gtk_clist_column_titles_hide( (GtkCList *)continentCList );
  gtk_signal_connect(GTK_OBJECT( continentCList), "select_row",
		     GTK_SIGNAL_FUNC(UpdateCountries),
		     (gpointer)Zone);
  gtk_widget_show (countryCList);
  gtk_container_add(GTK_CONTAINER(scrolled_window), countryCList);
  gtk_notebook_append_page (GTK_NOTEBOOK (*ZoneNotes), scrolled_window, label);


  /* list of cities and regions */
  label = gtk_label_new (RegionLabel);
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  /* for some reason the horizontal scroll bars are turning themselves on
     when policy is AUTOMATIC.  They shouldn't really be needed, so
     turn them off.  (alternative is to manually set the window size).
     Likewise for region list below, also */
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  gtk_widget_show (scrolled_window);

  regionCList = gtk_clist_new_with_titles( 1, NULL);
  Zone->regionCList = regionCList;
  gtk_clist_column_titles_hide( (GtkCList *)regionCList );
  gtk_signal_connect(GTK_OBJECT( countryCList), "select_row",
		     GTK_SIGNAL_FUNC(UpdateRegions),
		     (gpointer)Zone);
  gtk_signal_connect(GTK_OBJECT(regionCList), "select_row",
		     GTK_SIGNAL_FUNC(SelectRegion),
		     (gpointer)Zone);
  gtk_signal_connect(GTK_OBJECT(regionCList),
		     "button_press_event",
		     GTK_SIGNAL_FUNC(ButtonPressedInRegionList),
		     (gpointer)Zone);
  gtk_widget_show (regionCList);
  gtk_container_add(GTK_CONTAINER(scrolled_window), regionCList);
  gtk_notebook_append_page (GTK_NOTEBOOK (*ZoneNotes), scrolled_window, label);

  /* select Sydney, Australia (continent #6) as default */
  /* note, gtk_clist_moveto doesn't seem to do anything ;/  */
  gtk_clist_select_row( (GtkCList *)continentCList,6,0);
  gtk_clist_select_row( (GtkCList *)countryCList,0,0);
  gtk_clist_select_row( (GtkCList *)regionCList,3,0); 
  /*  gtk_clist_moveto( (GtkCList *)continentCList, 6, 0, 0.5, 0.0 );*/
}


void AddZoneToList(GtkWidget *w, gpointer NewZone)
{
  gchar *rowdata[2];
  gint row;
  extern gint changed;
  GtkTreeView *clocklist;
  GtkListStore *clocklistModel;
  GString *description;
  GString *TZ;
  GtkTreeIter iter;

  description = g_string_new(
     gtk_entry_get_text((GtkEntry *)((AddZoneStruct *)NewZone)->DescriptionEntry));
  TZ = g_string_new(
     gtk_entry_get_text((GtkEntry *)((AddZoneStruct *)NewZone)->TZEntry));

  /* check a time zone was given, if not, set to GMT */
  /* GST-0 is the "formal" TZ value for GMT */
  if ( TZ->len == 0 )
     g_string_assign(TZ,"GST-0");
  
  clocklist =  (GtkTreeView *)((AddZoneStruct *)NewZone)->clocklist;
  clocklistModel = GTK_LIST_STORE(gtk_tree_view_get_model( clocklist ));
  gtk_list_store_append ( clocklistModel, &iter);
  gtk_list_store_set ( clocklistModel, &iter,
		       TZ_NAME, TZ->str,
		       TZ_DESCRIPTION, description->str,
		       -1);     

  /* make new zone visible */
  gtk_tree_view_scroll_to_cell (clocklist,    
				gtk_tree_model_get_path(GTK_TREE_MODEL(clocklistModel), &iter),
				NULL, 
				/* attn: non use_align not implemented yet in GTK+
				   FALSE, 0, 0); */
				TRUE, 0, 1);
  gtk_tree_selection_select_iter( 
     gtk_tree_view_get_selection( clocklist ),
     &iter );

  SetTime((gpointer)clocklist);
  changed=1;

  g_string_free(description,TRUE);
  g_string_free(TZ,TRUE);
}

void AddZone( GtkWidget *w, gpointer clocklist )
{
  GtkWidget *window; 
  GtkWidget *vbox, *hbox;
  GtkWidget *AddButton, *DoneButton;
  GtkWidget *ZoneNotes;
  GtkWidget *Frame;
  GtkWidget *EntryDescription, *EntryTZ;
  static AddZoneStruct NewZone;

  NewZone.clocklist = (GObject *) clocklist;

  EntryDescription = gtk_entry_new();
  NewZone.DescriptionEntry = (GtkWidget *)EntryDescription;
  EntryTZ = gtk_entry_new();
  NewZone.TZEntry = (GtkWidget *) EntryTZ;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title ((GtkWindow *)window, _("Add Time Zone"));
  gtk_window_position((GtkWindow *)window,GTK_WIN_POS_CENTER);
  gtk_container_set_border_width (GTK_CONTAINER (window), 5);
  
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), vbox);
  gtk_widget_show (vbox);


  /* display zone choices as notebook:
     Continents on one page, cities/countries on other
  */
  PrepareZoneNotes(&ZoneNotes,&NewZone);
  gtk_box_pack_start (GTK_BOX (vbox), ZoneNotes, TRUE, FALSE, 5);


  /* place zone in text entry box (allowing for manual entry if desired) */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 5);
  gtk_widget_show (hbox);


  /* box for text description of zone */
  Frame = gtk_frame_new (_("Description"));
  gtk_frame_set_shadow_type( (GtkFrame *)Frame, GTK_SHADOW_NONE); 
  gtk_box_pack_start (GTK_BOX (hbox), Frame, TRUE, FALSE, 5);
  gtk_widget_show (Frame);
  /*  gtk_signal_connect (GTK_OBJECT (EntryDescription), "changed",
      GTK_SIGNAL_FUNC (GetDescriptionString), (gpointer)&NewZone );*/
  gtk_container_add (GTK_CONTAINER (Frame), EntryDescription);
  /* where did the entry boxes get their default size from?
     Not setting the size here (and below for TZ) makes the dialog box too large 
     Arguably too large, anyway - this is a question of taste of course */
  gtk_widget_set_usize( GTK_WIDGET(EntryDescription),125,0);
  gtk_widget_show (EntryDescription);


  /* box for TZ value of zone */
  Frame = gtk_frame_new (_("TZ value"));
  gtk_frame_set_shadow_type( (GtkFrame *)Frame, GTK_SHADOW_NONE); 
  gtk_box_pack_start (GTK_BOX (hbox), Frame, TRUE, FALSE, 5);
  gtk_widget_show (Frame);
  /*  gtk_signal_connect (GTK_OBJECT (EntryTZ), "changed",
      GTK_SIGNAL_FUNC (GetTZstring), (gpointer)&NewZone );*/
  gtk_container_add (GTK_CONTAINER (Frame), EntryTZ);
  gtk_widget_set_usize( GTK_WIDGET(EntryTZ),125,0);
  gtk_widget_show (EntryTZ);

  /* buttons to accept zone, or exit */
  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, FALSE, 5);
  gtk_widget_show (hbox);

  AddButton = gtk_button_new_with_label (_("Add Zone"));
  gtk_signal_connect (GTK_OBJECT (AddButton), "clicked",
		      GTK_SIGNAL_FUNC (AddZoneToList), (gpointer)&NewZone );
  gtk_box_pack_start (GTK_BOX (hbox), AddButton, TRUE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (EntryDescription), "key-press-event",
		      GTK_SIGNAL_FUNC (GotOK), (gpointer)AddButton);
  gtk_signal_connect (GTK_OBJECT (EntryTZ), "key-press-event",
		      GTK_SIGNAL_FUNC (GotOK), (gpointer)AddButton);
  gtk_widget_show (AddButton);

  DoneButton = gtk_button_new_with_label (_("Done"));
  gtk_signal_connect (GTK_OBJECT (DoneButton), "clicked",
		      GTK_SIGNAL_FUNC (DestroyWindow), (gpointer)window);
  gtk_box_pack_start (GTK_BOX (hbox), DoneButton, TRUE, FALSE, 0);
  gtk_widget_show (DoneButton);

  gtk_widget_show(window); 
}

void WriteZoneDescription(GtkDialog *dialog, gint responseId, gpointer Zone)
{
  extern gint changed;
  GString *description;
  GtkTreeView *clocklist;
  GtkTreeModel *clocklistModel;
  GtkTreeIter iter;

  if ( responseId == OK_BUTTON )
  {
     clocklist = GTK_TREE_VIEW(((AddZoneStruct *)Zone)->clocklist);
     if ( gtk_tree_selection_get_selected( 
	     gtk_tree_view_get_selection(clocklist),
	     &clocklistModel,
	     &iter) )
     {
	
	description = g_string_new(
	   gtk_entry_get_text((GtkEntry *)((AddZoneStruct *)Zone)->DescriptionEntry));
	
	gtk_list_store_set ( GTK_LIST_STORE(clocklistModel), 
			     &iter,
			     TZ_DESCRIPTION, description->str,
			     -1);     
	changed=1;

/*	gtk_tree_model_row_changed( clocklistModel,
				   gtk_tree_model_get_path( clocklistModel, &iter),
				    &iter);*/
//	resizeWindow( gtk_widget_get_toplevel( GTK_WIDGET(clocklist) ), 
	//clocklist );
     }
  }

  if ( responseId > 0 )
  {
     gtk_widget_destroy(GTK_WIDGET(dialog));
  }
}
  
void ChangeZoneDescription(GtkWidget *w, gpointer clocklist)
{
  gchar *description;
  GtkWidget *window, *vbox, *hbox;
  GtkWidget *DescriptionEntry, *OKButton, *CancelButton;
  static AddZoneStruct Zone;
  GString *title, *msg;
  gchar *button[]={"OK"};
  GtkTreeModel *clocklistModel;
  GtkTreeIter selectedRowIter;

  if ( ! gtk_tree_selection_get_selected( 
	  gtk_tree_view_get_selection( clocklist ),
	  &clocklistModel,
	  &selectedRowIter ) )
  {
    /* is this dialog box useful? */
     showMessageDialog( _("No zone chosen for changing."), GTK_MESSAGE_WARNING );
     return;
  }
  
  gtk_tree_model_get( GTK_TREE_MODEL(clocklistModel), &selectedRowIter,
		      TZ_DESCRIPTION, &description,
		      -1);

  Zone.clocklist = (GObject *) clocklist;

  window = gtk_dialog_new_with_buttons ( _("Change Zone Description"),
					 NULL,
					 GTK_DIALOG_NO_SEPARATOR,
					 "OK", OK_BUTTON,
					 "Cancel", CANCEL_BUTTON,
					 NULL);

  gtk_window_position((GtkWindow *)window,GTK_WIN_POS_CENTER);
  gtk_container_set_border_width (GTK_CONTAINER (window), 5);

  DescriptionEntry = gtk_entry_new();
  Zone.DescriptionEntry = DescriptionEntry;
  gtk_entry_set_text( (GtkEntry *) DescriptionEntry, description);
  gtk_signal_connect (GTK_OBJECT (DescriptionEntry), "key-press-event",
		      GTK_SIGNAL_FUNC (GotOKInDialog), (gpointer)window);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(window)->vbox), DescriptionEntry);
  gtk_widget_show (DescriptionEntry);

  g_signal_connect (GTK_OBJECT (window), "response",
			    G_CALLBACK(WriteZoneDescription), 
			    (gpointer)&Zone );

  gtk_widget_show(window);
}

