/*
 * Copyright (C) 2001-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: main.c,v 1.149 2005/11/10 18:42:15 dsalt Exp $
 *
 * gtk2 ui for xine
 *
 * main
 */

#include "globals.h"

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <X11/Xlib.h>

#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#ifdef HAVE_GETOPT_LONG
#include <getopt.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>

#include "engine.h"
#include "player.h"
#include "gtkvideo.h"
#include "info_widgets.h"
#include "wizards.h"
#include "xml_widgets.h"
#include "noskin_window.h"
#include "server.h"
#include "open_mrl.h"
#include "log_window.h"
#include "preferences.h"
#include "key_events.h"
#include "mediamarks.h"
#include "playlist.h"
#include "settings.h"
#include "snapshot.h"
#include "play_item.h"
#include "lirc.h"
#include "stream_info.h"
#include "ui.h"
#include "utils.h"
#include "vis.h"
#include "post.h"

/*
 * globals
 */

int             verbosity;
GtkWidget      *app = NULL;
gchar          *plugindir, *bindir, *logodir, *pixmapdir, *icondir, *miscdir;
gchar          *video_driver_id=NULL;
gchar          *audio_driver_id=NULL;
GAsyncQueue    *js_queue;

static void
gxine_try_remote (int argc, char *argv[], gboolean enqueue, gboolean autoplay)
{
  if (!server_client_connect())
    return;

  /*
   * pass on files to play
   */

  if (!enqueue)
    server_client_send ("playlist_clear();\n");

  if (optind<argc) {
    int i;

    for (i=optind; i<argc; i++)
    {
      char *mrl = argv[i];

      if (mrl[0] != '/' && !strstr (mrl, ":/"))
      {
	/* make filename an absolute pathname */
	char  buf[1024];
	getcwd (buf, sizeof (buf));
	mrl = g_strconcat (buf, "/", mrl, NULL);
      }	

      if (autoplay && i == optind)
        server_client_send ("playlist_play (");
      server_client_send ("playlist_add (\"");
      server_client_send (mrl);
      server_client_send ((autoplay && i == optind) ? "\"));\n" : "\");\n");
    }
  }

  exit (0);
}

static gboolean splash_destroy (gpointer splash)
{
  gtk_widget_destroy (splash);
  return FALSE;
}

static void splash_show (void)
{
  gchar     *pathname;
  GtkWidget *img;
  GtkWidget *splash;

  gdk_threads_enter();  

  splash = gtk_window_new (GTK_WINDOW_POPUP);

  gtk_window_set_decorated (GTK_WINDOW (splash), FALSE);

  gtk_window_set_type_hint (GTK_WINDOW (splash),
			    GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);
  gtk_window_set_position (GTK_WINDOW (splash), GTK_WIN_POS_CENTER_ALWAYS);
  gtk_window_set_keep_above (GTK_WINDOW (splash), TRUE);

  pathname = g_strconcat (pixmapdir, "/splash.png", NULL);
  img = gtk_image_new_from_file (pathname);
  g_free (pathname);

  gtk_container_add (GTK_CONTAINER (splash),img);

  gtk_widget_show_all (splash);
  gtk_timeout_add (4000, splash_destroy, splash);

  do_pending_events ();
  gdk_threads_leave();  
}

#ifndef GLIB_USES_SYSTEM_MALLOC
#include <assert.h>
gpointer xmalloc (gsize size)
{
  gpointer p = malloc (size);
  if (size)
    assert (p != NULL);
  return p;
}

gpointer xrealloc (gpointer p, gsize size)
{
  p = realloc (p, size);
  if (size)
    assert (p != NULL);
  return p;
}
#endif

/* Initialise some engine settings from saved preferences */
static gboolean post_init_configure (void)
{
  xine_cfg_entry_t entry;
  GtkVideo *v = GTK_VIDEO(gtv);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace", &entry))
    gtk_video_set_post_plugins_deinterlace (v, entry.str_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.video", &entry))
    gtk_video_set_post_plugins_video (v, entry.str_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.audio", &entry))
    gtk_video_set_post_plugins_audio (v, entry.str_value, audio_port);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.deinterlace_enable",
				&entry))
    gtk_video_set_use_post_plugins_deinterlace (v, entry.num_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.video_enable", &entry))
    gtk_video_set_use_post_plugins_video (v, entry.num_value);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.audio_enable", &entry))
    gtk_video_set_use_post_plugins_audio (v, entry.num_value, audio_port);

  if (xine_config_lookup_entry (xine, "gui.post_plugins.audio_visualisation",
				&entry))
    vis_set (entry.enum_values[entry.num_value]);

  return FALSE;
}

static gboolean main_start_play (gpointer data)
{
  playlist_play (*(int *)data);
  return FALSE;
}

static gboolean set_fullscreen (gpointer data)
{
  engine_exec ("vo_fullscreen.v = true;", NULL, NULL, NULL);
  return FALSE;
}

int main(int argc, char* argv[])
{
  gboolean enqueue, autoplay, fullscreen;
  int optstate;
  gboolean show_splash = TRUE;

#ifndef GLIB_USES_SYSTEM_MALLOC
  GMemVTable vtab = { xmalloc, xrealloc, free, NULL, NULL, NULL };
  g_mem_set_vtable (&vtab);
#endif

  setlinebuf (stdout);
  setlinebuf (stderr);

  /*
   * init paths here. defaults are compiled in and may
   * be overwritten by environment variables
   */

  plugindir = getenv("GXINE_PLUGINDIR")	? : g_strconcat(GXINE_PLUGINDIR, NULL);
  bindir    = getenv("GXINE_BINDIR")	? : g_strconcat(GXINE_BINDIR, NULL);
  logodir   = getenv("GXINE_LOGODIR")	? : g_strconcat(GXINE_LOGODIR, NULL);
  pixmapdir = getenv("GXINE_PIXMAPDIR")	? : g_strconcat(GXINE_PIXMAPDIR, NULL);
  icondir   = getenv("GXINE_ICONDIR")	? : g_strconcat(GXINE_ICONDIR, NULL);
  miscdir   = getenv("GXINE_MISCDIR")	? : g_strconcat(GXINE_MISCDIR, NULL);

  /* set up our gtkrc stuff */
  {
    gchar **file = gtk_rc_get_default_files ();
    gchar **copy;
    const gchar *home = g_get_home_dir ();
    guint homelen = strlen (home);
    guint length = 0, i;
    while (file[length])
      ++length;
    /* deep-copy the gtkrc filenames, allocating space for two more */
    copy = calloc (length + 3, sizeof (gchar *));
    for (i = 0; file[i]; ++i)
      copy[i] = strdup (file[i]);
    /* find the index of the first *user* gtkrc filename */
    for (i = 0; file[i]; ++i)
      if (!strncmp (file[i], home, homelen) && file[i][homelen] == '/')
	break;
    /* insert our system gtkrc filename before it (strdup for convenience) */
    memmove (copy + i + 1, copy + i, (length - i) * sizeof (gchar *));
    copy[i] = strdup (GXINE_CONFDIR "/gtkrc");
    /* now append our user gtkrc, terminate the list and tell GTK+ about it */
    copy[length + 1] = g_strconcat (home, "/.gxine/gtkrc", NULL);
    copy[length + 2] = NULL;
    gtk_rc_set_default_files (copy);
    /* free the list */
    for (i = 0; copy[i]; ++i)
      free (copy[i]);
    free (copy);
  }

  /*
   * init glib/gdk/gtk thread safe/aware
   */

#ifdef ENABLE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  bindtextdomain (PACKAGE".theme", LOCALEDIR);
  /* The next two lines prevent GTK errors caused by gettext's conversion of
   * text from UTF-8.
   */
  bind_textdomain_codeset (PACKAGE, "UTF-8");
  bind_textdomain_codeset (PACKAGE".theme", "UTF-8");
  bind_textdomain_codeset (LIB_PACKAGE, "UTF-8");
  textdomain (PACKAGE);
#endif

  g_thread_init (NULL);
  gdk_threads_init ();
  gtk_init(&argc, &argv);
  gtk_window_set_auto_startup_notification (FALSE);

  /*
   * parse command line arguments
   */

  verbosity = 0;
  enqueue = FALSE;
  autoplay = TRUE;
  fullscreen = FALSE;
  optstate = 0;
  for (;;)
  {
#define OPTS "hvaefs:V:A:S"
#ifdef HAVE_GETOPT_LONG
    static struct option longopts[] = {
      { "help", no_argument, NULL, 'h' },
      { "verbose", no_argument, NULL, 'v' },
      { "add", no_argument, NULL, 'a' },
      { "enqueue", no_argument, NULL, 'e' },
      { "full-screen", no_argument, NULL, 'f' },
      { "video", required_argument, NULL, 'V' },
      { "audio", required_argument, NULL, 'A' },
      { "no-splash", no_argument, NULL, 'S' },
      { NULL }
    };
    int index = 0;
    int opt = getopt_long (argc, argv, OPTS, longopts, &index);
#else
    int opt = getopt(argc, argv, OPTS);
#endif
    if (opt == -1)
      break;

    switch (opt) {
    case 'h':
      optstate |= 1;
      break;
    case 'v':
      verbosity++;
      break;
    case 'a':
      autoplay = TRUE;
      enqueue = TRUE;
      break;
    case 'e':
      autoplay = FALSE;
      enqueue = TRUE;
      break;
    case 'f':
      fullscreen = TRUE;
      break;
    case 'V':
      video_driver_id = optarg;
      break;
    case 'A':
      audio_driver_id = optarg;
      break;
    case 'S':
      show_splash = FALSE;
      break;
    default:
      optstate |= 2;
      break;
    }
  }

  if (optstate & 1)
    printf (_("\
gxine %s (%s)\n\
usage: %s [options] [MRLs...]\n\
options:\n\
  -h, --help		this help text\n\
  -A, --audio DRIVER	try to use this audio driver\n\
  -V, --video DRIVER	try to use this video driver\n\
  -S, --no-splash	don't show the splash window\n\
  -a, --add		don't clear the playlist, play the first new item\n\
  -e, --enqueue		don't clear the playlist, don't play the new items\n\
  -f, --full-screen	start in full-screen mode\n\
  -v, --verbose		be more verbose\n\
\n"), VERSION, VENDOR_PKG_VERSION, argv[0]);

  if (optstate & 2)
  {
    puts (_("gxine: invalid option (try -h or --help)"));
    return 1;
  }

  if (optstate)
    return 0;

  /*
   * make sure ~/.gxine exists
   */
  {
    char *fname = g_strconcat (g_get_home_dir(), "/.gxine", NULL);
    int r = ensure_path_exists (fname, 0700);
    if (r)
      display_warning
	(FROM_GXINE, _("File creation error"),
	 _("Couldn't create ~/.gxine: %s.\n"
	   "Configuration, playlist and media marks will not be saved."),
	 strerror (r));
    g_free (fname);
  }

  /*
   * find out if gxine is already running, if so
   * just pass on the files to play
   */

  gxine_try_remote (argc, argv, enqueue, autoplay);

  server_setup ();

  /* start using X... */
  if (!XInitThreads ())
  {
    fprintf (stderr, _("gtkvideo: XInitThreads failed - looks like you don't have a thread-safe xlib.\n"));
    return 2;
  }

  /*
   * a splash screen for the impatient
   */

  if (show_splash)
    splash_show ();

  /*
   * init xine, set up skript engine, main window
   */

  gdk_threads_enter ();
  do_pending_events ();

  js_queue = g_async_queue_new ();
  g_async_queue_ref (js_queue);

  engine_init ();
  player_init ();
  ui_init	   ();

  gtk_window_set_default_icon_name (GXINE_LOGO);

  do_pending_events ();
  gtk_window_set_auto_startup_notification (TRUE);
  noskin_main_init ();

  do_pending_events ();

  /*
   * create all dialogs (invisible)
   */

  file_dialog_init ();
  utils_init       ();
  open_mrl_init    ();
  log_window_init  ();
  preferences_init ();
  playlist_init    ();
  settings_init    ();
  mediamarks_init  ();
  play_item_init   ();
  key_events_init  ();
  gxine_lirc_init  ();
  snapshot_init    ();
  vis_init	   ();
  post_init	   ();
  stream_info_init ();
  wizards_init     ();

  engine_startup_script ();

  /*
   * wizards (first run only)
   */

  run_wizards (FALSE);
  gdk_threads_leave ();

  post_init_configure ();
  initialised = TRUE;

  /*
   * argument parsing
   */

  if (optind<argc) {
    int i;
    static int first = -1;

    if (!enqueue)
      playlist_clear();

    for (i=optind; i<argc; i++) {

      if (first<0) {
	first = playlist_add_mrl (argv[i], -1);
      } else {
	playlist_add_mrl (argv[i], -1);
      }
    }

    if (first < 0)
      gtk_init_add ((GtkFunction) playlist_logo, NULL);
    else
      gtk_init_add (main_start_play,  &first);
  }
  else
    gtk_init_add ((GtkFunction) playlist_logo, NULL);

  if (fullscreen)
    gtk_init_add (set_fullscreen, NULL);

  gtk_init_add ((GtkFunction) ui_post_init, NULL);
  server_start ();

  //gdk_threads_enter();
  gtk_main();
  //gdk_threads_leave();

  return 0;
}
