/*
 * Copyright (C) 2002-2005 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: settings.c,v 1.29 2005/11/11 21:06:01 dsalt Exp $
 *
 * settings implementation
 */

#include "globals.h"

#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "settings.h"
#include "ui.h"
#include "utils.h"

struct slider_s {
  const char *label;
  control_adjustment_t type;
};

struct slider_group_s {
  const char *label;
  const struct slider_s *sliders;
};

struct slider_tab_s {
  const gchar		*title;
  const struct slider_s *sliders;
  GtkWidget		*tabbox;
};

static const struct slider_s settings_video[] = {
  { N_("A/V sync"),	Control_AV_SYNC },
  { N_("Hue"),		Control_HUE },
  { N_("Saturation"),	Control_SATURATION },
  { N_("Contrast"),	Control_CONTRAST },
  { N_("Brightness"),	Control_BRIGHTNESS },
  { NULL }
};

static const struct slider_s settings_audio[] = {
  { N_("Volume"),	Control_VOLUME },
  { N_("Compressor"),	Control_COMPRESSOR },
  { N_("Amplifier"),	Control_AMPLIFIER },
  { NULL }
};

static const struct slider_s settings_equaliser[] = {
  { "30Hz",	Control_EQ_30 },
  { "60Hz",	Control_EQ_60 },
  { "125Hz",	Control_EQ_125 },
  { "250Hz",	Control_EQ_250 },
  { "500Hz",	Control_EQ_500 },
  { "1kHz",	Control_EQ_1K },
  { "2kHz",	Control_EQ_2K },
  { "4kHz",	Control_EQ_4K },
  { "8kHz",	Control_EQ_8K },
  { "16kHz",	Control_EQ_16K },
  { NULL }
};

static const struct slider_group_s settings_page[] = {
  { N_("Video"),	settings_video },
  { N_("Audio"),	settings_audio },
  { N_("Equaliser"),	settings_equaliser },
  { NULL }
};
#define NUM_PAGES (sizeof (settings_page) / sizeof (settings_page[0]) - 1)

struct slider_window_s {
  const gchar			*title;
  const struct slider_group_s	*sliders;
  GtkWidget			*window, *tabbox;
  gboolean			 is_visible;
};

static struct slider_window_s settings_window = {
  N_("Settings"), settings_page, NULL, FALSE
};

static const struct slider_s *
get_sliders (const struct slider_window_s *window, guint page)
{
  if (page >= NUM_PAGES)
  {
    page = (guint) gtk_notebook_get_current_page (GTK_NOTEBOOK(window->tabbox));
    if (page >= NUM_PAGES)
      return NULL;
  }
  return window->sliders[page].sliders;
}

#define SLIDERS_SET(FUNC) \
  static gboolean sliders_##FUNC (const struct slider_window_s *window, \
				  guint page, gboolean allow_vol) \
  { \
    const struct slider_s *slider = get_sliders (window, page); \
    if (slider) \
      for (; slider->label; ++slider) \
      { \
        if (slider->type != Control_VOLUME || allow_vol) \
	  ui_##FUNC##_control_adjustment (slider->type); \
      } \
    else \
      for (page = 0; page < NUM_PAGES; ++page) \
	sliders_##FUNC (window, page, allow_vol); \
    return FALSE; \
  }

#define JSFUNC(OBJ,FUNC) \
  static JSBool js_##OBJ##_##FUNC (JSContext *cx, JSObject *obj, uintN argc, \
				   jsval *argv, jsval *rval) \
  { \
    se_t *se = (se_t *) JS_GetContextPrivate(cx); \
    int page = -1; \
    se_log_fncall_checkinit (#OBJ"_"#FUNC); \
    se_argc_check_max (1, #OBJ"_"#FUNC); \
    if (argc) \
    { \
      se_arg_is_int_or_bool (0, #OBJ"_"#FUNC); \
      JS_ValueToInt32 (cx, argv[0], &page); \
    } \
    sliders_##FUNC (&OBJ##_window, page, TRUE); \
    return JS_TRUE; \
  }

#define JSDECL(OBJ,FUNC) \
  { #OBJ"_"#FUNC, js_##OBJ##_##FUNC, 0, 0,SE_GROUP_DIALOGUE, NULL, NULL }

static void sliders_show (struct slider_window_s *window, guint page,
			  gboolean unused)
{
  guint current =
    (guint) gtk_notebook_get_current_page (GTK_NOTEBOOK(window->tabbox));
  window->is_visible = (page != -1 && page != current)
		       ? TRUE : !window->is_visible;
  if (window->is_visible)
  {
    window_show (window->window, NULL);
    if (page < NUM_PAGES)
      gtk_notebook_set_current_page (GTK_NOTEBOOK(window->tabbox), page);
  }
  else
    gtk_widget_hide (window->window);
}

SLIDERS_SET(revert)
SLIDERS_SET(reset)
SLIDERS_SET(clear)

static void response_cb (GtkDialog *dbox, gint response, gpointer data)
{
  struct slider_window_s *window = data;
  switch (response)
  {
  case GTK_RESPONSE_CANCEL:
    sliders_revert (window, -1, TRUE);
    break;
  case 1:
    sliders_reset (window, -1, TRUE);
    break;
  default:
    window->is_visible = FALSE;
    gtk_widget_hide (window->window);
  }
}

JSFUNC (settings, show)
JSFUNC (settings, revert)
JSFUNC (settings, reset)
JSFUNC (settings, clear)

struct slider_data_s {
  GtkLabel *label;
  gdouble value;
};

static gboolean slider_set_text (struct slider_data_s *data)
{
  char n[12];
  sprintf (n, "%.0f", data->value);
  gtk_label_set_text (data->label, n);
  free (data);
  return FALSE;
}

static gchar *slider_format_value (GtkScale *scale, gdouble value,
				   gpointer data)
{
  struct slider_data_s *slider = malloc (sizeof (struct slider_data_s));
  slider->label = data;
  slider->value = value;
  g_idle_add ((GSourceFunc) slider_set_text, slider);
  return g_strdup ("");
}

static void sliders_create (struct slider_window_s *window)
{
  int page, lock;
  xine_cfg_entry_t entry;

  /* window */

  window->window = gtk_dialog_new_with_buttons (gettext (window->title),
						NULL, 0, NULL);
  gtk_dialog_add_action_widget (GTK_DIALOG(window->window),
				ui_button_new_stock_mnemonic
				  (GTK_STOCK_CLEAR, _("_Default")),
				1);
  gtk_dialog_add_buttons (GTK_DIALOG(window->window),
			GTK_STOCK_UNDO, GTK_RESPONSE_CANCEL,
			GTK_STOCK_CLOSE, GTK_RESPONSE_DELETE_EVENT,
			NULL);
  gtk_window_set_default_size (GTK_WINDOW (window->window), 500, 150);
  hide_on_delete (window->window, &window->is_visible);
  g_signal_connect (G_OBJECT(window->window), "response",
		    G_CALLBACK(response_cb), window);

  window->tabbox = gtk_notebook_new ();

  /* sliders */

  for (page = 0; page < NUM_PAGES; ++page)
  {
    GtkWidget *table = gtk_table_new (3, 2, FALSE);
    const struct slider_s *sliders = window->sliders[page].sliders;
    int i;
    gdouble min = INT_MAX, max = INT_MIN;

    for (i = 0; sliders[i].label; ++i)
    {
      GtkAdjustment *adj;
      GtkWidget *w, *v;

      w = gtk_label_new (gettext (sliders[i].label));
      gtk_misc_set_alignment (GTK_MISC (w), 1, 0.5);
      gtk_table_attach (GTK_TABLE(table), w,
			0, 1, i, i+1, GTK_FILL, GTK_FILL, 2, 2);

      v = gtk_label_new ("");
      gtk_misc_set_alignment (GTK_MISC (v), 1, 0.5);
      gtk_table_attach (GTK_TABLE(table), v,
			1, 2, i, i+1, GTK_FILL, GTK_FILL, 2, 2);

      adj = (GtkAdjustment *) ui_register_control_adjustment (sliders[i].type);
      if (adj->lower < min)
	min = adj->lower;
      if (adj->upper > max)
	max = adj->upper;
      w = gtk_hscale_new (adj);
      gtk_scale_set_value_pos (GTK_SCALE(w), GTK_POS_LEFT);
      gtk_scale_set_digits (GTK_SCALE(w), 0);
      g_signal_connect (G_OBJECT(w), "format-value",
			G_CALLBACK(slider_format_value), v);
      gtk_table_attach (GTK_TABLE(table), w,
			2, 3, i, i+1, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
    }

    min = min ? floor (log10 (fabs (min))) + 1 + (min < 0) : 1;
    max = max ? floor (log10 (fabs (max))) + 1 + (max < 0) : 1;
    if (min > max)
      max = min;

    { /* create an 'invisible' label to set the width of the second column */
      static const char pad[] = "____________";
      GtkWidget *w =
	gtk_label_new (pad + sizeof (pad) - 1 -
		       ((int) max > sizeof (pad) ? sizeof (pad) : (int) max));
      gtk_widget_set_size_request (w, -1, 1);
      gtk_table_attach (GTK_TABLE(table), w,
			1, 2, i, i+1, GTK_FILL, GTK_FILL, 2, 0);
      /* add the table to the notebook */
      w = gtk_label_new (gettext (window->sliders[page].label));
      gtk_notebook_insert_page (GTK_NOTEBOOK(window->tabbox), table, w, -1);
    }
  }

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(window->window)->vbox),
		      window->tabbox, TRUE, TRUE, 2);

  lock = xine_config_lookup_entry (xine, "audio.volume.remember_volume", &entry)
	 && !entry.num_value;

  for (page = 0; page < NUM_PAGES; ++page)
    sliders_revert (window, page, lock);
}

void settings_init (void)
{
  static const se_f_def_t defs[] = {
    JSDECL (settings, show),
    JSDECL (settings, revert),
    JSDECL (settings, reset),
    JSDECL (settings, clear),
    { NULL }
  };

  sliders_create (&settings_window);

  se_defuns (gse, NULL, defs);
}
