/*
 * Copyright (C) 2011 Hermann Meyer, Andreas Degert
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * ---------------------------------------------------------------------------
 *
 *        file: gxtuner.cpp      guitar tuner for jack
 *
 * ----------------------------------------------------------------------------
 */

#include "./gxtuner.h"
#include "./config.h"

#define P_(s) (s)   // FIXME -> gettext

enum {
    PROP_FREQ = 1,
};

static gboolean gtk_tuner_expose (GtkWidget *widget, GdkEventExpose *event);
static void draw_background(cairo_surface_t *surface_tuner);
static void gx_tuner_class_init (GxTunerClass *klass);
static void gx_tuner_base_class_finalize(GxTunerClass *klass);
static void gx_tuner_init(GxTuner *tuner);
static void gx_tuner_set_property(
    GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void gx_tuner_get_property(
    GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static const int tuner_width = 100;
static const int tuner_height = 90;
static const double rect_width = 100;
static const double rect_height = 60;

static const double dashes[] = {
    0.0,                    /* ink  */
    rect_height,            /* skip */
    10.0,                    /* ink  */
    10.0                    /* skip */
};

static const double dash_ind[] = {
    0,                        /* ink  */
    5,                        /* skip */
    rect_height-20,            /* ink  */
    100.0                    /* skip */
};

GType gx_tuner_get_type(void)
{
    static GType tuner_type = 0;

    if (!tuner_type) {
        const GTypeInfo tuner_info = {
            sizeof (GxTunerClass),
            NULL,                /* base_class_init */
            (GBaseFinalizeFunc) gx_tuner_base_class_finalize,
            (GClassInitFunc) gx_tuner_class_init,
            NULL,                /* class_finalize */
            NULL,                /* class_data */
            sizeof (GxTuner),
            0,                    /* n_preallocs */
            (GInstanceInitFunc) gx_tuner_init,
            NULL,                /* value_table */
        };
        tuner_type = g_type_register_static(
            GTK_TYPE_DRAWING_AREA, "GxTuner", &tuner_info, (GTypeFlags)0);
    }
    return tuner_type;
}

static void gx_tuner_class_init(GxTunerClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GTK_WIDGET_CLASS(klass)->expose_event = gtk_tuner_expose;
    gobject_class->set_property = gx_tuner_set_property;
    gobject_class->get_property = gx_tuner_get_property;
    g_object_class_install_property(
        gobject_class, PROP_FREQ, g_param_spec_double (
            "freq", P_("Frequency"),
            P_("The frequency for which tuning is displayed"),
            0.0, 1000.0, 0.0, GParamFlags(GTK_PARAM_READWRITE)));
    klass->surface_tuner = cairo_image_surface_create(
        CAIRO_FORMAT_ARGB32, tuner_width*3., tuner_height*3.);
    g_assert(klass->surface_tuner != NULL);
    draw_background(klass->surface_tuner);
}

static void gx_tuner_base_class_finalize(GxTunerClass *klass)
{
    if (klass->surface_tuner) {
        g_object_unref(klass->surface_tuner);
    }
}

static void gx_tuner_init (GxTuner *tuner)
{
    GtkWidget *widget = GTK_WIDGET(tuner);
    widget->requisition.width = tuner_width;
    widget->requisition.height = tuner_height;
}

void gx_tuner_set_freq(GxTuner *tuner, double freq)
{
    g_assert(GX_IS_TUNER(tuner));
    tuner->freq = freq;
    gtk_widget_queue_draw(GTK_WIDGET(tuner));
    g_object_notify(G_OBJECT(tuner), "freq");
}

GtkWidget *gx_tuner_new(void)
{
    return (GtkWidget*)g_object_new(GX_TYPE_TUNER, NULL);
}

static void gx_tuner_set_property(GObject *object, guint prop_id,
                                      const GValue *value, GParamSpec *pspec)
{
    GxTuner *tuner = GX_TUNER(object);

    switch(prop_id) {
    case PROP_FREQ:
        gx_tuner_set_freq(tuner, g_value_get_double(value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void gx_tuner_get_property(GObject *object, guint prop_id,
                                      GValue *value, GParamSpec *pspec)
{
    GxTuner *tuner = GX_TUNER(object);

    switch(prop_id) {
    case PROP_FREQ:
        g_value_set_double(value, tuner->freq);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static gboolean gtk_tuner_expose (GtkWidget *widget, GdkEventExpose *event)
{
    static const char* note[12] = {"A ","A#","B ","C ","C#","D ","D#","E ","F ","F#","G ","G#"};
    static const char* octave[9] = {"0","1","2","3","4","4","6","7"," "};
    static int indicate_oc = 0;
    GxTuner *tuner = GX_TUNER(widget);
    cairo_t *cr;

    double x0      = (widget->allocation.width - 100) * 0.5;
    double y0      = (widget->allocation.height - 90) * 0.5;

    cr = gdk_cairo_create(widget->window);
    cairo_save(cr);
    static double grow_w = 1.;
    static double grow_h = 1.;
    static double grow   = 0.;

    if(widget->allocation.width > widget->allocation.height +(10.*grow*3)) {
        grow = (widget->allocation.height/90.)/10.;
    } else {
        grow =  (widget->allocation.width/100.)/10.;
    }
    
    grow_h = (widget->allocation.height/90.)/3.;
    grow_w =  (widget->allocation.width/100.)/3.;
    
    cairo_translate(cr, -x0*grow_w, -y0*grow_h);
    cairo_scale(cr, grow_w, grow_h);
    cairo_set_source_surface(cr, GX_TUNER_CLASS(GTK_OBJECT_GET_CLASS(widget))->surface_tuner, x0, y0);
    cairo_paint (cr);
    cairo_restore(cr);

    cairo_save(cr);
    cairo_translate(cr, -x0*grow_w*3., -y0*grow_h*3.);
    cairo_scale(cr, grow_w*3., grow_h*3.);
    
    float scale = -0.5;
    if (tuner->freq) {
        float freq_is = tuner->freq;
        float fvis = 12 * log2f(freq_is/440.0);
        int vis = int(round(fvis));
        scale = (fvis-vis) / 2;
        vis = vis % 12;
        if (vis < 0) {
            vis += 12;
        }
        if (fabsf(scale) < 0.1) {
            if (freq_is < 31.78 && freq_is >0.0) {
                indicate_oc = 0;
            } else if (freq_is < 63.57) {
                indicate_oc = 1;
            } else if (freq_is < 127.14) {
                indicate_oc = 2;
            } else if (freq_is < 254.28) {
                indicate_oc = 3;
            } else if (freq_is < 509.44) {
                indicate_oc = 4;
            } else if (freq_is < 1017.35) {
                indicate_oc = 5;
            } else if (freq_is < 2034.26) {
                indicate_oc = 6;
            } else if (freq_is < 4068.54) {
                indicate_oc = 7;
            } else {
                indicate_oc = 8;
            }
        }else {
            indicate_oc = 8;
        }

        // display note
        cairo_set_source_rgba(cr, fabsf(scale)*2, 1-(scale*scale*4), 0.2,1-fabsf(scale)*2);
        cairo_set_font_size(cr, 18.0);
        cairo_move_to(cr,x0+50 -9 , y0+30 +9 );
        cairo_show_text(cr, note[vis]);
        cairo_set_font_size(cr, 8.0);
        cairo_move_to(cr,x0+54  , y0+30 +16 );
        cairo_show_text(cr, octave[indicate_oc]);
    }

    // display frequency
    char s[10];
    snprintf(s, sizeof(s), "%.0f Hz", tuner->freq);
    cairo_set_source_rgba (cr, 0.8, 0.8, 0.2,0.6);
    cairo_set_font_size (cr, 8.0);
    cairo_text_extents_t ex;
    cairo_text_extents(cr, s, &ex);
    cairo_move_to (cr, x0+90-ex.width, y0+58);
    cairo_show_text(cr, s);

    double ux=2., uy=2.;
    cairo_device_to_user_distance (cr, &ux, &uy);
    if (ux < uy)
        ux = uy;
    cairo_set_line_width (cr, ux + grow);


    // indicator (line)
    cairo_move_to(cr,x0+50, y0+rect_height-5);
    cairo_set_line_join(cr, CAIRO_LINE_JOIN_ROUND);
    cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND);
    cairo_set_dash (cr, dash_ind, sizeof(dash_ind)/sizeof(dash_ind[0]), 0);
    cairo_line_to(cr, (scale*2*rect_width)+x0+47, y0+(scale*scale*30)+2);
    cairo_set_source_rgb(cr,  0.5, 0.1, 0.1);
    cairo_stroke(cr);
    
    cairo_destroy(cr);

    return FALSE;
}

/*
** paint tuner background picture (the non-changing parts)
*/
static void draw_background(cairo_surface_t *surface)
{
    cairo_t *cr;

    double x0      = 0;
    double y0      = 0;

    cr = cairo_create(surface);
    cairo_scale(cr, 3, 3);
    // bottom part of gauge
    cairo_rectangle (cr, x0,y0+60,rect_width+1,30);
    cairo_set_source_rgb (cr, 0, 0, 0);
    cairo_fill (cr);
    cairo_set_source_rgb(cr,  0.2, 0.2, 0.2);
    cairo_set_line_width(cr, 5.0);
    cairo_move_to(cr,x0+2, y0+63);
    cairo_line_to(cr, x0+99, y0+63);
    cairo_stroke(cr);

    // upper part
    cairo_rectangle (cr, x0-1,y0-1,rect_width+2,rect_height+2);
    cairo_set_source_rgb (cr, 0, 0, 0);
    cairo_fill (cr);

    cairo_pattern_t*pat =
        cairo_pattern_create_radial (-50, y0, 5,rect_width-10,  rect_height, 20.0);
    cairo_pattern_add_color_stop_rgb (pat, 0, 1, 1, 1);
    cairo_pattern_add_color_stop_rgb (pat, 0.3, 0.4, 0.4, 0.4);
    cairo_pattern_add_color_stop_rgb (pat, 0.6, 0.05, 0.05, 0.05);
    cairo_pattern_add_color_stop_rgb (pat, 1, 0.0, 0.0, 0.0);
    cairo_set_source (cr, pat);
    cairo_rectangle (cr, x0+2,y0+2,rect_width-3,rect_height-3);
    cairo_fill (cr);

    cairo_set_source_rgb(cr,  0.2, 0.2, 0.2);
    cairo_set_line_width(cr, 2.0);
    cairo_move_to(cr,x0+98, y0+3);
    cairo_line_to(cr, x0+98, y0+64);
    cairo_stroke(cr);

    cairo_set_source_rgb(cr,  0.1, 0.1, 0.1);
    cairo_set_line_width(cr, 2.0);
    cairo_move_to(cr,x0+3, y0+64);
    cairo_line_to(cr, x0+3, y0+3);
    cairo_line_to(cr, x0+98, y0+3);
    cairo_stroke(cr);

    pat = cairo_pattern_create_linear (x0+50, y0,x0, y0);
    cairo_pattern_set_extend(pat, CAIRO_EXTEND_REFLECT);
    cairo_pattern_add_color_stop_rgb (pat, 0, 0.1, 0.8, 0.1);
    cairo_pattern_add_color_stop_rgb (pat, 1, 1, 0.1, 0.1);
    cairo_set_source (cr, pat);

    // division scale
    cairo_set_dash (cr, dashes, sizeof (dashes)/sizeof(dashes[0]), 100.0);
    cairo_set_line_width(cr, 3.0);
    for (int i = -5; i < 6; i++) {
        cairo_move_to(cr,x0+50, y0+rect_height-5);
        cairo_line_to(cr, (((i*0.08))*rect_width)+x0+50, y0+(((i*0.1*i*0.1))*30)+2);
    }
    cairo_stroke(cr);

    cairo_set_source_rgb(cr,  0.1, 1, 0.1);
    cairo_move_to(cr,x0+50, y0+rect_height-5);
    cairo_line_to(cr, x0+50, y0+2);
    cairo_stroke(cr);

    // indicator shaft (circle)
    cairo_set_dash (cr, dash_ind, sizeof(dash_ind)/sizeof(dash_ind[0]), 0);
    //cairo_set_dash (cr, 0, 0, 0); // (for full circle)
    cairo_move_to(cr, x0+50, y0+rect_height-5);
    cairo_arc(cr, x0+50, y0+rect_height-5, 2.0, 0, 2*M_PI);
    cairo_set_source_rgb(cr,  0.5, 0.1, 0.1);
    cairo_set_line_width(cr, 2.0);
    cairo_stroke(cr);
    cairo_pattern_destroy (pat);

    cairo_destroy(cr);
}

namespace gx_engine {
// downsampling factor
const int DOWNSAMPLE = 16;
// Number of times that the FFT is zero-padded to increase frequency resolution.
const int ZERO_PADDING_FACTOR = 64;
// Value of the threshold above which the processing is activated.
const float    SIGNAL_THRESHOLD_ON = 0.001;
// Value of the threshold below which the input audio signal is deactivated.
const float SIGNAL_THRESHOLD_OFF = 0.0009;
// Time between frequency estimates (in seconds)
const float TRACKER_PERIOD = 0.1;

void *PitchTracker::static_run(void *p) {
    (reinterpret_cast<PitchTracker *>(p))->run();
    return NULL;
}

PitchTracker::PitchTracker()
    : error(false),
    busy(false),
    tick(0),
    m_pthr(0),
    m_buffer(new float[MAX_FFT_SIZE]),
    m_bufferIndex(0),
    m_audioLevel(false),
    m_fftwPlanFFT(0),
    m_fftwPlanIFFT(0) {
    const int fftw_buffer_size = MAX_FFT_SIZE * ZERO_PADDING_FACTOR;
    m_fftwBufferTime = reinterpret_cast<float*>
                       (fftwf_malloc(fftw_buffer_size * sizeof(float)));
    m_fftwBufferFreq = reinterpret_cast<fftwf_complex*>
                       (fftwf_malloc(fftw_buffer_size * sizeof(fftwf_complex)));

    memset(m_buffer, 0, MAX_FFT_SIZE * sizeof(float));
    memset(m_fftwBufferTime, 0, fftw_buffer_size * sizeof(float));
    memset(m_fftwBufferFreq, 0, fftw_buffer_size * sizeof(fftwf_complex));

    sem_init(&m_trig, 0, 0);

    if (!m_buffer || !m_fftwBufferTime || !m_fftwBufferFreq) {
        error = true;
    }
}


PitchTracker::~PitchTracker() {
    fftwf_destroy_plan(m_fftwPlanFFT);
    fftwf_destroy_plan(m_fftwPlanIFFT);
    fftwf_free(m_fftwBufferTime);
    fftwf_free(m_fftwBufferFreq);
    delete[] m_buffer;
}


bool PitchTracker::setParameters(int sampleRate, int fftSize) {
    assert(fftSize <= MAX_FFT_SIZE);

    if (error) {
        return false;
    }
    m_sampleRate = sampleRate / DOWNSAMPLE;
    resamp.setup(sampleRate, m_sampleRate, 1, 16); // 16 == least quality

    if (m_fftSize != fftSize) {
        m_fftSize = fftSize;
        fftwf_destroy_plan(m_fftwPlanFFT);
        fftwf_destroy_plan(m_fftwPlanIFFT);
        m_fftwPlanFFT = fftwf_plan_dft_r2c_1d(
            m_fftSize, m_fftwBufferTime, m_fftwBufferFreq, FFTW_ESTIMATE); // FFT
        m_fftwPlanIFFT = fftwf_plan_dft_c2r_1d(
            ZERO_PADDING_FACTOR * m_fftSize, m_fftwBufferFreq,
            m_fftwBufferTime, FFTW_ESTIMATE); // IFFT zero-padded
    }

    if (!m_fftwPlanFFT || !m_fftwPlanIFFT) {
        error = true;
        return false;
    }

    if (!m_pthr) {
        start_thread();
    }
    pt_initialized = true;
    return !error;
}

void PitchTracker::start_thread() {
    int                min, max;
    pthread_attr_t     attr;
    struct sched_param  spar;
    int priority, policy;
    pthread_getschedparam(jack_client_thread_id(gx_jack::client), &policy, &spar);
    priority = spar.sched_priority;
    min = sched_get_priority_min(policy);
    max = sched_get_priority_max(policy);
    priority -= 6; // zita-convoler uses 5 levels
    if (priority > max) priority = max;
    if (priority < min) priority = min;
    spar.sched_priority = priority;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setschedpolicy(&attr, policy);
    pthread_attr_setschedparam(&attr, &spar);
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
    pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
    // pthread_attr_setstacksize(&attr, 0x10000);
    if (pthread_create(&m_pthr, &attr, static_run, reinterpret_cast<void*>(this))) {
        error = true;
    }
    pthread_attr_destroy(&attr);
}

int PitchTracker::find_minimum() {
    const int peakwidth = 3;
    float *p = &m_fftwBufferTime[peakwidth];
    for ( ; p < &m_fftwBufferTime[ZERO_PADDING_FACTOR * m_fftSize / 2 + 1 - peakwidth]; p++) {
        int i;
        for (i = -peakwidth; i <= peakwidth; i++) {
            if (*p > p[i]) {
                break;
            }
        }
        if (i > peakwidth) {
            break;
        }
    }
    return static_cast<int>((p - m_fftwBufferTime));
}

int PitchTracker::find_maximum(int l) {
    float    maxAutocorr            = 0.0;
    int        maxAutocorrIndex    = 0;
    while (l < ZERO_PADDING_FACTOR * m_fftSize / 2 + 1) {
        if (m_fftwBufferTime[l] > maxAutocorr) {
            maxAutocorr = m_fftwBufferTime[l];
            maxAutocorrIndex = l;
        }
        ++l;
    }
    if (maxAutocorr == 0.0) {
        return -1;
    }
    return maxAutocorrIndex;
}

float show_level(int n, float *buf) {
    float sum = 0.0;
    for (int k = 0; k < n; ++k) {
        sum += fabs(buf[k]);
    }
    return sum;
}

void PitchTracker::add(int count, float* input) {
    if (error) {
        return;
    }
    resamp.inp_count = count;
    resamp.inp_data = input;
    for (;;) {
        resamp.out_data = &m_buffer[m_bufferIndex];
        int n = MAX_FFT_SIZE - m_bufferIndex;
        resamp.out_count = n;
        resamp.process();
        n -= resamp.out_count; // n := number of output samples
        if (!n) { // all soaked up by filter
            return;
        }
        m_bufferIndex = (m_bufferIndex + n) % MAX_FFT_SIZE;
        if (resamp.inp_count == 0) {
            break;
        }
    }
    if (++tick * count >= m_sampleRate * DOWNSAMPLE * TRACKER_PERIOD) {
        if (busy) {
            return;
        }
        tick = 0;
        copy();
        sem_post(&m_trig);
    }
}

void PitchTracker::copy() {
    int start = (MAX_FFT_SIZE + m_bufferIndex - m_fftSize) % MAX_FFT_SIZE;
    int end = (MAX_FFT_SIZE + m_bufferIndex) % MAX_FFT_SIZE;
    int cnt = 0;
    if (start >= end) {
        cnt = MAX_FFT_SIZE - start;
        memcpy(m_fftwBufferTime, &m_buffer[start], cnt * sizeof(float));
        start = 0;
    }
    memcpy(&m_fftwBufferTime[cnt], &m_buffer[start], (end - start) * sizeof(float));
}

void PitchTracker::run() {
    for (;;) {
        busy = false;
        sem_wait(&m_trig);
        busy = true;
        if (error) {
            continue;
        }
        float sum = 0.0;
        for (int k = 0; k < m_fftSize; ++k) {
            sum += fabs(m_fftwBufferTime[k]);
        }
        float threshold = (m_audioLevel ? SIGNAL_THRESHOLD_OFF : SIGNAL_THRESHOLD_ON);
        m_audioLevel = (sum / m_fftSize >= threshold);
        if ( m_audioLevel == false ) {
            setEstimatedFrequency(0.0);
            continue;
        }

        /* Compute the transform of the autocorrelation given in time domain by
         *           k=-N
         *    r[t] = sum( x[k] * x[t-k] )
         *            N
         * or in the frequency domain (for a real signal) by
         *    R[f] = X[f] * X[f]' = |X[f]|^2 = Re(X[f])^2 + Im(X[f])^2
         * When computing the FFT with fftwf_plan_dft_r2c_1d there are only N/2+1
         * significant samples, so |.|^2 is computed for m_fftSize/2+1 samples only
         */
        int fftRSize = m_fftSize/2 + 1;
        fftwf_execute(m_fftwPlanFFT);
        for (int k = 0; k < fftRSize; ++k) {
            fftwf_complex& v = m_fftwBufferFreq[k];
            v[0] = v[0]*v[0] + v[1]*v[1];
            v[1] = 0.0;
        }

        // pad the FFT with zeros to increase resolution in time domain after IFFT
        int size_with_padding = ZERO_PADDING_FACTOR * m_fftSize - fftRSize;
        memset(&m_fftwBufferFreq[fftRSize][0], 0, size_with_padding * sizeof(fftwf_complex));
        fftwf_execute(m_fftwPlanIFFT);

        // search for a minimum and then for the next maximum to get the estimated frequency
        int maxAutocorrIndex = find_maximum(find_minimum());

        // compute the frequency of the maximum considering the padding factor
        if (maxAutocorrIndex >= 0) {
            setEstimatedFrequency(ZERO_PADDING_FACTOR * m_sampleRate
                                  / static_cast<float>(maxAutocorrIndex));
        } else {
            setEstimatedFrequency(0.0);
        }
        busy = false;
    }
}

void PitchTracker::setEstimatedFrequency(float freq) {
    gx_gui::fConsta4 = freq;
}

PitchTracker pitch_tracker;
}

static void destroy( GtkWidget *widget, gpointer   data ) {
    jack_port_unregister(gx_jack::client, gx_jack::input_port);
    jack_deactivate(gx_jack::client);
    jack_client_close(gx_jack::client);
    if (gx_gui::g_threads > 0) {
        g_source_remove(gx_gui::g_threads);
    }
    gtk_main_quit ();
}

static gboolean delete_event(GtkWidget *widget, GdkEvent  *event,
                             gpointer   data ) {
    return FALSE;
}

namespace gx_jack {

static bool gx_jack_init() {
    client_name =   "gxtuner";
    input_port  =   0;

    client = jack_client_open(client_name.c_str(), JackNoStartServer, &jackstat);
    if (client) {
        jack_sr = jack_get_sample_rate(client); // jack sample rate
        jack_bs = jack_get_buffer_size(client); // jack buffer size
        jack_set_process_callback(client, gx_jack_process, 0);
        jack_on_shutdown (client, jack_shutdown, 0);
        input_port = jack_port_register(client, "in_0", JACK_DEFAULT_AUDIO_TYPE,
                     JackPortIsInput|JackPortIsTerminal, 0);
    } else {
        fprintf (stderr, "connection to jack failed, . . exit\n");
        exit(1);
    }

    if (jack_activate (client)) {
        fprintf (stderr, "cannot activate client\n");
        exit (1);
    }
    return true;
}

void jack_shutdown (void *arg) {exit (1);}

static int gx_jack_process(jack_nframes_t nframes, void *arg) {
    float *input = static_cast<float *>
                       (jack_port_get_buffer(input_port, nframes));
    gx_engine::pitch_tracker.add(nframes, input);
    return 0;
}
}

static gboolean gx_update_all_gui(gpointer arg) {
    gx_tuner_set_freq(GX_TUNER(gx_gui::tuner),gx_gui::fConsta4);
    return true;
}

int main(int argc, char *argv[]) {
    gx_gui::g_threads = 0;
    g_thread_init(NULL);
    gx_jack::gx_jack_init();
    gx_engine::pitch_tracker.init();
    gtk_init (&argc, &argv);
    GError* err = NULL;
    GtkWidget *window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_default_size(GTK_WINDOW(window), 120,100);
    gtk_window_set_icon_from_file(GTK_WINDOW(window),
        (std::string(PIXMAPS_DIR) + "/gxtuner.png").c_str(),&err);
    if (err != NULL) g_error_free(err);
    gx_gui::tuner = gx_tuner_new();
    gtk_container_add (GTK_CONTAINER (window), gx_gui::tuner);

    g_signal_connect (window, "delete-event",
            G_CALLBACK (delete_event), NULL);
    g_signal_connect (window, "destroy",
            G_CALLBACK (destroy), NULL);
    gx_tuner_set_freq(GX_TUNER(gx_gui::tuner),gx_gui::fConsta4);
    gtk_widget_show_all  (window);
    gx_gui::g_threads = g_timeout_add(100, gx_update_all_gui, 0);
    gtk_main ();

    return 0;
}
