#!/usr/bin/env python
# Copyright (c) 2012 Google Inc. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.
"""
Verifies that the user can override the compiler and linker using CC/CXX/LD
environment variables.
"""

import TestGyp
import os
import copy
import sys

here = os.path.dirname(os.path.abspath(__file__))

if sys.platform == 'win32':
  # cross compiling not support by ninja on windows
  # and make not supported on windows at all.
  sys.exit(0)

# Clear any existing compiler related env vars.
for key in 'CC', 'CXX', 'LD', 'CC_host', 'CXX_host', 'LD_host':
  if key in os.environ:
    del os.environ[key]


def CheckCompiler(test, gypfile, check_for, run_gyp):
  if run_gyp:
    test.run_gyp(gypfile)
  test.build(gypfile)

  # We can't test to presence of my_ld.py in the output since
  # ninja will use CXX_target as the linker regardless
  test.must_contain_all_lines(test.stdout(), check_for)


test = TestGyp.TestGyp(formats=['ninja', 'make'])

def TestTargetOveride():
  # Check that CC, CXX and LD set target compiler
  oldenv = os.environ.copy()
  try:
    os.environ['CC'] = 'python %s/my_cc.py FOO' % here
    os.environ['CXX'] = 'python %s/my_cxx.py FOO' % here
    os.environ['LD'] = 'python %s/my_ld.py FOO_LINK' % here

    CheckCompiler(test, 'compiler.gyp',
                  ['my_cc.py', 'my_cxx.py', 'FOO', 'FOO_LINK'],
                  True)
  finally:
    os.environ.clear()
    os.environ.update(oldenv)

  # Run the same tests once the eviron has been restored.  The
  # generated should have embedded all the settings in the
  # project files so the results should be the same.
  CheckCompiler(test, 'compiler.gyp',
                ['my_cc.py', 'my_cxx.py', 'FOO', 'FOO_LINK'],
                False)

def TestTargetOverideCompilerOnly():
  # Same test again but with that CC, CXX and not LD
  oldenv = os.environ.copy()
  try:
    os.environ['CC'] = 'python %s/my_cc.py FOO' % here
    os.environ['CXX'] = 'python %s/my_cxx.py FOO' % here

    CheckCompiler(test, 'compiler.gyp',
                  ['my_cc.py', 'my_cxx.py', 'FOO'],
                  True)
  finally:
    os.environ.clear()
    os.environ.update(oldenv)

  # Run the same tests once the eviron has been restored.  The
  # generated should have embedded all the settings in the
  # project files so the results should be the same.
  CheckCompiler(test, 'compiler.gyp',
                ['my_cc.py', 'my_cxx.py', 'FOO'],
                False)


def TestHostOveride():
  # Check that CC_host sets host compilee
  oldenv = os.environ.copy()
  try:
    os.environ['CC_host'] = 'python %s/my_cc.py HOST' % here
    os.environ['CXX_host'] = 'python %s/my_cxx.py HOST' % here
    os.environ['LD_host'] = 'python %s/my_ld.py HOST_LINK' % here
    CheckCompiler(test, 'compiler-host.gyp',
                  ['my_cc.py', 'my_cxx.py', 'HOST', 'HOST_LINK'],
                  True)
  finally:
    os.environ.clear()
    os.environ.update(oldenv)

  # Run the same tests once the eviron has been restored.  The
  # generated should have embedded all the settings in the
  # project files so the results should be the same.
  CheckCompiler(test, 'compiler-host.gyp',
                ['my_cc.py', 'my_cxx.py', 'HOST', 'HOST_LINK'],
                False)


TestTargetOveride()
TestTargetOverideCompilerOnly()
TestHostOveride()

test.pass_test()
