{-# OPTIONS -fglasgow-exts -fallow-undecidable-instances #-}
module HAppS.ACID 
{-# DEPRECATED "Use MACID instead" #-}
where
-- Copyright (C) 2004 HAppS.org. All Rights Reserved.
{--
  System for elimination of concurrency headaches, maximal application 
  throughput, and robustness with respect to power outages.

  Requirements
  * all transactive state can fit in memory (like Google!)
  * you don't need atomicity accross events (e.g. all events are http and smtp)
  * your events and state are serializable
  * for high performance use battery backed writeCaching disk subsystem

  SEE INSTRUCTIONS BELOW
--}

--import Data.FiniteMap
import qualified Data.Map as Map
import Control.Concurrent 
import Control.Exception(try)
import System.Directory
import System.Locale
import System.Random
import System.IO
import System

import Maybe
import Char
import List
import Monad
import Time 

import HAppS.Util.Concurrent
import HAppS.Util.Common

{--
  INSTRUCTIONS

  Results and SideEffects

  The appserver takes events in order and produces Results that may
  update state, put SideEffects on SideEffect queues, and respond synchornously
  or just treat the event as a query and only respond syncrhonously.

  The system guarantees that no SideEffect on any queue will be executed 
  before every prior SideEffect on that queue has COMPLETED.  The "" queue
  has no guaranteed order and everything is executed on it ASAP but also
  AT LEAST ONCE.

--}

data Result state queueId response --event 
    = Action {resultState::state
             ,sideEffects::[SideEffect queueId]  -- event
	     ,response::response}
    | Query {response::response} -- intermediary, no write-ahead log required

u_resultState f res = res {resultState = f $ resultState res}
u_response f res = res {response = f $ response res}

data SideEffect queueId -- event 
    = SideEffect {sideAction::IO ()}
    | QueueEffect {queueId::queueId,sideAction::IO ()}
    --  CallBack   {queueId::queueId,callback::IO()->event}
                  --sideQueueId::Maybe queueId,

castSideEffect (SideEffect x)    = SideEffect x
castSideEffect (QueueEffect q x) = QueueEffect q x

{--
  need to make sure callbacks can't be replayed in a different order
  therefore callbacks can only happen when sideeffects complete
  because there is a total order on sideeffect completion
  
  need to writeahead log the callback event w/ a time and a sideeffectId
  and only then mark the sideeffect as DONE.
  then we have a deterministic merge algorithm on the incoming events and 
  the sideeffect producing events
  we also need a on-recovery process that marks a sideeffect as done 
  if it has written to the log but not been marked done to protect
  against dying between the atomic logwrite and the atomic sideeffect done.
--}

{--
  
  EventHandlers

  The core of your app is the eventhandler.  EventHandlers produce Results.
  You need to define the type of your app state and the type of your events.
  Typical events are HTTP requests or SMTP envelopes.
  If the server is unplugged, the system replays events from the logfile.
  
--}

{--
type EventHandler state event response respChan =
	state           --current state of your application
 -> Callback state  --function to allow sideeffects to update/query state
 -> Chan respChan   --channel for sideeffects to transmit data to response
 -> EventContext event --everything you need to know to produce a Result
 -> (Result state response) --because of respChan, response is actual IO response
--}

{--
  !!!we need to modify callbacks so they only happen at the
  end of sideeffects.  so there is a total order on them.
  otherwise replaying will end up being incorrect.
  for now, simply disable callbacks?

--}

data EventContext event = 
    EventContext {eventTime::CalendarTime
		 ,eventId::EventId
		 ,eventRandom::Integer
		 ,subject::event} deriving (Read,Show)

type EventHandler state event queueId response respChan =
    AppContext state event respChan -> Result state queueId response --event 

data AppContext state event respChan = 
    AppContext {state::state
                --,callback::Callback state
	       ,respChan::Chan respChan
	       ,appTime::CalendarTime
	       ,seqNo::EventId
	       ,appRand::Integer
	       ,event::event
	       ,afterRespVar::SampleVar (IO ())
	       }
eSecs = epochSeconds . appTime
u_event f appCtx = appCtx {event = f $ event appCtx}
a_event event appCtx = appCtx {event=event}
a_state state appCtx = appCtx {state=state}
doAfterResp = writeSampleVar . afterRespVar 
result fromObj fromAppCtx f = u_response (>>=return.fromObj) . f . fromAppCtx

{--subHandler :: (inner -> outer -> outer, outer -> inner) 
           -> ievent
           -> EventHandler inner ievent c d e
           -> EventHandler outer oevent c d e
--}
subHandler (from,to) ev fun (AppContext st a b c d _ f) =
  case fun $ AppContext (to st) a b c d ev f of
    Query x         -> Query x
    Action st' ss x -> Action (from st' st) (map castSideEffect ss) x

subResHandler :: (ires -> ores)
           -> iev
           -> EventHandler st iev c ires e
           -> EventHandler st oev c ores e
subResHandler conv iev eh (AppContext st a b c d _ f) =
  case eh $ AppContext st a b c d iev f of
    Query x         -> Query $ conv x
    Action st ss x  -> Action st (map castSideEffect ss) $ conv x

subResConvHandler :: (inner -> outer -> outer, outer -> inner)
           -> (ires -> ores)
           -> iev
           -> EventHandler inner iev c ires e
           -> EventHandler outer oev c ores e
subResConvHandler (from,to) ir ev eh (AppContext st a b c d _ f) =
  case eh $ AppContext (to st) a b c d ev f of
    Query x         -> Query $ ir x
    Action st' ss x -> Action (from st' st) (map castSideEffect ss) $ ir x

{--callback--} --callback 
mkAppCtx state chan afterRespVar (EventContext appTime seqNo appRand appEvent) =
	AppContext state chan appTime seqNo appRand appEvent afterRespVar

type EventId=Integer -- total order on logged requests
type Callback state = (state -> (String,state)) -> IO (SampleVar String)

{--
  
  SimpleConfig

  When you start experimenting, use simpleConfig to create configs.
  If fast recovery is important to you and your app state takes a long
  time to read from disk, you may want to create your own.  E.g.
  Reading a big FiniteMap will take a while but there are lots of ways 
  to optimize reading FiniteMaps depending on how you wrote them.

  You may also want to adjust how frequently logfiles are rotated.
  The system defaults to hourly.

--}

simpleConfig 
  dir -- the directory the app should use for state,logs,and sideeffect tracking
  startState -- the initial state of you app before any event has arrrived
  eventHandler --the eventhandler for your app
    = AppConfig 
      {loadState= \fpath ->mbReadFile ("",0,0,startState) read (sdir fpath)
      ,saveState= \fp stateCtx -> atomicWriteFile (sdir fp) $ show stateCtx
      ,getLogName= \time->
       (formatCalendarTime defaultTimeLocale "%Y_%m_%d_%H.log" time)
       ,baseDir=dir
       ,handler=eventHandler
  }
    where sdir fpath = fpath ++ "/state"


data AppConfig state event queueId response respChan handler
    = AppConfig
      {loadState::FilePath -> IO (LogName,LogPos,EventId,state)
      ,saveState::FilePath -> (LogName,LogPos,EventId,state) -> IO ()
      ,getLogName::CalendarTime -> LogName
      ,baseDir::FilePath -- create if doesn't already exist
      ,handler::handler
      --EventHandler state event queueId response respChan 
      }
type LogName=String
type LogPos=Integer

{--

  StartApp

  Pass a config in here and get back a function that takes events and
  puts them on the appqueue.  Pass the returned function to your various
  socket listeners and when they get a complete request, envelope, etc, 
  have them  put them on the queue 

--}

startApp config =
    do 
    mkDirs config
    x@(logName,logPos,lastEventId,state) <- loadState config stateDir
    (pushEvContext, pullResultCtx) <- startEventProcessor state (handler config)
    runSides <-startSideRunner (baseDir config++"/sides") 
    toSaver <- startSaver config stateDir
    logInfo@(lastEventId',logName,logPos) <- 
	if logName=="" then return (0,"begin",0)
	else recover config pushEvContext logName logPos lastEventId 
    startLogger config logDir logInfo pullResultCtx runSides toSaver  
    queueReq <- startEventHandling lastEventId' pushEvContext
    return queueReq 
    where
    mkDirs config = 
	mapM (\dir -> createDirectory dir `catch` (\x->print x>> print dir))
		 [baseDir config,stateDir,sideDir,logDir,workDir]
    base dir = baseDir config ++ '/':dir
    sideDir=base "sides"
    logDir=base "logs"
    stateDir=base "state"
    workDir=base "work"

startEventHandling lastId pushEvContext = 
    do
    inChan <- newChan
    forkEverSt (doEvents (readChan inChan)) (lastId+1)
    return (writeChan inChan)
    where
    doEvents nextEvent eventId =
	do
	(cliHandle,event) <- nextEvent
        current <- getClockTime >>= return.toUTCTime
	rand <- randomIO
	pushEvContext (Just cliHandle,EventContext current eventId rand event)
	return $ eventId+1

recover config pushEv logName logPos lastEventId = 
    do
    (backLog,logName,logPos) <- getEntries config logName logPos []
    lastEventId <- pushEvents lastEventId backLog
    return (lastEventId,logName,logPos)
    where
    pushEvents lastEventId [] = return lastEventId
    pushEvents _ (evCtx:rest) = 
	pushEv (Nothing,evCtx) >> pushEvents (eventId evCtx) rest

getEntries config logName pos soFar = 
    do
    h <- openFile (getLogPath config logName) ReadMode
    hSetBinaryMode h True
    hSeek h AbsoluteSeek pos
    x@(s,entries,pos)<- hGetLogEntries h
    hClose h 
    either (\_ ->return (soFar++entries,logName,pos)) 
	       (getEntries' $ soFar++entries)
	       (readEither s)
    where
    getEntries' entries (LogPointer name) = getEntries config name 0 entries

getLogPath config logName = baseDir config ++ "/logs/" ++ logName

hGetLogEntries log =
    do
    pos <- hTell log
    line <- hGetLine log `catch` (\_->return "")
    (s,rest,pos') <- if null line then return ("",[],pos) 
                     else hGetLogEntries log
    return $ either (\s'->(s',[],pos)) (\entry->(s,entry:rest,pos')) 
               (readEither line)

newtype LogPointer = LogPointer LogName deriving (Read,Show)


startLogger config logDir logStat pullResultCtx runSides toSaver = 
    openLog >>= forkEverSt doLog
    where
    (lastEventId,startLogName,startLogPos) = logStat
    isNew evContext = eventId evContext > lastEventId
    openLog = do
              logPath <- return (getLogPath config startLogName)
	      h <- openFile logPath ReadWriteMode
	      hSetBinaryMode h True
	      hSeek h AbsoluteSeek startLogPos
	      curPos <- hTell h
	      return (logPath,h,curPos)
    doLog logState@(logPath,h,curPos) = 
	do
	resultCtx@(mbCliH,evCtx,result,_) <- pullResultCtx -- !!!!?
	logState@(logPath,h,curPos) <- doNewActions resultCtx
	runSides (evCtx,result)
        doClient resultCtx
	return logState
	where
	doNewActions resultCtx@(_,evCtx,result,_)=
	    if isNew evCtx && isAction result
	    then do
		 (logPath,h,pos) <- reOpenLog logState evCtx
		 pos <- writeLogEntry h pos evCtx
		 toSaver (pos,evCtx,result) 
		 return (logPath,h,pos)
	    else return logState
    closeLog h nextName = 
	hPutStrLn h (show $ LogPointer nextName) >> hFlush h >> hClose h
    reOpenLog oldLogState@(logPath,h,curPos) evCtx = 
	if evPath == logPath then return oldLogState
	else do closeLog h evName
		h<-openFile evPath AppendMode
		hSetBinaryMode h True		
		return (evPath,h,0)
	where
	evPath = getLogPath config (getLogName config $ eventTime evCtx)
	evName = (getLogName config $ eventTime evCtx)
    doClient resultCtx@(mbCliH,_,result,afterRespVar) =
	maybe (return ()) (fork_ . doResp) mbCliH 
	where 
	doResp c = do response result >>= c; 
                      after<-readSampleVar afterRespVar; after

writeLogEntry h lastPos evCtx =
    do
    entry <- return $ show evCtx
    hPutStrLn h entry
    hFlush h
    return $ lastPos + (fromIntegral $ length entry) + 1 
    -- hFileSize is another disk hit!?

startSaver config stateDir =
    do
    saveVar <- newEmptySampleVar
    forkEver $ saver config saveVar
    return $ writeSampleVar saveVar
    where
    saver config saveVar = 
        do threadDelay 1000 -- save every second
           (logPos,evContext,result) <- readSampleVar saveVar
	   logName <- return $ getLogName config $ eventTime evContext
	   saveState config stateDir (logName, logPos, 
                                      (eventId evContext),
                                      (resultState result))

startSideRunner sideDir =
    do
    sideChan <- newChan
    forkEverSt (doSides (readChan sideChan)) Map.empty
    return $ writeChan sideChan
    where
    doSides nextResult queues = 
	do
	(evCtx,result) <- nextResult
	sides <- return $ if isAction result 
		 then zip3 (repeat $ eventId evCtx) [0..] (sideEffects result)
		 else []
	doQueues sides queues 
    doQueues [] queues = return queues
    doQueues (sideCtx@(eventId,sideNo,side):sides) queues = 
        maybe newQueue addToQueue $ Map.lookup qName queues
	where
	qName = case side of
		SideEffect _ ->("priv3361_"++
				(show $ mod (eventId*eventId*eventId) 1000))
		QueueEffect queueId _ -> show queueId
	newQueue = 
            do queueHandle <- makeQueueHandler qName 
	       queueHandle sideCtx
               doQueues sides $ Map.insert  qName queueHandle queues
	addToQueue queueHandle = queueHandle sideCtx >> doQueues sides queues 
    makeQueueHandler qName =
	do
	lastSide <- mbReadFile (0,0) read (sideFileName qName)
	chan <- newChan
	fork $ doSide chan lastSide
	return (writeChan chan)
	where
	doSide sideChan lastSide =
	    do 
	    sideCtx@(eventId,sideNo,side) <- readChan sideChan
	    if (eventId,sideNo) <= lastSide 
               then doSide sideChan lastSide
	       else unGetChan sideChan sideCtx >> doSide' sideChan
	doSide' sideChan =
            do sideCtx@(eventId,sideNo,side)<-readChan sideChan
	       sideAction side
	       writeFile (sideFileName qName) $ show (eventId,sideNo)
    sideFileName qName = sideDir++'/':qName++".side"

startEventProcessor state handler =
    do
    inChan <- newChan
    outChan <- newChan
    forkEverSt (doEvent inChan (writeChan outChan)) state
    return (writeChanLeft inChan,readChan outChan)
    where
    pullResult chan=readChan chan
    doEvent inChan writeResult state = ignoreFail state (readChan inChan >>= 
  				       either doEvContext doCallBack)
	where
	doEvContext (mbClientHandle,evContext) =
	    do
	    chan <- newChan
	    afterRespVar <- newSampleVar (return ())
	    appCtx <- return $ mkAppCtx state chan afterRespVar evContext
	    result <- 
                let ?state=state; ?respChan=chan; ?appTime=eventTime evContext;
                    ?appSecs=epochSeconds $ eventTime evContext;
                                    ?eventId=eventId evContext;
				    ?evRand=eventRandom evContext;
				    ?event=subject evContext;
				    ?doAfterResp=writeSampleVar afterRespVar;
				    ?appCtx=appCtx
		in try (return $! handler appCtx)
		{--(writeCB inChan)--}
            case result of
              Left e  -> do putStrLn ("EVENT "++show (eventId evContext)
                                      ++" CAUSED AN EXCEPTION: "++show e)
                            return state
              Right r -> do writeResult (mbClientHandle,evContext,r,afterRespVar)
	                    return $ if isAction r then resultState r else state
	doCallBack (responseVar,f) =
	    do
	    (resp,newState)<-return $ f state
	    writeSampleVar responseVar resp
	    return newState
	writeCB chan f =
	    do
	    responseVar <- newEmptySampleVar
	    writeChanRight chan (responseVar,f)
	    return responseVar

isAction (Action _ _ _ )= True
--isAction (ActionA _ _ _ _)= True
isAction _ = False

{--
Goal: Take advantage of Haskell's laziness, concurrency and type system to create 
an app server for which it is truly easy to develop and that scales well on modern 
hardware (lots of memory, multiple write-cached harddrives, multiple cpus).  
Ideally, we want to allow the developer to ignore issues of 
persistence and concurrency, to focus almost exclusively what the app does, 
rather than how it does it!  (app specific declarative semantics).
--}

{--
  Instructions:
  1. Instantiate App YourState YourReqType YourRespType
  2. Pick state,logPath,and side directories - should have write caches!
  3. Write function that listens for requests passes to queue
  4. main = serveApp startState execRequest statePath logPath sideDir
--}

{--
ACID 
Assuming requests are Atomic!, serveApp takes care of making sure that state
is Consistent and Durable by:
* write ahead logging all requests before processing them
* forcing apps to be deterministic about state
* saving state and rotating logs hourly to maximize recovery speed
It achieves Isolation by relying on Haskell's referential transparensee.
IO is achieved via logged sideeffects that are guaranteed to execute at least once.
Sideeffects may have a partial order via multiple queues.
Sideeffects may modify application state via callbacks.
Responses may do IO and have a channel from sideeffects if desired.
Responses are NOT guaranteed to execute.
--}
